/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "pxmchem-oligomer.h"
#include "pxmchem-monomer.h"
#include "pxmchem-masscalc.h"



/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS ...
 */
PxmOligomer *
pxmchem_oligomer_new (void)
{
  PxmOligomer *oligomer = g_malloc0 (sizeof (PxmOligomer));
  
  oligomer->propGPA = g_ptr_array_new ();

  /* 'masspair' member is not allocated because an oligo may not need
     to have masses computed for it.
  */

  return oligomer;
}


PxmOligomer *
pxmchem_oligomer_dup (const PxmOligomer *oligomer, gint how_dup)
{
  PxmOligomer *oligomer_new = NULL;
  
  PxmOligomerDup how = (PxmOligomerDup) how_dup;
  

  g_assert (oligomer != NULL);
  
  /*
    Do not allocate prop array, we'll copy it later if 'how_dup'
    requires it.
  */
  oligomer_new = g_malloc0 (sizeof (PxmOligomer)); 
  
  if (oligomer->name != NULL)
    oligomer_new->name = g_strdup (oligomer->name);
  
  oligomer_new->polymer = oligomer->polymer;
  
  if (oligomer->name != NULL)
    oligomer_new->name = g_strdup (oligomer->name);

  oligomer_new->start_idx = oligomer->start_idx;
  oligomer_new->end_idx = oligomer->end_idx;

  /* 'masspair' member may not be allocated because an oligo may not need
     to have masses computed for it.
  */
  if (oligomer->masspair != NULL)
    oligomer_new->masspair = libpolyxmass_masspair_dup (oligomer->masspair);

  g_assert (oligomer->propGPA != NULL);
  
  if (how == PXM_OLIGOMER_DUP_DEEP_YES)
    {
      oligomer_new->propGPA = libpolyxmass_prop_GPA_dup (oligomer->propGPA, how);
    }
  else
    {
      /*
	All we do is allocate the array as if we had used 
	pxmchem_oligomer_new ().
      */
      oligomer_new->propGPA = g_ptr_array_new ();
    }
  
  return oligomer_new;
}


PxmProp *
pxmchem_oligomer_prop_dup (const PxmProp *prop, gint how_dup)
{
  PxmProp *prop_new = NULL;
  PxmOligomer *oligomer_new = NULL;
  
  
  g_assert (prop != NULL);
  

  prop_new = libpolyxmass_prop_new ();
  libpolyxmass_prop_set_name (prop_new, prop->name);

  oligomer_new = pxmchem_oligomer_dup ((PxmOligomer *) prop->data, how_dup);
  prop_new->data = (gpointer) oligomer_new;
  
  /*
    And now set the pointer to the prop's housekeeping functions:
  */
  g_assert (prop->custom_dup != NULL);
  prop_new->custom_dup = prop->custom_dup ;

  g_assert (prop->custom_cmp != NULL);
  prop_new->custom_cmp = prop->custom_cmp;

  g_assert (prop->custom_free != NULL);
  prop_new->custom_free = prop->custom_free;
  
  return prop_new;
}


/*
  This function does not set the name of the oligomer, the 
  caller is responsible for this.
*/
PxmOligomer *
pxmchem_oligomer_with_options_new (PxmPolymer *polymer,
				   gint start_idx,
				   gint end_idx,
				   PxmCalcOpt *calcopt,
				   PxmIonizerule *ionizerule,
				   PxmPlmChement plm_chement,
				   PxmPolchemdef *polchemdef)
{
  gint length = 0;

  PxmOligomer *oligomer = NULL;
    
  PxmProp *prop_new = NULL;
  PxmProp *prop_found = NULL;
  
  
  g_assert (polymer != NULL);
  length = polymer->monomerGPA->len;

  g_assert (ionizerule != NULL);
  
  /*
    If calcopt is NULL, that means that no mass calculation is
    performed. If calcopt is NULL, polchemdef may be NULL also, since
    it is not required.
  */
  oligomer = pxmchem_oligomer_new ();

  oligomer->polymer = polymer;
  oligomer->start_idx = start_idx;
  oligomer->end_idx = end_idx;
  
  /*
    ONLY INFORMATIONAL STUFF, NOT FOR MASS CALCULATIONS...
    ------------------------------------------------------
     
    Here we want to set properties to the oligomer only if the following
    data are required to be checked:

    1. if the oligomer should know if it corresponds to the left
    end of the parent polymer.

    2.  if the oligomer should know if it corresponds to the right
    end of the parent polymer.
  */
  if (0 != (plm_chement & PXMCHEMENT_PLM_LEFT_MODIF) ||
      0 != (plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF))
    {
      if (oligomer->start_idx == 0 && oligomer->end_idx == length - 1)
	{
	  /*
	    The oligomer is both the left end-terminal and right
	    end-terminal oligomer.
	  */
	  prop_new = 
	    libpolyxmass_prop_both_strings_new ("LEFT_RIGHT_END_OLIGOMER",
					    "BOTH_ENDS");
	  
	  g_ptr_array_add (oligomer->propGPA, prop_new);
	  
	  /*
	    Now that we know that the current oligomer corresponds in
	    fact to the whole polymer (both left end-terminal and
	    right end-terminal oligomer of one polymer), we must check
	    if these two ends are modified or not.
	  */
	  prop_found = libpolyxmass_prop_find_prop (polymer->propGPA,
						NULL,
						NULL,
						"LEFT_END_MODIF",
						NULL,
						PXM_UNDEF_PROP_CMP);
	  if (NULL != prop_found)
	    {
	      if (prop_found->data != NULL)
		{
		  prop_new = libpolyxmass_prop_dup (prop_found,
						PXM_MODIF_PROP_DUP_DEEP_YES);
		  
		  g_ptr_array_add (oligomer->propGPA, prop_new);
		}
	    }
	  
	  prop_found = libpolyxmass_prop_find_prop (polymer->propGPA,
						NULL,
						NULL,
						"RIGHT_END_MODIF",
						NULL,
						PXM_UNDEF_PROP_CMP);
	  if (NULL != prop_found)
	    {
	      if (prop_found->data != NULL)
		{
		  prop_new = libpolyxmass_prop_dup (prop_found,
						PXM_MODIF_PROP_DUP_DEEP_YES);
		  
		  g_ptr_array_add (oligomer->propGPA, prop_new);
		}
	    }
	}
      /* end of 
	 if (oligomer->start_idx == 0 && oligomer->end_idx == length - 1)
      */
    }
  else if (0 != (plm_chement & PXMCHEMENT_PLM_LEFT_MODIF) )
    {
      if (oligomer->start_idx == 0)
	{
	  prop_new = 
	    libpolyxmass_prop_both_strings_new ("LEFT_RIGHT_END_OLIGOMER",
					    "LEFT_END");
	  
	  g_ptr_array_add (oligomer->propGPA, prop_new);
	  
	  /*
	    Now that we know that the current oligomer corresponds in
	    fact to the left end-terminal oligomer of the polymer),
	    we must check if these two ends are modified or not.
	  */
	  prop_found = libpolyxmass_prop_find_prop (polymer->propGPA,
						NULL,
						NULL,
						"LEFT_END_MODIF",
						NULL,
						PXM_UNDEF_PROP_CMP);
	  
	  if (NULL != prop_found)
	    {
	      if (prop_found->data != NULL)
		{
		  prop_new = libpolyxmass_prop_dup (prop_found,
						PXM_MODIF_PROP_DUP_DEEP_YES);
		  
		  g_ptr_array_add (oligomer->propGPA, prop_new);
		}
	    }
	}
      /* end of 
	 else if (oligomer->start_idx == 0)
      */
    }
  else if (0 != (plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF))
    {
      if (oligomer->end_idx == length - 1)
	{
	  prop_new = 
	    libpolyxmass_prop_both_strings_new ("LEFT_RIGHT_END_OLIGOMER",
					    "RIGHT_END");
	  
	  g_ptr_array_add (oligomer->propGPA, prop_new);
	  
	  /*
	    Now that we know that the current oligomer corresponds in
	    fact to the left end-terminal oligomer of the polymer),
	    we must check if these two ends are modified or not.
	  */
	  prop_found = libpolyxmass_prop_find_prop (polymer->propGPA,
						NULL,
						NULL,
						"RIGHT_END_MODIF",
						NULL,
						PXM_UNDEF_PROP_CMP);
	  
	  if (NULL != prop_found)
	    {
	      if (prop_found->data != NULL)
		{
		  prop_new = libpolyxmass_prop_dup (prop_found,
						PXM_MODIF_PROP_DUP_DEEP_YES);
		  
		  g_ptr_array_add (oligomer->propGPA, prop_new);
		}
	    }
	}
      /* end of 
	 else if (oligomer->end_idx == 0)
      */
    }

  /* MASS CALCULATION PROPER STUFF.
   */

  /*
    Now check if there are mass calculation options available,
    so that we may calculate the mass of the oligomer just created.
  */
  if (calcopt != NULL)
    {
      /* polchemdef param is needed for the mass calculations to be
	 performed.
      */
      g_assert (polchemdef != NULL);
      
      /*
	Create a masspair instance so that masses are going to be
	calculated in it. Since we are CREATING an oligomer, the member
	'masspair' is certainly NULL.
      */
      g_assert (oligomer->masspair == NULL);
      oligomer->masspair = libpolyxmass_masspair_new ();
      
      /*
	Here we need to check something serious. I decided that in
	order to make code reuse optimal in the polyxmass project, a
	number of functions are written to take parameters that means
	different things depending on the context. The function that
	we'll call below is one of these functions: depending on the
	context the parameters passed to it are calculated in
	different manners so that the function does the expected
	work. For example, when a selection is made in the sequence
	editor, and that we need to calculate the masses for the
	selected oligomer, the way the numbering of the monomers is
	done in the sequence editor is different than the way the
	numbering is done for defining the borders of an oligomer
	obtained through the cleavage of a polymer. In the former
	case, the end_idx is incremented by one respect to the latter
	case, which is why in the function
	masscalc_noncapped_monomer_GPA () we have the following
	control loop : for (iter = calcopt->start_idx; iter <
	calcopt->end_idx; iter++). Thus, since here the end_idx is
	one value lower than the one reported upon selection of an
	oligomer in the sequence editor window, we have to increment
	it by one prior to passing its value to the calculation
	function :
      */
      calcopt->start_idx = oligomer->start_idx;
      calcopt->end_idx = oligomer->end_idx + 1;
      
      /*
	Note that the ionization of the oligomer is performed in the
	function call below.
      */
      if (FALSE == pxmchem_masscalc_oligomer (oligomer,
					      polchemdef,
					      calcopt,
					      NULL,
					      ionizerule,
					      oligomer->masspair))
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: failed computing the masses for oligomer '%s'\n"),
		 __FILE__, __LINE__, oligomer->name);	
	}
    }
  /* end of
     if (calcopt != NULL)
  */

  return oligomer;
}


gboolean
pxmchem_oligomer_set_polymer (PxmOligomer *oligomer, PxmPolymer *polymer)
{
  g_assert (oligomer != NULL && polymer != NULL);
  
  oligomer->polymer = polymer;
  
  return TRUE;
  
}

gboolean
pxmchem_oligomer_set_name (PxmOligomer *oligomer, gchar *name)
{
  g_assert (oligomer != NULL && name != NULL);
  

  /*
    The member data may be NULL, as this function can be called
    right after pxmchem_oligomer_new () which leaves the members
    NULL (except the propGPA which is allocated).
  */
  if (oligomer->name != NULL)
    g_free (oligomer->name);
  
  oligomer->name = g_strdup (name);
  
  return TRUE;
  
}

gboolean
pxmchem_oligomer_set_mono (PxmOligomer *oligomer, gdouble mass)

{
  g_assert (oligomer != NULL);
  
  if (oligomer->masspair == NULL)
    oligomer->masspair = libpolyxmass_masspair_new ();
  
  oligomer->masspair->mono = mass;

  return TRUE;
}


gboolean
pxmchem_oligomer_set_avg (PxmOligomer *oligomer, gdouble mass)

{
  g_assert (oligomer != NULL);
  
  if (oligomer->masspair == NULL)
    oligomer->masspair = libpolyxmass_masspair_new ();

  oligomer->masspair->avg = mass;

  return TRUE;
}

/* COMPARISON FUNCTIONS
 */

gint
pxmchem_oligomer_cmp (PxmOligomer *oligomer1, PxmOligomer *oligomer2, 
		      gint how_cmp)
{
  gint result = 0;
  gint iter = 0;
  gint jter = 0;
  
  PxmOligomerCmp how = (PxmOligomerCmp) how_cmp;
  
  PxmProp *prop1 = NULL;
  PxmProp *prop2 = NULL;
  
  gboolean found_prop = FALSE;

  g_assert (oligomer1 != NULL);
  g_assert (oligomer2 != NULL);

  g_assert (how & PXM_OLIGOMER_CMP_NONE
	    || how & PXM_OLIGOMER_CMP_POLYMER
	    || how & PXM_OLIGOMER_CMP_NAME
	    || how & PXM_OLIGOMER_CMP_COORDS
	    || how & PXM_OLIGOMER_CMP_MASSPAIR
	    || how & PXM_OLIGOMER_CMP_PROP_ARRAY_SUBSET1
	    || how & PXM_OLIGOMER_CMP_PROP_ARRAY_SUBSET2);

  if (how & PXM_OLIGOMER_CMP_NONE)
    {
      return -1;
    }

  if (how & PXM_OLIGOMER_CMP_POLYMER)
    {
      /*
	If both pointers are identical, the returned value is 0.
      */
      result += (oligomer1->polymer != oligomer2->polymer);
    }
  
  if (how & PXM_OLIGOMER_CMP_NAME)
    {
      result += strcmp (oligomer1->name, oligomer2->name);
    }
  
  if (how & PXM_OLIGOMER_CMP_COORDS)
    {
      result += (oligomer1->start_idx != oligomer2->start_idx);
      result += (oligomer1->end_idx != oligomer2->end_idx);
    }
  
  if (how & PXM_OLIGOMER_CMP_MASSPAIR)
    {
      result += (oligomer1->masspair->mono !=
		 oligomer2->masspair->mono);
      
      result += (oligomer1->masspair->avg !=
		 oligomer2->masspair->avg);
    }
  
  if (how & PXM_OLIGOMER_CMP_PROP_ARRAY_SUBSET1)
    {
      /*
	It is admitted that the two oligomers are not absolutely
	identical, only if all the prop objects in oligomer1->propGPA
	are in oligomer2->propGPA (ie oligomer1 is a subset of
	oligomer2). Thus, a very first easy check is to ensure that
	the number of objects in oligomer1->propGPA is lower or equal to
	oligomer2->propGPA.
      */
      if (oligomer1->propGPA->len > oligomer2->propGPA->len)
	{
	  return ++result;
	}

      for (iter = 0; iter < oligomer1->propGPA->len; iter++)
	{
	  found_prop = FALSE;
	  
	  prop1 = g_ptr_array_index (oligomer1->propGPA, iter);
	  g_assert (prop1 != NULL);
	  
	  for (jter = 0; jter < oligomer2->propGPA->len; jter++) 
	    {
	      prop2 = g_ptr_array_index (oligomer2->propGPA, jter);
	      g_assert (prop2 != NULL);
	      
	      /*
		The comparison here is the most thoroughful it can be.
	      */
	      if (0 == 
		  libpolyxmass_prop_cmp (prop1, prop2,
				     PXM_DEFAULT_PROP_CMP_NAME
				     | PXM_DEFAULT_PROP_CMP_DATA
				     | PXM_DEFAULT_PROP_CMP_DEEP
				     | PXM_DEFAULT_PROP_CMP_DATA_SUBSET1
				     | PXM_DEFAULT_PROP_CMP_DATA_SUBSET2))
		{
		  found_prop = TRUE;
		  
		  break;
		}
	      else
		continue;
	    }
	  
	  if (found_prop == FALSE)
	    {
	      return ++result;
	    }
	}
    }

  if (how_cmp & PXM_OLIGOMER_CMP_PROP_ARRAY_SUBSET2)
    {
      /*
	It is admitted that the two oligomers are not absolutely
	identical, only if all the prop objects in oligomer2->propGPA
	are in oligomer1->propGPA (ie oligomer2 is a subset of
	oligomer1). Thus, a very first easy check is to ensure that the
	number of objects in oligomer2->propGPA is lower or equal to
	oligomer1->propGPA.
      */
      if (oligomer2->propGPA->len > oligomer1->propGPA->len)
	{
	  return ++result;
	}

      for (iter = 0; iter < oligomer2->propGPA->len; iter++)
	{
	  found_prop = FALSE;
	  
	  prop2 = g_ptr_array_index (oligomer2->propGPA, iter);
	  g_assert (prop2 != NULL);
	  
	  for (jter = 0; jter < oligomer1->propGPA->len; jter++) 
	    {
	      prop1 = g_ptr_array_index (oligomer1->propGPA, jter);
	      g_assert (prop1 != NULL);
	      
	      /*
		The comparison here is the most thoroughful it can be.
	      */
	      if (0 == 
		  libpolyxmass_prop_cmp (prop1, prop2,
				     PXM_DEFAULT_PROP_CMP_NAME
				     | PXM_DEFAULT_PROP_CMP_DATA
				     | PXM_DEFAULT_PROP_CMP_DEEP
				     | PXM_DEFAULT_PROP_CMP_DATA_SUBSET1
				     | PXM_DEFAULT_PROP_CMP_DATA_SUBSET2))
		{
		  found_prop = TRUE;
		  
		  break;
		}
	      else
		continue;
	    }
	  
	  if (found_prop == FALSE)
	    {
	      return ++result;
	    }
	}
    }
  
  return result;
}

  
gint
pxmchem_oligomer_prop_cmp (PxmProp *prop1, PxmProp *prop2, gint how_cmp)
{
  gint result = 0;
  
  PxmOligomer *oligomer1 = NULL;
  PxmOligomer *oligomer2 = NULL;
  

  g_assert (prop1 != NULL);
  g_assert (prop2 != NULL);

  oligomer1 = (PxmOligomer *) prop1->data;
  oligomer2 = (PxmOligomer *) prop2->data;
  
  g_assert (oligomer1 != NULL);
  g_assert (oligomer2 != NULL);
  

  /* Remember that all the prop_cmp functions can be called with a
     number of default parameters:
     
     how_cmp & PXM_DEFAULT_PROP_CMP_NONE
     
     how_cmp & PXM_DEFAULT_PROP_CMP_NAME

     how_cmp & PXM_DEFAULT_PROP_CMP_DATA

     how_cmp & PXM_DEFAULT_PROP_CMP_DEEP

     how_cmp & PXM_DEFAULT_PROP_CMP_DATA_SUBSET1

     how_cmp & PXM_DEFAULT_PROP_CMP_DATA_SUBSET2

  */
  if (how_cmp & PXM_DEFAULT_PROP_CMP_NONE
      || how_cmp & PXM_OLIGOMER_PROP_CMP_NONE)
    {
      return -1;
    }
  
  if (how_cmp & PXM_DEFAULT_PROP_CMP_NAME
      || how_cmp & PXM_OLIGOMER_PROP_CMP_NAME)
    {
      result += strcmp (prop1->name, prop2->name);
    }
  
  if (how_cmp & PXM_DEFAULT_PROP_CMP_DATA
      || how_cmp & PXM_OLIGOMER_PROP_CMP_DATA)
    {
      result += pxmchem_oligomer_cmp (oligomer1, oligomer2,
				      PXM_OLIGOMER_CMP_POLYMER
				      | PXM_OLIGOMER_CMP_NAME
				      | PXM_OLIGOMER_CMP_COORDS
				      | PXM_OLIGOMER_CMP_MASSPAIR);
    }
  
  if (how_cmp & PXM_DEFAULT_PROP_CMP_DEEP
      || how_cmp & PXM_OLIGOMER_PROP_CMP_DEEP)
    {
      if (how_cmp & PXM_DEFAULT_PROP_CMP_DATA_SUBSET1
	  || how_cmp & PXM_OLIGOMER_PROP_CMP_PROP_ARRAY_SUBSET1)
	result += pxmchem_oligomer_cmp (oligomer1, oligomer2,
				       PXM_OLIGOMER_CMP_PROP_ARRAY_SUBSET1);

      if (how_cmp & PXM_DEFAULT_PROP_CMP_DATA_SUBSET2
	  || how_cmp & PXM_OLIGOMER_PROP_CMP_PROP_ARRAY_SUBSET2)
	result += pxmchem_oligomer_cmp (oligomer1, oligomer2,
					PXM_OLIGOMER_CMP_PROP_ARRAY_SUBSET2);
    }

  return result;
}



/*
  INTEGRITY CHECKING FUNCTIONS
*/
gboolean
pxmchem_oligomer_check_boundaries (PxmOligomer *oligomer,
				   PxmPolymer *polymer)
{
  gint len = 0;
  
  g_return_val_if_fail (oligomer != NULL, FALSE);
  g_return_val_if_fail (polymer != NULL, FALSE);
  
  len = polymer->monomerGPA->len;
  
  if (oligomer->start_idx < 0)
    return FALSE;
  
  if (oligomer->end_idx > polymer->monomerGPA->len - 1)
    return FALSE;
  
  return TRUE;
}




/*  LOCATING FUNCTIONS
 */
PxmOligomer *
pxmchem_oligomer_get_ptr_by_name (GPtrArray *GPA, gchar *name)
{
  gint iter = 0;
  PxmOligomer *oligomer = NULL;
  

  g_assert (GPA != NULL);
  g_assert (name != NULL);
  
  for (iter = 0; iter < GPA->len; iter ++)
    {
      oligomer = g_ptr_array_index (GPA, iter);
      
      if (oligomer->name != NULL)
	{
	  if (0 == strcmp (oligomer->name, name))
	    return oligomer;
	}
    }
  
  return NULL;
}


PxmOligomer *
pxmchem_oligomer_find_that_encompasses_idx (gint monomer_idx,
					    GPtrArray *GPA,
					    gint *item_idx)
{
  gint iter = 0;
  gint search_idx = 0
    ;
  PxmOligomer * oligomer = NULL;

  g_assert (GPA != NULL);

  if (GPA->len <= 0)
    return NULL;

  if (item_idx == NULL)
    search_idx = 0;
  else if (*item_idx >= GPA->len)
    return NULL;
  else 
    search_idx = *item_idx;

  for (iter = search_idx; iter < GPA->len; iter++)
    {
      oligomer = g_ptr_array_index (GPA, iter);
      
      if (monomer_idx >= oligomer->start_idx 
	  &&
	  monomer_idx <= oligomer->end_idx)
	{
	  if (item_idx != NULL)
	    *item_idx = iter;
	  return oligomer;
	}
    }

  return NULL;
}


PxmProp *
pxmchem_oligomer_find_prop (PxmOligomer *oligomer,
			    PxmProp *prop,
			    gint how_cmp,
			    gint *idx)
{
  gint iter = 0;
  
  PxmProp *prop_found = NULL;

  
  for (iter = 0; iter < oligomer->propGPA->len ; iter++)
    {
      prop_found = g_ptr_array_index (oligomer->propGPA, iter);
      g_assert (prop_found != NULL);
      
      if (0 == libpolyxmass_prop_cmp (prop, prop_found,
				  how_cmp))
	{
	  if (idx != NULL)
	    *idx = iter;
	  
	  return prop_found;
	}
    }
  
  return NULL;
}


PxmProp *
pxmchem_oligomer_find_prop_in_monomers (PxmOligomer *oligomer, 
					PxmProp *prop,
					gint how_cmp,
					gint *idx)
{
  gint iter = 0;
  gint len = 0;
  
  PxmProp *prop_found = NULL;
  PxmMonomer *monomer = NULL;
  

  g_assert (oligomer != NULL);
  g_assert (oligomer->polymer != NULL);
  g_assert (oligomer->polymer->monomerGPA != NULL);
  g_assert (prop != NULL);


  len = oligomer->polymer->monomerGPA->len;

  /*
    Make sure that the boundaries of the oligomer are OK.
  */
  g_assert (oligomer->start_idx >= 0);
  g_assert (oligomer->end_idx < len);
  
  /*
    If the polymer is zero-length, of course we'll not find any 
    prop object in it !
  */
  if (len == 0)
    return NULL;

  /*
    What we are doing, is searching for a given prop object in all
    the monomers that are comprising a given oligomer. The way the
    comparison is to be performed is governed by the how_cmp
    parameter which has to be a PxmDefaultPropCmp param. This is
    because we canot figure with anticipation what type is going to
    be the prop->data object.
  */
  g_assert (how_cmp & PXM_DEFAULT_PROP_CMP_NONE
	    || how_cmp & PXM_DEFAULT_PROP_CMP_NAME
	    || how_cmp & PXM_DEFAULT_PROP_CMP_DATA
	    || how_cmp & PXM_DEFAULT_PROP_CMP_DEEP
	    || how_cmp & PXM_DEFAULT_PROP_CMP_DATA_SUBSET1
	    || how_cmp & PXM_DEFAULT_PROP_CMP_DATA_SUBSET2);
  
  
  /*
    Now start iterating in the oligomer, and for each monomer
    iterated make sure we can find a prop object in it. As soon as we
    find a monomer that has a prop identical to the 'prop' passed a
    param, we return that prop object's pointer. If 'idx' is
    non-NULL, then we set it to the index of the monomer that
    contained that prop. This way, the calling function can use the
    returned idx value to modify the oligomer->start_idx member in
    order to recursively iterate in all the oligomer.
  */
  for (iter = oligomer->start_idx ; iter < oligomer->end_idx + 1 ; iter++)
    {
      monomer = g_ptr_array_index (oligomer->polymer->monomerGPA, iter);
      g_assert (monomer != NULL);
      
      prop_found = pxmchem_monomer_find_prop (monomer, prop, how_cmp, NULL);
      
      if (prop_found != NULL)
	{
	  if (idx != NULL)
	    *idx  = iter;

	  return prop_found;
	}
    }
    
  return NULL;
}

  



/*
  UTILITY FUNCTIONS
*/



/*
  XML-format TRANSACTIONS
*/
gchar *
pxmchem_oligomer_format_xml_string_olm (PxmOligomer *oligomer, 
					gchar *indent, gint offset)
{
  return NULL;
}


PxmOligomer *
pxmchem_oligomer_render_xml_node_olm (xmlDocPtr xml_doc,
				      xmlNodePtr xml_node,
				      gpointer user_data)
{
  return NULL;
}




/*
  FREE'ING FUNCTIONS
*/
gboolean
pxmchem_oligomer_free (PxmOligomer *oligomer)
{
  g_assert (oligomer != NULL);
  
  /*
    Free inner material first.
  */
  if (oligomer->propGPA != NULL)
    libpolyxmass_prop_GPA_free (oligomer->propGPA);
  
  if (oligomer->name != NULL)
    g_free (oligomer->name);
  
  if (oligomer->masspair != NULL)
    libpolyxmass_masspair_free (oligomer->masspair);
  
  g_free (oligomer);
  
  return TRUE;
}


/*
  GPtrArray-RELATED FUNCTIONS
*/
gint
pxmchem_oligomer_GPA_free (GPtrArray *GPA)
{
  gint count = 0;
  
  PxmOligomer *oligomer = NULL;
  

  g_assert (GPA != NULL);
  
  while (GPA->len > 0)
    {
      oligomer = g_ptr_array_remove_index (GPA, 0);
      g_assert (oligomer != NULL);
      pxmchem_oligomer_free (oligomer);
      count++;
    }
  
  g_ptr_array_free (GPA, TRUE);

  return count;
}


/*
  Frees an array of arrays of oligomers.
*/
gint
pxmchem_oligomer_GPA_of_GPA_free (GPtrArray *GPA)
{
  gint count = 0;
  
  GPtrArray *oligGPA = NULL;
  

  g_assert (GPA != NULL);
  
  while (GPA->len > 0)
    {
      oligGPA = g_ptr_array_remove_index (GPA, 0);
      g_assert (oligGPA != NULL);

      count += pxmchem_oligomer_GPA_free (oligGPA);
    }
  
  g_ptr_array_free (GPA, TRUE);

  return count;
}






