#!/usr/local/bin/pike

#include <module.h>
inherit "module";
inherit "roxenlib";
inherit Stdio;


// CRD 09/04/2000 cdawson@webiphany.com
//
//  Secure insert
//
// Used to insert macro files in a secure way.  Files are stored in a 
// root outside of the nromal http root.  Fiesl cannot be referenced
// via normal get requests.  So, you can have information that will
// be inserted into your files and parsed only once, but there is no 
// security risk if someone were to figure out your paths.
//
// v 0.1 09/04/2000 CRD
//

#define false 0
#define true 1
#define SCRIPT_ERROR "Error with the script"
 
class cached_file {

  string contents;
  string filename;
  int timestamp;

}

int last_index = 0;

// Cache for one hour
int threshold = ( 60 * 60 );

#define FILE_CACHE_SIZE 100

array( object( cached_file ) ) cfiles = ({});

#define SECURE_DIR "secdir"

#define FILE_NOT_FOUND( x )  ( "<!-- File not found: " + x + " -->" )

void create() {
  defvar( SECURE_DIR,	 
	  "NONE",
	  "Secured include directory",
	  TYPE_DIR,
	  "This directory is a secure place to place files which don't "
  "need to be parsed (or can't) and insert them outside the "
	  "normal http directory tree.  Usage is &lt;sinsert&gt;.  "
	  "Use the cache attribute to cache files and speed up insertion." );
}

string secure_insert(string tag, mapping args, object id );

string real_get_file( string name ) {

  string contents = "";
  string dir = query( SECURE_DIR );
  object file = Stdio.File();

  if( file->open( dir + name, "r" ) ) {
    contents = file->read();
  }

  return contents;
}

string get_contents( string name, int cache ) {

  string contents = "";
  int current = time();
  int found = false;

  int index = 0;
  // Start at the last_index, instead of zero...
  int better_index = 0;

  if( !cfiles || sizeof( cfiles ) == 0 ) {
    cfiles = ({});

    for( index = 0; index < FILE_CACHE_SIZE; index++ ) {
      cfiles += ({ cached_file() });
    }
  }

  if( cache ) {
      // See if we have it, and check the timeout
      for( index = 0; index < sizeof( cfiles ); index++ ) {
	better_index = ( index + last_index ) % FILE_CACHE_SIZE;
	if( // cfiles[ better_index ] && 
	    cfiles[ better_index ]->filename &&
	    cfiles[ better_index ]->filename == name ) {

	  if( cfiles[ better_index ]->timestamp + threshold < current ) {
	    // Cached has expired, reload
	    contents = real_get_file( name );
	
	    if( "" != contents ) {
	      cfiles[ better_index ]->timestamp = current;
	      cfiles[ better_index ]->contents = contents;
	    }
	    else {
	      // Yikes, clear it here.
	      cfiles[ better_index ] = cached_file();
	      contents = FILE_NOT_FOUND( name );
	    }

	  }
	  else {
	    contents = cfiles[ better_index ]->contents;
	  }

	  // Escape the loop
	  index = FILE_CACHE_SIZE;
	  found = true;
	}
      }

    if( !found ) {
      last_index = ( last_index+1 ) % FILE_CACHE_SIZE;

      // kick out the one at the last index
      contents = real_get_file( name );

      if( "" != contents ) {
	cfiles[ last_index ]->contents = contents;
	cfiles[ last_index ]->filename = name;
	cfiles[ last_index ]->timestamp = current;
      }
      else {
	contents = FILE_NOT_FOUND( name );
      }
    }
  } // if ( cache )
  else { 
    contents = real_get_file( name );
    if( "" == contents ) {
      contents = FILE_NOT_FOUND( name );
    }
  }
  
  return contents;

}

string usage() {
  return "<br><br><font size=\"+2\">&lt;sinsert&gt;</font>"
    "<br><br>\n<i>attributes</i>:<br>"
    "<b>filename</b>: filename, added to the directory specified in "
    "the module, used to resolve the file<br><b>nocache</b>: don't cache the file, "
    "for more rapid access<br><b>debug</b>: print out debugging "
    "information<br><br>Use no attributes for this usage message<br><br>";
}

string debug() {
  string return_string = "";

  int index = 0;
  int better_index = 0;
  int current = time();

  return_string += "<br>Summary of cached files, sorted by timestamp:<br>";

  if( !cfiles || sizeof( cfiles ) == 0 ) {
    return_string += "No files cached<br>";
  }
  else 
    {
      for( index = 0; 
	   index < FILE_CACHE_SIZE; 
	   index++ ) {
	better_index = ( index + last_index + 1 ) % FILE_CACHE_SIZE;
	if( cfiles[ better_index ] &&
	    cfiles[ better_index ]->filename &&
	    cfiles[ better_index ]->filename != "" ) {
	return_string += "<i>File #" + (string)index + "</i> ";
	  return_string +=
	    "<b>filename</b>: " + cfiles[ better_index ]->filename + 
	    " expires in " + 
	    (string)( ( current + threshold )  - 
		      cfiles[ better_index ]->timestamp ) +
	    " seconds" +
	    "<br>\n";
	}
	else {
	  //	  return_string += "</b>uncached</b>\n<br>";
	}
      }
    }
  
  return return_string;
}

string secure_insert(string tag, mapping args, object id ) {

  string return_string = "";

  // Resolve file
  string filename = args->filename;
  int cache = ( !args->nocache );

  if( !filename || filename == "" ) {
    if( args->debug ) {
      return_string += debug();
    }
    else {
      return_string += usage();
    }
  }
  else {
    return_string = get_contents( filename, cache );
  }

  return return_string;
}

mapping query_tag_callers() { 
  return ( [ "sinsert" :  secure_insert ] );
}

array register_module()
{
  return ({ MODULE_PARSER,
	    "Secure insert",
	    "This module is used to inesrt files securely.  The files are "
	    "setup in a special directory outside the normal http directory." +
	    usage(), 
	    0, 1	     
  });
}





