
// #define PHOENIX_SINGLETON_DEBUG 0
#include <PACKAGE_NAMESPACE/class_loader.h>

/**
	It is important to understand that any given class_loader<T>
	instantiation works only for types which are exactly of type
	T, not for subtypes of T. As this essentially kills any chance
	of using polymorphic types, this is obviously a problem! That
	said, we can coerce the classloader into using sub-types of T
	by feeding it factories which creates (SubT*) and returns
	(T*). class_loader<T>::register_subtype<X>() provides a very
	simple way to register a default factory for type X, returning
	pointers to T (where X must be a subclass of T).

        The sample code below exemplifies the requirements placed on
        clients of class_loader:

        1. Classes to be loaded from a given classoader type must
        derive from a common base type, defined by the paramaterized
        type T.

        2. Clients calling load() must use a class_loader<T> instance.

        3. Proper lookup keys must be registered with the appropriate
        classloader(s).
**/

#include <PACKAGE_NAMESPACE/cl_debuggering_macros.h> // just what it says

/**
   First we define a class hierarchy:
*/
struct A {
        A(){
                // We echo the ctors so we can see when classes
                // are instantiated by class_loader.
                CERR << "A()"<<std::endl;
        }
        virtual ~A(){
                // We echo the dtors so we can see if shared-instance
                // class_loaders clean up when they should (and not
                // when the shouldn't!).
                CERR << "~A()"<<std::endl;
        }
        bool no_op;
};
struct B : public A {
        B(){
                CERR << "B()"<<std::endl;
        }
        virtual ~B(){
                CERR << "~B()"<<std::endl;
        }

};
struct C : public B {
        C(){
                CERR << "C()"<<std::endl;
        }
        virtual ~C(){
                CERR << "~C()"<<std::endl;
        }
};

// Now we register classes. We show both approaches, first
// the easy way (one macro) or the hard way (a couple lines
// of code).


// Define USE_REGISTER to 0 to disable the use of CLASSLOADER_REGISTER
// and use manual factory registration instead. This is for testing
// both approaches, and they should function identically.
#define USE_REGISTER 1

#if USE_REGISTER
// Corresponds to the ACL classloader in the example code below:
CLASSLOADER_REGISTER(A,A);
CLASSLOADER_REGISTER(A,B);
CLASSLOADER_REGISTER(A,C);
// ^^^ normally clients would only use these 3, but the ones coming up
// next are useful in some cases:

// Corresponds to the BCL classloader in the example code below:
CLASSLOADER_REGISTER(B,B);
CLASSLOADER_REGISTER(B,C);

// Corresponds to the CCL classloader in the example code below:
CLASSLOADER_REGISTER(C,C);

#endif // USE_REGISTER
// if we USE_REGISTER, we're done with the classloader registration
// process!

#include "LoadableClass.h" // sample BaseType for demonstating DLLs
#include <PACKAGE_NAMESPACE/dll_loader.h> // a DLL-aware class_loader implementation


int main()
{

        using namespace PACKAGE_NAMESPACE;

        // Define some shortcuts:
        typedef class_loader<A,std::string,true> ACL;
        //typedef class_loader<A> ACL;// for testing cases which shouldn't work.

        CERR << "Note: class_loader<A> uses shared objects." << std::endl;
        typedef class_loader<B> BCL;
        typedef class_loader<C> CCL;

#if ! USE_REGISTER
        /******************************
	If we do not use CLASSLOADER_REGISTER then the
	following code is necessary to register loaders
	based on A, B and C.

	If you'll look closely you'll notice that these calls
	correspond one-to-one to the above macro calls.
        ****************************/

        // Set up a loader for class A:
        ACL::register_factory( "A" ); // factory creating/returning A*
        ACL::register_subtype<B>( "B" ); // factory creating B*, returning A*
        ACL::register_subtype<C>( "C" ); // factory creating C*, returning A*

        // Set up a loader for class B:
        BCL::register_factory( "B" ); // factory creating B*, returning B*
        BCL::register_subtype<C>( "C" ); // factory creating C*, returning B*
        // BCL::register_subtype<A>( "A" ); // ERROR: there is no conversion for A* to B*

        // Set up a loader for class C:
        CCL::register_factory( "C" ); // factory creating C*, returning C*
        // CCL::register_subtype<A>( "A" ); // ERROR: there is no conversion from A* to C*
        // CCL::register_subtype<B>( "B" ); // ERROR: there is no conversion from B* to C*

        // reminder: the above ERRORs are compile-time errors.
#endif

        // make sure i've set up ACL properly:
        assert( ACL::use_shared_instances && "Error: ACL doesn't seem to be sharing objects." );


        // let's register an alias for class C:
        // brute-force way:
        CCL::register_factory( "bogo::C" );
        // or the gentler way:
        CCL::alias( "bogo::C", "C"  );
        // ^^^ The first one creates (or accepts) an additional factory, but
        // the second one requires that the client know the stringified class name.
        // For *most* cases those are functionaly identical: if you provide your
        // own factories there is a subtle difference (see the docs).


        // It may be useful to define configurable classes types:
        ACL::alias( "default_document_class", "C" );
        // For example, an app's config file may provide the default
        // implementation class for a type of plugin, document class,
        // or whatever.


        // Now we run some tests on our loaders. Any test which fails will
        // cause the app to exit with a non-zero status.

#define TESTLOAD(OBJ,ClassloadeR,ClassnamE,ShouldpasS) \
	CERR << "BEGIN TEST:\n" << # OBJ << " = "<<#ClassloadeR<<".load("<<ClassnamE<<");"<<std::endl; \
	OBJ = ClassloadeR.load( ClassnamE ); \
 	std::cerr << "   = " << std::hex<<OBJ<<std::endl; \
	if( ShouldpasS && (!OBJ) ) { CERR << "TEST FAILED! :(" << std::endl; exit(1); } \
	else CERR << "TEST PASSED :)" << std::endl; \
	if( OBJ && ! ClassloadeR.use_shared_instances ) delete( OBJ );



        // test out class_loader<A>:
        ACL acl;
        A * a = 0;
        TESTLOAD(a,acl,"A",true); // creates A*
        TESTLOAD(a,acl,"B",true); // creates B*, disguised as A*
        TESTLOAD(a,acl,"B",true); // test the shared object support. Should be the same as previous call.
        TESTLOAD(a,acl,"C",true); // creates C*, disguised as A*
        TESTLOAD(a,acl,"C",true); // test shared again...
        TESTLOAD(a,acl,"X",false); // NULL: key "X" was never registered with ACL
        TESTLOAD(a,acl,"default_document_class",true); // creates C*

        // test out class_loader<B>:
        BCL bcl;
        B * b = 0;
        TESTLOAD(b,bcl,"B",true); // creates B*
        TESTLOAD(b,bcl,"A",false); // NULL: key "A" was never registered with BCL
        TESTLOAD(b,bcl,"C",true); // creates a C*, disguised as a B*

        // test out class_loader<C>:
        CCL ccl;
        C * c = 0;
        TESTLOAD(c,ccl,"C",true); // creates a C*
        TESTLOAD(c,ccl,"bogo::C",true); // creates a C*
        TESTLOAD(c,ccl,"default_document_class",false); // NULL. Was never registered as an alias with CCL.


        // Now try some DLL loading. The base-most class (LoadableClass) is known
        // at compile-time, but LoadableSubClass is not: it is known here only
        // by a name, and the DLL loader loads it for us:
        typedef dll_loader<LoadableClass> LCL;
        LCL lcl;
        LoadableClass * lc = 0;
        TESTLOAD(lc,lcl,"LoadableClass",true);
        TESTLOAD(lc,lcl,"LoadableSubClass",true); // a DLL, we hope
        LCL::alias( "AnAlias", "LoadableSubClass" );
        TESTLOAD(lc,lcl,"AnAlias",true); // creates a LoadableSubClass*
        TESTLOAD(lc,lcl,"ANonAlias",false); // NULL. Unregistered class name.


        // For demonstration purposes, LoadableSubClass does some extra
        // CLASSLOADER_REGISTER calls in it's implementation file, allowing us
        // to use an int-keyed class_loader:
        typedef class_loader<LoadableClass,int> INTCL;
        INTCL intcl;
        INTCL::alias( 42, 7 );
        TESTLOAD(lc,intcl,7,true);  // see LoadableSubClass.cpp for why this works
        TESTLOAD(lc,intcl,1,false);  // ... and why this doesn't. (tip: never registered)
        TESTLOAD(lc,intcl,42,true); // This works because of the above alias(42,7).

        // Caveat: INTCL works only because the string-keyed
        // dll_loader has already loaded LoadableSubClass: that
        // triggered the registrations for the int variants. In client
        // code such registrations must normally be done in some
        // app-initialization code, or perhaps in the objects'
        // header files, depending on how closely the objects
        // want to be tied to the class_loader.


        CERR << "Exiting main(). Any loaders using shared object instances should clean up now..." << std::endl;
        return 0;
}

