/* 
   Project: MPDCon

   Copyright (C) 2004

   Author: Daniel Luederwald

   Created: 2004-05-12 17:59:14 +0200 by flip
   
   Custom View : Shows Trackinfos etc.

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/

#include <AppKit/AppKit.h>
#include "PlayView.h"


/* ---------------------
   - Private Interface -
   ---------------------*/

@interface PlayView(Private)
- (NSString *) _getTime;

- (void) _drawTrack;
- (void) _drawArtist;
- (void) _drawAlbum;
- (void) _drawTitle;
- (void) _drawTime;
@end

@implementation PlayView

/* --------------------------
   - Initialization Methods -
   --------------------------*/

+ (id) init
{
  self = [super init];
  

  return self;
}

- (void) dealloc
{
  RELEASE(dispSong);

  [super dealloc];
}

/* -------------------
   - Display Methods -
   -------------------*/


- (void) drawRect: (NSRect)rect
{
  NSRect myBounds;
  
  if (! notificationAdded) 
    {
      notificationAdded = YES;
      [[NSNotificationCenter defaultCenter] 
	addObserver: self
	   selector: @selector(songChanged:)
	       name: SongChangedNotification
             object: nil];
    }
  
  [[NSColor grayColor] set];

  myBounds = [self bounds];
  [NSBezierPath fillRect: myBounds];

  if (displayEnabled) 
    {
      [self _drawTrack];
      [self _drawArtist];
      [self _drawAlbum];
      [self _drawTitle];
      [self _drawTime];
    } 
  else 
    {
      NSImage *gnustepImage;
      
      gnustepImage = [NSImage imageNamed: @"GNUstep.tiff"];

      [gnustepImage compositeToPoint: NSMakePoint(130, 20) 
		            fromRect: myBounds 
		           operation: NSCompositeSourceOver];
  }
}

- (void) setDisplaySong: (PlaylistItem *)newSong
{
  RELEASE(dispSong);

  dispSong = RETAIN(newSong);
  [self setNeedsDisplay: YES];
}

- (void) setCurrentSong: (int)newSong
{
  currentSong = newSong;
}

- (void) setTotalSongs: (int)newSong
{
  totalSongs = newSong;
}

- (void) setElapsedTime: (int)newTime
{
  if (dispSong)
    {
      [dispSong setElapsedTime: newTime];
    }
  
  [self setNeedsDisplay: YES];
}

- (void) setReversedTime: (BOOL)reversed
{
  reversedTime = reversed;
}

- (void) enableDisplay: (BOOL)enabled
{
  displayEnabled = enabled;
  [self setNeedsDisplay: YES];
}

/* ------------------------
   - Notification Methods -
   ------------------------*/

- (void) songChanged: (NSNotification *)aNotif
{
  artistScroll = 0;
  titleScroll = 0;
  albumScroll = 0;
  artistScrollForward = YES;
  titleScrollForward = YES;
  albumScrollForward = YES;
}

/* --------------------
   - Delegate Methods -
   --------------------*/

- (void) mouseUp: (NSEvent *)anEvent
{
  if (reversedTime) 
    {
      reversedTime = NO;
    }
  else 
    {
      reversedTime = YES;
    }

  [[NSUserDefaults standardUserDefaults] setBool: reversedTime 
					  forKey: @"reversedTime"];
}

@end
/* -------------------
   - Private Methods -
   -------------------*/

@implementation PlayView(Private)
- (NSString *) _getTime
{
  int totalTime, elapsedTime;

  int tSecs, eSecs;
  int tMin, eMin;


  totalTime = [dispSong getTotalTime];
  elapsedTime = [dispSong getElapsedTime];
  tSecs = (totalTime % 60);
  tMin = (int)totalTime/60;
  
  eSecs = 0;
  eMin = 0;

  if (! reversedTime) 
    {
      eSecs = (elapsedTime % 60);
      eMin = (int)elapsedTime/60;
      return [NSString stringWithFormat: @"%d:%02d/%d:%02d"
		       , eMin, eSecs, tMin, tSecs];
    } 
  else 
    {
      eSecs = ((totalTime - elapsedTime) % 60);
      eMin = (int) (totalTime - elapsedTime) / 60;
      return [NSString stringWithFormat: @"- %d:%02d/%d:%02d"
		       , eMin, eSecs, tMin, tSecs];
    }
}

- (void) _drawTrack
{
  [[NSString stringWithFormat: _(@"Playing Track %d of %d")
	     , currentSong, totalSongs] 
    drawAtPoint: NSMakePoint(3, 66) withAttributes: nil];
}

- (void) _drawArtist
{
  NSString *theArtist;
  NSRect myBounds;

  NSMutableDictionary *attributes;
  NSFont *theFont;

  int enableScroll;


  theArtist = [dispSong getArtist];
  myBounds = [self bounds];
  
  attributes = [NSMutableDictionary dictionary];
  
  enableScroll =  [[NSUserDefaults standardUserDefaults] 
		    integerForKey: @"enableScroll"];
  
  theFont = [NSFont boldSystemFontOfSize: 18];
  [attributes setObject: theFont forKey: NSFontAttributeName];
  
  if (([theArtist sizeWithAttributes: attributes].width 
       > myBounds.size.width-10) && enableScroll) 
    {
      if ([theArtist sizeWithAttributes: attributes].width-artistScroll 
	  < myBounds.size.width - 20) 
	{
	  artistScrollForward=NO;
	}
      
      if (artistScroll == 0) 
	{
	  artistScrollForward=YES;
	}
      
      if (artistScrollForward)
	{
	  artistScroll+=5;
	}
      else
	{
	  artistScroll-=5;
	}
    }
  
  [theArtist drawAtPoint: NSMakePoint(10-artistScroll, 43) 
	  withAttributes: attributes];
}

- (void) _drawAlbum
{
  NSString *theAlbum;
  NSRect myBounds;

  NSMutableDictionary *attributes;
  NSFont *theFont;

  int enableScroll;


  myBounds = [self bounds];
  
  attributes = [NSMutableDictionary dictionary];
  
  enableScroll =  [[NSUserDefaults standardUserDefaults] 
		    integerForKey: @"enableScroll"];
  
  theAlbum = [dispSong getAlbum];
  
  theFont = [NSFont systemFontOfSize: 13];

  [attributes setObject: theFont forKey: NSFontAttributeName];
  
  if (([theAlbum sizeWithAttributes: attributes].width 
       > myBounds.size.width-3) && enableScroll) 
    {
      if ([theAlbum sizeWithAttributes: attributes].width-albumScroll 
	  < myBounds.size.width - 10) 
	{
	  albumScrollForward=NO;
	}
      
      if (albumScroll == 0) 
	{
	  albumScrollForward=YES;
	}
    
      if (albumScrollForward)
	{
	  albumScroll+=5;
	}
      else
	{
	  albumScroll-=5;
	}
    }
  
  [theAlbum drawAtPoint: NSMakePoint(3-albumScroll, 3) 
	 withAttributes: attributes];
}

- (void) _drawTitle
{
  NSString *theTitle;
  NSString *theTrack;
  NSRect myBounds;

  NSMutableDictionary *attributes;
  NSFont *theFont;

  int enableScroll;


  myBounds = [self bounds];
  
  attributes = [NSMutableDictionary dictionary];
  
  enableScroll =  [[NSUserDefaults standardUserDefaults] integerForKey: @"enableScroll"];
  
  theTrack = [dispSong getTrackNr];
  
  theTitle = [theTrack stringByAppendingString: 
			 [@" - " stringByAppendingString: 
			     [dispSong getTitle]]];
  
  
  theFont = [NSFont boldSystemFontOfSize: 13];
  
  [attributes setObject: theFont forKey: NSFontAttributeName];
  
  if (([theTitle sizeWithAttributes: attributes].width 
       > myBounds.size.width-3) && enableScroll) 
    {
      if ([theTitle sizeWithAttributes: attributes].width-titleScroll 
	  < myBounds.size.width - 10) 
	{
	  titleScrollForward=NO;
	}
    
      if (titleScroll == 0) 
	{
	  titleScrollForward=YES;
	}
    
      if (titleScrollForward)
	{
	  titleScroll+=5;
	}
      else
	{
	  titleScroll-=5;
	}
    }
  
  [theTitle drawAtPoint: NSMakePoint(3-titleScroll, 23) 
         withAttributes: attributes];
}

- (void) _drawTime
{
  NSRect myBounds;

  NSMutableDictionary *attributes;
  NSFont *theFont;

  
  myBounds = [self bounds];
  
  attributes = [NSMutableDictionary dictionary];
  
  theFont = [NSFont boldSystemFontOfSize: 17];

  [attributes setObject: theFont forKey: NSFontAttributeName];
  
  NSString *timeString = [self _getTime];
  NSSize timeSize = [timeString sizeWithAttributes: attributes];
  [timeString drawAtPoint: NSMakePoint(myBounds.size.width-timeSize.width-2
				       , 63) 
	   withAttributes: attributes];
}

@end
