/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "MRI.h"

struct MRIColorSpaceData {
	MRI	*mri;
	MRI_Link *next;
	int	width;
	int	height;
	int	debug;
	cmsHPROFILE hInProfile, hOutProfile;
	cmsHTRANSFORM hTransform;
        cmsCIEXYZ *whitePoint;
};

void
MRIColorSpaceStart (void *private, int width, int height, int freedata)
{
	struct MRIColorSpaceData *wd = private;

	wd->width = width;
	wd->height = height;
	(*wd->next->start) (wd->next->private, width, height, freedata);
}

void
MRIColorSpaceRow (void *private, void *data)
{
	struct MRIColorSpaceData *wd = private;
	struct MRI_ScanLine *sl = data;
	unsigned short tin[3], tout[3];
	int j;

	if (sl->scanLineType == LINETYPE_SHORT) {
	    unsigned short *R = sl->R;
	    unsigned short *G = sl->G;
	    unsigned short *B = sl->B;
	    for (j = 0; j < wd->width; j++) {
		tin[0] = R[j];
		tin[1] = G[j];
		tin[2] = B[j];
		/* if (wd->debug) { if (tin[1] < 10000) tin[1] = 10000; } */
		cmsDoTransform(wd->hTransform, tin, tout, 1);
		if (wd->debug && tout[0] == 0 ) {
			/* fprintf (stderr, "Debug: %d,%d,%d -> %d,%d,%d\n", tin[0], tin[1], tin[2], tout[0], tout[1], tout[2]); */
			tout[0] = 1;
		}
		R[j] = tout[0];
		G[j] = tout[1];
		B[j] = tout[2];
	    }
	}
	else if (sl->scanLineType == LINETYPE_FLOAT) {
	    float *R = sl->R;
	    float *G = sl->G;
	    float *B = sl->B;
	    for (j = 0; j < wd->width; j++) {
		tin[0] = R[j] < 0.0 ? 0 : R[j] > 65535.0 ? 65535 : R[j];
		tin[1] = G[j] < 0.0 ? 0 : G[j] > 65535.0 ? 65535 : G[j];
		tin[2] = B[j] < 0.0 ? 0 : B[j] > 65535.0 ? 65535 : B[j];
		/* if (wd->debug) { if (tin[1] < 10000) tin[1] = 10000; } */
		cmsDoTransform(wd->hTransform, tin, tout, 1);
		if (wd->debug && tout[0] == 0 ) {
			/* fprintf (stderr, "Debug: %d,%d,%d -> %d,%d,%d\n", tin[0], tin[1], tin[2], tout[0], tout[1], tout[2]); */
			tout[0] = 1;
		}
		R[j] = tout[0];
		G[j] = tout[1];
		B[j] = tout[2];
	    }
	}

	(*wd->next->row) (wd->next->private, sl);
}

void
MRIColorSpaceRowSpecial (void *private, void *data)
{
	struct MRIColorSpaceData *wd = private;
	struct MRI_ScanLine *sl = data;

	MRI_ConvertScanLineToLAB (wd->mri, sl);
	(*wd->next->row) (wd->next->private, sl);
}

void
MRIColorSpaceClose (void *private)
{
	struct MRIColorSpaceData *wd = private;
	(*wd->next->close) (wd->next->private);
	free (wd->next);
#if 0
	cmsDeleteTransform(wd->hTransform);
	cmsCloseProfile(wd->hInProfile);
	cmsCloseProfile(wd->hOutProfile);
#endif
	free (wd);
}

struct spacer {
	char	*inName, *outName;
	int     intent, useSpecial;
	cmsHPROFILE hInProfile, hOutProfile;
	cmsHTRANSFORM hTransform;
	struct spacer *next;
};

static struct spacer *transform_cache = NULL;

MRI_Link *
MRI_GenColorSpacer (MRI *mri,
		    const char *inputProfileFileName,
		    const char *outputProfileFileName,
		    int intent,
		    MRI_Link *next)
{
	int iColorSpace, oColorSpace;
	int useSpecial;
	cmsCIExyY wPt;
	struct spacer *sp, *new;
	struct MRIColorSpaceData *wd = malloc (sizeof (struct MRIColorSpaceData));
	MRI_Link *ep = malloc (sizeof (*ep));

#if 0
	fprintf (stderr, "Converting from %s -> %s\n", inputProfileFileName, outputProfileFileName);
#endif
	if (wd == (struct MRIColorSpaceData *)0 || ep == (MRI_Link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = MRIColorSpaceStart;
	ep->row = MRIColorSpaceRow;
	ep->close = MRIColorSpaceClose;
	ep->private = wd;
	wd->next = next;

	wd->mri = mri;
	wd->whitePoint = MRI_GetWhitePoint (mri);

	for (sp = transform_cache; sp != NULL; sp = sp->next)
	    if (sp->intent == intent &&
	        strcmp (sp->inName, inputProfileFileName) == 0 &&
	        strcmp (sp->outName, outputProfileFileName) == 0)
		break;
	if (sp == NULL) {
	    cmsXYZ2xyY (&wPt, wd->whitePoint);
	    if (strcmp (inputProfileFileName, "__LAB__") == 0) {
		wd->hInProfile = cmsCreateLabProfile (&wPt);
		iColorSpace = TYPE_Lab_16;
	    }
	    else {
		if (strcmp (inputProfileFileName, "__sRGB__") == 0)
			wd->hInProfile = cmsCreate_sRGBProfile ();
		else if (strcmp (inputProfileFileName, MRI_NATIVE_PROFILE) == 0)
			wd->hInProfile = MRI_GetNativeProfile (mri);
		else
			wd->hInProfile  = cmsOpenProfileFromFile (inputProfileFileName, "r");
		iColorSpace = TYPE_RGB_16;
	    }


            useSpecial = strcmp (inputProfileFileName, MRI_NATIVE_PROFILE) == 0 &&
	                 strcmp (outputProfileFileName, "__LAB__") == 0;

	    if (useSpecial) {
		ep->row = MRIColorSpaceRowSpecial;
		wd->hOutProfile = cmsCreateXYZProfile ();
		oColorSpace = TYPE_XYZ_16;
		/* fprintf (stderr, "Using builtin Dimage->Lab converter.\n"); */
	    }
	    else if (strcmp (outputProfileFileName, "__LAB__") == 0) {
		wd->hOutProfile = cmsCreateLabProfile (&wPt);
		oColorSpace = TYPE_Lab_16;
	    }
	    else {
		if (strcmp (outputProfileFileName, "__sRGB__") == 0)
			wd->hOutProfile = cmsCreate_sRGBProfile ();
		else if (strcmp (outputProfileFileName, MRI_NATIVE_PROFILE) == 0)
			wd->hOutProfile = MRI_GetNativeProfile (mri);
		else
			wd->hOutProfile = cmsOpenProfileFromFile (outputProfileFileName, "r");
		oColorSpace = TYPE_RGB_16;
	    }

	    wd->debug = (iColorSpace == TYPE_RGB_16) && (oColorSpace == TYPE_Lab_16);
	    wd->hTransform = cmsCreateTransform(wd->hInProfile, iColorSpace,
                                            wd->hOutProfile, oColorSpace,
#if 0
                                            INTENT_PERCEPTUAL, 0);
					    INTENT_PERCEPTUAL, cmsFLAGS_NOTPRECALC);
					    INTENT_ABSOLUTE_COLORIMETRIC, cmsFLAGS_NOTPRECALC);
#endif
					    intent, cmsFLAGS_HIGHRESPRECALC);

	    sp = malloc (sizeof(*sp));
	    sp->intent = intent;
	    sp->inName = strdup (inputProfileFileName);
	    sp->outName = strdup (outputProfileFileName);
	    sp->next = transform_cache;
	    transform_cache = sp;
	    sp->useSpecial = useSpecial;
	    sp->hInProfile = wd->hInProfile;
	    sp->hOutProfile = wd->hOutProfile;
	    sp->hTransform = wd->hTransform;
	}
	else {
	    wd->hInProfile = sp->hInProfile;
	    wd->hOutProfile = sp->hOutProfile;
	    wd->hTransform = sp->hTransform;
	    if (sp->useSpecial)
		ep->row = MRIColorSpaceRowSpecial;
	}
	return ep;
}
