/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <netdude/nd.h>
#include <netdude/nd_debug.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_protocol_registry.h>
#include <netdude/nd_misc.h>
#include <netdude/nd_raw_protocol.h>
#include <netdude/nd_gettext.h>

#include <nd_icmp.h>
#include <nd_icmp_callbacks.h>


static ND_ProtoField icmp_fields[] = {
  { ND_VAL_FIELD, N_("Type (%s)"),         N_("ICMP message type"),  8, nd_icmp_type_cb   },
  { ND_VAL_FIELD, N_("Code (%s)"),         N_("ICMP message code"),  8, nd_icmp_code_cb   },
  { ND_VAL_FIELD, N_("Checksum (0x%.4x)"), N_("ICMP checksum"),     16, nd_icmp_cksum_cb  },
  { 0, NULL, NULL, 0, NULL }
};

static ND_ProtoField icmp_req_seq_fields[] = {
  { ND_VAL_FIELD, N_("ID (%u)"),           N_("Request identifier"), 16, nd_icmp_id_cb  },
  { ND_VAL_FIELD, N_("Seq. (%u)"),         N_("Sequence number"),    16, nd_icmp_seq_cb },
  { 0, NULL, NULL, 0, NULL }
};

static ND_ProtoField icmp_addr_mask_fields[] = {
  { ND_VAL_FIELD, N_("Subnet mask (%s)"),  N_("IP subnet mask"),     32, nd_icmp_ip_cb },
  { 0, NULL, NULL, 0, NULL }
};

static ND_ProtoField icmp_tstamp_fields[] = {
  { ND_VAL_FIELD, N_("Orig. TS (%lu)"),     N_("Originate timestamp"), 32, nd_icmp_ts_orig_cb },
  { ND_VAL_FIELD, N_("Recv. TS (%lu)"),     N_("Receive timestamp"),   32, nd_icmp_ts_recv_cb },
  { ND_VAL_FIELD, N_("Trans. TS (%lu)"),    N_("Transmit timestamp"),  32, nd_icmp_ts_trans_cb },
  { 0, NULL, NULL, 0, NULL }
};

static ND_ProtoField icmp_error_fields[] = {
  { ND_VAL_FIELD, N_("Data (%s)"),          N_("Data field"),          32, nd_icmp_error_cb },
  { 0, NULL, NULL, 0, NULL }
};

static ND_ProtoField icmp_router_adv_fields[] = {
  { ND_VAL_FIELD, N_("Number of addrs. (%u)"),     N_("Number of addresses"),      8,  nd_icmp_adv_num_addr_cb },
  { ND_VAL_FIELD, N_("Entry size (%u)"),           N_("Address entry size"),       8,  nd_icmp_adv_addr_entry_cb },
  { ND_VAL_FIELD, N_("Lifetime (%u s)"),           N_("Address lifetime"),        16,  nd_icmp_adv_lifetime_cb },
  { ND_VAL_FIELD, N_("Router address (%s)"),       N_("Router address"),          32,  nd_icmp_adv_ip_cb },
  { ND_VAL_FIELD, N_("Pref. level (0x%.8x)"),      N_("Preference level"),        32,  nd_icmp_adv_pref_cb },
  { 0, NULL, NULL, 0, NULL }
};

ND_MenuData icmp_menu_type_data[] = {
  { "Echo reply", NULL,                           ICMP_ECHOREPLY,      nd_icmp_type_value_cb },
  { "Destination unreachable", NULL,              ICMP_DEST_UNREACH,   nd_icmp_type_value_cb },
  { "Source quench", NULL,                        ICMP_SOURCE_QUENCH,  nd_icmp_type_value_cb },
  { "Redirect", NULL,                             ICMP_REDIRECT,       nd_icmp_type_value_cb },
  { "Echo request", NULL,                         ICMP_ECHO,           nd_icmp_type_value_cb },
  { "Router advertisement", NULL,                 9,                   nd_icmp_type_value_cb },
  { "Router solicitation", NULL,                  10,                  nd_icmp_type_value_cb },
  { "Time exceeded", NULL,                        ICMP_TIME_EXCEEDED,  nd_icmp_type_value_cb },
  { "Parameter problem", NULL,                    ICMP_PARAMETERPROB,  nd_icmp_type_value_cb },
  { "Timestamp request", NULL,                    ICMP_TIMESTAMP,      nd_icmp_type_value_cb },
  { "Timestamp reply", NULL,                      ICMP_TIMESTAMPREPLY, nd_icmp_type_value_cb },
  { "Information request", NULL,                  ICMP_INFO_REQUEST,   nd_icmp_type_value_cb },
  { "Information reply", NULL,                    ICMP_INFO_REPLY,     nd_icmp_type_value_cb },
  { "Address request", NULL,                      ICMP_ADDRESS,        nd_icmp_type_value_cb },
  { "Address reply", NULL,                        ICMP_ADDRESSREPLY,   nd_icmp_type_value_cb },
  { "Custom",     N_("Custom type value"),        -1,                  nd_icmp_type_custom_cb },
  { NULL, NULL, 0, NULL}
};


ND_MenuData icmp_menu_unreach_code_data[] = {
  { "Network unreachable", NULL,                          ICMP_NET_UNREACH,    nd_icmp_code_value_cb },
  { "Host unreachable", NULL,                             ICMP_HOST_UNREACH,   nd_icmp_code_value_cb },
  { "Protocol unreachable", NULL,                         ICMP_PROT_UNREACH,   nd_icmp_code_value_cb },
  { "Port unreachable", NULL,                             ICMP_PORT_UNREACH,   nd_icmp_code_value_cb },
  { "Fragmentation needed", NULL,                         ICMP_FRAG_NEEDED,    nd_icmp_code_value_cb },
  { "Source route failed", NULL,                          ICMP_SR_FAILED,      nd_icmp_code_value_cb },
  { "Destination network unknown", NULL,                  ICMP_NET_UNKNOWN,    nd_icmp_code_value_cb },
  { "Destination host unknown", NULL,                     ICMP_HOST_UNKNOWN,   nd_icmp_code_value_cb },
  { "Destination host isolated", NULL,                    ICMP_HOST_ISOLATED,  nd_icmp_code_value_cb },
  { "Destination net administratively prohibited", NULL,  ICMP_NET_ANO,        nd_icmp_code_value_cb },
  { "Destination host administratively prohibited", NULL, ICMP_HOST_ANO,       nd_icmp_code_value_cb },
  { "Network unreachable for TOS", NULL,                  ICMP_NET_UNR_TOS,    nd_icmp_code_value_cb },
  { "Host unreachable for TOS", NULL,                     ICMP_HOST_UNR_TOS,   nd_icmp_code_value_cb },
  { "Communication prohibited by filtering", NULL,        ICMP_PKT_FILTERED,   nd_icmp_code_value_cb },
  { "Host precedence violated", NULL,                     ICMP_PREC_VIOLATION, nd_icmp_code_value_cb },
  { "Precedence cutoff", NULL,                            ICMP_PREC_CUTOFF,    nd_icmp_code_value_cb },
  { "Custom",     N_("Custom code value"),                -1,                  nd_icmp_code_custom_cb },
  { NULL, NULL, 0, NULL}
};

ND_MenuData icmp_menu_redirect_code_data[] = {
  { "Redirect for network", NULL,                   ICMP_REDIR_NET,      nd_icmp_code_value_cb },
  { "Redirect for host", NULL,                      ICMP_REDIR_HOST,     nd_icmp_code_value_cb },
  { "Redirect for TOS and network", NULL,           ICMP_REDIR_NETTOS,   nd_icmp_code_value_cb },
  { "Redirect for TOS and host", NULL,              ICMP_REDIR_HOSTTOS,  nd_icmp_code_value_cb },
  { "Custom",     N_("Custom code value"),          -1,                  nd_icmp_code_custom_cb },
  { NULL, NULL, 0, NULL}
};

ND_MenuData icmp_menu_time_ex_code_data[] = {
  { "TTL is zero during transit", NULL,             ICMP_EXC_TTL,        nd_icmp_code_value_cb },
  { "TTL is zero during reassembly", NULL,          ICMP_EXC_FRAGTIME,   nd_icmp_code_value_cb },
  { "Custom",     N_("Custom code value"),          -1,                  nd_icmp_code_custom_cb },
  { NULL, NULL, 0, NULL}
};

ND_MenuData icmp_menu_param_code_data[] = {
  { "IP header bad", NULL,                          0,                   nd_icmp_code_value_cb },
  { "Option missing", NULL,                         1,                   nd_icmp_code_value_cb },
  { "Custom",     N_("Custom code value"),          -1,                  nd_icmp_code_custom_cb },
  { NULL, NULL, 0, NULL}
};

ND_MenuData icmp_menu_data[] = {
  { N_("Fix Checksums"),  N_("Fixes the ICMP checksums of the current selection"), 0, nd_icmp_cksum_fix_cb },
  { NULL, NULL, 0, NULL}
};


static ND_Protocol *icmp;


static ND_Protocol *
icmp_get_ip(void)
{
  static ND_Protocol *ip = NULL;

  if (!ip)
    {
      ip = nd_proto_registry_find(ND_PROTO_LAYER_NET, 0x0800);
      
      if (ip->id == 1)
	ip = NULL;
    }

  return ip;
}


static struct ip *
icmp_get_last_ip_before_icmp(const ND_Packet *packet, struct icmphdr **icmphdr)
{
  ND_Protocol *ip;
  GList *l;
  ND_ProtoData *pd;
  struct ip *iphdr = NULL;

  if (!packet || !(ip = icmp_get_ip()))
    return NULL;

  for (l = packet->pd; l; l = g_list_next(l))
    {
      pd = (ND_ProtoData *) l->data;

      if (pd->inst.proto == icmp)
	{
	  if (icmphdr)
	    *icmphdr = (struct icmphdr*) pd->data;
	  break;
	}

      if (pd->inst.proto == ip)
	iphdr = (struct ip*) pd->data;
    }

  return iphdr;
}


/**
 * icmp_header_complete -
 *
 * checks whether the ICMP message is complete by looking at the
 * IP length field -- easier since ICMP messages can be of varying sizes.
 */
static gboolean
icmp_header_complete(const ND_Packet *packet)
{
  struct ip *iphdr = NULL;

  iphdr = icmp_get_last_ip_before_icmp(packet, NULL);
  if (!iphdr)
    return FALSE;

  return (((guchar*) iphdr) + ntohs(iphdr->ip_len) <= nd_packet_get_end(packet));
}




/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return _("ICMP Plugin");
}


const char *
description(void)
{
  return _("A plugin providing support for ICMPv4.\n");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return ("0.1.0");
}


ND_Protocol *
init(void)
{
  icmp = nd_proto_new("ICMP", ND_PROTO_LAYER_NET | ND_PROTO_LAYER_TRANS, IPPROTO_ICMP);
			     		
  icmp->create_gui      = nd_icmp_create_gui;
  icmp->set_gui         = nd_icmp_set_gui;
  icmp->init_packet     = nd_icmp_init_packet;
  icmp->header_complete = nd_icmp_header_complete;
  icmp->fix_packet      = nd_icmp_fix_packet;

  /* We're using a button table to display the protocol content,
     so we need to hook them in here: */
  icmp->fields = icmp_fields;

  icmp->header_width = 32;

  /* We provide a little menu that allows us to fix checksums. */
  icmp->menu = nd_gui_create_menu(icmp_menu_data);

  return icmp;
}


/* Protocol method implementations: ------------------------------------ */

GtkWidget *
nd_icmp_create_gui(ND_Trace *trace, ND_ProtoInfo *pinf)
{
  GtkWidget *table;

  table = nd_gui_proto_table_create(trace, pinf);

  return table;
}


void       
nd_icmp_set_gui(const ND_Packet *packet, ND_ProtoInfo *pinf)
{
  struct icmphdr *icmphdr;

  icmphdr = (struct icmphdr*) nd_packet_get_data(packet, icmp, 0);

  nd_icmp_set_gui_type(pinf, icmphdr);
  nd_icmp_set_gui_code(pinf, icmphdr);
  nd_icmp_set_gui_cksum(pinf, icmphdr, packet);

  if (nd_icmp_header_is_error(icmphdr))
    {
      nd_icmp_set_gui_data(pinf, icmphdr, packet);
      return;
    }

  switch (icmphdr->type)
    {
    case ICMP_ECHO:
    case ICMP_ECHOREPLY:
      nd_icmp_set_gui_echo(pinf, icmphdr, packet);
      break;

    case ICMP_TIMESTAMP:
    case ICMP_TIMESTAMPREPLY:
      nd_icmp_set_gui_timestamp(pinf, icmphdr, packet);
      break;

    case ICMP_INFO_REQUEST:
    case ICMP_INFO_REPLY:
      nd_icmp_set_gui_info(pinf, icmphdr, packet);
      break;

    case ICMP_ADDRESS:
    case ICMP_ADDRESSREPLY:
      nd_icmp_set_gui_addr(pinf, icmphdr, packet);
      break;

    case 9:
      nd_icmp_set_gui_router_adv(pinf, icmphdr, packet);
      break;

    case 10:
      nd_icmp_set_gui_data(pinf, icmphdr, packet);
      break;

    default:
    }
}


void       
nd_icmp_init_packet(ND_Packet *packet, guchar *data, guchar *data_end)
{
  ND_Protocol *ip;
  struct icmphdr *icmphdr;
  struct ip *iphdr;

  /* Cast the data pointer into your protocol's header */
  icmphdr = (struct icmphdr *) data;

  if (!icmp_header_complete(packet))
    {
      nd_raw_proto_get()->init_packet(packet, data, data_end);
      return;
    }

  nd_packet_add_proto_data(packet, icmp, data, data_end);  

  /* We'll need IP data below, make sure we can find the IP protocol.
   */
  if (! (ip = icmp_get_ip()))
    {
      D(("WARNING -- IP not found\n"));
      return;
    }

  /* For some ICMP messages we offer the hex editor to edit the data.
   * Otherwise, we do nothing, as ICMP is the innermost protocol. */

  if (nd_icmp_header_is_error(icmphdr))
    {      
      ip->init_packet(packet, data + 8, data_end);
      return;
    }
  
  iphdr = icmp_get_last_ip_before_icmp(packet, NULL);
  if (!iphdr)
    return;

  if (((guchar*) icmphdr) + 8 >= ((guchar*) iphdr) + ntohs(iphdr->ip_len))
    return;

  D(("Pointers: %p %p, difference: %u\n", 
     (((guchar*) icmphdr) + 8), (((guchar*) iphdr) + ntohs(iphdr->ip_len)),
     (((guchar*) iphdr) + ntohs(iphdr->ip_len)) - (((guchar*) icmphdr) + 8)));

  switch (icmphdr->type)
    {
      /* Echo requests and replies can have optional data that must
	 be echoed. we display that using the raw data plugin:
      */
    case ICMP_ECHO:
    case ICMP_ECHOREPLY:
      nd_raw_proto_get()->init_packet(packet, data + 8, data_end);
    }
}


gboolean
nd_icmp_header_complete(const ND_Packet *packet, guint nesting)
{
  if (!packet)
    return FALSE;

  return icmp_header_complete(packet);
  TOUCH(nesting);
}


gboolean
nd_icmp_message_complete(const ND_Packet *packet)
{
  struct icmphdr *icmphdr;
  struct ip *iphdr;

  if (!packet)
    return FALSE;

  icmphdr = (struct icmphdr *) nd_packet_get_data(packet, icmp, 0);

  if (!icmphdr)
    return FALSE;

  /* If it's an error message, check whether we have the full
   * datagram that contains the ICMP message.
   */
  if (nd_icmp_header_is_error(icmphdr))
    {
      iphdr = (struct ip *) ((guchar *) icmphdr + 8);
      
      return ((guchar *) iphdr + (iphdr->ip_hl << 2) + 8 <= nd_packet_get_end(packet));
    }

  /* For all other cases, check individually.
   */
  switch (icmphdr->type)
    {
    case ICMP_ECHO:
    case ICMP_ECHOREPLY:

      iphdr = icmp_get_last_ip_before_icmp(packet, NULL);
      if (!iphdr)
	return FALSE;

      return ((guchar *) iphdr + ntohs(iphdr->ip_len) <= nd_packet_get_end(packet));

    case ICMP_TIMESTAMP:
    case ICMP_TIMESTAMPREPLY:
      return ((guchar *) icmphdr + 20 <= nd_packet_get_end(packet));

    case ICMP_INFO_REQUEST:
    case ICMP_INFO_REPLY:
    case 10:
      return ((guchar *) icmphdr + 8 <= nd_packet_get_end(packet));

    case ICMP_ADDRESS:
    case ICMP_ADDRESSREPLY:
      return ((guchar *) icmphdr + 12 <= nd_packet_get_end(packet));

    case 9:
      {
	struct nd_icmphdr *icmphdr_adv = (struct nd_icmphdr *) icmphdr;

	return ((guchar *) icmphdr + 8 + icmphdr_adv->un.adv.num_addr * 8 <= nd_packet_get_end(packet));
      }

    default:
    }

  return FALSE;
}


guint16    
nd_icmp_checksum(const ND_Packet *packet)
{
  guint16         old_sum, sum;
  struct icmphdr *icmphdr;
  struct ip      *iphdr;

  
  iphdr = icmp_get_last_ip_before_icmp(packet, &icmphdr);
  
  old_sum = icmphdr->checksum;
  icmphdr->checksum = 0;
  sum = nd_misc_in_cksum((u_short *)icmphdr, ntohs(iphdr->ip_len) - (iphdr->ip_hl << 2), 0);
  icmphdr->checksum = old_sum;
  
  return sum;
}


gboolean   
nd_icmp_csum_correct(const ND_Packet *packet, guint16 *correct_sum)
{
  struct icmphdr *icmphdr;
  guint16 sum;

  if (!packet)
    return FALSE;

  icmphdr = (struct icmphdr *) nd_packet_get_data(packet, icmp, 0);
  
  sum = nd_icmp_checksum(packet);
  
  if (correct_sum)
    *correct_sum = sum;
  
  return (icmphdr->checksum == sum);
}


gboolean
nd_icmp_fix_packet(ND_Packet *packet, int index)
{
  struct icmphdr *icmphdr;
  guint16         correct_sum;
  
  if (!packet)
    return FALSE;
  
  if (!nd_icmp_csum_correct(packet, &correct_sum))
    {
      icmphdr = (struct icmphdr*) nd_packet_get_data(packet, icmp, 0);
      icmphdr->checksum = correct_sum;
      nd_packet_modified_at_index(packet, index);
      return TRUE;
    }
  
  return FALSE;
}

/* Misc helper stuff below --------------------------------------------- */

void
nd_icmp_set_gui_type(ND_ProtoInfo *pinf, struct icmphdr *icmphdr)
{
  nd_proto_field_set_for_menu(pinf, &icmp_fields[0], DATA_TO_PTR(icmphdr->type),
			      icmp_menu_type_data, "%u");
}


void
nd_icmp_set_gui_code(ND_ProtoInfo *pinf, struct icmphdr *icmphdr)
{
  char            s[MAXPATHLEN];
  
  switch (icmphdr->type)
    {
    case ICMP_DEST_UNREACH:
      nd_proto_field_set_for_menu(pinf, &icmp_fields[1], DATA_TO_PTR(icmphdr->code),
				  icmp_menu_unreach_code_data, "%u");
      break;
    case ICMP_REDIRECT:
      nd_proto_field_set_for_menu(pinf, &icmp_fields[1], DATA_TO_PTR(icmphdr->code),
				  icmp_menu_redirect_code_data, "%u");
      break;
    case ICMP_TIME_EXCEEDED:
      nd_proto_field_set_for_menu(pinf, &icmp_fields[1], DATA_TO_PTR(icmphdr->code),
				  icmp_menu_time_ex_code_data, "%u");
      break;
    case ICMP_PARAMETERPROB:
      nd_proto_field_set_for_menu(pinf, &icmp_fields[1], DATA_TO_PTR(icmphdr->code),
				  icmp_menu_param_code_data, "%u");
      break;
    default:
      g_snprintf(s, MAXPATHLEN, "%u", icmphdr->code);
      nd_proto_field_set(pinf, &icmp_fields[1], s);
    }
}


void
nd_icmp_set_gui_cksum(ND_ProtoInfo *pinf,
		      struct icmphdr *icmphdr,
		      const ND_Packet *packet)
{
  nd_proto_field_set(pinf, &icmp_fields[2], DATA_TO_PTR(ntohs(icmphdr->checksum)));

  if (!nd_icmp_message_complete(packet))
    {
      nd_proto_info_field_set_state(pinf, 
				    &icmp_fields[2],
				    ND_FIELD_STATE_UNKN);
      return;
    }
  
  if (!nd_icmp_csum_correct(packet, NULL))
    nd_proto_info_field_set_state(pinf, 
				  &icmp_fields[2],
				  ND_FIELD_STATE_ERROR);
  else
    nd_proto_info_field_set_state(pinf,
				  &icmp_fields[2],
				  ND_FIELD_STATE_NORMAL);
}


void
nd_icmp_set_gui_data(ND_ProtoInfo *pinf,
		     struct icmphdr *icmphdr,
		     const ND_Packet *packet)
{
  char            s[MAXPATHLEN];
  guchar         *data;

  nd_gui_proto_table_clear(packet->trace, pinf);
  
  data = ((guchar*) icmphdr) + 4;
  g_snprintf(s, MAXPATHLEN, "%.2x:%.2x:%.2x:%.2x", data[0], data[1], data[2], data[3]);	     
  nd_gui_proto_table_add(packet->trace, pinf, &icmp_error_fields[0], s, FALSE);
}


void
nd_icmp_set_gui_echo(ND_ProtoInfo *pinf,
		     struct icmphdr *icmphdr,
		     const ND_Packet *packet)
{
  nd_gui_proto_table_clear(packet->trace, pinf);

  nd_gui_proto_table_add(packet->trace, pinf, &icmp_req_seq_fields[0], DATA_TO_PTR(ntohs(icmphdr->un.echo.id)), FALSE);
  nd_gui_proto_table_add(packet->trace, pinf, &icmp_req_seq_fields[1], DATA_TO_PTR(ntohs(icmphdr->un.echo.sequence)), FALSE);  
}


void
nd_icmp_set_gui_timestamp(ND_ProtoInfo *pinf,
			  struct icmphdr *icmphdr,
			  const ND_Packet *packet)
{
  struct nd_icmphdr *icmphdr_ts = (struct nd_icmphdr *) icmphdr;

  nd_gui_proto_table_clear(packet->trace, pinf);

  nd_gui_proto_table_add(packet->trace, pinf, &icmp_req_seq_fields[0], DATA_TO_PTR(ntohs(icmphdr->un.echo.id)), FALSE);
  nd_gui_proto_table_add(packet->trace, pinf, &icmp_req_seq_fields[1], DATA_TO_PTR(ntohs(icmphdr->un.echo.sequence)), FALSE);  

  nd_gui_proto_table_add(packet->trace, pinf, &icmp_tstamp_fields[0],
			 DATA_TO_PTR(ntohl(icmphdr_ts->un.ts.orig_ts)), FALSE);

  nd_gui_proto_table_add(packet->trace, pinf, &icmp_tstamp_fields[1],
			 DATA_TO_PTR(ntohl(icmphdr_ts->un.ts.recv_ts)), FALSE);  

  nd_gui_proto_table_add(packet->trace, pinf, &icmp_tstamp_fields[2],
			 DATA_TO_PTR(ntohl(icmphdr_ts->un.ts.trans_ts)), FALSE);    
}


void
nd_icmp_set_gui_info(ND_ProtoInfo *pinf,
		     struct icmphdr *icmphdr,
		     const ND_Packet *packet)
{
  nd_gui_proto_table_clear(packet->trace, pinf);

  nd_gui_proto_table_add(packet->trace, pinf, &icmp_req_seq_fields[0], DATA_TO_PTR(ntohs(icmphdr->un.echo.id)), FALSE);
  nd_gui_proto_table_add(packet->trace, pinf, &icmp_req_seq_fields[1], DATA_TO_PTR(ntohs(icmphdr->un.echo.sequence)), FALSE);  
}


void
nd_icmp_set_gui_addr(ND_ProtoInfo *pinf,
		     struct icmphdr *icmphdr,
		     const ND_Packet *packet)
{
  struct nd_icmphdr *icmphdr_in = (struct nd_icmphdr *) icmphdr;
  
  nd_gui_proto_table_clear(packet->trace, pinf);

  nd_gui_proto_table_add(packet->trace, pinf, &icmp_req_seq_fields[0], DATA_TO_PTR(ntohs(icmphdr->un.echo.id)), FALSE);
  nd_gui_proto_table_add(packet->trace, pinf, &icmp_req_seq_fields[1], DATA_TO_PTR(ntohs(icmphdr->un.echo.sequence)), FALSE);  

  nd_gui_proto_table_add(packet->trace, pinf, &icmp_addr_mask_fields[0],
			 inet_ntoa(icmphdr_in->un.in.in), FALSE);    
}


void       
nd_icmp_set_gui_router_adv(ND_ProtoInfo *pinf,
			   struct icmphdr *icmphdr,
			   const ND_Packet *packet)
{
  int i;
  struct nd_icmphdr *icmphdr_adv = (struct nd_icmphdr *) icmphdr;

  nd_gui_proto_table_clear(packet->trace, pinf);

  nd_gui_proto_table_add(packet->trace, pinf, &icmp_router_adv_fields[0], DATA_TO_PTR(icmphdr_adv->un.adv.num_addr), FALSE);
  nd_gui_proto_table_add(packet->trace, pinf, &icmp_router_adv_fields[1], DATA_TO_PTR(icmphdr_adv->un.adv.addr_size), FALSE);
  nd_gui_proto_table_add(packet->trace, pinf, &icmp_router_adv_fields[2], DATA_TO_PTR(ntohs(icmphdr_adv->un.adv.lifetime)), FALSE);

  for (i = 0; i < icmphdr_adv->un.adv.num_addr; i++)
    {
      guchar *data;
      struct in_addr *in;
      guint32 *prec;

      data = &icmphdr_adv->un.adv.addr_data + i * 8;

      if (data + 8 > nd_packet_get_end(packet))
	break;

      in = (struct in_addr *) data;
      prec = (guint32 *) (data + 4);

      nd_gui_proto_table_add(packet->trace, pinf, &icmp_router_adv_fields[3],
			     inet_ntoa(*in), FALSE);
      nd_gui_proto_table_add(packet->trace, pinf, &icmp_router_adv_fields[4],
			     DATA_TO_PTR(ntohl(*prec)), FALSE);
    }
}


ND_Protocol *
nd_icmp_get(void)
{
  return icmp;
}


gboolean
nd_icmp_header_is_error(struct icmphdr *icmphdr)
{
  if (!icmphdr)
    return FALSE;

  switch (icmphdr->type)
    {
    case ICMP_DEST_UNREACH:
    case ICMP_SOURCE_QUENCH:
    case ICMP_REDIRECT:
    case ICMP_TIME_EXCEEDED:
    case ICMP_PARAMETERPROB:
      return TRUE;
    }

  return FALSE;
}
