/*
 * PAM module for MySQL 
 *
 * Original Version written by: Gunay ARSLAN <arslan@gunes.medyatext.com.tr>
 * This version by: James O'Kane <jo2y@midnightlinux.com>
 * Modifications by Steve Brown, <steve@electronic.co.uk>
 *                  B.J. Black, <bj@taos.com>
 *                  Kyle Smith, <kyle@13th-floor.org>
 *
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <stdarg.h>
#ifdef __linux__
#include <alloca.h>
#endif
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>


/* AFAIK, only FreeBSD has MD5Data() defined in md5.h
 * better MD5 support will appear in 0.5
 */
#ifdef HAVE_MD5DATA
#include <md5.h>
#endif

#include <mysql/mysql.h>

/*
 * here, we make definitions for the externally accessible functions
 * in this file (these definitions are required for static modules
 * but strongly encouraged generally) they are used to instruct the
 * modules include file to define their prototypes.
 */

#define PAM_SM_AUTH
#define PAM_SM_ACCOUNT
#define PAM_SM_SESSION
#define PAM_SM_PASSWORD

#define PAM_MODULE_NAME  "pam_mysql"
#define PLEASE_ENTER_PASSWORD "Password:"
#define PLEASE_ENTER_OLD_PASSWORD "(Current) Password:"
#define PLEASE_ENTER_NEW_PASSWORD "(New) Password:"
#define PLEASE_REENTER_NEW_PASSWORD "Retype (New) Password:"
#define DEBUG 

#include <security/pam_modules.h>
#include <security/pam_misc.h>

struct optionstruct {
	char host[257];
	char where[257];
	char database[17];
	char dbuser[17];
	char dbpasswd[17];
	char table[17];
	char usercolumn[17];
	char passwdcolumn[17];
	int crypt;
	int md5;
	int sqllog;
	char logtable[17];
	char logmsgcolumn[17];
	char logpidcolumn[17];
	char logusercolumn[17];
	char loghostcolumn[17];
	char logtimecolumn[17];
};				/*Max length for most MySQL fields is 16 */


/* Global Variables */
MYSQL *mysql_auth = NULL;

struct optionstruct options = {
	"localhost",
	"",
	"mysql",
	"nobody",
	"",
	"user",
	"User",
	"Password",
	0,
	-1,
	-1,
	"",
	"",
	"",
	"",
	"",
	""
};

/* 
 * General PAM Prototypes 
 */
int converse (pam_handle_t * pamh,
	      int nargs,
	      struct pam_message **message,
	      struct pam_response **response);

int _set_auth_tok (pam_handle_t * pamh,
		   int flags, int argc,
		   const char **argv);

PAM_EXTERN int pam_sm_authenticate(pam_handle_t *pamh,
		int flags, int argc, const char **argv);
PAM_EXTERN int pam_sm_chauthtok(pam_handle_t *pamh, int flags, int argc,
		const char **argv);
PAM_EXTERN int pam_sm_acct_mgmt(pam_handle_t *pamh, int flags, int argc,
		const char **argv);
PAM_EXTERN int pam_sm_setcred(pam_handle_t *pamh, int flags, int argc,
		const char **argv);
PAM_EXTERN int pam_sm_open_session(pam_handle_t *pamh, int flags, int argc,
		const char **argv);
PAM_EXTERN int pam_sm_close_session(pam_handle_t *pamh, int flags, int argc,
		const char **argv);
/*
 * Internal function Prototypes
 */

int breakArgs( const char *in, char **lhs, char **rhs );
void parseArgs( int argc, const char **argv );
int askForPassword(pam_handle_t *pamh, int pwtype, const char *pwprompt);
void saltify( char *salt, const char *salter );


/*
 * MySQL Related Prototypes
 */

int db_connect (MYSQL * auth_sql_server);
void db_close( void );
int updatePasswd( MYSQL *my, const char *user, const char *oldpass, 
		const char *newpass, int isRoot );


/* 
 * SQL Logging Prototypes
 */

int sqlLog(MYSQL *auth_sql_server, const char *msg, const char *user);

/* breakArgs() breaks up a long string argument into its component chunks,
   accounting for escape chars and quoted strings as PAM doesn't (yet).
   It also looks for name-value pairs, so it probably still won't go away
   totally when PAM gets smarter :-) */
int breakArgs( const char *in, char **lhs, char **rhs ) {
	int i = 0;	/* array index for in */
	int j = 0;	/* array index for lhs and rhs */
	int k = 0;	/* array index for tmp */
	int dis = 0;	/* is the current character escaped */
	int quoteOn = 0; /* are we inside "" */
	int left = -1; /* FIXME document me */
	char *tmp;		/* FIXME document me */

	tmp = (char *) malloc(sizeof(char) * (strlen(in) + 1));
	
	while (in[i] != '\0') {
		switch( in[i] ) {
			case ' ':
				if (!quoteOn && !dis) {
					tmp[k] = '\0';
					if (left) {
						rhs[j] = tmp;
						rhs[j] = malloc(sizeof(char));
						rhs[j][0] = '\0';
					} else {
						rhs[j] = tmp;
					}
					j++;
					k = 0;
					tmp = (char *) malloc(sizeof(char) *
						(strlen(in) + 1));
					left = -1;
				} else {
					tmp[k++] = in[i];
					dis = 0;
				}
			break;;
			case '=':
				if (!quoteOn && !dis && left) {
					tmp[k] = '\0';
					lhs[j] = tmp;
					left = 0;
					tmp = (char *) malloc(sizeof(char) *
						(strlen(in) + 1));
					k = 0;
				} else {
					tmp[k++] = in[i];
					dis = 0;
				}
			break;;
			case '\\':
				if (dis)
					tmp[k++] = in[i];
				dis = !dis;
			break;;
			case '"':
				if (!dis) {
					quoteOn = !quoteOn;
				} else {
					tmp[k++] = in[i];
					dis = 0;
				}
			break;;
			default:
				tmp[k++] = in[i];
				dis = 0;
		}
		i++;
	}
	tmp[k] = '\0';
	if (left) {
		lhs[j] = tmp;
		rhs[j] = (char *) malloc(sizeof(char));
		rhs[j][0] = '\0';
	} else {
		rhs[j] = tmp;
	}
	return j+1;
}

void parseArgs(int argc, const char **argv) {
	int argc2, i, cc = 0;
	char **lhs, **rhs, *tmp;

	for (i=0; i<argc; i++) {
		cc += strlen(argv[i]) + 1;
	}
	
	tmp = (char *) malloc(sizeof(char) * cc);
	strcpy(tmp, argv[0]);
	for (i=1; i<argc; i++)
		sprintf (tmp, "%s %s", tmp, argv[i]);
	lhs = (char **) malloc(sizeof(char *) * (strlen(tmp) + 1));
	rhs = (char **) malloc(sizeof(char *) * (strlen(tmp) / 2 + 1));
	argc2 = breakArgs( tmp, lhs, rhs );

	/* Parse arguments taken from pam_listfile.c */
	for (i = 0; i < argc2; i++) {
		char mybuf[256], myval[256];

		if (rhs[i][0] != '\0') {
			strncpy (mybuf, lhs[i], 255);
			strncpy (myval, rhs[i], 255);
			if (!strcasecmp ("host", mybuf)) {
				strncpy (options.host, myval, 255);
#ifdef DEBUG
				syslog(LOG_ERR, "host changed.");
#endif
			} else if (!strcasecmp ("where", mybuf)) {
				strncpy (options.where, myval, 256);
#ifdef DEBUG
				syslog(LOG_ERR, "where changed.");
#endif
			} else if (!strcasecmp ("db", mybuf)) {
				strncpy (options.database, myval, 16);
#ifdef DEBUG
				syslog(LOG_ERR, "database changed.");
#endif
			} else if (!strcasecmp ("user", mybuf)) {
				strncpy (options.dbuser, myval, 16);
#ifdef DEBUG
				syslog(LOG_ERR, "dbuser changed.");
#endif
			} else if (!strcasecmp ("passwd", mybuf)) {
				strncpy (options.dbpasswd, myval, 16);
#ifdef DEBUG
				syslog(LOG_ERR, "dbpasswd changed.");
#endif
			} else if (!strcasecmp ("table", mybuf)) {
				strncpy (options.table, myval, 16);
#ifdef DEBUG
				syslog(LOG_ERR, "table changed.");
#endif
			} else if (!strcasecmp ("usercolumn", mybuf)) {
				strncpy (options.usercolumn, myval, 16);
#ifdef DEBUG
				syslog(LOG_ERR, "usercolumn changed.");
#endif
			} else if (!strcasecmp ("passwdcolumn", mybuf)) {
				strncpy (options.passwdcolumn, myval, 16);
#ifdef DEBUG
				syslog(LOG_ERR, "passwdcolumn changed.");
#endif
			} else if (!strcasecmp ("crypt", mybuf)) {
				if ((!strcmp (myval, "1")) ||
					(!strcasecmp (myval, "Y"))) {
						options.crypt = 1;
				} else if ((!strcmp(myval, "2")) ||
					(!strcasecmp(myval, "mysql"))) {
						options.crypt = 2;
#ifdef HAVE_MD5DATA
				} else if ((!strcmp(myval, "3")) ||
					(!strcasecmp(myval, "MD5"))) {
						options.crypt = 3;
#endif
				} else {
					options.crypt = 0;
				}
#ifdef DEBUG
				syslog(LOG_ERR, "crypt changed.");
#endif
			} else if (!strcasecmp ("md5", mybuf)) {
				if ((!strcmp (myval, "0")) ||
					(!strcasecmp (myval, "N"))) {
						options.md5 = 0;
				} else {
						options.md5 = -1;
				}
#ifdef DEBUG
				syslog(LOG_ERR, "md5 changed.");
#endif
			} else if (!strcasecmp("sqllog", mybuf)){
				if ((!strcmp(myval, "0")) || (!strcasecmp(myval, "N"))) {
					options.sqllog = 0;
				} else {
					options.sqllog = -1;
				}
#ifdef DEBUG
				syslog(LOG_ERR, "sqllog changed.");
#endif
			} else if (!strcasecmp("logtable", mybuf)) {
				strncpy(options.logtable, myval, 16);
#ifdef DEBUG
				syslog(LOG_ERR, "logtable changed.");
#endif
			} else if (!strcasecmp("logpidcolumn", mybuf)) {
				strncpy(options.logpidcolumn, myval, 16);
#ifdef DEBUG
				syslog(LOG_ERR, "logpidcolumn changed.");
#endif
			} else if (!strcasecmp("logmsgcolumn", mybuf)) {
				strncpy(options.logmsgcolumn, myval, 16);
#ifdef DEBUG
				syslog(LOG_ERR, "logmsgcolumn changed.");
#endif
			} else if (!strcasecmp("logusercolumn", mybuf)) {
				strncpy(options.logusercolumn, myval, 16);
#ifdef DEBUG
				syslog(LOG_ERR, "logusercolumn changed.");
#endif
			} else if (!strcasecmp("loghostcolumn", mybuf)) {
				strncpy(options.loghostcolumn, myval, 16);
#ifdef DEBUG
				syslog(LOG_ERR, "loghostcolumn changed.");
#endif
			} else if (!strcasecmp("logtimecolumn", mybuf)) {
				strncpy(options.logtimecolumn, myval, 16);
#ifdef DEBUG
				syslog(LOG_ERR, "logtimecolumn changed.");
#endif
			}else {
#ifdef DEBUG
				syslog(LOG_ERR, "Unknown option: %s=%s", mybuf, myval);
#endif
			}
		} else {
			char *error;
			error = (char *) malloc(sizeof(char)
				* (strlen(lhs[i])+14));
			if ( error )
			{
				sprintf (error, "Unknown option: %s", lhs[i]);
#ifdef DEBUG
				syslog(LOG_ERR, "%s", error);
#endif
			}
		}
	}/* for loop */
	for (i=0; i<argc2; i++) {
		free (lhs[i]);
		free (rhs[i]);
	}
	free (lhs);
	free (rhs);
}


/* MySQL access functions */
int db_connect (MYSQL *auth_sql_server) {
	int retvalue = PAM_AUTH_ERR;

#ifdef DEBUG
	syslog(LOG_ERR, "db_connect  called.");
#endif

	if (mysql_auth != NULL)
		return PAM_SUCCESS;

	mysql_init(auth_sql_server);
	mysql_auth = mysql_real_connect(auth_sql_server, options.host,
	                                options.dbuser, options.dbpasswd,
	                                options.database, 0, NULL, 0);

	if (mysql_auth != NULL)
		if (!mysql_select_db(auth_sql_server, options.database))
			retvalue = PAM_SUCCESS;

	if (retvalue != PAM_SUCCESS)
		syslog(LOG_INFO, "pam_mysql: MySQL err %s\n", mysql_error(auth_sql_server));

#ifdef DEBUG	
	syslog(LOG_ERR, "returning %i .",retvalue);
#endif

	return retvalue;
}

void db_close (void) {
	if (mysql_auth == NULL)
		return; /* closed already */

	mysql_close(mysql_auth);
	mysql_auth = NULL;
}

static int db_checkpasswd (MYSQL *auth_sql_server, const char *user, const char *passwd)
{
	char *sql = NULL;		/* Buffer for SQL query */
	int querysize = 0;		/* Calculated size of query */
	char *escapeUser = NULL;	/* User provided stuff MUST be escaped */
	char *encryptedPass = NULL;	/* Buffer for encrypted password */
	char *salt = NULL;		/* Buffer for salt */
#ifdef HAVE_MD5DATA
	char *md5buf = NULL;		/* temp buffer for MD5Data() */
#endif
	MYSQL_RES *result;
	MYSQL_ROW row;
	int retvalue = PAM_AUTH_ERR;
	int i;

#ifdef DEBUG
	syslog(LOG_ERR, "db_checkpasswd called.");
#endif

	/* To avoid putting a plain password in the MySQL log file and on the wire
	   more than needed we will request the encrypted password from MySQL. We
	   will check encrypt the passed password against the one returned from MySQL.
	*/

	escapeUser = malloc(sizeof(char) * (strlen(user) * 2) + 1);

	if (escapeUser == NULL) {
		syslog(LOG_ERR, "%s", "pam_mysql: Insufficient memory to allocate user escape string");
		return PAM_BUF_ERR;
	}

#ifdef HAVE_MYSQL_REAL_ESCAPE_STRING
	mysql_real_escape_string(auth_sql_server, escapeUser, user, strlen(user));
#else
	mysql_escape_string(escapeUser, user, strlen(user));
#endif	   

	querysize = strlen("select  from  where  ='' and ='' and ()") + 
		strlen(options.passwdcolumn) + 
		strlen(options.table) +
		strlen(options.usercolumn) + 
		strlen(escapeUser) +
		strlen(options.where);

	sql = malloc(sizeof(char) * querysize);

	if (sql == NULL)
		return PAM_BUF_ERR;

	snprintf(sql, querysize, "SELECT %s FROM %s WHERE %s='%s'",
		options.passwdcolumn, options.table,
		options.usercolumn, escapeUser);

	/* escapeUser is no longer needed */
	free(escapeUser);

	if (strlen(options.where) > 0){
		strncat(sql, " AND (", (querysize - strlen(sql)));
		strncat(sql, options.where, (querysize - strlen(sql)));
		strncat(sql, ")", (querysize - strlen(sql)));
	}


#ifdef DEBUG
	syslog(LOG_ERR, "pam_mysql: where clause = %s", options.where);
#endif

#ifdef DEBUG
/* These lines are commented even though its inside a ifdef DEBUG
  Why? Because it exposes user names and passwords into syslog and is
  a MASSIVE security hole. Uncomment only if you REALLY need to see
  the query. Remember, usernames and attmpted passwords will be
  exposed into syslog though!
*/
/*
	syslog(LOG_ERR, "pam_mysql: %s", sql);
*/
	syslog(LOG_ERR, "%s", sql);
	
#endif

	mysql_query(auth_sql_server, sql);

	free(sql);

	result = mysql_store_result(auth_sql_server);

	if (result == NULL) {
		syslog(LOG_ERR, "%s", mysql_error(auth_sql_server));
		mysql_free_result(result);
		return PAM_AUTH_ERR;
	}

	if (mysql_num_rows(result) != 1) {
		syslog(LOG_ERR, "%s", "pam_mysql: select returned more than one result");
		mysql_free_result(result);
		return PAM_AUTH_ERR;
	}

	/* Grab the password from RESULT_SET. */
	row = mysql_fetch_row(result);

	if (row == NULL) {
		syslog(LOG_ERR, "%s", mysql_error(auth_sql_server));
		return PAM_AUTH_ERR;
	}

	/* I really wish someone would explain how this was decided upon. */
	encryptedPass = malloc(sizeof(char) * (strlen(passwd) + 31 + 1));

	if (encryptedPass == NULL) {
		syslog(LOG_ERR, "%s", "pam_mysql: Insufficient memory to allocate encrypted password");
		return PAM_BUF_ERR;
	}

	switch (options.crypt) {
		/* PLAIN */
		case 0: strcpy(encryptedPass, passwd);
			break;
		/* ENCRYPT */
		case 1: if (strlen(row[0]) < 12) {
				/* strlen < 12 isn't a valid encrypted password. */
				syslog(LOG_ERR, "%s", "pam_mysql: select returned an invalid encrypted password");
				break;
			}

			salt = malloc(sizeof(char) * strlen(row[0]) + 1);

			if (salt == NULL) {
				syslog(LOG_ERR, "%s", "pam_mysql: Insufficient memory to allocate salt");
				return PAM_BUF_ERR;
			}

			if (strncmp("$1$", row[0], 3) == 0) {
				/* A MD5 salt starts with "$1$" and is 12 bytes long */
				strncpy(salt, row[0], 12);
				salt[12] = '\0';
			} else {
				/* If it's not MD5, assume DES and a 2 byte salt.  */
				strncpy(salt, row[0], 2);
				salt[2] = '\0';
			}

			strcpy(encryptedPass, crypt(passwd, salt));
			free(salt);
			break;

		/* PASSWORD */
		case 2: make_scrambled_password(encryptedPass, passwd);
			break;

#ifdef HAVE_MD5DATA
		/* MD5 hash (not MD5 crypt()) */
		case 3: strcpy(encryptedPass, MD5Data(passwd, strlen(passwd), md5buf));
			if (md5buf != NULL)
				free(md5buf);
			break;
#endif /* HAVE_MD5DATA */
	}

	if (!strcmp(row[0], encryptedPass))
		retvalue = PAM_SUCCESS;

	if (retvalue == PAM_SUCCESS)
		sqlLog(auth_sql_server, "AUTH SUCCESSFUL", user);
	
	/* overwrite encryped password in memory with 0 */
	for (i = strlen(encryptedPass); i >= 0; i--)
		encryptedPass[i] = 0;

	free(encryptedPass);

#ifdef DEBUG
	syslog(LOG_ERR, "returning %i .",retvalue);
#endif
	mysql_free_result(result);
	return retvalue;
}


/* Global PAM functions stolen from other modules */


int converse(pam_handle_t *pamh, int nargs, struct pam_message **message,
             struct pam_response **response)
{
	int retval;
	struct pam_conv *conv;

	retval = pam_get_item(pamh, PAM_CONV, (const void **) &conv); 

	if (retval == PAM_SUCCESS) {
		retval = conv->conv(nargs,
		                    (const struct pam_message **) message,
		                    response, conv->appdata_ptr);
		if ((retval != PAM_SUCCESS) && (retval != PAM_CONV_AGAIN))
			syslog(LOG_DEBUG, "pam_mysql: conversation failure [%s]",
			       pam_strerror(pamh, retval));
	} else {
		syslog(LOG_ERR, "pam_mysql: couldn't obtain coversation function [%s]",
		       pam_strerror(pamh, retval));
	}

	return retval;	/* propagate error status */
}

/* Create a random salt for use with CRYPT() when changing passwords */
void saltify(char *salt, const char *salter)
{
	unsigned int i, j = 0, k = 0;
	static const char saltstr[] = 
		"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789./";

#ifdef DEBUG
	syslog(LOG_ERR, "Called.");
#endif

	if( salter != NULL )
		for( i=0; i<strlen(salter); i++ )
			j += salter[i];
	j = (unsigned int)((time(NULL) + j) % 65535);
	srand(j);
	if( options.md5 ) {
		strcpy(salt, "$1$");
		j = 3;
		k = 8;
	} else {
		j = 0;
		k = 2;
	}
	for( i=0; i<k; i++ )
		salt[i+j] = saltstr[rand() % 64];
	if( options.md5 )
		salt[k+(j++)] = '$';
	salt[j+k] = '\0';

#ifdef DEBUG
	syslog(LOG_ERR, "Returning with salt = %s", salt);
#endif
}

/* Update the password in MySQL */
/* To reduce the number of calls to the DB, I'm now assuming that the old
   password has been verified elsewhere, so I only check for null/not null
   and isRoot. */
int updatePasswd(MYSQL *my, const char *user, const char *oldpass,
	const char *newpass, int isRoot)
{
	char *escUser = NULL, *escNew = NULL, *salt;
	char *sql = NULL, *encNew = NULL, *tmp = NULL;
#ifdef HAVE_MD5DATA
	char *md5buf = NULL;
#endif
	long ret=PAM_AUTH_ERR;
#ifdef HORRIBLE_HACK_FOR_MYSQL_3_23_27
	char mem[1000];
#endif

#ifdef DEBUG
	syslog(LOG_ERR, "updatePasswd  called.");
#endif

	if ( user == NULL || newpass == NULL )
	{
		if ( oldpass == NULL && !isRoot )
			syslog(LOG_ERR, "%s",
			"pam_mysql: User, OldPass, or NewPass is NULL!");
		else
			syslog(LOG_ERR, "%s",
			"pam_mysql: User or NewPass is NULL!");
		syslog(LOG_ERR, "%s", "pam_mysql: UNABLE TO CHANGE PASSWORD");
		return PAM_BUF_ERR;
	}

	if (!isRoot && oldpass == NULL) {
		syslog(LOG_ERR,
			"Old password is null for %s.  PASSWORD NOT UPDATED!",
			user);
		return PAM_BUF_ERR;
	}

	switch(options.crypt) {
		case 0:
			encNew = malloc(sizeof(char) * (strlen(newpass)+1));
			sprintf(encNew, "%s", newpass);
		break;
		case 1:
			if(options.md5)
				salt = malloc(sizeof(char) * 17);
			else
				salt = malloc(sizeof(char) * 4);

			saltify(salt, NULL);
			tmp = crypt(newpass, salt);
			free(salt);
			encNew = malloc(sizeof(char) * (strlen(tmp) + 2));
			strncpy(encNew, tmp, strlen(tmp) + 1);
#ifdef DEBUG
			syslog(LOG_ERR, "encNew = %s", encNew);
#endif
			free(tmp);
		break;
		case 2:
			encNew = malloc(sizeof(char) * 20);
			make_scrambled_password(encNew, newpass);
		break;
#ifdef HAVE_MD5DATA
		case 3:
			md5buf = NULL;
			encNew = malloc(sizeof(char) * 
				(strlen(newpass) + 31 + 1));
			strcpy( encNew, MD5Data(newpass,
				strlen(newpass), md5buf));
			if (md5buf != NULL)
			{
				free(md5buf);
				md5buf = NULL;
			}
		break;
#endif
		default:
			encNew = malloc(sizeof('\0'));
			encNew[0]='\0';
	}

	escUser = malloc(sizeof(char) * (strlen(user) * 2) + 1);
	escNew = malloc(sizeof(char) * (strlen(encNew) * 2) + 1);

	if ( escUser == NULL || escNew == NULL )
	{
		syslog(LOG_ERR, "%s",
			"pam_mysql: Insufficient memory to allocate user or password escape strings");
		syslog(LOG_ERR, "%s", "pam_mysql: UNABLE TO CHANGE PASSWORD");
		return PAM_BUF_ERR;
	}

#ifdef HAVE_MYSQL_REAL_ESCAPE_STRING
	mysql_real_escape_string(my, escUser, user, strlen(user));
	mysql_real_escape_string(my, escNew, encNew, strlen(encNew));
#else
	mysql_escape_string(escUser, user, strlen(user));
	mysql_escape_string(escNew, encNew, strlen(encNew));
#endif
	free(encNew);

	sql = malloc(sizeof(char) *
		(strlen("update  set  = '' where  = ''") +
		strlen(options.table) + strlen(options.passwdcolumn) +
		strlen(escNew) + strlen(options.usercolumn) +
		strlen(escUser)) + 2);
	sprintf(sql,
		"update %s set %s = '%s' where %s = '%s'",
		options.table, options.passwdcolumn, escNew,
		options.usercolumn, escUser);

	free(escUser);
	free(escNew);

	if (mysql_query(my, sql))
	{
		syslog (LOG_ERR, "pam_mysql: Query Error \"%s\"",
			mysql_error(my));
		ret = PAM_AUTH_ERR;
	}
	else
		ret = PAM_SUCCESS;

	free (sql);
	return ret;
}

int askForPassword(pam_handle_t *pamh, int pwtype, const char *pwprompt)
{
	struct pam_message msg[1], *mesg[1];
	struct pam_response *resp = NULL;
	char *prompt = NULL;
	int i = 0;
	int retval;

	prompt = malloc(strlen(pwprompt));

	if (prompt == NULL) {
		syslog(LOG_ERR, "%s", "pam_mysql: askForPassword(), out of memory!?");
		return PAM_BUF_ERR;
	} else {
		sprintf(prompt, pwprompt);
		msg[i].msg = prompt;
	}

	msg[i].msg_style = PAM_PROMPT_ECHO_OFF;
	mesg[i] = &msg[i];

	retval = converse(pamh, ++i, mesg, &resp);

	if (prompt) {
		_pam_overwrite(prompt);
		_pam_drop(prompt);
	}

	if (retval != PAM_SUCCESS) {
		if (resp != NULL)
			_pam_drop_reply(resp,i);
		return ((retval == PAM_CONV_AGAIN)
		        ? PAM_INCOMPLETE : PAM_AUTHINFO_UNAVAIL);
	}

	/* we have a password so set AUTHTOK
	 */
	return pam_set_item(pamh, pwtype, resp->resp);
}

int sqlLog(MYSQL * auth_sql_server, const char *msg, const char *user) {
	char *sql = NULL;   /* Buffer for SQL query */
	char *escapeUser = NULL;  /* User provided stuff MUST be escaped */
	char *escapeMsg = NULL;   /* Escape message just to be safe */
	char *remote_host = NULL; /* Buffer for remote hostname */
	pid_t pid;      /* process id */
	pid_t tmppid;     /* hold pid while calculating length */
	int pidlen = 2;     /* length of process id as a string */
	socklen_t salen;    /* sizeof(socaddr_in) */
	struct sockaddr_in remoteaddr;  /* struct to get remote host info */
	int retval = PAM_SUCCESS; /* return value */
	
#ifdef DEBUG
	syslog(LOG_ERR, "sqlLog called.");
#endif
	
	if (!options.sqllog)
		return retval;

	if (!strcmp(options.logtable, "")) {
		syslog(LOG_ERR, "%s",
				"pam_mysql: error: sqllog set but logtable not set");
		retval = PAM_AUTH_ERR;
	}
	if (!strcmp(options.logmsgcolumn, "")) {
		syslog(LOG_ERR, "%s",
				"pam_mysql: error: sqllog set but logmsgcolumn not set");
		retval = PAM_AUTH_ERR;
	}
	if (!strcmp(options.logusercolumn, "")) {
			syslog(LOG_ERR, "%s",
					"pam_mysql: error: sqllog set but logusercolumn not set");
			retval = PAM_AUTH_ERR;
	}
	if (!strcmp(options.loghostcolumn, "")) {
		syslog(LOG_ERR, "%s",
				"pam_mysql: error: sqllog set but loghostcolumn not set");
		retval = PAM_AUTH_ERR;
	}
	if (!strcmp(options.logtimecolumn, "")) {
		syslog(LOG_ERR, "%s",
				"pam_mysql: error: sqllog set but logtimecolumn not set");
		retval = PAM_AUTH_ERR;
	}
	if (retval != PAM_SUCCESS)
		return retval;

	escapeUser = malloc(sizeof(char) * (strlen(user) * 2 + 1));
	if (escapeUser == NULL) {
		syslog(LOG_ERR, "%s",
				"pam_mysql: Insufficient memory to allocate user escape string");
		return PAM_BUF_ERR;
	}

	escapeMsg = malloc(sizeof(char) * (strlen(msg) * 2 + 1));

	if (escapeMsg == NULL) {
		syslog(LOG_ERR, "%s",
				"pam_mysql: Insufficient memory to allocate message escape string");
		free(escapeUser);
		return PAM_BUF_ERR;
	}
#ifdef HAVE_MYSQL_REAL_ESCAPE_STRING
	mysql_real_escape_string(auth_sql_server, escapeUser, user,
			strlen(user));
	mysql_real_escape_string(auth_sql_server, escapeMsg, msg, strlen(msg));
#else
	mysql_escape_string(escapeUser, user, strlen(user));
	mysql_escape_string(escapeMsg, msg, strlen(msg));
#endif

	/* get the remote host the hard way */
	salen = sizeof(remoteaddr);
	if (getpeername(0, (struct sockaddr *) &remoteaddr, &salen) == 0 &&
			remoteaddr.sin_family == AF_INET)
		remote_host = strdup(inet_ntoa(remoteaddr.sin_addr));
	else
		remote_host = strdup("");

	if (remote_host == NULL) {
		syslog(LOG_ERR, "%s",
				"pam_mysql: Insufficient memory to allocate remote host string");
		free(escapeUser);
		free(escapeMsg);
		return PAM_BUF_ERR;
	}

	pid = getpid();

	for (tmppid = pid; tmppid > 10; tmppid /= 10)
		pidlen++;

	sql = malloc(sizeof(char) * (strlen("insert into %s (%s, %s, %s, %s, %s) values('%s', '%s', '%s', '%i', NOW())") + 1 + strlen(options.logtable) +
				strlen(options.logmsgcolumn) + strlen(options.logusercolumn) +
				strlen(options.loghostcolumn) + strlen(options.logtimecolumn) +
				strlen(escapeUser) + strlen(remote_host) + pidlen));

	if (sql == NULL) {
		syslog(LOG_ERR, "%s",
				"pam_mysql: Insufficient memory to allocate SQL buffer");
		free(remote_host);
		free(escapeUser);
		free(escapeMsg);
		return PAM_BUF_ERR;
	}

	sprintf(sql,
			"insert into %s (%s, %s, %s, %s, %s) values('%s', '%s', '%s', '%i', NOW())",
			options.logtable, options.logmsgcolumn, options.logusercolumn,
			options.loghostcolumn, options.logpidcolumn,
			options.logtimecolumn, escapeMsg, escapeUser, remote_host, pid);

	syslog(LOG_ERR, "%s", sql);
	retval = mysql_real_query(auth_sql_server, sql, strlen(sql));

	free(sql);  
	free(remote_host);
	free(escapeUser);
	free(escapeMsg);

	retval = (retval == 0) ? PAM_SUCCESS : PAM_AUTH_ERR;

	if (retval != PAM_SUCCESS)
		syslog(LOG_INFO, "pam_mysql: MySQL err %s\n",
				mysql_error(auth_sql_server));

#ifdef DEBUG
	syslog(LOG_ERR, "Returning %i", retval);
#endif

	return retval;
}


		 
			

/* PAM Authentication functions */

PAM_EXTERN int pam_sm_authenticate (pam_handle_t * pamh,
				    int flags,
				    int argc,
				    const char **argv)
{
	int retval;
	const char *user;
	char *passwd = NULL;
#ifdef HORRIBLE_HACK_FOR_MYSQL_3_23_27
	char mem[1000];
#endif
	MYSQL auth_sql_server;

#ifdef DEBUG
	syslog(LOG_ERR, "pam_sm_authenticate called.");
#endif

	parseArgs(argc, argv);

	/* Get User */

	retval = pam_get_user(pamh, &user, NULL);

	if (retval != PAM_SUCCESS || user == NULL) {
		syslog(LOG_ERR, "%s", "pam_mysql: no user specified");
#ifdef DEBUG
		syslog(LOG_ERR, "returning.");
#endif
		return PAM_USER_UNKNOWN;
	} 
	
	retval = pam_get_item(pamh, PAM_AUTHTOK, (const void **) &passwd);

	if (passwd == NULL)
		askForPassword(pamh, PAM_AUTHTOK, PLEASE_ENTER_PASSWORD);

	retval = pam_get_item(pamh, PAM_AUTHTOK, (const void **)&passwd);

	if (passwd == NULL)
		return PAM_AUTHINFO_UNAVAIL;
		
	if ((retval = db_connect(&auth_sql_server)) != PAM_SUCCESS) {
		db_close();
#ifdef DEBUG
		syslog(LOG_ERR, "returning %i after db_connect.",retval);
#endif
		return retval;
	}

	if ((retval = db_checkpasswd(&auth_sql_server, user, passwd)) != PAM_SUCCESS) {
#ifdef DEBUG
		syslog(LOG_ERR, "returning %i after db_checkpasswd.",retval);
#endif
		db_close();
		return retval;
	}

#ifdef DEBUG
	syslog(LOG_ERR, "returning %i.",retval);
#endif
	db_close();
	return retval;

} /* pam_sm_authenticate */


/* --- account management functions --- */
PAM_EXTERN int pam_sm_acct_mgmt (pam_handle_t * pamh, int flags, int argc
				 ,const char **argv)
{
#ifdef DEBUG
	syslog (LOG_INFO, "%s", "pam_mysql: acct_mgmt called but not implemented. Dont panic though :)");
#endif
	return PAM_SUCCESS;
}

PAM_EXTERN
int pam_sm_setcred(pam_handle_t *pamh,int flags,int argc
		   ,const char **argv)
{
#ifdef DEBUG
	syslog(LOG_INFO, "%s", "pam_mysql: setcred called but not implemented.");
#endif
	return PAM_SUCCESS;
}

/* --- password management --- */

PAM_EXTERN
int pam_sm_chauthtok(pam_handle_t *pamh,int flags,int argc
		     ,const char **argv)
{
	int retval;
	const char *user, *oldpass, *newpass;
	char *sTmp;

#ifdef HORRIBLE_HACK_FOR_MYSQL_3_23_27
	char mem[1000];
#endif
	MYSQL auth_sql_server;

#ifdef DEBUG
	syslog(LOG_ERR, "pam_sm_chauthtok called.");
#endif

	parseArgs(argc, argv);

	/* Get User */

	retval = pam_get_user (pamh, &user, NULL);
	if (retval != PAM_SUCCESS || user == NULL) {
		syslog (LOG_ERR, "%s", "pam_mysql: no user specified");
#ifdef DEBUG
		syslog(LOG_ERR, "returning.");
#endif
		return PAM_USER_UNKNOWN;
	}

#ifdef DEBUG
	syslog(LOG_ERR, "%s", "pam_mysql: in pam_sm_chauthtok()");
#endif

	if ((retval = db_connect (&auth_sql_server)) != PAM_SUCCESS) {
		db_close();
		return retval;
	}
	if( flags & PAM_PRELIM_CHECK )
	{
#ifdef DEBUG
		syslog(LOG_ERR, "PAM_PRELIM_CHECK");
#endif
		if ( getuid() == 0 && !(flags & PAM_CHANGE_EXPIRED_AUTHTOK) )
			retval = PAM_SUCCESS;
		else {
			retval = pam_get_item(pamh, PAM_OLDAUTHTOK,
				(const void **) &oldpass);
			if (oldpass == NULL )
			{
				askForPassword(pamh, PAM_OLDAUTHTOK,
					PLEASE_ENTER_OLD_PASSWORD);
			}
			retval = pam_get_item(pamh, PAM_OLDAUTHTOK,
				(const void **) &oldpass);
			if (oldpass == NULL )
				retval = PAM_AUTHTOK_ERR;
			else
			{
				if (!db_checkpasswd(&auth_sql_server,
					user, oldpass))
					retval = PAM_SUCCESS;
				else
					retval = PAM_AUTHTOK_ERR;
			}
		}
		db_close();
#ifdef DEBUG
		if (retval==PAM_SUCCESS)
			syslog(LOG_ERR, "Returning PAM_SUCCESS");
		else
			syslog(LOG_ERR, "Returning %d", retval);
#endif
		return retval;
	}
	if( flags & PAM_UPDATE_AUTHTOK )
	{
#ifdef DEBUG
		syslog(LOG_ERR, "PAM_UPDATE_AUTHTOK");
#endif
		retval = pam_get_item(pamh, PAM_AUTHTOK,
			(const void **) &newpass);
		if( retval )
			return PAM_AUTHTOK_ERR;

		if( newpass == NULL ) {
#ifdef DEBUG
			syslog(LOG_ERR, "Asking for new password(1)");
#endif
			askForPassword(pamh, PAM_AUTHTOK,
				PLEASE_ENTER_NEW_PASSWORD);
			retval = pam_get_item(pamh, PAM_AUTHTOK,
				(const void **) &newpass);
			if( retval )
				return PAM_AUTHTOK_ERR;
			sTmp = (char *) malloc(sizeof(char) *
				(strlen(newpass) + 2));
			strncpy (sTmp, newpass, strlen(newpass) + 2);
#ifdef DEBUG
			syslog(LOG_ERR, "Asking for new password(1)");
#endif
			/* Can anybody see a problem with doing it this
			   way?  I can't, but that doesn't mean that it
			   won't bite me later... */
			askForPassword(pamh, PAM_AUTHTOK,
				PLEASE_REENTER_NEW_PASSWORD);
			retval = pam_get_item(pamh, PAM_AUTHTOK,
				(const void **) &newpass);
			if( newpass == NULL || strcmp(sTmp, newpass) )
				return PAM_AUTHTOK_ERR;
			free(sTmp);
		}

		retval = pam_get_item(pamh, PAM_OLDAUTHTOK,
			(const void **) &oldpass);
		if( retval )
			return PAM_AUTHTOK_ERR;

#ifdef DEBUG
		syslog(LOG_ERR, "Calling updatePasswd(%s,%s)", oldpass, newpass);
#endif
		return updatePasswd(&auth_sql_server, user, oldpass, newpass,
			getuid() == 0 && !(flags & PAM_CHANGE_EXPIRED_AUTHTOK)
		);

	}
	db_close();
	return PAM_SUCCESS;
}

/* --- session management --- */

PAM_EXTERN
int pam_sm_open_session(pam_handle_t *pamh,int flags,int argc
			,const char **argv)
{
#ifdef DEBUG
	syslog(LOG_INFO, "%s", "pam_mysql: open_session called but not implemented.");
#endif
	return PAM_SUCCESS;
}

PAM_EXTERN
int pam_sm_close_session(pam_handle_t *pamh,int flags,int argc
			 ,const char **argv)
{
#ifdef DEBUG
	syslog(LOG_INFO, "%s", "pam_mysql: close_session called but not implemented.");
#endif
	return PAM_SUCCESS;
}

/* end of module definition */

#ifdef PAM_STATIC

/* static module data */

struct pam_module _pam_permit_modstruct = {
    "pam_permit",
    pam_sm_authenticate,
    pam_sm_setcred,
    pam_sm_acct_mgmt,
    pam_sm_open_session,
    pam_sm_close_session,
    pam_sm_chauthtok
};

#endif
