/* ----------------------------------------------------------------------------
 * configfile.c
 * functions to read or write the configfile.
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>
#include <sys/stat.h>

#include <pbb.h>

#include "input_manager.h"
#include "module_powersave.h"
#include "module_pmac.h"
#include "gettext_macros.h"
#include "configfile.h"

/* The options in the config file are not case sensitive. */

  static struct cftitem cft[] = {
    /* system */
    { 0, "userallowed",       TAG_USERALLOWED,        0,
	    cft_read_string,  cft_write_string,
	    N_("user who is allowed to use IPC") },
    { 0, "autorescan",        TAG_AUTORESCAN,         0,
	    cft_read_bool,    cft_write_bool,
	    N_("automatic rescan of event devices") },
    { 0, "CmdTimeout",        TAG_TIMEFORCMD,         0,
	    cft_read_int,     cft_write_int,
	    N_("timeout in seconds for launched scripts, etc.") },
    /* module powersave */
    { 1, "onAC_Policy",       TAG_ONAC_POLICY,          0,
 	    cft_read_policy,  cft_write_policy,
 	    N_("'"POLICY_NONE_NAME"', '"POLICY_POWERSAVE_NAME"', '"POLICY_USER_NAME"' or '"POLICY_PERFORMANCE_NAME"'") },
    { 1, "onAC_TimerAction",  TAG_ONAC_TIMERACTION,     0,
	    cft_read_action,  cft_write_action, 
	    N_("'"ACTION_NONE_NAME"', '"ACTION_TORAM_NAME"', '"ACTION_TODISK_NAME"' or '"ACTION_BLANK_NAME"'") },
    { 1, "onAC_CoverAction",  TAG_ONAC_COVERACTION,     0,
	    cft_read_action,  cft_write_action, 
	    N_("see TimerAction for possible values") },
    { 1, "onAC_KeyAction",    TAG_ONAC_KEYACTION,       0,
	    cft_read_action,  cft_write_action, 
	    N_("Action (see TimerAction) for the sleepkey") },
    { 1, "onAC_SuspendTime",  TAG_ONAC_TIMESUSPEND,     0,
	    cft_read_int,     cft_write_int,
	    NULL },
    { 1, "onAC_DimTime",      TAG_ONAC_TIMEDIM,         0,
	    cft_read_int,     cft_write_int,
	    NULL },
	    
    { 1, "onBattery_policy",  TAG_ONBATT_POLICY,        0,
	    cft_read_policy,  cft_write_policy,
	    N_("'"POLICY_NONE_NAME"', '"POLICY_POWERSAVE_NAME"', '"POLICY_USER_NAME"' or '"POLICY_PERFORMANCE_NAME"'") },
    { 1, "onBattery_TimerAction",TAG_ONBATT_TIMERACTION,0,
	    cft_read_action,  cft_write_action, 
	    N_("'"ACTION_NONE_NAME"', '"ACTION_TORAM_NAME"', '"ACTION_TODISK_NAME"' or '"ACTION_BLANK_NAME"'") },
    { 1, "onBattery_CoverAction",TAG_ONBATT_COVERACTION,0,
	    cft_read_action,  cft_write_action, 
	    N_("see TimerAction for possible values") },
    { 1, "onBattery_KeyAction",TAG_ONBATT_KEYACTION,    0,
	    cft_read_action,  cft_write_action, 
	    N_("Action (see TimerAction) for the sleepkey") },
    { 1, "onBattery_SuspendTime",TAG_ONBATT_TIMESUSPEND,0,
	    cft_read_int,     cft_write_int,
	    NULL },
    { 1, "onBattery_DimTime", TAG_ONBATT_TIMEDIM,       0,
	    cft_read_int,     cft_write_int,
	    NULL },

/* --- depreciated config options - read only --- */
    { 1, "onAC_sleep",        TAG_ONACSLEEP,          0,
	    cft_read_bool,    NULL,
	    N_("yes=sleep, no=blank after sleep timeout") },
    { 1, "onAC_coversleep",   TAG_ONACCOVERSLEEP,     0,
	    cft_read_bool,    NULL,
	    NULL },
    { 1, "onAC_Tsleep",       TAG_ONACTIMESLEEP,      0,
	    cft_read_int,     NULL,
	    NULL },
    { 1, "onAC_Tdim",         TAG_ONACTIMEDIM,        0,
	    cft_read_int,     NULL,
	    NULL },
    { 1, "onBattery_sleep",   TAG_ONBATTERYSLEEP,     0,
	    cft_read_bool,    NULL,
	    NULL },
    { 1, "onBattery_coversleep", TAG_ONBATTERYCOVERSLEEP, 0,
	    cft_read_bool,    NULL,
	    NULL },
    { 1, "onBattery_Tsleep",  TAG_ONBATTERYTIMESLEEP, 0,
	    cft_read_int,     NULL,
	    NULL },
    { 1, "onBattery_Tdim",    TAG_ONBATTERYTIMEDIM,   0,
	    cft_read_int,     NULL,
	    NULL },
    { 1, "SleepKeySleep",     TAG_SLEEPKEYSLEEP,       0,
	    cft_read_bool,    NULL,
	    NULL },
/* --- */

    { 1, "SleepKey",          TAG_SLEEPKEY,           TAG_SLEEPMOD,
	    cft_read_key,     cft_write_key,
	    NULL },
    { 1, "SleepKeyDelay",     TAG_SLEEPKEYDELAY,      0,
	    cft_read_int,     cft_write_int,
	    N_("values > 0 may be dangerous, if the power key is used to trigger sleep") },
    { 1, "BWL_first",         TAG_BWLFIRST,           0,
	    cft_read_int,     cft_write_int,
	    N_("first battery warnlevel, time in minutes") },
    { 1, "BWL_second",        TAG_BWLSECOND,          0,
	    cft_read_int,     cft_write_int,
	    N_("second battery warnlevel, time in minutes") },
    { 1, "BWL_last",          TAG_BWLLAST,            0,
	    cft_read_int,     cft_write_int,
	    N_("last battery warnlevel, time in minutes") },
    { 1, "Script_PMCS",       TAG_SCRIPTPMCS,  0,
	    cft_read_string,  cft_write_string,
	    NULL },
    { 1, "EmergencyAction",   TAG_EMERGENCYACTION,    0,
	    cft_read_ema,     cft_write_ema,
	    N_("action, if battery is critically low") },
    { 1, "HeartbeatBeep",     TAG_HEARTBEATBEEP,      0,
	    cft_read_bool,    cft_write_bool,
	    N_("beep, if nothing else showed that the computer lives") },
    /* sleeplocks */
    { 1, "CPULoad_sleeplock", TAG_CPULOADSLEEPLOCK,   0,
	    cft_read_bool,    cft_write_bool,
	    NULL },
    { 1, "CPULoad_min",       TAG_CPULOADMIN,         0,
	    cft_read_int,     cft_write_int,
	    N_("value in percent") },
    { 1, "CPULoad_period",    TAG_CPULOADPERIOD,      0,
	    cft_read_int,     cft_write_int,
	    N_("time in seconds") },
    { 1, "NETLoad_sleeplock", TAG_NETLOADSLEEPLOCK,   0,
	    cft_read_bool,    cft_write_bool,
	    NULL },
    { 1, "NETLoad_min",       TAG_NETLOADMIN,         0,
	    cft_read_int,     cft_write_int,
	    N_("traffic in Bytes/s") },
    { 1, "NETLoad_period",    TAG_NETLOADPERIOD,      0,
	    cft_read_int,     cft_write_int,
	    N_("time in seconds") },
    { 1, "NETLoad_device",    TAG_NETLOADDEV,         0,
	    cft_read_string,  cft_write_string,
	    NULL },
    /* module display */
    { 2, "LCD_Brightness",    TAG_LCDBRIGHTNESS,      0,
	    cft_read_int,     cft_write_int,
	    N_("initial LCD brightness level") },
    { 2, "LCD_FadingSpeed",   TAG_LCDFADINGSPEED,     0,
	    cft_read_int,     cft_write_int,
	    N_("0 = no smooth fading") },
    { 2, "LCD_AutoAdjust",    TAG_LCDAUTOADJUST,      0,
	    cft_read_bool,    cft_write_bool,
	    N_("only on Aluminum PowerBooks") },
    { 2, "LCD_IllumUpKey",    TAG_LCDILLUMUPKEY,      TAG_LCDILLUMUPMOD,
	    cft_read_key,     cft_write_key,
	    NULL },
    { 2, "LCD_IllumDownKey",  TAG_LCDILLUMDOWNKEY,    TAG_LCDILLUMDOWNMOD,
	    cft_read_key,     cft_write_key,
	    NULL },
    { 2, "LCD_Threshold",     TAG_LCDTHRESHOLD,       0,
	    cft_read_int,     cft_write_int,
	    N_("ambient light threshold in percent for backlight autoadj.") },
    { 2, "LCD_AutoAdjMin_Bat",TAG_LCDAUTOADJMINBAT,   0,
	    cft_read_int,     cft_write_int,
	    N_("autoadjustment range parameters") },
    { 2, "LCD_AutoAdjMax_Bat",TAG_LCDAUTOADJMAXBAT,   0,
	    cft_read_int,     cft_write_int,
	    NULL },
    { 2, "LCD_AutoAdjMin_AC", TAG_LCDAUTOADJMINAC,    0,
	    cft_read_int,     cft_write_int,
	    NULL },
    { 2, "LCD_AutoAdjMax_AC", TAG_LCDAUTOADJMAXAC,    0,
	    cft_read_int,     cft_write_int,
	    NULL },
    { 2, "KBD_Brightness",    TAG_KBDBRIGHTNESS,      0,
	    cft_read_int,     cft_write_int,
	    N_("initial keyboard illumination level") },
    { 2, "KBD_OnBrightness",  TAG_KBDONBRIGHTNESS,    0,
	    cft_read_int,     cft_write_int,
	    N_("initial level if KBD on/off key is pressed") },
    { 2, "KBD_FadingSpeed",   TAG_KBDFADINGSPEED,     0,
	    cft_read_int,     cft_write_int,
	    N_("0 = no smooth fading") },
    { 2, "KBD_AutoAdjust",    TAG_KBDAUTOADJUST,      0,
	    cft_read_bool,    cft_write_bool,
	    N_("only on Aluminum PowerBooks") },
    { 2, "KBD_IllumUpKey",    TAG_KBDILLUMUPKEY,      TAG_KBDILLUMUPMOD,
	    cft_read_key,     cft_write_key,
	    NULL },
    { 2, "KBD_IllumDownKey",  TAG_KBDILLUMDOWNKEY,    TAG_KBDILLUMDOWNMOD,
	    cft_read_key,     cft_write_key,
	    NULL },
    { 2, "KBD_IllumOnKey",    TAG_KBDILLUMONKEY,      TAG_KBDILLUMONMOD,
	    cft_read_key,     cft_write_key,
	    NULL },
    { 2, "KBD_Threshold",     TAG_KBDTHRESHOLD,   0,
	    cft_read_int,     cft_write_int,
	    N_("ambient light threshold in percent for keyboard light autoadj.")},
    { 2, "dev_FrameBuffer",   TAG_FRAMEBUFFERDEVICE,  0,
	    cft_read_string,  cft_write_string,
	    NULL },
    { 2, "UseFBBlank",        TAG_BLANKFRAMEBUFFER,   0,
	    cft_read_bool,    cft_write_bool,
	    NULL },
    { 2, "DimFullyDark",      TAG_DIMFULLYDARK,       0,
	    cft_read_bool,    cft_write_bool,
	    NULL },
    /* module ossmixer */
    { 3, "Volume",            TAG_VOLUME,             0,
	    cft_read_int,     cft_write_int,
	    N_("initial volume level") },
    { 3, "Speakers_muted",    TAG_MUTE,               0,
	    cft_read_bool,    cft_write_bool,
	    N_("mute after startup?") },
    { 3, "VolumeUpKey",       TAG_VOLUMEUPKEY,        TAG_VOLUMEUPMOD,
	    cft_read_key,     cft_write_key,
	    NULL },
    { 3, "VolumeDownKey",     TAG_VOLUMEDOWNKEY,      TAG_VOLUMEDOWNMOD,
	    cft_read_key,     cft_write_key,
	    NULL },
    { 3, "MuteKey",           TAG_MUTEKEY,            TAG_MUTEMOD,
	    cft_read_key,     cft_write_key,
	    NULL },
    { 3, "dev_Mixer",         TAG_MIXERDEVICE,        0,
	    cft_read_string,  cft_write_string,
	    NULL },
    { 3, "MixerInitDelay",    TAG_MIXERINITDELAY,     0,
	    cft_read_bool,    cft_write_bool,
	    NULL },
    { 3, "MixerChannels",     TAG_MIXERCHANNELS,      0,
	    cft_read_string,  cft_write_string,
	    NULL },
    /* module cdrom */
    { 4, "dev_CDROM",         TAG_CDROMDEVICE,        0,
	    cft_read_string,  cft_write_string, 
	    NULL },
    { 4, "EjectCDKey",        TAG_EJECTCDKEY,         TAG_EJECTCDMOD,
	    cft_read_key,     cft_write_key,
	    NULL },
    { 4, "EjectCDKeyDelay",   TAG_EJECTCDKEYDELAY,    0,
	    cft_read_int,     cft_write_int,
	    NULL },
    /* module pmac */
    { 5, "dev_PMU",           TAG_PMUDEVICE,          0,
	    cft_read_string,  cft_write_string,
	    NULL },
    { 5, "dev_ADB",           TAG_ADBDEVICE,          0,
	    cft_read_string,  cft_write_string,
	    NULL },
    { 5, "TPModeUpKey",       TAG_TPMODEUPKEY,        TAG_TPMODEUPMOD,
	    cft_read_key,     cft_write_key,
	    NULL },
    { 5, "TPModeDownKey",     TAG_TPMODEDOWNKEY,      TAG_TPMODEDOWNMOD,
	    cft_read_key,     cft_write_key,
	    NULL },
    { 5, "TPMode",            TAG_TPMODE,             0,
	    cft_read_tpmode,  cft_write_tpmode,
	    N_("'"TRACKPAD_NOTAP_NAME"', '"TRACKPAD_TAP_NAME"', '"TRACKPAD_DRAG_NAME"' or '"TRACKPAD_LOCK_NAME"'")},
    { 5, "KBDMode",           TAG_KBDMODE,            0,
	    cft_read_kbdmode, cft_write_kbdmode,
	    N_("'"KEYBOARD_FNBACK_NAME"' or '"KEYBOARD_FNTOP_NAME"'")},
    { 5, "Batlog",            TAG_BATLOG,             0,
	    cft_read_batlogmode, cft_write_batlogmode,
	    N_("'"BATLOG_NONE_NAME"', '"BATLOG_CYCLE_NAME"' or '"BATLOG_LOG_NAME"'")}
  };

struct tagitem *
read_configfile (char *configfile)
{
  struct stat stat_buf;
  struct tagitem *taglist;
  char *buffer, *strptr, linebuffer[STDBUFFERLEN], *token, *arg;
  int n, bufferlen;
  FILE *fd;

  if ((stat(configfile, &stat_buf)) == -1) {
    print_error (_("WARNING: Couldn't read configfile [%s, %s], using defaults.\n"), configfile, strerror(errno));
    return NULL;
  }
  bufferlen = stat_buf.st_size;

  if ((buffer = (char *) malloc (bufferlen)) == 0) {
    print_error (_("WARNING: Couldn't read configfile [%s, out of memory], using defaults.\n"), configfile);
    return NULL;
  }

	taglist = (struct tagitem *) buffer;    /* moving pointer */
	strptr = buffer + bufferlen;         /* moving pointer */

	if ((fd = fopen(configfile, "r"))) {
		while (fgets(linebuffer, sizeof(linebuffer), fd)) {
			if (linebuffer[0] != '#') {
				cleanup_buffer(linebuffer);
				if ((token = strtok(linebuffer, "=\n"))) {
					arg = strtok(0, "#;=\n");
					for (n=(sizeof(cft) / sizeof(struct cftitem) - 1); n >= 0; n--)
						if (!strcasecmp (cft[n].configtag, token)) {
							cft[n].read (&cft[n], &taglist, &strptr, arg);
							break;
						}
					if (n == -1)
						strtok(0,"=\n");
				}
			}
		}
		taglist->tag = TAG_END;
		fclose(fd);
	}
	return (struct tagitem *) buffer;
}

void
write_configfile (char *configfile)
{
	struct tagitem *taglist;
	int n, count, section;
	long fpos1, fpos2;
	FILE *fd;
	char *sectionhead[] = {
			"\n# [SYSTEM]\n",
			"\n# [MODULE POWERSAVE]\n",
			"\n# [MODULE DISPLAY]\n",
			"\n# [MODULE OSSMIXER]\n",
			"\n# [MODULE CDROM]\n",
			"\n# [MODULE PMAC]\n" };

	count = sizeof(cft) / sizeof(struct cftitem);
	if ((taglist = (struct tagitem*) malloc (2*sizeof(struct tagitem)*count)) == 0) {
	    print_error (_("WARNING: Couldn't write configfile [%s, out of memory].\n"), configfile);
		return;
	}

	taglist_init (taglist);  /* create taglist containing all possible config settings */
	for (n=0; n < count; n++) {
		taglist_add (taglist, cft[n].tag1, -1);
		if (cft[n].tag2 != 0)
			taglist_add (taglist, cft[n].tag2, -1);
	}
	process_queue (QUERYQUEUE, taglist);   /* ask modules for config settings */

	section = -1;
	if ((fd = fopen(configfile, "w"))) {
		fputs (_("# Configuration file for pbbuttonsd >= version 0.5\n# for options see man pbbuttonsd.conf\n"), fd);
		for (n=0; n < count; n++) {
			if (cft[n].section != section) {
				section = cft[n].section;
				fputs(sectionhead[section], fd);
			}
			if (cft[n].write != NULL) {
				if (cft[n].tag1 == (TAG_USERALLOWED)) {
					if (strlen ((char*) tagfind (taglist, cft[n].tag1, (long) "")) == 0)
						fputs ("# ", fd);
				}
				fprintf (fd, "%-22s= ", cft[n].configtag);
				fpos1 = ftell(fd);
				cft[n].write (fd, &cft[n], taglist);
				if (cft[n].comment) {
					fpos2 = ftell(fd);
					if ((fpos2 - fpos1) < 2)
						fprintf(fd, "\t");
					fprintf (fd, "\t; %s", _(cft[n].comment));
				}
				fputs("\n", fd);
			}
		}
		fclose(fd);
	}
	free(taglist);  /* free memory for taglist */
}

void
cft_read_string (struct cftitem *citem, struct tagitem **taglist, char **strptr, char *arg)
{
	if (arg != NULL) {  /* prevent segfault with empty arg -> fallback to default*/
		*strptr = *strptr - strlen (arg) - 1;
		strcpy (*strptr, arg);    /* copy argument before end of current stringbuffer */
		(*taglist)->tag = citem->tag1;
		(*taglist)->data = (long) *strptr;
		(*taglist)++;
	}
}

void
cft_write_string (FILE *fd, struct cftitem *citem, struct tagitem *taglist)
{
    char *string = (char*) tagfind (taglist, citem->tag1, (long) "");
	fprintf (fd, "\"%s\"", string);
}

void
cft_read_key (struct cftitem *citem, struct tagitem **taglist, char **strptr, char *arg)
{
	int key, mod;

	decode_key_and_modifier(arg, &key, &mod);
	(*taglist)->tag = citem->tag1;
	(*taglist)->data = (long) key;
	(*taglist)++;
	(*taglist)->tag = citem->tag2;
	(*taglist)->data = (long) mod;
	(*taglist)++;
}

void
cft_write_key (FILE *fd, struct cftitem *citem, struct tagitem *taglist)
{
	int key, mod;

    key = tagfind (taglist, citem->tag1, 0);
    mod = tagfind (taglist, citem->tag2, 0);
	fprintf (fd, "%s", code_key_and_modifier(key, mod));
}

void
cft_read_bool (struct cftitem *citem, struct tagitem **taglist, char **strptr, char *arg)
{
	(*taglist)->tag = citem->tag1;
	if (!strncmp("yes", arg, 3))
		(*taglist)->data = 1;
	else
		(*taglist)->data = 0;
	(*taglist)++;
}

void
cft_write_bool (FILE *fd, struct cftitem *citem, struct tagitem *taglist)
{
    int val = tagfind (taglist, citem->tag1, 0);
	fprintf (fd, "%s", val ? "yes" : "no");
}

void
cft_read_invbool (struct cftitem *citem, struct tagitem **taglist, char **strptr, char *arg)
{
	(*taglist)->tag = citem->tag1;
	if (!strncmp("yes", arg, 3))
		(*taglist)->data = 0;
	else
		(*taglist)->data = 1;
	(*taglist)++;
}

void
cft_write_invbool (FILE *fd, struct cftitem *citem, struct tagitem *taglist)
{
    int val = tagfind (taglist, citem->tag1, 0);
	fprintf (fd, "%s", val ? "no" : "yes");
}

void
cft_read_int (struct cftitem *citem, struct tagitem **taglist, char **strptr, char *arg)
{
	(*taglist)->tag = citem->tag1;
	(*taglist)->data = atoi (arg);
	(*taglist)++;
}

void
cft_write_int (FILE *fd, struct cftitem *citem, struct tagitem *taglist)
{
	int val = tagfind (taglist, citem->tag1, 0);
	fprintf (fd, "%d", val);
}

void
cft_read_tpmode (struct cftitem *citem, struct tagitem **taglist, char **strptr, char *arg)
{
	char *names[] = {TRACKPAD_NOTAP_NAME, TRACKPAD_TAP_NAME, TRACKPAD_DRAG_NAME, TRACKPAD_LOCK_NAME};

	(*taglist)->tag = citem->tag1;
	if (((*taglist)->data = read_label (arg, names, TRACKPAD_LAST)) != -1)
		(*taglist)++;
}

void
cft_write_tpmode (FILE *fd, struct cftitem *citem, struct tagitem *taglist)
{
	char *names[] = {TRACKPAD_NOTAP_NAME, TRACKPAD_TAP_NAME, TRACKPAD_DRAG_NAME, TRACKPAD_LOCK_NAME};
	int val = tagfind (taglist, citem->tag1, 0);
	if (val > TRACKPAD_LAST) val = 0;
	fprintf (fd, "%s", names[val]);	
}

void
cft_read_kbdmode (struct cftitem *citem, struct tagitem **taglist, char **strptr, char *arg)
{
	char *names[] = {KEYBOARD_FNBACK_NAME, KEYBOARD_FNTOP_NAME};

	(*taglist)->tag = citem->tag1;
	if (((*taglist)->data = read_label (arg, names, KEYBOARD_LAST)) != -1)
		(*taglist)++;
}

void
cft_write_kbdmode (FILE *fd, struct cftitem *citem, struct tagitem *taglist)
{
	char *names[] = {KEYBOARD_FNBACK_NAME, KEYBOARD_FNTOP_NAME};
	int val = tagfind (taglist, citem->tag1, 0);
	if (val > KEYBOARD_LAST) val = 0;
	fprintf (fd, "%s", names[val]);	
}

void
cft_read_batlogmode (struct cftitem *citem, struct tagitem **taglist, char **strptr, char *arg)
{
	char *names[] = {BATLOG_NONE_NAME, BATLOG_CYCLE_NAME, BATLOG_LOG_NAME};

	(*taglist)->tag = citem->tag1;
	if (((*taglist)->data = read_label (arg, names, BATLOG_LAST)) != -1)
		(*taglist)++;
}

void
cft_write_batlogmode (FILE *fd, struct cftitem *citem, struct tagitem *taglist)
{
	char *names[] = {BATLOG_NONE_NAME, BATLOG_CYCLE_NAME, BATLOG_LOG_NAME};
	int val = tagfind (taglist, citem->tag1, 0);
	if (val > BATLOG_LAST) val = 0;
	fprintf (fd, "%s", names[val]);	
}

void
cft_read_ema (struct cftitem *citem, struct tagitem **taglist, char **strptr, char *arg)
{
	char *names[] = {EMA_SLEEP_NAME, EMA_SIGNAL_NAME, EMA_COMMAND_NAME};

	(*taglist)->tag = citem->tag1;
	if (((*taglist)->data = read_label (arg, names, EMA_LAST)) != -1)
		(*taglist)++;
}

void
cft_write_ema (FILE *fd, struct cftitem *citem, struct tagitem *taglist)
{
	char *names[] = {EMA_SLEEP_NAME, EMA_SIGNAL_NAME, EMA_COMMAND_NAME};
	int val = tagfind (taglist, citem->tag1, 0);
	if (val > EMA_LAST) val = EMA_LAST;
	fprintf (fd, "%s", names[val]);	
}

void
cft_read_policy (struct cftitem *citem, struct tagitem **taglist, char **strptr, char *arg)
{
	char *names[] = {POLICY_NONE_NAME, POLICY_POWERSAVE_NAME, POLICY_USER_NAME, POLICY_PERFORMANCE_NAME};

	(*taglist)->tag = citem->tag1;
	if (((*taglist)->data = read_label (arg, names, POLICY_LAST)) != -1)
		(*taglist)++;
}

void
cft_write_policy (FILE *fd, struct cftitem *citem, struct tagitem *taglist)
{
	char *names[] = {POLICY_NONE_NAME, POLICY_POWERSAVE_NAME, POLICY_USER_NAME, POLICY_PERFORMANCE_NAME};
	int val = tagfind (taglist, citem->tag1, 0);
	if (val > POLICY_LAST) val = 0;
	fprintf (fd, "%s", names[val]);	
}

void
cft_read_action (struct cftitem *citem, struct tagitem **taglist, char **strptr, char *arg)
{
	char *names[] = {ACTION_NONE_NAME, ACTION_TORAM_NAME, ACTION_TODISK_NAME, ACTION_BLANK_NAME};
 
	(*taglist)->tag = citem->tag1;
	if (((*taglist)->data = read_label (arg, names, ACTION_LAST)) != -1)
		(*taglist)++;
}
 
void
cft_write_action (FILE *fd, struct cftitem *citem, struct tagitem *taglist)
 {
	char *names[] = {ACTION_NONE_NAME, ACTION_TORAM_NAME, ACTION_TODISK_NAME, ACTION_BLANK_NAME};

	int val = tagfind (taglist, citem->tag1, 0);
	if (val > ACTION_LAST) val = 0;
	fprintf (fd, "%s", names[val]);	
}
 
int
read_label(char *label, char *names[], int cnt)
{
	for (; cnt >= 0; cnt--)
		if (!strcmp(names[cnt], label))
			return cnt;
	return -1; /* unkown label */
}
		
void
decode_key_and_modifier(char *arg, int *key, int *mod)
{
	char *tag;
	int k;

	*mod = 0; *key = 0;
	tag = strtok(arg, "+\n");
	while (tag) {
		k = atoi (tag);
		if (k == 0) {
			/* while (*tag == ' ') tag++;    ignore leading spaces */
			if (!strncmp("shift", tag, 5))
				*mod |= MOD_SHIFT;
			else if (!strncmp("alt", tag, 3))
				*mod |= MOD_ALT;
			else if (!strncmp("ctrl", tag, 4))
				*mod |= MOD_CTRL;
		} else
			*key = k;
		tag = strtok(0, "+\n");
	}
}

char*
code_key_and_modifier (int key, int mod)
{
	static char buffer[30];

	snprintf (buffer, sizeof(buffer)-1, "%d", (key & 255));
	if (mod & MOD_SHIFT)
		strcat (buffer, " + shift");
	if (mod & MOD_ALT)
		strcat (buffer, " + alt");
	if (mod & MOD_CTRL)
		strcat (buffer, " + ctrl");
	return buffer;
}
