/*
 * compact_smpl.c - compact output sampling module for all IPF PMU 
 *
 * Copyright (C) 2002-2003 Hewlett-Packard Co
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This file is part of pfmon, a sample tool to measure performance 
 * of applications on Linux/ia64.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 */

#include "pfmon.h"

#include <perfmon/pfmlib_generic_ia64.h>
#include <perfmon/perfmon_default_smpl.h>


#define SMPL_MOD_NAME			"compact-ia64"
#define CHECK_VERSION(h)		(PFM_VERSION_MAJOR((h)) != PFM_VERSION_MAJOR(PFM_DEFAULT_SMPL_VERSION))
#define COMPACT_DFL_SMPL_ENTRIES	2048UL

typedef struct {
	unsigned long smpl_entries;
} compact_options_t;

static compact_options_t compact_options;

/*
 * forward declaration
 */
pfmon_smpl_module_t compact_ia64_smpl_module;

static int
compact_process_samples(pfmon_smpl_desc_t *csmpl)
{
	pfm_default_smpl_hdr_t *hdr;
	pfm_default_smpl_entry_t *ent;
	FILE *fp = csmpl->smpl_fp;
	pfm_gen_ia64_pmd_reg_t *reg;
	unsigned long entry, i;
	uint64_t count;
	unsigned int n, npmds, last_npmds = 0;
	unsigned char ovfl_pmd;
	unsigned char last_ovfl_pmd = PFMON_MAX_PMDS-1; /* undefined PMD */
	int ret;

	hdr 	   = csmpl->smpl_hdr;
	ent	   = (pfm_default_smpl_entry_t *)(hdr+1);
	entry      = options.opt_aggr ? *csmpl->aggr_count : csmpl->entry_count;
	count      = hdr->hdr_count;

	DPRINT(("hdr_count=%lu hdr=%p\n", count, hdr));

	for(i=0; i < count; i++) {

		ret = fprintf(fp, "%-8lu %-8d %-2d 0x%016lx 0x%016lx %3u %lu ",
				entry,
				ent->pid,
				ent->cpu,
				ent->ip,
				ent->tstamp,
				(unsigned int)ent->ovfl_pmd,
				-1*ent->last_reset_val);

		ovfl_pmd = ent->ovfl_pmd;

		reg = (pfm_gen_ia64_pmd_reg_t *)(ent+1);

		npmds = ent->ovfl_pmd == last_ovfl_pmd ? last_npmds: bit_weight(options.rev_smpl_pmds[ovfl_pmd]);
		last_npmds = npmds;
		for (n = npmds; n ; n--) {
			ret = fprintf(fp, "0x%016lx ", reg->pmd_val);
			reg++;
		}
		ret = fputc('\n', fp);

		last_ovfl_pmd = ovfl_pmd;

		/* fprintf() error detection */
		if (ret == -1) goto error;

		ent  = (pfm_default_smpl_entry_t *)reg;	
		entry++;
	}
	/*
	 * when aggregation is used, for are guaranteed sequential access to
	 * this routine by higher level lock
	 */
	if (options.opt_aggr) {
		*csmpl->aggr_count += hdr->hdr_count;
	} else {
		csmpl->entry_count += hdr->hdr_count;
	}
	csmpl->last_count = count;

	return 0;
error:
	warning("cannot write to sampling file: %s\n", strerror(errno));
	/* not reached */
	return -1;
}

static int
compact_check_new_samples(pfmon_smpl_desc_t *csmpl)
{
	pfm_default_smpl_hdr_t *hdr; 
	uint64_t last_ovfl;

	hdr       = csmpl->smpl_hdr;
	last_ovfl = csmpl->last_ovfl;

	if (hdr->hdr_overflows <= last_ovfl && last_ovfl != ~0UL && hdr->hdr_count == csmpl->last_count) {
		DPRINT(("skipping identical set of samples %lu <= %lu\n", hdr->hdr_overflows, last_ovfl)); 
		return -1;
	}
	/*
	 * XXX: module specific field in generic structure (last_ovfl)
	 */
	csmpl->last_ovfl = hdr->hdr_overflows;

	return 0;
}

/*
 * 000-255   reserved for generic options
 * 400-499   reserved for PMU specific options
 * 500-599   reserved for format specific options
 */
static struct option compact_cmd_options[]={
	{ "smpl-entries", 1, 0, 500},
	{ NULL, 0, 0, 0}
};

static void
compact_show_options(void)
{
	printf("\t--smpl-entries=val\t\tset number of entries for sampling buffer (default %lu)\n", COMPACT_DFL_SMPL_ENTRIES);
}

/*
 * 0  means we understood the option
 * -1 unknown option
 */
static int
compact_parse_options(int code, char *optarg, pfmon_lib_param_t *evt)
{
	char *endptr = NULL;

	switch(code) {
		case  500:
			if (compact_options.smpl_entries != COMPACT_DFL_SMPL_ENTRIES) 
				fatal_error("smpl-entries already defined\n");

			compact_options.smpl_entries = strtoul(optarg, &endptr, 0);

			if (*endptr != '\0') 
				fatal_error("invalid number of entries: %s\n", optarg);
			break;
		default:
			return -1;
	}
	return 0;

}


static size_t
compact_get_fmt_arg_size(void)
{
	return sizeof(pfm_default_smpl_ctx_arg_t);
}

#define MAX_PMD_COUNT		64
#define FUDGE_FACTOR		(sizeof(pfm_default_smpl_entry_t)+(sizeof(unsigned long)*MAX_PMD_COUNT))
#define ENTRY_SIZE(npmd)	(sizeof(pfm_default_smpl_entry_t)+((npmd)*sizeof(unsigned long)))
static int
compact_initialize_ctx_arg(pfmon_ctx_arg_t *arg, unsigned max_pmds_sample)
{
	pfm_default_smpl_ctx_arg_t *ctx_arg = (pfm_default_smpl_ctx_arg_t *)arg;
	unsigned long entry_size;

	entry_size = ENTRY_SIZE(max_pmds_sample);
	
	/*
	 * the fudge factor allows us to get exactly the number of entries specified by the
	 * user (or the default). The kernel module is such that it will consider the buffer
	 * full if less than PFM_DEFAULT_MAX_ENTRY_SIZE bytes are left in the buffer. Any
	 * entry size is <= PFM_DEFAULT_MAX_ENTRY_SIZE, therefore we will always record
	 * less than the specified number unless we increase the buffer size slightly.
	 */
	ctx_arg->buf_arg.buf_size = sizeof(pfm_default_smpl_hdr_t)
		                  + (FUDGE_FACTOR-entry_size)
				  + compact_options.smpl_entries*entry_size;

	/*
	 * copy the uuid of the format we are using
	 */
	memcpy(ctx_arg->ctx_arg.ctx_smpl_buf_id, compact_ia64_smpl_module.uuid, sizeof(pfm_uuid_t));

	vbprintf("buffer entries : %lu max PMD/entry=%u\n", compact_options.smpl_entries, max_pmds_sample);

	DPRINT(("max_pmds_sample=%u buf_size=%lu fudge=%lu buffer_header=%lu entry_header=%lu (max)entry_size=%lu\n", 
		max_pmds_sample, 
		ctx_arg->buf_arg.buf_size, 
		FUDGE_FACTOR, 
		sizeof(pfm_default_smpl_hdr_t), 
		sizeof(pfm_default_smpl_entry_t), 
		entry_size));
		

	return 0;
}

/*
 * module initialization
 */
static int
compact_initialize_module(void)
{
	compact_options.smpl_entries = COMPACT_DFL_SMPL_ENTRIES;

	return pfmon_register_smpl_mod_options(compact_cmd_options, sizeof(compact_cmd_options));
}

static int
compact_print_header(pfmon_smpl_desc_t *csmpl)
{
	FILE *fp = csmpl->smpl_fp;
	unsigned long msk;
	int ev;
	unsigned int i, j, col;
	char name[PFMON_MAX_EVTNAME_LEN];

	fprintf(fp, "# sampling buffer entries: %lu\n#\n", compact_options.smpl_entries);

	fprintf(fp, "# description of columns:\n"
		    "#\tcolumn  1: entry number\n"
	 	    "#\tcolumn  2: process id\n"
		    "#\tcolumn  3: cpu number\n"
		    "#\tcolumn  4: instruction pointer\n"
		    "#\tcolumn  5: unique timestamp\n"
		    "#\tcolumn  6: overflowed PMD index\n"
		    "#\tcolumn  7: initial value of first overflowed PMD\n");

	for(i=0; i < PFMON_MAX_PMDS; i++) {

		ev = options.rev_pc[i];

		if (ev == -1) continue;

		col = 8;

		pfm_get_event_name(options.events[ev].event, name, PFMON_MAX_EVTNAME_LEN);

		if ((options.short_rates[ev].flags & PFMON_RATE_VAL_SET) == 0) continue;

		fprintf(fp, "#\n#\twhen PMD%u(%s) overflows: \n", i, name);

		msk = options.smpl_pmds[ev];

		for(j=0; msk; msk >>=1, j++) {	
			if ((msk & 0x1) == 0) continue;

			fprintf(fp, "#\t\tcolumn %2u: PMD%u\n", col, j);

			col++;
		}
	}
	return 0;
}

static int
compact_check_version(pfmon_smpl_desc_t *csmpl)
{
	pfm_default_smpl_hdr_t *hdr; 

	hdr   = csmpl->smpl_hdr;

	if (CHECK_VERSION(hdr->hdr_version)) {
		warning("format %s expect format v%u.x not v%u.%u %u\n", 
				SMPL_MOD_NAME,
				PFM_VERSION_MAJOR(PFM_DEFAULT_SMPL_VERSION),
				PFM_VERSION_MAJOR(hdr->hdr_version),
				PFM_VERSION_MINOR(hdr->hdr_version), hdr->hdr_version);
		return -1;
	}
	return 0;
}

pfmon_smpl_module_t compact_ia64_smpl_module ={
	.name		    = SMPL_MOD_NAME,
	.pmu_mask	    = PFMON_PMU_MASK(PFMLIB_GENERIC_IA64_PMU),
	.description	    = "Column-style raw values",
	.check_version	    = compact_check_version,
	.process_samples    = compact_process_samples,
	.get_fmt_arg_size   = compact_get_fmt_arg_size,
	.initialize_ctx_arg = compact_initialize_ctx_arg,
	.check_new_samples  = compact_check_new_samples,
	.show_options       = compact_show_options,
	.parse_options      = compact_parse_options,
	.initialize_module  = compact_initialize_module,
	.print_header       = compact_print_header,
	.uuid		    = PFM_DEFAULT_SMPL_UUID,
};
