/*
 * Graphics routines for Fig code
 *
 * Original routines by Dr. George Lueker (for imPress language output)
 * Modified by Tim Morgan
 *
 * Modified by Micah Beck to produce Fig code
 * Modified by Micah Beck again to produce Fig 1.4 code
 */

#include <stdio.h>
#include <ctype.h>
#include "object.h"

#define DEF_PEN_SIZE	8	/* Default pen diameter */
#define	MAXPOINTS	300	/* Max number of points in a path */

#define bool int
#define TRUE (-1)
#define FALSE 0

#define	P_WHITE		0	/* Graphics drawing modes ("commands") */
#define	P_BLACK		15
#define	TEXTURE		3
#define ORTEXTURE	7

extern	double	sqrt(), cos(), sin();

#define PI			3.14159265358979323846
#define TWOPI			(2*PI)
#define RES			1000.0
#define Pix_To_In(x)		(((float) x) / RES)

#define FIGRES			80
#define FIGCANV_W		(8*FIGRES)
#define FIGCANV_H		(10*FIGRES)
#define Pix_To_Figpix(x)	((int)(FIGRES * Pix_To_In(x)))
#define Pen_to_Figpix(x)	((x*FIGRES + 500) / 1000)


extern	int dbg;		/* Non-zero when debugging info wanted */
static	int xx[MAXPOINTS],yy[MAXPOINTS];	/* Current path */
static	int xmin, xmax, ymin, ymax; /* Extremes of the graph, Imagen coords */
static	float xslope,yslope,xbase,ybase;/* Convert Window to Viewport */
static	int arraylen;		/* Number of points in current path */
static	bool pathsent;		/* True if path has been defined */
static	int ipensize;		/* Desired current pen size */
extern	FILE *TEXFILE;		/* Output file */


/*
 * Set scaling factors and graph base offset onto imagen virtual page.
 * Page measurements are in inches, but window measurements are in
 * arbitrary units.  Reversing hi and lo ypage flips the y-axis, which
 * runs backwards on the Imagen.
 */
fig_window(lox, loy, hix, hiy, loxpage, hiypage, hixpage, loypage)
float	lox, loy, hix, hiy, loxpage, loypage, hixpage, hiypage;
{
	xslope = RES * (hixpage-loxpage) / (hix-lox);
	xbase  = RES * loxpage - lox*xslope;
	yslope = RES * (hiypage-loypage) / (hiy-loy);
	ybase  = RES * loypage - loy*yslope;
	if (dbg)
	    printf("Coefficients: %10.5f%10.5f%10.5f%10.5f\n",
		xslope,xbase,yslope,ybase);
}


/*
 * Set the size of the virtual pen used in drawing
 */
pensize(x)
int	x;
{
	if (x != ipensize) {
	    ipensize = Pen_to_Figpix(x);
	    if (!pathsent) sendpath();
	    clearpath();
	}
}


/*
 * Define the texture to be used for shading
 */
texture (pcount, bitpattern)
int pcount, *bitpattern;
{
    int     i;

    if (!pathsent)
	sendpath ();
    clearpath ();

    fprintf (stderr, "pic2fig: textures not implemented (tx %d)", pcount);
/*
    for (i = 0; i < pcount; i++)
	fprintf (TEXFILE, " %x", *bitpattern++);
    fprintf (TEXFILE, "}%%\n");
*/
}


/*
 * Clear the path memory before beginning a new figure
 */
clearpath()
{
	arraylen = 0;
	pathsent = FALSE;
}


/*
 * Keep up with the min and max points on the virtual page
 */
static ckbounds(x, y)
int x,y;
{
	if (x < xmin) xmin = x;
	if (x > xmax) xmax = x;
	if (y < ymin) ymin = y;
	if (y > ymax) ymax = y;
}


/*
 * Routine to convert from window coords to Imagen coords.  Also,
 * keeps xmin, etc, up to date.  Note that ix and iy are passed by
 * reference, and they are returned in units of INCHES.
 */
map(x, y, ix, iy, bounds)
float		x, y;
register int	*ix, *iy;
bool	bounds;
{
	*ix = xbase + xslope * x;
	*iy = ybase + yslope * y;
	if (bounds) ckbounds(*ix, *iy);
}


/*
 * Place the text string at the indicated position.
 */
fig_text_at(text, x, y, position)
char	*text;
float	x, y;
int	position;	/* Indicates what part of text is at (x,y) */
{
    int	    ix, iy, offset;
    char    *cp;
    map(x, y, &ix, &iy, 1);

    for (cp = text; *cp && isspace(*text); cp++);
    if (!*cp) return;

#ifdef TEXT_LEFT_ONLY
    if (position != T_LEFT_JUSTIFIED) offset = strlen(cp)*8;
    if (position == T_CENTER_JUSTIFIED) offset = offset/2;
    position = T_LEFT_JUSTIFIED;
#else
    offset = 0;
#endif TEXT_LEFT_ONLY

    fprintf(TEXFILE, "%d %d %d %d %d %d %6.3f %d %d %d %d %d %s%c\n",
	O_TEXT, position, -1, -1, -1, -1, 0.0, -1, 16, strlen(cp)*8,
	Pix_To_Figpix(ix) - offset, Pix_To_Figpix(iy), cp, '\01');
}


/*
 * Add a new point to the current path in the window
 */
fig_drawto(x, y)
float	x, y;
{
	int x0, y0;

	map(x, y, &x0, &y0, TRUE);
	fig_raw_drawto(x0, y0);
}


/*
 * This routine is used by the spline routine, where the coords have
 * already been converted to window coords.
 */
fig_raw_drawto(x, y)
int	x,y;
{
    if (arraylen == 0 || x != xx[arraylen] || y != yy[arraylen]) {
	arraylen++;
	xx[arraylen] = x;
	yy[arraylen] = y;
	ckbounds(x, y);
	pathsent = FALSE;
    }
}


/*
 * Download the currently defined path to the Imagen.  Remember that
 * it has now been sent.
 */
static sendpath()
{
	register int i;

	if (dbg) printf("Sending path ...%d\n", arraylen);
	if (arraylen > 1) {

	    fprintf(TEXFILE, "%d %d %d %d %d %d %d %d %d %d %d\n",
		O_POLYLINE, T_POLYLINE, SOLID_LINE,
		ipensize, -1, -1, -1, -1, -1, 0, 0);
	    for (i=1; i<=arraylen; i++)
		fprintf(TEXFILE, "%d %d ",
			Pix_To_Figpix(xx[i]), Pix_To_Figpix(yy[i]));
	    fprintf(TEXFILE, "9999 9999\n");
	}

	pathsent = TRUE;
}


/*
 * Same as sendpath(), but produces dashed or dotted lines.  There had better
 * be exactly 2 points in the path array!
 */
send_dashed_path(inchesperdash, dotted)
float inchesperdash;		/* Dashs/inch */
int dotted;			/* boolean */
{
	register int i;

	if (dbg) printf("Sending path ...%d\n", arraylen);
	if (arraylen < 2) return;
	else {
	    fprintf(TEXFILE, "%d %d %d %d %d %d %d %d %d %d %d\n",
		O_POLYLINE, T_POLYLINE,
		(dotted ? DOTTED_LINE : DASH_LINE), ipensize,
		-1, -1, -1, -1, -1, 0, 0);
	    for (i=1; i<=arraylen; i++)
		fprintf(TEXFILE, "%d %d ",
			Pix_To_Figpix(xx[i]), Pix_To_Figpix(yy[i]));
	    fprintf(TEXFILE, "9999 9999\n");
	}

	pathsent = TRUE;
}


/*
 * Cause the current figure to be defined to the imagen and to be
 * drawn.
 */
static draw()
{	
	if (!pathsent) sendpath();
	clearpath();
}


/*
 * If a drawing has been defined, flush it, then move the pen to the
 * indicated position.
 */
fig_hvflush(x, y)
float x, y;
{
    int ix, iy;
    if (arraylen > 0 && !pathsent) {
	map(x, y, &ix, &iy, 0);
	if (ix == xx[arraylen] && iy == yy[arraylen]) return;/* Already there */
	draw();
    }
    fig_drawto(x, y);
}


/*
 * Guarantee that the last drawing has actually been drawn.
 */
fig_flush()
{
    if (!pathsent && arraylen > 0) draw();
}


/*
 * Draw the indicated arc.  Angles in radians
 */
fig_arc(xcenter, ycenter, xradius, yradius, start, stop)
float	xcenter, ycenter, xradius, yradius, start, stop;
{
    int X0, Y0, xrad, yrad, xend, yend;
    extern double cos(), sin();

    map(xcenter, ycenter, &X0, &Y0, FALSE);
    xrad = xradius * xslope;
    yrad = yradius * yslope;
    if (is_onarc(0.0, start, stop)) ckbounds(X0+xrad, Y0);
    if (is_onarc(PI/2, start, stop)) ckbounds(X0, Y0+yrad);
    if (is_onarc(PI, start, stop)) ckbounds(X0-xrad, Y0);
    if (is_onarc(1.5*PI, start, stop)) ckbounds(X0, Y0-yrad);
    xend = X0 + xrad * cos(start) + 0.5;
    yend = Y0 + yrad * sin(start) + 0.5;
    ckbounds(xend, yend);
    xend = X0 + xrad * cos(stop);
    yend = Y0 + yrad * sin(stop);
    ckbounds(xend, yend);

    fprintf(TEXFILE, "%d %d %d %d %d, %d %d %d %d %d %d %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f\n",
		O_ELLIPSE, T_ELLIPSE_BY_RAD,
	        SOLID_LINE, ipensize, -1, -1, -1, -1, -1, 1, 0.0,
		Pix_To_Figpix(X0), Pix_To_Figpix(Y0),
		Pix_To_Figpix(xrad), Pix_To_Figpix(yrad), 0.0, 0.0, 0.0, 0.0);
}


/*
 * Draw a circle with the indicated parameters
 */
fig_circle(xcenter, ycenter, radius)
float	xcenter, ycenter, radius;
{
	if (dbg) printf("Circle: %6.3f, %6.3f, %6.3f\n", xcenter, ycenter, radius);
	fig_arc(xcenter, ycenter, radius, radius, 0.0, TWOPI);
}


/*
 * Draw an ellipse with the indicated parameters
 */
fig_ellipse(xcenter, ycenter, xradius, yradius)
float	xcenter, ycenter, xradius, yradius;
{
	if (dbg) printf("Ellipse: %6.3f, %6.3f, %6.3f, %6.3f\n", xcenter, ycenter,
		xradius, yradius);
	fig_arc(xcenter, ycenter, xradius, yradius, 0.0, TWOPI);
}


/*
 * Initialize a drawing
 */
fig_begin_drawing()
{
  	fprintf(TEXFILE, "#FIG 1.4\n%d %d\n", FIGRES, 2);

	fig_window(0.0, 0.0, 10.0, 10.0, 0.0, 0.0, 3.0, 3.0);
	xmin = 30000;	xmax = -30000;
	ymin = 30000;	ymax = -30000;
	arraylen = 0;
	pathsent = FALSE;
	ipensize = Pen_to_Figpix(DEF_PEN_SIZE);
}


/*
 * Routine to clean up after a drawing.  Makes sure last figure was output.
 */
fig_end_drawing()
{
	int ixzero, iyzero;

	if (dbg) printf("END DRAWING\n");
	fig_flush();
	map(0.0, 0.0, &ixzero, &iyzero, 0);
	if (dbg) {
	    printf("Origin is %8.4fin from left edge of graph,",
		Pix_To_In(ixzero-xmin));
	    printf("%8.4fin from top edge of graph,\n",
	    	Pix_To_In(ymax-iyzero));
	    printf("and graph is %6.3fin wide", Pix_To_In(xmax - xmin));
	    printf(" and %6.3fin high.\n", Pix_To_In(ymax - ymin));
	}
}


/*
 * Pretend to draw a spline.
 */
draw_fig_spline(xx, yy, N)
int xx[], yy[], N;
{
    register int i;

    if (!pathsent) {
	sendpath();
	clearpath();
    }

    fprintf(TEXFILE, "%d %d %d %d %d %d %d %d %d %d %d\n",
	O_SPLINE, T_OPEN_NORMAL,
	SOLID_LINE, ipensize, -1, -1, -1, -1, -1, 0, 0);
    for (i=1; i<N; i++) {
	ckbounds(xx[i], yy[i]);
	fprintf(TEXFILE, "%d %d ",
		Pix_To_Figpix(xx[i]), Pix_To_Figpix(yy[i]));
      }
    fprintf(TEXFILE, "9999 9999\n");	
}


/*
 * The previous box/circle/ellipse should be shaded
 */
fig_shade()
{
    fig_flush();
    fprintf(stderr, "pic2fig: shading not implemented\n");
}


/*
 * The previous box/circle/ellipse should be whitened inside
 */
fig_whiten()
{
    fig_flush();
    fprintf(stderr, "pic2fig: whitening not implemented\n");
}
