// ------------------------------------------------------------------------- //
// $Id: node.cpp,v 1.61 2004/01/02 15:13:52 pandr Exp $
// ------------------------------------------------------------------------- //

/*
 * Copyright (c) 2002 
 *				see AUTHORS list
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

// ------------------------------------------------------------------------- //

#if HAVE_CONFIG_H
# include <config.h>
#endif

#if STDC_HEADERS
# include <stdio.h>
#endif

#if HAVE_GL_GLU_H
# ifndef __GNUC__
#  ifdef _WIN32
#   include <windows.h>
#  endif
# endif
# include <GL/glu.h>
#endif

#if HAVE_OPENGL_GLU_H	// MAC OS X
# include <OpenGL/glu.h>
#endif

#if HAVE_TYPEINFO
# include <typeinfo>
#endif

#if HAVE_ALGORITHM
# include <algorithm>
#endif

#include "node.h"
#include "group.h"
#include "nodecontroller.h"
#include "common.h"
#include "mainwindow.h"
#include "primitive.h"

// ------------------------------------------------------------------------- //

int Node::_controllers_run = 0;

Node::Node(const std::string& name) : _color(1.0f, 1.0f, 1.0f, 1.0f),
	_alpha(1.0f), _parent(0), _name(name), _pos(0.0f, 0.0f, 0.0f),
	_visible(true), _size(0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f)
{
	//std::cout << "Made node " << name << " at " << this << std::endl;
	_mat.makeIdentity();
}

Node::~Node()
{
	//std::cout << "Destroyed node " << _name << " at " << this << std::endl;
	for (nc_iter i = _controllers.begin(); i != _controllers.end(); ++i) {
		delete (*i);
	}
}

void Node::frame_update()
{
	for (nc_iter i = _controllers.begin(); i != _controllers.end(); ++i) {
		if ((*i)->get_active()) {
			_controllers_run++;
			(*i)->frame_update();
		}
	}
}

std::string Node::full_name() const
{
	return _parent ? _parent->full_name() + "/" + name() : name();
}

void Node::dump_tree(uint indent)
{
	for (uint i = 0 ; i < indent; i++) { std::cout << " "; }
	const char* type = typeid(*this).name();
	while ('0' <= *type && *type <= '9') type++;
	std::cout << type << " (" << name() << ")" << std::endl;
}

void Node::add_controller(NodeController* cont)
{
	_controllers.push_back(cont);
}

// ------------------------------------------------------------------------ //

Leaf::~Leaf()
{
	for (primitive_iter i = _primitives.begin();
	     i != _primitives.end(); ++i) {
		delete (*i);
	}
}

void Leaf::add_primitive(Primitive* prim)
{
	_primitives.push_back(prim);
}

bool Leaf::remove_primitive(Primitive* prim)
	// FIXME consider use of algorithm remove together with clean (josuttis 327, 378)
{
	primitive_iter i = std::find(_primitives.begin(), _primitives.end(), prim);
	if (i == _primitives.end())
		return false;
	_primitives.erase(i);
	return true;
}

void Leaf::draw_prims(float alpha)
{
	if (!get_visible()) return;

	if (_dropshadow) {
		// Drop shadow hack
		glPushMatrix();
		v3 oldpos = get_pos();
		set_pos(get_pos() + v3(_dropshadow, -_dropshadow, 0.0f));
		_load_GL_matrix();

		rgba black(0.0f, 0.0f, 0.0f, 0.5f*alpha);

		for (primitive_iter i = _primitives.begin(); i != _primitives.end(); ++i)
		{
			(*i)->draw(black);
		}
		set_pos(oldpos);
		glPopMatrix();
	}

	glPushMatrix();
	_load_GL_matrix();

	rgba color(_color);
	color.a() = byte((float)color.a()*alpha*_alpha);

	for (primitive_iter i = _primitives.begin(); i != _primitives.end(); ++i)
	{
		(*i)->draw(color);
	}
	glPopMatrix();
}

// ------------------------------------------------------------------------ //

Letter::Letter(const std::string &name, FreeTypeFont::handle font, 
               const Glyph *glyph, unsigned char letter)
	: Leaf(name), _font(font), _glyph(glyph), _kern(0.0f)
{
	set_size(size3(0, _glyph->_advance.x(), _glyph->_offset.y(), 
				_glyph->_size.y() + _glyph->_offset.y(), 0.0f, 0.0f));
	Primitive *p = new Primitive(_glyph->_tile);
	p->offset(_glyph->_offset.x(), _glyph->_offset.y(), 0.0f);
	add_primitive(p);
	set_letter(letter);
}

Letter::~Letter()
{
}

float Letter::get_kerning_with_prev_node(Node *n)
{
	return n->get_kerning_with_next_node(this);
}

float Letter::get_kerning_with_next_node(Letter *n)
{
	// Enhancement: check fonts on both letters
	return _font->get_kerning( get_glyph()->_index , n->get_glyph()->_index );
}

// ------------------------------------------------------------------------ //

