########################################################################
# $Header: /var/local/cvsroot/4Suite/test/Xml/Xslt/Exslt/test_datetime.py,v 1.1 2005/02/14 00:15:51 jkloth Exp $
"""Tests for EXSLT Dates and Times"""

import re

from Ft.Xml.Xslt.Exslt import DateTime

DATE = r'\d\d\d\d-[01]\d-[0-3]\d'
TIME = r'[012]\d:[0-5]\d:[0-6]\d(Z|[+\-]\d\d:?\d\d)?'
DATETIME = DATE + 'T' + TIME

TESTS = []

# date:date-time()
def test_DateTime(tester):
    tester.startTest('date:date-time() syntax')
    result = DateTime.DateTime(None)
    tester.compare(True, re.search('^%s$' % DATETIME, result) is not None,
                   'Result %r does not match regexp %r' % (result, DATETIME))
    tester.testDone()
    return
TESTS.append(test_DateTime)


# date:date()
def test_Date(tester):
    tester.startTest('date:date() syntax')
    result = DateTime.Date(None)
    tester.compare(True, re.search('^%s$' % DATE, result) is not None,
                   'Result %r does not match regexp %r' % (result, DATE))
    tester.testDone()

    # Our implementation only supports ISO8601 date, time, and date/time
    # formats for input & output.  xs:dateTime values are the same as
    # ISO8601 date/time strings.  xs:date values can have an optional
    # time zone on the end, unlike ISO8601 date strings.  xs:time values
    # do not start with 'T', unlike ISO8601 time strings.
    dates = [
        (u'2003-09-12T23:59:59-0400', u'2003-09-12'),
        (u'2003-09-12T23:59:59-04:00', u'2003-09-12'), # offset in extended format
        (u'2001-01-01T00:00:00Z', u'2001-01-01'),
        (u'2000-01-01T00:00:00', u'2000-01-01'),
        (u'2005-05-05', u'2005-05-05'),
        (u'Jan 01, 2001', u''),
        #(u'2005-05-05+0100', u'2005-05-05+0100'), # tz in = tz out? spec unclear
        ]
    for source, expected in dates:
        tester.startTest("date:date('%s')" % source)
        result = DateTime.Date(None, source)
        tester.compare(expected, result)
        tester.testDone()
    return
TESTS.append(test_Date)


# date:time()
def test_Time(tester):
    tester.startTest('date:time() syntax')
    result = DateTime.Time(None)
    tester.compare(True, re.search('^%s$' % TIME, result) is not None,
                   'Result %r does not match regexp %r' % (result, TIME))
    tester.testDone()

    # see note above re: ISO 8601 vs XSD
    # Our implementation always returns with a timezone.
    # It's not supposed to, if none was given in the input.
    times = [
        (u'2003-09-12T23:59:59-0400', [u'23:59:59-04:00',
                                       u'23:59:59-0400',
                                       u'03:59:59Z']),
        (u'2003-09-12T23:59:59-04:00', [u'23:59:59-04:00',
                                        u'23:59:59-0400',
                                        u'03:59:59Z']),
        (u'2001-01-01T00:00:00Z', u'00:00:00Z'),
        (u'2000-01-01T00:00:00', u'00:00:00Z'),
        (u'00:00:00', u'00:00:00'), # xs:date input good!
        (u'T00:00:00', u''),        # ISO 8601 date input bad!
        (u'02:22:22 PM', u''),
        ]
    for time, expected in times:
        tester.startTest("date:time('%s')" % time)
        result = DateTime.Time(None, time)
        if isinstance(expected, list):
            tester.compareIn(expected, result)
        else:
            tester.compare(expected, result)
        tester.testDone()
    return
TESTS.append(test_Time)


def NotTested(tester, title):
    tester.startTest(title)
    tester.warning('Not tested')
    tester.testDone()
    return

for title in ('add',
              'add-duration',
              'day-abbreviation',
              'day-in-month',
              'day-in-week',
              'day-in-year',
              'day-name',
              'day-of-week-in-month',
              'difference',
              'duration',
              'format-date',
              'hour-in-day',
              'leap-year',
              'minute-in-hour',
              'month-abbreviation',
              'month-in-year',
              'month-name',
              'parse=date',
              'second-in-minute',
              'seconds',
              'sum',
              'week-in-month',
              'week-in-year',
              'year',
              ):
    TESTS.append(lambda tester, title='date:%s()' % title:
                 NotTested(tester, title))

def Test(tester):
    tester.startGroup('Dates and Times')
    for test in TESTS:
        test(tester)
    tester.groupDone()
    return
