/*
 * robdefines.cpp
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// robdefines.cpp: Implementierung der Klasse DefineInputStream.
//
//////////////////////////////////////////////////////////////////////

#include "robdefines.h"

#include "robvars.h"

#include <rtchar.h>
#include <rtmath.h>
#include <rtstring.h>
#include <rtcollect.h>
#include <rtstreams.h>
#include <rtmap.h>

using namespace lrt;

namespace rt {

const String DefineInputStream::whitespace("\r\n\t ");

DefineInputStream::DefineInputStream(InputStream *in, const String &separators, 
									 const char commentStart) :
	infos(1), streams(1), defineMap(false), separators(separators), commentStart(commentStart), 
	markLineNum(1),	ignoreComments(true), autoTrim(true), defines(false), curStream(0)
{
	streams[0] = in;
	this->separators += commentStart;
	initValidIdentifierChars();
}

void DefineInputStream::close()
{
	streams[0]->close();
}

DefineInputStream::~DefineInputStream()
{
	close();
	deleteAll(streams);
}

void DefineInputStream::setIgnoreComments(bool val)
{
	ignoreComments = val;
}

void DefineInputStream::setAutoTrim(bool val)
{
	autoTrim = val;
}

void DefineInputStream::setEnableDefines(bool val)
{
	defines = val;
}
	
void DefineInputStream::mark()
{
	if(curStream > 0) return; // mark only if in 0th stream
	markLineNum = infos[0].lineNum;
	streams[0]->mark();
}

void DefineInputStream::reset()
{
	if(curStream > 0) { // return to 0th stream
		for(int s = streams.length() - 1; s >= 1; s--) {
			deleteAt(streams, s);
			infos.remove(s);
		}
		curStream = 0;
	}
	infos[0].lineNum = markLineNum;
	streams[0]->reset();
}

bool DefineInputStream::markSupported()
{
	if(curStream > 0) return false;
	return streams[0]->markSupported();
}

bool DefineInputStream::fail()
{
	return streams[0]->fail(); // the other streams won't fail (they're StringInStreams)
}

int DefineInputStream::read()
{
	StreamInfo& ci = infos[curStream];
	if(ci.eol) { 
		ci.lineNum++; ci.eol = false; 
	}
	int ret;
	if(ci.buf >= 0) { ret = ci.buf; ci.buf = -1; } // read from buf
	else ret = streams[curStream]->read(); // read from stream
	if(ret == '\n') ci.eol = true;
	if((curStream > 0) && (ret == -1)) { // current stream has ended
		deleteAt(streams, curStream);
		infos.remove(curStream);
		curStream--;
	}
	return ret;
}

//! Can't handle multiple successive unreads! Characters will get lost!
void DefineInputStream::unread(int c)
{
	infos[curStream].buf = c;
	if(c == '\n') infos[curStream].eol = false;
}

RobLoadReturnType DefineInputStream::getWord(String& word, const String& specialSeps)
{
	word.clear();
	const String& separators = (specialSeps == "" ? this->separators : specialSeps);
	bool eof = false;
	bool first = true;
	int c;
	while(true) {
		eof = (curStream == 0);
		c = read();
		if(c >= 0) eof = false;
		else break; 
		
		if(first) { // for first char of word, things are a bit different...
			if(c == commentStart) {
				if(!ignoreComments) // return the whole comment as a word
				{ unread(c); word = getLine(); return rlOK; }
				else // skip comment: return first word of next line
				{ skipRestOfLine(); return getWord(word, specialSeps); }
			}
			else {
				word += (char)c; // separators are allowed as first char
				if(separators.indexOf((char)c) >= 0) break;
			}
			first = false;
		}
		else { // more chars to the word
			if(separators.indexOf((char)c) >= 0) 
			{ unread(c); break; } // break on separator, put it to next word
			word += (char)c;
		}
	}

	if(autoTrim) {
		word = word.trim();
	}
	if(!eof && (word.length() == 0)) // word is empty => fetch next word
		return getWord(word, specialSeps);
	if(!defines) return rlOK;
	
	/// Defines...
	RobLoadReturnType ret;
	int parstart = 0;
	if((parstart = word.indexOf('?')) >= 0) // a define parameter
	{
		String defPar = word.substring(parstart);
		if(!infos[curStream].params.isSet(defPar))
			return rlFailWrongDefinePar;
		word = word.substring(0, parstart) + infos[curStream].params[defPar];
		return rlOK;
	}
	if(!word.compareIgnoreCase("define")) // a define declaration
	{
		setEnableDefines(false); // disable defines inside defines
		ret = createDefine();
		setEnableDefines(true); // re-enable defines 
		if(ret) return ret;
		return getWord(word, specialSeps);
	}
	if(word.startsWith("&")) // a define instantiation
	{
		if(streams.length() >= rlMaxRecurse) // too many recursions
			return rlFailRecurse;
		if(!defineMap.isSet(word)) // define undefined
			return rlFailUndefinedDefine;
		Define& def = defineMap.get(word);
		StreamInfo info(def);
		if(def.params.length() > 0) { // read params from original stream, if it has any
			ret = readParams(info.params, def.params);
			if(ret) return ret;
		}
		infos += info;
		streams += def.getStream();
		curStream = streams.length() - 1; // switch to the define stream
		return getWord(word, specialSeps); // read word from within the define

	}
	return rlOK;
}

// special define functions
RobLoadReturnType DefineInputStream::readParams(StringMap<String>& params, Vector<String>& paramNames)
{
	String seps = whitespace + "(),";
	RobLoadReturnType ret;
	String word;

	// expect (
	ret = getWord(word, seps);
	if(ret) return ret;
	if(word != "(")
		return rlFailExpectOpen;

	for(int i = 0; i < paramNames.length(); i++)
	{
		ret = getWord(word, seps);	// read parameter
		if(ret) return ret;
		params[paramNames[i]] = word;

		if(i != paramNames.length() - 1) {
			ret = getWord(word, seps);	// read comma between parameters
			if(ret) return ret;
			if(word != ",")
				return rlFailExpectComma;
		}
	}

	// expect )
	ret = getWord(word, seps);
	if(ret) return ret;
	if(word != ")")
		return rlFailExpectClose;

	return rlOK;
}

// Processes a define declaration.
RobLoadReturnType DefineInputStream::createDefine()
{
	String seps = whitespace + "{(),";

	RobLoadReturnType ret;
	String word;
	ret = getWord(word, seps); // read define name
	if(ret) return ret;
	if(!word.startsWith("&")) // name has to start with &
		return rlFailExpectAnd;
	if(!isIdentifierValid(word.substring(1)))
		return rlFailInvalidIdentifier;
	//if(defineMap.isSet(word)) // name has to be unique
	//	return rlFailRedefinedDefine;
	Define def(word);
	def.lineOffset = getLineNumber() - 1;
	ret = getWord(word, seps); // read { or (
	if(ret) return ret;
	
	if(word == "(") // this define has params
	{
		do{
			ret = getWord(word, seps); // read param name
			if(ret) return ret;
			if(!isIdentifierValid(word.substring(1)))
				return rlFailInvalidIdentifier;
			if(!word.startsWith("?"))
				return rlFailExpectQuestionMark;
			def.params += word;
			if(def.params.length() > 10)
				return rlFailTooManyDefinePar;

			ret = getWord(word, seps); // read comma
			if(ret) return ret;
			if(word == ")") break;	// finished with params

			if(word != ",")
				return rlFailExpectComma;
		}while(true);

		ret = getWord(word, seps); // read {
		if(ret) return ret;
	}

	if(word != "{")
		return rlFailExpectBOpen;
	// read contents
	int c;
	bool inComment = false; // is true if we're in a endline comment 
	while(true) {
		c = read();
		if(c < 0) return rlFailExpectBClose; // stream must not end before }
		if(!inComment && (c == '}')) break;
		if(c == commentStart) inComment = true;
		if(c == '\n') inComment = false;
		if(!inComment) def.value += ((char)c);
	}
	// add define to map
	defineMap.put(def.name, def);
	return rlOK;
}

String DefineInputStream::getLine(bool skipEmpty)
{
	String ret;
	bool eof = false;
	int c;
	while(true) {
		eof = (curStream == 0);
		c = read();
		if(c >= 0) eof = false;
		else break;
		if(c == '\n') break; // never ever append the newline to the string
		ret += (char)c;
	}
	if(ignoreComments) {
		int commentIndex = ret.lastIndexOf(commentStart);
		if(commentIndex >= 0)
			ret = ret.substring(0, commentIndex);
	}
	if(autoTrim) {
		ret = ret.trim();
	}
	if(skipEmpty && !eof && (ret.length() == 0)) // line is empty => return next line
		return getLine(skipEmpty);
	else return ret;	// return this line
}

int DefineInputStream::getLineNumber()
{
	return infos[curStream].lineNum + infos[curStream].lineOffset;
}

bool DefineInputStream::skipRestOfLine()
{
	int c;
	while(true) {
		c = read();
		if(c < 0) break;
		if(c == '\n') break;
	}
	return (c >= 0);
}

bool DefineInputStream::eos()
{
	if(curStream > 0) return false;
	if(infos[0].buf >= 0) return false;
	infos[0].buf = read();
	return (infos[0].buf < 0);
}

bool DefineInputStream::isIdentifierValid(const String& str)
{
	if(str.length() == 0)
		return false;
	if(!(validAsIdentifierChar[(unsigned char) str[0]] & identCharValidFirst))
		return false;
	for(int i = 1; i < str.length(); i++)
		if(!(validAsIdentifierChar[(unsigned char) str[i]] & identCharValidNext))
			return false;
	return true;
}

void DefineInputStream::initValidIdentifierChars()
{
	int c = 0;
	for(c = 0; c < 192; c++)
		validAsIdentifierChar[c] = identCharInvalid;

	for(c = '0'; c <= '9'; c++)
		validAsIdentifierChar[c] = identCharValidAlways;
	for(c = 'a'; c <= 'z'; c++)
		validAsIdentifierChar[c] = identCharValidAlways;
	for(c = 'A'; c <= 'Z'; c++)
		validAsIdentifierChar[c] = identCharValidAlways;
	for(c = 192; c < 256; c++)
		validAsIdentifierChar[c] = identCharValidAlways;
	
	validAsIdentifierChar['.'] = identCharValidNext;
	validAsIdentifierChar['_'] = identCharValidNext;
}


///////////// inner classes ////////////


DefineInputStream::StreamInfo::StreamInfo() : 
	lineNum(2), lineOffset(0), buf(-1), eol(false), params(false)
{}
DefineInputStream::StreamInfo::StreamInfo(Define& define) :
	lineNum(1), lineOffset(define.lineOffset), buf(-1), eol(false), 
	params(false)
{
}

DefineInputStream::StreamInfo::~StreamInfo()
{
}

DefineInputStream::Define::Define(const String& name) :
	name(name), value(), params(0), lineOffset(0)
{}

InputStream* DefineInputStream::Define::getStream()
{
	return new StringInStream(value);
}


DefineInputStream::StringInStream::StringInStream(const String& str) : str(str), pos(0)
{}

int DefineInputStream::StringInStream::read()
{
	if(pos < str.length())
		return str[pos++];
	else return -1;
}

bool DefineInputStream::StringInStream::eos()
{
	return (pos >= str.length());
}


} // namespace

