/*
 * robtopmgr.h
 * 
 * Copyright (c) 2000-2004 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** @file
  * Declares the top mode tournament manager class
  */

#ifndef __RT_TOP_MANAGER__
#define __RT_TOP_MANAGER__

#include <rtstring.h>
#include <rtfile.h>

#include "robbase.h"
#include "robmanage.h"

using namespace lrt;

namespace rt {

class TopInfo;
class TopTourResult;

/** Manages a top mode tournament. Can replace RobServ. */
class TopManager : public SimManager{

public:
	/** Creates a new TopManager.
	  * @param importOld Whether old, partial results from a previous run of robotour
	  *                  should be used.
	  * @param size The chart size for this competition.
	  * @param competitionFolder The folder in which the result files should be stored. 
	  */
	TopManager(bool importOld, int size, const String& competitionFolder);
	virtual ~TopManager();

	virtual void setup(SimOptions* const options);
	virtual void run();

	// Note that in all of the TopManager classes, "botname" means "bot file name"
	// unlike in the rest of RoboTour where it means "bot program name". 
	// All of these botnames must be file names localized against the competition folder

	String getResultFileName(const String& botname); // get the resultfile filename
	String getInfoFileName(const String& botname);   // get the infofile filename
	String getId(const String& botname);
	bool setIfInTop(const String& botname, const String& top);

	bool saveRes(const String& bot1, const String& bot2, int wins, int looses, int ties);
	TopInfo* updateBotInfo(const String& botname);
	bool doFights(const String& botname); // let a bot fight against the charts
	/** Reads the charts ini file into the given store, which is cleared first. 
	  * This method is not intended for usage in TopManager itself (which should always 
	  * calculate the results from the results files, not from the charts), but for
	  * usage in HTML (or other readable formats) generators. */
	void readCharts(const IniFile& charts, Vector<TopTourResult>& store);
	/** Calculates the charts store from the results files. The store is cleared first.
	  * When done, sorts the store so that it is ordered after points, highest first. */
	void calcCharts(const IniFile& charts, Vector<TopTourResult>& store);
	/** Sorts the store so that it is ordered after points, highest first. */
	void sortCharts(Vector<TopTourResult>& store);
	/** Removes all entries from the store and the charts which are not eligible for the
	  * charts anymore. */
	void checkChartsLength(IniFile& charts, Vector<TopTourResult>& store);

	/** Updates the charts with the values from the given store. */
	bool updateCharts(IniFile& charts, const Array<TopTourResult>& store);

	/** Try to add a program to the currently loaded programs. Checks if the new program
	  * can be assembled correctly. If it doesn't, it is not added and <tt>false</tt> is
	  * returned.  */
	bool addAndCheck(const String& botname);

protected:
	TourInitInfo getTourInitInfo();

private:
	bool importOld;
	int size;
	String chartsFileName;
	File competitionFolder;
	Vector<String> newFiles;
};

/** Holds special information about a certain robot. This information is only used
  * by TopManager and its HTML generators. */
class TopInfo {
public:
	TopInfo(const String& fileName);
	~TopInfo();

	String fileName;

	StringMap<Program::HeaderField> headers; 

	String id;           ///< ID of one bot
	Time date;         ///< date of Info-file-creation
	String top;          ///< whether the bot is in the top

	void fetchDate();	 ///< fetches current date into this TopInfo
	void incID();
	bool read();
	bool write();
};

/** Extends TourResult to store additional information as required by TopManager. */
class TopTourResult : public TourResult {
public:
	static int compareByPoints(const TopTourResult& t1, const TopTourResult& t2);
	static int compareByDate(const TopTourResult& t1, const TopTourResult& t2);
	
	/// DON't use!!
	TopTourResult() : TourResult(), id(0), infoId(0), top(), date() {}
	/// Use this one instead.
	TopTourResult(const StringMap<Program::HeaderField>& headers, const String& fileName) :
	    TourResult(headers, fileName), id(0), infoId(0), top(), date()
	{}

	int id; ///< required only by the TopManager
	int infoId; ///< required only for det htmls (version check)
	String top; ///< required only for det htmls ?
	Time date;
};

} // namespace

# endif
