#include "module.h"
#include "module_registry.h"
#include "utils.h"

#include <algorithm>
#include <string>
using namespace std;

void Module::service()
{
        for (vector<Module*>::iterator i = childModules.begin(); i < childModules.end(); ++i)
                (*i)->service();
}

void Module::updateParent(const string& keyName, const DataSet& data)
{
        if (parentModule != NULL) {
                addStickyToOutput();
                
                if (output->count() > 0) {
                        addToOutput(keyName, data);

                        sendOutput();
                } else {
                        parentModule->update(keyName, data);
                        parentModule->updated(keyName, data);
                }
        }
}

void Module::addStickyToOutput()
{
        /* send sticky attributes through children */
        if (env.validIndex("sticky")) {
                DataSet& sticky = env["sticky"];
                for (int i = 0; i < sticky.count(); ++i) {
                        string attributeName = sticky.getString(i);
                        if (env.validIndex(attributeName)) {
                                addToOutput(attributeName, env[attributeName]);
                        }
                }
        }
}

void Module::sendOutput()
{
        addStickyToOutput();
        parentModule->receiveOutput(*output);
        clearOutput();
}

void Module::receiveOutput(const DataSetMap& rOutput)
{
        env.integrate(rOutput);
        outputReceived(rOutput);
}

void Module::outputReceived(const DataSetMap& rOutput)
{
        for (DataSetMap::const_iterator i = rOutput.begin(); i != rOutput.end(); ++i) {
                updated(i->first, i->second);
        }
}

Module::~Module()
{
        if (childModules.size() != 0)
                rpwarning("module destroyed while it still has children");
}

Module* Module::newChild(const string& typeName)
{
        const ModuleInfo &m = ModuleRegistry::instance()->module_info(typeName);
        ModuleInfo::ModuleConstructor *c = m.constructor();
        Module* aNewModule = c();
        aNewModule->setParent(this);
        addChild(aNewModule);
        return aNewModule;
}

void Module::delChild(Module* m)
{
        childModules.erase(find(childModules.begin(), childModules.end(), m));
        delete m;
}

// for each default that isn't set, set it
void Module::setDefaults()
{
        const DataSetMap& minf = get_module_info();
        DataSet defs = minf["supported_var_defaults"];
        DataSet vars = minf["supported_vars"];
        for (int i = 0; i < vars.count(); i++) {
                string v = vars.getString(i);
                if (!env.validIndex(v)) {
                        string d = defs.getString(i);
                        if (d != "") {
                                set(v, DataSet(d));
                        }
                }
        }
}

string Module::send_on_command(const Path& path, const string& command, const string& parameter)
{
        Path npath(path);
        npath.remove_first();
        
        int child_num = path.first();
        assert(child_num >= 0 && child_num < int(childModules.size()));
        Module& child = *childModules[child_num];

        return child.issue_command(npath, command, parameter);
}

string Module::execute_command(const string& command_name, const string& parameter)
{
        return default_execute_command(command_name, parameter);
}

string Module::default_execute_command(const string& command_name, const string& parameter)
{
        if (command_name == "get_value") {
                if (env.validIndex(parameter))
                        return env[parameter].asString();
        } else if (command_name == "set_value") {
                const char *lhs_start = parameter.c_str();
                char *rhs_start = strchr(lhs_start, '=');
                string lhs(parameter, 0, rhs_start - lhs_start);
                string rhs(parameter, rhs_start - lhs_start + 1);
                set(lhs, DataSet(rhs));
        } else if (command_name == "get_type") {
                return getName();
        } else if (command_name == "create_child") {
                newChild(parameter);
        } else if (command_name == "delete_child") {
                int child_num = strtol(parameter.c_str(), 0, 10);
                assert(child_num >= 0 && child_num < int(childModules.size()));
                delChild(childModules[child_num]);
        } else if (command_name == "child_count") {
                return int_to_string(childModules.size());
        } else if (command_name == "post_startup") {
                postStartup();
        } else if (command_name == "pre_shutdown") {
                preShutdown();
        } else if (command_name == "overlay_defaults") {
                setDefaults();
        } else {
                cerr << "command not recognised \"" << command_name << "\"\n";
        }
        return "";
}

string Module::issue_command(const Path& path, const string& command,
                             const string& parameter)
{
        if (path.length() == 0) {
                return execute_command(command, parameter);
        } else {
                return send_on_command(path, command, parameter);
        }
}

void Module::postStartup()
{
}
        
void Module::preShutdown()
{
}

int Module::child_num(Module *ch)
{
        for (int i = 0; i < int(childModules.size()); i++) {
                if (childModules[i] == ch)
                        return i;
        }
        return -1;
}

void Module::return_message(const DataSet& msg, Path path)
{
        if (parentModule != NULL) 
        {
                path.add_start(parentModule->child_num(this));
                parentModule->return_message(msg, path);
        }
}
