#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include "sg_include.h"

/* This program sends a user specified number of TEST UNIT READY commands
   to the given sg device. Since TUR is a simple command involing no
   data transfer (and no REQUEST SENSE command iff the unit is ready)
   then this can be used for timing per SCSI command overheads.

*  Copyright (C) 2000-2002 D. Gilbert
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2, or (at your option)
*  any later version.

   Invocation: sg_turs -n=<number_of_turs> <sg_device>
   The value give to "-n=" can have the following multipliers:
     'c','C'  *1       'b','B' *512      'k' *1024      'K' *1000
     'm' *(1024^2)     'M' *(1000^2)     'g' *(1024^3)  'G' *(1000^3)

   Version 02.05 (20021212)

6 byte TEST UNIT READY command:
[0x00][   |lu][res   ][res   ][res   ][res   ]

*/

#define OFF sizeof(struct sg_header)
#define TUR_CMD_LEN 6

#define ME "sg_turs: "

#define EBUFF_SZ 512


int get_num(char * buf)
{
    int res, num;
    char c;

    res = sscanf(buf, "%d%c", &num, &c);
    if (0 == res)
        return -1;
    else if (1 == res)
        return num;
    else {
        switch (c) {
        case 'c':
        case 'C':
            return num;
        case 'b':
        case 'B':
            return num * 512;
        case 'k':
            return num * 1024;
        case 'K':
            return num * 1000;
        case 'm':
            return num * 1024 * 1024;
        case 'M':
            return num * 1000000;
        case 'g':
            return num * 1024 * 1024 * 1024;
        case 'G':
            return num * 1000000000;
        default:
            fprintf(stderr, "unrecognized multiplier\n");
            return -1;
        }
    }
}


int main(int argc, char * argv[])
{
    int sg_fd, k;
    unsigned char turCmdBlk [TUR_CMD_LEN] =
                                {0x00, 0, 0, 0, 0, 0};
    unsigned char turBuff[OFF + TUR_CMD_LEN];
    int turInLen = sizeof(turBuff);
    struct sg_header * sghp = (struct sg_header *)turBuff;
    char * file_name = 0;
    char ebuff[EBUFF_SZ];
    int num_turs = 0;
    int num_errs = 0;
    int do_time = 0;
    struct timeval start_tm, end_tm;
    struct stat a_st;
    int block_dev = 0;

    for (k = 1; k < argc; ++k) {
        if (0 == strncmp("-n=", argv[k], 3)) {
            num_turs = get_num(argv[k] + 3);
            if (num_turs < 0) {
                printf("Couldn't decode number after '-n' switch\n");
                file_name = 0;
                break;
            }
        }
        else if (0 == strcmp("-t", argv[k]))
            do_time = 1;
        else if (*argv[k] == '-') {
            printf("Unrecognized switch: %s\n", argv[k]);
            file_name = 0;
            break;
        }
        else if (0 == file_name)
            file_name = argv[k];
        else {
            printf("too many arguments\n");
            file_name = 0;
            break;
        }
    }
    if ((0 == file_name) || (num_turs <= 0)) {
        printf("Usage: 'sg_turs -n=<num_of_test_unit_readys> <sg_device>'\n");
        return 1;
    }

    if ((sg_fd = open(file_name, O_RDWR)) < 0) {
        snprintf(ebuff, EBUFF_SZ, ME "error opening file: %s", file_name);
        perror(ebuff);
        return 1;
    }
    if (fstat(sg_fd, &a_st) < 0) {
        fprintf(stderr, "could do fstat() on fd ??\n");
        close(sg_fd);
        return 1;
    }
    if (S_ISBLK(a_st.st_mode))
        block_dev = 1;
    /* Just to be safe, check we have a sg device by trying an ioctl */
    if (block_dev || (ioctl(sg_fd, SG_GET_TIMEOUT, 0) < 0)) {
        printf("sg_turs: %s doesn't seem to be an sg device\n", file_name);
        close(sg_fd);
        return 1;
    }

    /* Prepare TEST UNIT READY command */
    sghp->reply_len = OFF;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;     /* some apps assume this is done ?!? */
    sghp->sense_buffer[0] = 0; /* only needed for old sg driver */
    memcpy(turBuff + OFF, turCmdBlk, TUR_CMD_LEN);
    if (do_time) {
        start_tm.tv_sec = 0;
        start_tm.tv_usec = 0;
        gettimeofday(&start_tm, NULL);
    }

    for (k = 0; k < num_turs; ++k) {
        /* Send TEST UNIT READY command */
        sghp->pack_id = k;
        if (write(sg_fd, turBuff, turInLen) < 0) {
            perror("sg_turs: write error");
            close(sg_fd);
            return 1;
        }
        
        /* Read back status (sense_buffer) */
        if (read(sg_fd, turBuff, OFF) < 0) {
            perror("sg_turs: read error");
            close(sg_fd);
            return 1;
        }
        if (k != sghp->pack_id) /* this shouldn't happen */
            printf("Test Unit Ready pack_id mismatch, wanted=%d, got=%d\n", 
                   k, sghp->pack_id);

        if (0 != sghp->sense_buffer[0])
            ++num_errs;
    }
    if ((do_time) && (start_tm.tv_sec || start_tm.tv_usec)) {
        struct timeval res_tm;
        double a, b;

        gettimeofday(&end_tm, NULL);
        res_tm.tv_sec = end_tm.tv_sec - start_tm.tv_sec;
        res_tm.tv_usec = end_tm.tv_usec - start_tm.tv_usec;
        if (res_tm.tv_usec < 0) {
            --res_tm.tv_sec;
            res_tm.tv_usec += 1000000;
        }
        a = res_tm.tv_sec;
        a += (0.000001 * res_tm.tv_usec);
        b = (double)num_turs;
        printf("time to perform commands was %d.%06d secs",
               (int)res_tm.tv_sec, (int)res_tm.tv_usec);
        if (a > 0.00001)
            printf("; %.2f operations/sec\n", b / a);
        else
            printf("\n");
    }

    printf("Completed %d Test Unit Ready commands with %d errors\n",
            num_turs, num_errs);
    close(sg_fd);
    return 0;
}
