//  libsigcperl -- a helper library for writing XSUB wrappers of libsigc++
//  Copyright (C) 2002 Ron Steinke
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Library General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Library General Public License for more details.
//
//  You should have received a copy of the GNU Library General Public
//  License along with this library; if not, write to the 
//  Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
//  Boston, MA  02111-1307  USA.

#ifndef SIGC_PERL_DATA_H
#define SIGC_PERL_DATA_H

#include <assert.h>
#include <vector>

extern "C" {
#include "EXTERN.h"
#include "perl.h"
#ifdef PERL_IMPLICIT_CONTEXT
#include "XSUB.h"
#endif
}

namespace SigCPerl {

class Data
{
 public:
  class Iter {
   public:
    friend class Data;
    typedef std::vector<SV*>::const_iterator IType;
    Iter(IType i) : m_i(i) {}

    Iter& operator++() {++m_i; return *this;}
    Iter operator++(int) {Iter new_i(*this); ++m_i; return new_i;}
    SV* operator*() {return sv_mortalcopy(*m_i);}
    bool operator==(const Iter &i) {return m_i == i.m_i;}
    bool operator!=(const Iter &i) {return m_i != i.m_i;}
    Iter operator+(int diff) {Iter new_i(*this); new_i.m_i += diff; return new_i;}

   private:
    IType m_i;
  };

  Data() throw() : m_data(0) {}
  Data(const Data&) throw();
  Data(const Iter&, const Iter&) throw(); // array slice
  Data(const Data&, const Data&) throw(); // concatenation
  ~Data() throw();

  Data& operator=(const Data&) throw();

  // Ordinary contaier ops
  void push_back(SV *sv) throw() {m_data.push_back(newSVsv(sv));}
  template<class R> // Need to include convert.h to use this
    void push_back(R r) throw() {m_data.push_back(GetSV(r));}
  Iter begin() const {return m_data.begin();}
  Iter end() const {return m_data.end();}
  unsigned int size() const {return m_data.size();}
  SV* operator[](unsigned int i) const {return sv_mortalcopy(m_data[i]);}

  // Push/Pop onto/off of the perl stack
  void push_stack() const throw();
  static Data pop_stack(unsigned int count) throw();

 private:
  Data(unsigned int count) : m_data(count) {}

  std::vector<SV*> m_data;
};

// Some macros which interact with the Perl XSUB macros

#define SIGCPERL_GET_STACK_DATA_FIRST(data, first) \
  for(int i = first; i < items; ++i) data.push_back(ST(i));

#define SIGCPERL_GET_STACK_DATA(data) SIGCPERL_GET_STACK_DATA_FIRST(data, 1)

} // namespace SigCPerl

#endif // SIGC_PERL_DATA_H
