/*
 * raw2image.cc -- Converts  a Raw DV stream to a saved image (various formats)
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <unistd.h>

#include <iostream>
#include <string>
#include <cmath>
using std::string;
using std::cout;
using std::cerr;
using std::cin;
using std::endl;

#include <preferences.h>
#include <Diagnostics.h>
#include <RawDVFileInput.h>
#include <Threader.h>
#include <extensions.h>
#include <time.h>
#include <pthread.h>
#include <gdk_imlib.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

class Raw2Image : public Threader
{
	private:
		int deinterlace;
		int count;
		string stem;
		string extension;
		bool multiple;
		uint8_t *image;
		int quality;
		int width;
		int height;
		bool ppm;
		int pump_size;

	public:
		Raw2Image( ) : 
			deinterlace( 0 ),
			count( 0 ),
			stem( "" ),
			extension( "" ),
			multiple( false ),
			quality( 100 ),
			width( 0 ),
			height( 0 ),
			ppm( false ),
			pump_size( 25 )
		{
			image = new uint8_t[ 720 * 576 * 3 ];
		}

		virtual ~Raw2Image( )
		{
			delete image;
		}

		virtual string LogId( )
		{
			return "Raw2Image";
		}

		void SetDeinterlace( int _deinterlace )
		{
			deinterlace = _deinterlace;
		}

		void SetFile( string file )
		{
			if ( stem != "" || extension != "" )
				throw "Image file name can only be specified once.";

			extension = file.substr( file.rfind( "." ) );
			stem = file.substr( 0, file.rfind( "." ) );

			if ( stem == "" || extension == "" )
				throw "Filename specified was invalid. Must be name.type (ie: kino.jpeg).";
		}

		void SetMultiple( bool _multiple )
		{
			multiple = _multiple;
		}

		void SetSize( char *value )
		{
			sscanf( value, "%dx%d", &width, &height );
		}

		void SetQuality( int _quality )
		{
			quality = _quality;
		}

		void SetPPMOutput( bool _value )
		{
			ppm = _value;
			SetMultiple( true );
		}

		void SetPumpSize( int _pump_size )
		{
			pump_size = _pump_size;
		}

	protected:

		bool OutputPPM( uint8_t *image, int width, int height )
		{
			cout << "P6 " << width << " " << height << " 255" << endl;
			return fwrite( image, width * height * 3, 1, stdout ) == 1;
		}

		bool OutputFrame( Frame &frame )
		{
			bool ret = true;
			char filename[ 132 ];

			if ( multiple )
				sprintf( filename, "%s%04d%s", stem.c_str( ), count ++, extension.c_str( ) );
			else
				sprintf( filename, "%s%s", stem.c_str( ), extension.c_str( ) );

			GdkImlibSaveInfo saveInfo = {0, 0, 0, 0, 0, 0};

			saveInfo.quality = (int) ( (float) quality / 100.0 * 256.0 );
			Preferences::getInstance().deinterlacePreview = deinterlace;

			frame.decoder->quality = DV_QUALITY_BEST;

			frame.ExtractPreviewRGB( image );

			if ( ppm )
			{
				if ( width != 0 && height != 0 )
				{
					GdkPixbuf *input = gdk_pixbuf_new_from_data( image, GDK_COLORSPACE_RGB, FALSE, 8, frame.GetWidth( ), frame.GetHeight( ), frame.GetWidth( )*3, NULL, NULL );
					GdkPixbuf *scaled = gdk_pixbuf_scale_simple( input, width, height, GDK_INTERP_HYPER );
					ret = OutputPPM( gdk_pixbuf_get_pixels( scaled ), width, height );
					gdk_pixbuf_unref( scaled );
					gdk_pixbuf_unref( input );
				}
				else
				{
					ret = OutputPPM( image, frame.GetWidth( ), frame.GetHeight( ) );
				}
			}
			else
			{
				GdkImlibImage *original = gdk_imlib_create_image_from_data( image, NULL, frame.GetWidth( ), frame.GetHeight( ) );

				if ( width != 0 && height != 0 )
				{
					GdkImlibImage *im = gdk_imlib_clone_scaled_image( original, width, height );
					gdk_imlib_save_image( im, filename, &saveInfo );
					gdk_imlib_destroy_image( im );
				}
				else
				{
					gdk_imlib_save_image( original, filename, &saveInfo );
				}

				gdk_imlib_destroy_image( original );
			}


			return ret;
		}

		void Thread( )
		{
			bool running = true;

			RawDVFileInput input;
			input.SetPumpSize( pump_size );
			input.SetFile( stdin );

			input.ThreadStart( );

			while( running && ThreadIsRunning( ) )
			{
				if ( input.GetOutputAvailable( ) > 0 )
				{
					Frame &frame = input.GetOutputFrame( );
					running = OutputFrame( frame );
					fflush( stdout );
					input.QueueInputFrame( );
				}
				else
				{
					running = input.ThreadIsRunning( ) && input.PumpIsNotCleared( );
				}
			}

			input.ThreadStop( );
		}
};

static void Usage( )
{
	cerr << "Usage: raw2image [ options ] file.extension" << endl;
	cerr << "Where: options are" << endl;
	cerr << "       -ppm         - PPM pipe output" << endl;
	cerr << "       -s WxH       - size of output image" << endl;
	cerr << "       -i type      - deinterlacing type (0 = none, 1 = simple)" << endl;
	cerr << "       -m           - multiple raw frames provided" << endl;
	cerr << "       -q quality   - quality of output image" << endl;
	exit( 0 );
}

int main( int argc, char **argv )
{
	Raw2Image output;

	if ( isatty( fileno( stdin ) ) )
	{
		cerr << "Input must must be obtained from a pipe or a file." << endl;
		Usage( );
	}

	Diagnostics::SetApp( "raw2image" );

	try
	{
		for ( int i = 1; i < argc; i ++ )
		{
			if ( !strcmp( argv[ i ], "-i" ) )
				output.SetDeinterlace( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-p" ) || !strcmp( argv[ i ], "-ppm" ) )
				output.SetPPMOutput( true );
			else if ( !strcmp( argv[ i ], "-m" ) )
				output.SetMultiple( true );
			else if ( !strcmp( argv[ i ], "-q" ) )
				output.SetQuality( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-s" ) )
				output.SetSize( argv[ ++ i ] );
			else if ( !strcmp( argv[ i ], "-u" ) )
				output.SetPumpSize( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-v" ) )
				Diagnostics::SetLevel( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "--help" ) )
				Usage( );
			else
				output.SetFile( string( argv[ i ] ) );
		}

		output.ThreadExecute( );
	}
	catch ( const string exc )
	{
		cerr << "Exception thrown: " << exc << endl;
	}

	return 0;
}
