/*
   SwingWT
   Copyright(c)2003-2004, R. Rawson-Tetley

   For more information on distributing and using this program, please
   see the accompanying "COPYING" file.

   Contact me by electronic mail: bobintetley@users.sourceforge.net

   Log is at the end of this file now since it had gotten rather large.

*/


package swingwtx.swing;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.*;

import java.io.*;
import java.net.*;
import java.util.*;

/**
 * Utilities required for SwingWT. Handles management
 * of SWT display, SwingWT event pump and some handy stuff
 * for calculating renderered text size, determining
 * platform, etc.
 *
 * @author Robin Rawson-Tetley
 */
public abstract class SwingWTUtils {

    private static boolean isDebug = true;
    public static boolean showInternalSWTExceptions = false;
    
    private final static String VERSION = "0.85 (021104)";
    private final static String COPYRIGHT = "This is SwingWT (http://swingwt.sourceforge.net)\n" +
                                            "Version: " + VERSION + "\n" +
                                            "Copyright(c)2003-2004, R.Rawson-Tetley and other contributors.\n\n" +
                                            "This library is distributed in the hope that it will be useful,\n" +
                                            "but WITHOUT ANY WARRANTY; without even the implied warranty of\n" + 
                                            "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU\n" +
                                            "Lesser General Public Licence for more details.";
    
    private static Display display = null;
    
    /** Whether we have a dispatch thread going */
    private static boolean eventsDispatching = false;
    /** Who owns the event dispatcher */
    private static Object eventsDispatchOwner = null;
    /** Have we set a shutdown hook to remove temp files? */
    private static boolean setDeleteTempOnShutdown = false;
    /** List of temporary files created in this session */
    private static Vector tempFileList = new Vector();
    /** Dispatch thread retarding for performance */
    private static boolean retardEventDispatch = false;
    private static int retardationms = 2000;
    /** Window reference count. If it reaches 0 then we end the dispatch thread */
    private static int windowReferenceCount = 0;
    /** Whether or not to use SWT fast virtuals. 
     *  NOTE: As at M8 there is an issue with Win32 environments 
     *  and virtual tables so it defaults to off for Windows.
     */
    private static boolean useSWTFastVirtualTables = !isWindows();
    
    private static String tempDirectory = System.getProperty("user.home") + File.separator + "tmp" + File.separator + "swingwt";
    
    public static String getVersion() { return VERSION; }
    
    /** The height of the MacOS X menubar in pixels */
    public static int MACOSX_MENUBAR_BOTTOM = 20;
    
    public static synchronized void incrementWindowReferences() {
        windowReferenceCount++;    
    }
    
    public static synchronized void decrementWindowReferences() {
        windowReferenceCount--;
        // TODO: (Dan) This is not the best solution, but killing the dispatch thread
        // causes problems if another window is created after this.  Commenting out for now.
        //if (windowReferenceCount == 0) stopEventDispatchRunning();
    }
    
    public static void setShowSwingWTInfoOnStartup(boolean b) {
        isDebug = b;    
    }
    
    /** Determines whether the event dispatch thread is retarded for extra performance */
    public static synchronized void setRetardDispatchThread(boolean b) {
        retardEventDispatch = b;    
    }
    /** Determines whether the event dispatch thread is retarded for extra performance */
    public static synchronized void setRetardDispatchThread(boolean b, int ms) {
        retardEventDispatch = b;    
        retardationms = ms;
    }
    
    /** 
     * If your code is an Eclipse plugin (ie. Something outside
     * of SwingWT and your program has started using SWT components
     * in the current VM and an event dispatcher is already going), 
     * Call this routine with (true) to have SwingWT use the other
     * program's event dispatcher (rather than it's own thread).
     */
    public synchronized static void setEclipsePlugin(boolean b) {
        if (b) {
            eventsDispatching = true;
	    display = Display.getDefault();
	}
	else {
            eventsDispatching = false;
	}
    }
    
    /** Returns whether SwingWT is using fast SWT virtual tables */
    public static boolean isUseSWTFastVirtualTables() { return useSWTFastVirtualTables; }
    /** Set this to true to use fast SWT Virtual tables (off by default) */
    public static void setUsesSWTFastVirtualTables(boolean b) { useSWTFastVirtualTables = b; }
    
    /**
     * Startup routine for MacOSX programs. This isn't very pretty, but the threading
     * model for java under MacOSX is fundamentally broken and requires the event
     * pump to run on the main thread.
     *
     * @param run The startup code to run (this call will block)
     */
    public static void initialiseMacOSX(Runnable run) {
	
	// Shove it onto the event pump
        getDisplay().asyncExec(run);

        // Start the event pump running
	checkEventDispatcher();
	
    }
    
    /** 
     *  Checks whether the event dispatch thread is running, and starts it
     *  if it isn't.
     */
    public synchronized static void checkEventDispatcher() {
        if (!eventsDispatching) {
            
            eventsDispatching = true;
            
            // Make sure there is a VM hook to dump the temp directory when we're finished
            // and do any necessary cleaning up
            checkShutdownHook();
            
            // Event dispatch thread itself
            Thread pump = new Thread() {
                public void run() {
                    
                    // Make sure display is set
                    display = Display.getDefault();
                    
                    // Debug info
                    if (isDebug)
                        System.out.println(COPYRIGHT);
                    
                    while (eventsDispatching) {
                        try {
                            if (!display.readAndDispatch()) {
                                // Send this thread to sleep to allow other
                                // processes a go :)
                                display.sleep();
                            }
                            // If thread retardation is on, sleep
                            if (SwingWTUtils.isRetardDispatchThread()) {
                                try {
                                    Thread.sleep(SwingWTUtils.getRetardationInterval());
                                }
                                catch (InterruptedException e) {}
                            }
                        }
                        // By catching all exceptions and errors here, we
                        // can prevent the app going down for small event
                        // related problems. User code can still catch where
                        // necessary.
                        catch (org.eclipse.swt.SWTException e) {
			    
			    // If we have a "Device is disposed" error, and
			    // we're on MacOS X, then somebody has hit the
			    // Quit button from the app menu and we need to
			    // stop the dispatcher running so the program
			    // can exit gracefully.
			    if (e.getMessage().startsWith("Device is disposed") &&
			        isMacOSX()) {
				stopEventDispatchRunning();
				return;
			    }

			    // If we are showing exceptions, dump the stack
			    // trace out to the console.
                            if (showInternalSWTExceptions)
                                e.printStackTrace();
			    
                        }
                        catch (Error e) {
                            e.printStackTrace();
                        }
                        catch (Exception e) {
                            e.printStackTrace();    
                        }
                    }
                }
            };

	    // MacOSX needs to be on the main thread, so this call should
	    // block. Other platforms can spawn a new thread to handle the
	    // pump like Swing.
	    if (isMacOSX())
		pump.run();
	    else
		pump.start();
        }
    }
    
    /** @return true if the dispatch thread is currently retarded */
    public static boolean isRetardDispatchThread() { return retardEventDispatch; }
    /** @return The dispatch thread retardation in milliseconds */
    public static int getRetardationInterval() { return retardationms; }
    
    public static synchronized Display getDisplay() {
        
        // For MacOSX, we need to grab the display on this
	// thread and return it
	if (isMacOSX()) {
	    display = Display.getDefault();
	    return display;
	}
	    
        // Make sure event dispatch is running (since this
        // creates the display).
        checkEventDispatcher();
        
        // Wait for the display to be set
        while (display == null) {
            try {
                Thread.sleep(50);
            }
            catch (InterruptedException e) {}
        }
        
        return display;
    }
    
    /** Checks to see if this users temp directory is there
     *  and creates it if it isn't. 
     *  @throws IOException if the directory couldn't be created.
     */
    private static void checkForTempDirectory() throws IOException {
        File f = new File(tempDirectory);
        if (!f.exists())
            f.mkdirs();
    }
    
    /** Determines if an SWT peer is available for use
     */
    public static boolean isSWTControlAvailable(org.eclipse.swt.widgets.Control c) {
        if (c == null) return false;
        if (c.isDisposed()) return false;
        return true;
    }
    
    /** Determines if an SWT menu peer is available for use
     */
    public static boolean isSWTMenuControlAvailable(org.eclipse.swt.widgets.MenuItem c) {
        if (c == null) return false;
        if (c.isDisposed()) return false;
        return true;
    }
    
    public synchronized static boolean isEventDispatchRunning() {
        return eventsDispatching;    
    }
    
    /**
     * Stops the event dispatch thread running (if it isn't running.
     * does nothing).
     */
    public static synchronized void stopEventDispatchRunning() {
        eventsDispatching = false;
    }
    
    /**
     * @param url The path to inspect.
     * @return TRUE if the passed URL is contained in a JAR file.
     * @throws IOException
     */
    public static boolean isUrlInJar(URL url) throws IOException {
        return url == null ? false : url.toString().startsWith("jar:");
    }
    
    /**
     * Given a byte array of content, writes it to a temporary file and then
     * returns the path to it as a URL
     * @param contents The content of the file
     * @param type The file extension to use
     * @throws IOException if an error occurs
     */
    public static URL stringToTempFile(byte[] contents, String type) throws IOException {
        
        // Make sure we have a temp directory
        checkForTempDirectory();
        
        // Generate a random file name and keep doing it until we get a unique one
        File f = null;
        String fName = null;
        do {
            fName = tempDirectory + File.separator +
                ((int) (Math.random() * 10000000)) + "." + type;
            f = new File(fName);
        } while (f.exists());
        
        System.out.println("TEMP: Creating temp file " + fName);
        FileOutputStream out = new FileOutputStream(f);
        out.write(contents);
        out.close();
        
        // Remember this file for later deletion
        tempFileList.add(fName);
        
        return new URL("file://" + fName);
    }
    
    public static void clearTempDirectory() {
        // Abandon if we created no files in this session
        if (tempFileList.size() == 0) return;
        Iterator i = tempFileList.iterator();
        while (i.hasNext()) {
            try {
                File f = new File((String) i.next());
                f.delete();
            }
            catch (Exception e) { e.printStackTrace(); }
        }    
    }
    
    public static void checkShutdownHook() {
        if (!setDeleteTempOnShutdown) {
            Runtime.getRuntime().addShutdownHook( new Thread() {
                public void run() {
                    
                    clearTempDirectory();  
                    
                    // Unfortunately, we use some finalizer based
                    // cleanup for manually created Color/Font/Graphics (only way really)
                    // so we want to make sure they run before closing so we don't 
                    // lose OS resources. Might as well do a garbage collection too :-)
                    
                    // This is a real "damned if you do, damned if you don't" situation -
                    // the problem is that we have to use finalizers to kill the native
                    // resources because we know they should run at some point when the
                    // object is finished with. In practice, the best you can hope for
                    // is "on shutdown, but maybe before". The other option would be to
                    // manage the damn things myself, but by keeping a list, I ensure
                    // the objects won't be collected until I destroy them, and I have
                    // no way of knowing whether they are being used in on-screen 
                    // components - I can guarantee it at shutdown however, but ideally
                    // I want them to run before that! So I'm stuck with fucking
                    // finalizers. Best thing to do is call System.runFinalization()
                    // yourself in your app when you've finished with awt.Color/Graphics/Font
                    // (or, each of those AWT mapped objects has a dispose() method to
                    // kill off the native resource).
                    
                    // Another option could have been the use of WeakReferences, however
                    // this would be very difficult to code, and not all VMs support them
                    // (we care more about free VMs than Sun's)
                    System.runFinalization();
                    System.gc();
                    
                }
            });
            setDeleteTempOnShutdown = true;
        }
    }
    
    private static boolean checkedOS = false;
    private static boolean isWin = false;
    private static boolean isMacOS = false;
    /** Returns true if this is a windows platform */
    public static boolean isWindows() {
        if (!checkedOS)
            isWin = SWT.getPlatform().equals("win32");
        return isWin;
    }
    /** Returns true if this is a MacOS X platform */
    public static boolean isMacOSX() {
	if (!checkedOS)
	   isMacOS = SWT.getPlatform().equals("carbon");
	return isMacOS;
    }
    
    /**
     * Saves an image to a JPEG. Not sure where to put this
     * since the only support Java has for this, is in a hidden
     * com.sun.jpeg.JPEGEncoder class which is not part of the
     * formal interface.
     * Anyway, I'm sticking it here.
     */
    public static void saveImageToJPG(swingwt.awt.Image image, OutputStream stream) {
        org.eclipse.swt.graphics.ImageLoader il = new org.eclipse.swt.graphics.ImageLoader();
        il.data = new org.eclipse.swt.graphics.ImageData[] { image.image.getImageData() };
        il.save(stream, org.eclipse.swt.SWT.IMAGE_JPEG);
        il = null;
    }
    
    /**
     * Saves an image to a GIF. Not sure where to put this
     * since the only support Java has for this, is in a hidden
     * com.sun.gif.GIFEncoder class which is not part of the
     * formal interface.
     * Anyway, I'm sticking it here.
     */
    public static void saveImageToGIF(swingwt.awt.Image image, OutputStream stream) {
        org.eclipse.swt.graphics.ImageLoader il = new org.eclipse.swt.graphics.ImageLoader();
        il.data = new org.eclipse.swt.graphics.ImageData[] { image.image.getImageData() };
        il.save(stream, org.eclipse.swt.SWT.IMAGE_GIF);
        il = null;
    }
    
        /**
     * Saves an image to a JPEG. Not sure where to put this
     * since the only support Java has for this, is in a hidden
     * com.sun.png.PNGEncoder class which is not part of the
     * formal interface.
     * Anyway, I'm sticking it here.
     */
    public static void saveImageToPNG(swingwt.awt.Image image, OutputStream stream) {
        org.eclipse.swt.graphics.ImageLoader il = new org.eclipse.swt.graphics.ImageLoader();
        il.data = new org.eclipse.swt.graphics.ImageData[] { image.image.getImageData() };
        il.save(stream, org.eclipse.swt.SWT.IMAGE_PNG);
        il = null;
    }
    
    /** Used as a return val from renderStringWidth routines */
    private static int intretval = 0;
    /** 
     * Given a piece of text, this routine will evaluate how many pixels wide it will
     * be when renderered in the default system font. This is used by JTable and
     * JList to determine the column widths
     */
    public static int getRenderStringWidth(final String text) {
        SwingUtilities.invokeSync( new Runnable() {
            public void run() {
                org.eclipse.swt.graphics.GC gc = null;
                org.eclipse.swt.graphics.Image im = null;
                if (getDisplay().getActiveShell() == null) {
                    im = new org.eclipse.swt.graphics.Image(getDisplay(), getDisplay().getBounds()); 
                    gc = new org.eclipse.swt.graphics.GC(im);
                }
                else
                    gc = new org.eclipse.swt.graphics.GC(getDisplay().getActiveShell());

                // We need to calculate differently for Windows platforms. Isn't it always the
                // way? This is because Windows shows the 3 dots - even if the text just fits! So we
                // have to make it even larger. What a piece of shit Windows is
                org.eclipse.swt.graphics.Point p = null;
                if (isWindows())
                    p = gc.stringExtent(text + "WWW");
                else
                    p = gc.stringExtent(text + "W");
                int width = p.x;
                gc.dispose();
                gc = null;
                if (im != null) im.dispose();
                im = null;
                intretval = width;
            }
        });
        return intretval;
    }
    
   /** 
     * Given a piece of text, this routine will evaluate how many pixels high it will
     * be when renderered in the default system font. This is used by JPanel to
     * determine the extra height needed on the insets for laying out purposes
     */
    public static int getRenderStringHeight(final String text) {
        SwingUtilities.invokeSync( new Runnable() {
            public void run() {
                org.eclipse.swt.graphics.GC gc = null;
                org.eclipse.swt.graphics.Image im = null;
                if (getDisplay().getActiveShell() == null) {
                    im = new org.eclipse.swt.graphics.Image(getDisplay(), getDisplay().getBounds()); 
                    gc = new org.eclipse.swt.graphics.GC(im);
                }
                else
                    gc = new org.eclipse.swt.graphics.GC(getDisplay().getActiveShell());

                org.eclipse.swt.graphics.Point p = gc.stringExtent(text + "W");
                int height = p.y;
                gc.dispose();
                gc = null;
                if (im != null) im.dispose();
                im = null;
                intretval = height;
            }
        });
        return intretval;
    }
    
    /**
     * Because GCJ/GIJ Classpath doesn't support StringBuffer.indexOf, we have
     * to have a replacement that uses Strings instead.
     * @param buffer The StringBuffer to find in
     * @param string The String to find
     * @return The index of the string or -1 if it wasn't found
     */
    public static int getStringBufferIndexOf(StringBuffer buffer, String string) {
        return buffer.toString().indexOf(string);
    }
    /**
     * Because GCJ/GIJ Classpath doesn't support StringBuffer.indexOf, we have
     * to have a replacement that uses Strings instead.
     * @param buffer The StringBuffer to find in
     * @param string The String to find
     * @param fromIndex The char index to search from
     * @return The index of the string or -1 if it wasn't found
     */
    public static int getStringBufferIndexOf(StringBuffer buffer, String string, int fromIndex) {
        return buffer.toString().indexOf(string, fromIndex);    
    }
    
    /**
     * Renders a Swing Icon onto an SWT image. Used by SwingWT to
     * render all images onto components.
     * 
     * @author Robin Rawson-Tetley
     *
     * @param component the SwingWT component this icon is being rendered for
     * @param icon The icon to render
     * @return An SWT image to assign to the component
     */
    public static org.eclipse.swt.graphics.Image getSWTImageFromSwingIcon(swingwt.awt.Component c, swingwtx.swing.Icon icon) {
        
        if (icon == null) return null;
        
        // Is it an ImageIcon? If so, we can cheat since it already
        // has an SWT image
        if (icon instanceof ImageIcon)
            return ((ImageIcon) icon).getImage().image;
        
        // Otherwise, we need to render the icon onto an image -
        
        // create the image and drawing context
        org.eclipse.swt.graphics.Image img = new org.eclipse.swt.graphics.Image(getDisplay(), icon.getIconWidth(), icon.getIconHeight());
        org.eclipse.swt.graphics.GC gc = new org.eclipse.swt.graphics.GC(img);
        swingwt.awt.SWTGraphics2DRenderer g = new swingwt.awt.SWTGraphics2DRenderer(gc, false);
        
        // Tell the icon to paint itself
        icon.paintIcon(c, g, 0, 0);
        
        // Destroy the graphics context
        g.dispose();
        
        // return the renderered image
        return img;
        
    }
    
    /** Translates the alignment part of the Swing constants */
    public static int translateSwingAlignmentConstant(int constant) {
        int ret = 0;
        switch (constant) {
            case (SwingConstants.CENTER): ret = SWT.CENTER; break;
            case (SwingConstants.TOP): ret = SWT.TOP; break;
            case (SwingConstants.LEFT): ret = SWT.LEFT; break;
            case (SwingConstants.BOTTOM): ret = SWT.BOTTOM; break;
            case (SwingConstants.RIGHT): ret = SWT.RIGHT; break;
            case (SwingConstants.LEADING): ret = SWT.LEFT; break;
            case (SwingConstants.TRAILING): ret = SWT.RIGHT; break;
        }
        return ret;
    }
    
    /** Translates the orientation part of the Swing constants */
    public static int translateSwingOrientationConstant(int constant) {
        int ret = 0;
        switch (constant) {
            case (SwingConstants.HORIZONTAL): ret = SWT.HORIZONTAL; break;
            case (SwingConstants.VERTICAL): ret = SWT.VERTICAL; break;
        }
        return ret;
    }
    
    /*** Removes HTML tags from a string
     * @param s The string to remove HTML tags from
     */
    public static String removeHTML(String s) {

	// If we have an opener and no closer, then
	// it ain't HTML and we shouldn't break it
	int opener = s.indexOf("<");
	int closer = s.indexOf(">");
	if (opener != -1 && closer == -1)
	    return s;
	    
        int i = s.indexOf("<");
        while (i != -1) {
            // Find end pos
            int e = s.indexOf(">", i);
            if (e == -1) e = s.length();
            // Strip from the string
            s = s.substring(0, i) + 
                ( e < s.length() ? s.substring(e + 1, s.length())
                                 : "" );
            // Find again
            i = s.indexOf("<");
        }
        
        // Replace known sequences with characters
        s = replace(s, "&nbsp;", " ");
        s = replace(s, "&nbsp", " ");
        s = replace(s, "&amp;", " ");
        s = replace(s, "&lt;", "<");
        s = replace(s, "&gt;", ">");
        
        // Compress whitespace
        s = s.trim();
        StringBuffer o = new StringBuffer();
        boolean lastWasSpace = false;
        for (i = 0; i < s.length(); i++) {
            if (s.substring(i, i + 1).equals(" ")) {
                if (!lastWasSpace) {
                    lastWasSpace = true;
                    o.append(" ");
                }
            }
            else {
                o.append(s.substring(i, i + 1));
                lastWasSpace = false;
            }
        }
        s = o.toString();

        return s;
    }
    
    
    /** Looks in findin for all occurrences of find and replaces them with replacewith 
     * @param findin The string to find occurrences in
     * @param find The string to find
     * @param replacewith The string to replace found occurrences with
     * @return A string with all occurrences of find replaced.
     */
    public static String replace(String findin, String find, String replacewith) {
        
        StringBuffer sb = new StringBuffer(findin);
        int i = 0;
        try {
            while (i <= sb.length() - find.length()) {
                if (sb.substring(i, i + find.length()).equalsIgnoreCase(find)) {
                    sb.replace(i, i + find.length(), replacewith);
                }
                i++;
            }
        }
        catch (StringIndexOutOfBoundsException e) {
            // We hit the end of the string - do nothing and carry on
        }
            
        return sb.toString();
    }
    
}

/**
    $Log: SwingWTUtils.java,v $
    Revision 1.88  2004/11/02 11:28:56  bobintetley
    Versioned off for 0.85

    Revision 1.87  2004/11/02 11:26:05  bobintetley
    Fix to default size of JTextArea/JTextField

    Revision 1.86  2004/11/01 12:16:45  bobintetley
    Fixed JFileChooser drive box bug, Dialog size/display bug and a bug in the
    classpath file for Eclipse editing

    Revision 1.85  2004/10/30 20:11:57  bobintetley
    Code cleanup

    Revision 1.84  2004/10/29 10:24:06  bobintetley
    Separate EditorPane implementations to decouple platform specifics,
    stubbing of HTMLEditorKit

    Revision 1.83  2004/10/08 15:10:57  bobintetley
    *** empty log message ***

    Revision 1.82  2004/10/07 10:09:15  bobintetley
    Bug fix to GTK2 to prevent Windows intermittently opening with a 1x1 pixel size

    Revision 1.81  2004/09/24 14:07:30  bobintetley
    JScrollPane/Toolbar fixes

    Revision 1.80  2004/09/24 12:39:01  bobintetley
    MacOSX specific fixes for window position and use of platform toolbuttons

    Revision 1.79  2004/09/17 19:11:11  dannaab
    Load documents in-memory and set with Browser.setText() instead of creating temp files

    Revision 1.78  2004/09/10 16:10:44  dannaab
    decrementWindowReferences(): comment out call to shut down event dispatch
    thread when window count == 0 (was causing swt "invalid thread access"
    exceptions on the default display when thread recreated).  temp fix?

    Revision 1.77  2004/09/07 09:21:35  bobintetley
    Bugs added to TODO and simple frame for dynamic Swing->SwingWT replacement

    Revision 1.76  2004/06/29 16:29:26  bobintetley
    *** empty log message ***

    Revision 1.75  2004/06/24 12:24:14  bobintetley
    MacOSX specific fixes

    Revision 1.74  2004/06/23 11:05:57  bobintetley
    MacOSX fixes for when the Quit item on the application menu is selected

    Revision 1.73  2004/06/23 07:34:16  bobintetley
    MacOSX users are now first class citizens and SwingWT now works properly

    Revision 1.72  2004/06/18 12:56:55  bobintetley
    Filechooser shows location now and JFileChooser(String/File) constructors work correctly again

    Revision 1.71  2004/06/11 09:17:32  bobintetley
    Various fixes and things

    Revision 1.70  2004/06/10 11:32:19  bobintetley
    Makefile fixes for Cygwin/Win32

    Revision 1.69  2004/06/07 15:01:33  bobintetley
    (Daniel Spiewak) JToolbar rollover support, JProgressBar orientation and indeterminate support, plus correct LAF for Win32

    Revision 1.68  2004/06/07 14:03:00  bobintetley
    Only turns of VIRTUAL for Win32 (since it's fine under GTK2)

    Revision 1.67  2004/06/07 14:01:32  bobintetley
    JTable fixes for Win32, and ability to turn off SWT VIRTUAL tables. Off by
    default due to Win32 bugs in 3.0M8

    Revision 1.66  2004/06/04 14:21:09  bobintetley
    JTextArea default size fixes and new computePreferredSize() method that
       delegates to peer to size a component

    Revision 1.65  2004/05/26 13:02:20  bobintetley
    Replaced deleted scripts and updated pre version

    Revision 1.64  2004/05/11 09:41:41  bobintetley
    Fixes to JInternalFrame ordering

    Revision 1.63  2004/05/10 14:33:17  bobintetley
    MDI support and JPanel border fixes

    Revision 1.62  2004/05/07 12:11:18  bobintetley
    Default layout fixes and correct behaviour for null layout

    Revision 1.61  2004/05/06 12:35:22  bobintetley
    Parity with Swing constants for Binary Compatibility + fixes to JDesktopPane

    Revision 1.60  2004/05/05 13:24:32  bobintetley
    Bugfixes and Laurent's patch for binary compatibility on Container.add()

    Revision 1.59  2004/05/04 09:49:00  bobintetley
    removeHTML fix to not misidentify HTML

    Revision 1.58  2004/05/04 09:31:43  bobintetley
    PlainDocument/View support and implementation. Build script supports java/javax
    packages - fix to build script to use nested args in bootclasspath (single path broke on my Ant 1.6.1/Linux)

    Revision 1.57  2004/04/30 21:54:21  bobintetley
    Moved log to the end of commonly changed files

   Revision 1.56  2004/04/30 21:38:04  bobintetley
   Fixes for layout interactions with GridBagLayout

   Revision 1.55  2004/04/30 13:54:00  bobintetley
   Fixes to getRenderStringHeight() to return the height, rather than width

   Revision 1.54  2004/04/30 13:20:43  bobintetley
   Fix to unrealised peer preferred sizes, forwarding window events to
   content panes and fix for mouse drag events.

   Revision 1.53  2004/04/27 13:50:18  bobintetley
   Build script fixes

   Revision 1.52  2004/04/23 10:24:53  bobintetley
   Version update and note on non-refreshed tables

   Revision 1.51  2004/04/21 10:44:50  bobintetley
   Code cleanup and native build script fix

   Revision 1.50  2004/04/19 14:18:42  bobintetley
   SWT Exception suppression

   Revision 1.49  2004/04/19 12:49:37  bobintetley
   JTaskTray implementation (and demo), along with Frame repaint fix

   Revision 1.48  2004/04/18 15:11:00  bobintetley
   Updated todo items and version

   Revision 1.47  2004/04/16 14:38:47  bobintetley
   Table and Tree cell editor support

   Revision 1.46  2004/04/15 11:24:33  bobintetley
   (Dan Naab) ComponentUI, UIDefaults/UIManager and Accessibility support.
   (Antonio Weber) TableColumnModelListener implementation and support

   Revision 1.45  2004/04/06 12:30:53  bobintetley
   JTable thread safety, ListSelectionModel implementation for JList/JTable

   Revision 1.44  2004/03/31 08:41:19  bobintetley
   Fixes to whitespace stripper for better looking HTML->Text

   Revision 1.43  2004/03/30 10:42:46  bobintetley
   Many minor bug fixes, event improvements by Dan Naab. Full swing.Icon support

   Revision 1.42  2004/03/26 12:04:42  bobintetley
   Fixed bug in TreeModel that caused events not to fire down to JTree

   Revision 1.41  2004/03/23 15:22:06  bobintetley
   SystemColor/Dialog fix and SwingWTUtils.setEclipsePlugin(true) implementation

   Revision 1.40  2004/03/22 15:10:22  bobintetley
   JRootPane and JLayeredPane implementation

   Revision 1.39  2004/03/19 15:39:00  bobintetley
   Updated version

   Revision 1.38  2004/03/18 14:42:11  bobintetley
   Fix to Window hierarchy to match Swing, and fix to allow MDI apps
      to work under SWT 2.x

   Revision 1.37  2004/03/16 22:17:18  bobintetley
   Fixes for JDK 1.3

   Revision 1.36  2004/03/12 11:27:26  bobintetley
   Memory leak fixes

   Revision 1.35  2004/03/12 11:05:24  bobintetley
   Fixed memory leak in container destruction

   Revision 1.34  2004/03/04 15:32:28  bobintetley
   JInternalFrame methods now modify their peers after creation

   Revision 1.33  2004/03/03 13:49:36  bobintetley
   Updating of README, FAQ and VERSION

   Revision 1.32  2004/03/03 09:13:12  bobintetley
   JList threading fixed and top level error handling

   Revision 1.31  2004/03/02 08:48:35  bobintetley
   Version update

   Revision 1.30  2004/03/01 15:58:47  bobintetley
   Various little bug fixes

   Revision 1.29  2004/02/27 16:16:15  bobintetley
   Threading fixes

   Revision 1.28  2004/02/23 11:15:54  bobintetley
   Fixes to popup

   Revision 1.27  2004/02/22 08:38:20  bobintetley
   Fixed scrollbar interaction with JTextArea

   Revision 1.26  2004/02/20 14:00:18  bobintetley
   Updated version

   Revision 1.25  2004/02/20 13:57:01  bobintetley
   JLookupPopup - A high speed alternative to JComboBox

   Revision 1.24  2004/02/20 10:22:07  bobintetley
   GCJ/GIJ compatible replacement for StringBuffer.indexOf()

   Revision 1.23  2004/02/19 09:58:44  bobintetley
   Various small bug fixes and JTextArea should be much faster/lighter

   Revision 1.22  2004/01/27 09:05:12  bobintetley
   ListModel and List Selection implemented. ScrollPane fix so all components
      scrollable

   Revision 1.21  2004/01/26 12:02:49  bobintetley
   JPanel titled border support

   Revision 1.20  2004/01/26 10:57:45  bobintetley
   HTML handling (throws it away - SWT can't do anything with it)

   Revision 1.19  2004/01/23 08:05:51  bobintetley
   JComboBox fixes and better Action implementation

   Revision 1.18  2004/01/20 09:17:15  bobintetley
   Menu class overhaul for compatibility, Action support and thread safety

   Revision 1.17  2004/01/16 09:35:47  bobintetley
   Full event dispatch thread support!

   Revision 1.16  2004/01/15 09:55:13  bobintetley
   Fixed multiple display create problem (thanks Sachin)

   Revision 1.15  2004/01/07 11:32:52  bobintetley
   Fix to calculation under Windows for rendering widths

   Revision 1.14  2004/01/07 09:26:25  bobintetley
   Render widths now work correctly

   Revision 1.13  2004/01/06 15:56:51  bobintetley
   Render width now works without an active shell

   Revision 1.12  2004/01/06 15:38:30  bobintetley
   Adjusted render width calculations. Fixed horrible button border under Win32

   Revision 1.11  2004/01/06 15:31:02  bobintetley
   New render width function for accurate auto-sizing

   Revision 1.10  2004/01/06 10:29:21  bobintetley
   Dispatch thread retardation code

   Revision 1.9  2003/12/22 14:46:23  bobintetley
   Oops, forgot to make them static :-)

   Revision 1.8  2003/12/22 14:44:31  bobintetley
   Image encode/save support

   Revision 1.7  2003/12/17 15:24:33  bobintetley
   Threading fixes

   Revision 1.6  2003/12/16 13:14:33  bobintetley
   Use of SwingWTUtils.isSWTControlAvailable instead of null test

   Revision 1.5  2003/12/16 09:19:02  bobintetley
   Various small fixes to match Swing more closely

   Revision 1.4  2003/12/14 09:13:38  bobintetley
   Added CVS log to source headers

*/
