/*
 * dvi_fb.c - frame buffer
 * by Hirotsugu Kakugawa
 *
 *  31 Jul 1998  First implementation
 *  10 Dec 1998  Added DVI_fb_boundingbox().
 *   9 Oct 1999  Debuged clipping in DVI_fb_put_bitmap().
 */
/*
 * Copyright (C) 1998-1999  Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This file is part of the DVIlib Library.  This library is free
 * software; you can redistribute it and/or modify it under the terms of
 * the GNU Library General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  This library is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_STDARG_H
#  include <stdarg.h>
#else
#  include <vararg.h>
#endif
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#if defined(HAVE_STRING_H)
#  include  <string.h>
#endif
#if defined(HAVE_STRINGS_H)
#  include  <strings.h>
#endif

#include "dvi-2_6.h"
#include "defs.h"

static unsigned char ith_bits[]    = {
  0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01};
static unsigned char rect_bits_l[] = {
  0xff, 0x7f, 0x3f, 0x1f, 0x0f, 0x07, 0x03, 0x01};
static unsigned char rect_bits_r[] = {
  0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfc, 0xfe, 0xff};



static int   fb_boundingbox_mono(DVI_FRAME_BUFFER, long*, long*, long*, long*);
static int   fb_boundingbox_rgb(DVI_FRAME_BUFFER, long*, long*, long*, long*);




Public DVI_FRAME_BUFFER 
DVI_fb_create(long w, long h)
{
  return  DVI_fb_create_mono(w, h);
}


Public DVI_FRAME_BUFFER 
DVI_fb_create_mono(long w, long h)
{
  DVI_FRAME_BUFFER   fb;

  if ((fb = calloc(1, sizeof(struct dvi_s_frame_buffer))) == NULL)
    return NULL;

  if (w <= 0)
    w = 1;
  if (h <= 0)
    h = 1;

  fb->type = DVI_FB_TYPE_MONO;
  fb->width   = w;
  fb->height  = h;
  fb->w_bytes = (w + 7) / 8;
  fb->raster  = (w + 7) / 8;
  if ((fb->data = (unsigned char*)malloc(fb->raster * fb->height)) == NULL){
    free(fb);
    return NULL;
  }

  return fb;
}

Public DVI_FRAME_BUFFER 
DVI_fb_create_rgb(long w, long h)
{
  DVI_FRAME_BUFFER   fb;

  if ((fb = calloc(1, sizeof(struct dvi_s_frame_buffer))) == NULL)
    return NULL;

  if (w <= 0)
    w = 1;
  if (h <= 0)
    h = 1;

  fb->type = DVI_FB_TYPE_RGB;
  fb->width   = w;
  fb->height  = h;
  fb->w_bytes = 3*w;
  fb->raster  = 3*w;
  fb->rgb_fg_r = fb->rgb_fg_g = fb->rgb_fg_b = 0x00;
  fb->rgb_bg_r = fb->rgb_bg_g = fb->rgb_bg_b = 0xff;
  
  if ((fb->data = malloc(3*fb->width*fb->height)) == NULL){
    free(fb);
    return NULL;
  }

  return fb;
}


Public void
DVI_fb_rgb_foreground(DVI_FRAME_BUFFER fb, int r, int g, int b)
{
  fb->rgb_fg_r = r;
  fb->rgb_fg_g = g;
  fb->rgb_fg_b = b;
}


Public void
DVI_fb_rgb_background(DVI_FRAME_BUFFER fb, int r, int g, int b)
{
  fb->rgb_bg_r = r;
  fb->rgb_bg_g = g;
  fb->rgb_bg_b = b;
}



Public void
DVI_fb_dispose(DVI_FRAME_BUFFER fb)
{
  if (fb != NULL){
    if (fb->data != NULL){
      free(fb->data);
      fb->data = NULL;
    }
    free(fb);
    fb = NULL;
  }
}


Public void
DVI_fb_clear(DVI_FRAME_BUFFER fb)
{
  unsigned char *p;
  long           xw, y, z;

  if ((fb == NULL) || (fb->data == NULL))
    return;

  switch (fb->type){
  case DVI_FB_TYPE_MONO:
    for (y = 0; y < fb->height; y++){
      p = &fb->data[y * fb->raster];
      for (xw = 0; xw < fb->w_bytes; xw++)
	*(p++) = 0;
    }
    break;

  case DVI_FB_TYPE_RGB:
    xw = fb->width * fb->height;
    for (y = 0; y < xw; y++){
      z = 3 * y;
      fb->data[z + 0] = fb->rgb_bg_r;
      fb->data[z + 1] = fb->rgb_bg_g;
      fb->data[z + 2] = fb->rgb_bg_b;
    }
    break;
  }
}



Public int 
DVI_fb_put_bitmap(DVI_FRAME_BUFFER fb, DVI_BITMAP bm,
		  long pos_x, long pos_y)
{
  int             xr, xl;
  long            xb, by, fy, nb, i, z;
  long            clip_x, clip_y, clip_w, clip_h;
  unsigned char  *fp, *bp;

  if ((fb == NULL) || (fb->data == NULL))
    return -1;

  if ((pos_x >= fb->width-1) || (pos_y >= fb->height-1))
    return 0;
  if ((bm->width <= 0) || (bm->height <= 0))
    return 0;    

  clip_w = bm->width;
  clip_h = bm->height;
  clip_x = 0;
  clip_y = 0;
  if (pos_x < 0){
    clip_x = -pos_x;
    clip_w += pos_x;
    pos_x = 0;
  }
  if (pos_y < 0){
    clip_y = -pos_y;
    clip_h += pos_y;
    pos_y = 0;
  }
  if (pos_x + clip_w > fb->width)
    clip_w = fb->width - pos_x;
  if (pos_y + clip_h > fb->height)
    clip_h = fb->height - pos_y;

  if ((clip_w <= 0) || (clip_h <= 0))
    return 0;
  
  switch (fb->type){
  case DVI_FB_TYPE_MONO:
    xb = pos_x / 8;
    xr = (pos_x + clip_x) % 8;
    xl = 8 - xr;
    nb = (clip_w - clip_x + 7) / 8;
    for (by = clip_y, fy = pos_y; by < clip_h; by++, fy++){
      bp = &bm->bitmap[bm->raster * by + (clip_x/8)];
      fp = &fb->data[fb->raster * fy + xb];
      for (i = nb; i > 0; --i){
	*(fp+1) |= (*bp     << xl);
	*(fp++) |= (*(bp++) >> xr);
      }
    }
    break;

  case DVI_FB_TYPE_RGB:
    for (by = clip_y, fy = pos_y; by < clip_h; by++, fy++){
      for (i = 0; i < clip_w; i++){
	if ((bm->bitmap[bm->raster * by + (clip_x+i)/8]
	    & ith_bits[(clip_x + i)%8]) != 0){
	  z = fb->raster * fy + 3*(pos_x + i);
	  fb->data[z + 0] = fb->rgb_fg_r;
	  fb->data[z + 1] = fb->rgb_fg_g;
	  fb->data[z + 2] = fb->rgb_fg_b;
	}
      }
    }
    break;
  }

  return 0;
}



Public int 
DVI_fb_put_rectangle(DVI_FRAME_BUFFER fb, long pos_x, long pos_y,
		     long rect_w, long rect_h)
{
  int             xb;
  long            y, x, z, nb;
  unsigned char  *fp, wl, wr, w;

  if ((fb == NULL) || (fb->data == NULL))
    return -1;

  if (pos_x < 0){
    rect_w = rect_w + pos_x;
    pos_x = 0;
  }
  if (pos_y < 0){
    rect_h = rect_h + pos_y;
    pos_y = 0;
  }
  if ((rect_h <= 0) || (rect_w <= 0))
    return 0;
  if (pos_x + rect_w > fb->width)
    rect_w = fb->width - pos_x;
  if (pos_y + rect_h > fb->height)
    rect_h = fb->height - pos_y;
  if ((rect_h <= 0) || (rect_w <= 0))
    return 0;

  switch (fb->type){
  case DVI_FB_TYPE_MONO:
    xb = pos_x / 8;
    wl = rect_bits_l[pos_x % 8];
    wr = rect_bits_r[(pos_x + rect_w) % 8];
    fp = &fb->data[fb->w_bytes * pos_y + xb];
    nb = (pos_x + rect_w)/8 - (pos_x/8) + 1;
    if (nb > 1){
      for (y = 0; y < rect_h; y++){
	fp[0] |= wl;
	for (x = 1; x < nb-1; x++)
	  fp[x] = 0xff;
	fp[nb-1] |= wr;
	fp = fp + fb->w_bytes;
      }
    } else {
      w = wl & wr;
      for (y = 0; y < rect_h; y++){
	fp[0] |= w;
	fp = fp + fb->w_bytes;
      }
    }
    break;

  case DVI_FB_TYPE_RGB:
    for (y = 0; y < rect_h; y++){
      for (x = 0; x < rect_w; x++){
	z = fb->raster*(pos_y + y) + 3*(pos_x + x);
	fb->data[z + 0] = fb->rgb_fg_r;
	fb->data[z + 1] = fb->rgb_fg_g;
	fb->data[z + 2] = fb->rgb_fg_b;
      }
    }
    break;
  }

  return 0;
}


Public int 
DVI_fb_put_pixmap_rgb(DVI_FRAME_BUFFER fb, DVI_PIXMAP_RGB bm,
		      long pos_x, long pos_y)
{
  int             xr, xl;
  long            xb, by, fy, nb, i, z, b;
  long            clip_x, clip_y, clip_w, clip_h;
  unsigned char  *bp, *fp;

  if ((fb == NULL) || (fb->data == NULL))
    return -1;

  if ((pos_x >= fb->width-1) || (pos_y >= fb->height-1))
    return 0;
  if ((bm->width <= 0) || (bm->height <= 0))
    return 0;    

  clip_w = bm->width;
  clip_h = bm->height;
  clip_x = 0;
  clip_y = 0;
  if (pos_x < 0){
    clip_x = -pos_x;
    clip_w += pos_x;
    pos_x = 0;
  }
  if (pos_y < 0){
    clip_y = -pos_y;
    clip_h += pos_y;
    pos_y = 0;
  }
  if (pos_x + clip_w > fb->width)
    clip_w = fb->width - pos_x;
  if (pos_y + clip_h > fb->height)
    clip_h = fb->height - pos_y;

  if ((clip_w <= 0) || (clip_h <= 0))
    return 0;
  
  switch (fb->type){
  case DVI_FB_TYPE_MONO:
    for (by = clip_y, fy = pos_y; by < clip_h; by++, fy++){
      for (i = 0; i < clip_w; i++){
	b = 3 * (by * bm->raster + clip_x + i);
	if (bm->bitmap[b+0] + bm->bitmap[b+1] + bm->bitmap[b+2] >= 0.5)
	  fb->data[fb->raster * fy + (pos_x + i)/8]
	    |= ith_bits[(pos_x + i) % 8];
      }
    }
    break;

    xb = pos_x / 8;
    xr = (pos_x + clip_x) % 8;
    xl = 8 - xr;
    nb = (clip_w - clip_x + 7) / 8;
    for (by = clip_y, fy = pos_y; by < clip_h; by++, fy++){
      bp = &bm->bitmap[bm->raster * by + (clip_x/8)];
      fp = &fb->data[fb->raster * fy + xb];
      for (i = nb; i > 0; --i){
	*(fp+1) |= (*bp     << xl);
	*(fp++) |= (*(bp++) >> xr);
      }
    }
    break;

  case DVI_FB_TYPE_RGB:
    for (by = clip_y, fy = pos_y; by < clip_h; by++, fy++){
      for (i = 0; i < clip_w; i++){
	b = by * bm->raster + 3*(clip_x + i);
	z = fb->raster * fy + 3*(pos_x + i);
	fb->data[z + 0] = bm->bitmap[b+0];
	fb->data[z + 1] = bm->bitmap[b+1];
	fb->data[z + 2] = bm->bitmap[b+2];
      }
    }
    break;
  }

  return 0;
}




Public unsigned short*
DVI_fb_antialias(DVI_FRAME_BUFFER fb, int aa_factor, 
		 long pos_x, long pos_y, long w, long h,
		 long *aab_w_p, long *aab_h_p)
{
  long            x0, x1, y0, y1, aabw, aabh, x, y, z1, z2;
  int             i;
  unsigned char  *fp;
  unsigned short *ap;
  unsigned short *aa_buff  = NULL;

  if ((fb == NULL) || (fb->data == NULL))
    return NULL;

  x0 =  pos_x - (pos_x % aa_factor);
  y0 =  pos_y - (pos_y % aa_factor);
  x1 =  (pos_x + w) + aa_factor - 1 - ((pos_x + w) % aa_factor);
  y1 =  (pos_y + h) + aa_factor - 1 - ((pos_y + h) % aa_factor);
  *aab_w_p = aabw = (x1 - x0 + 1) / aa_factor;
  *aab_h_p = aabh = (y1 - y0 + 1) / aa_factor;

#if 0
  fprintf(stderr, "!! %ld,%ld  %ld,%ld  %ld,%ld %ld,%ld\n",
	  x0, y0, x1, y1, aabw, aabh, x1-x0, y1-y0);
#endif

  switch (fb->type){

  case DVI_FB_TYPE_MONO:
    if ((aa_buff = malloc(aabw * aabh * sizeof(unsigned short))) == NULL)
      return NULL;
    for (i = 0; i < aabh * aabw; i++)
      aa_buff[i] = 0;
    for (y = y0; y <= y1; y++){
      ap = &aa_buff[aabw * ((y-y0)/aa_factor)];
      fp = &fb->data[fb->raster * y];
      for (x = x0; x <= x1; x++){
	if ((fp[x/8] & ith_bits[x%8]) != 0)
	  ap[(x-x0)/aa_factor] += 1;
      }
    }
    break;
    
  case DVI_FB_TYPE_RGB:    
    if ((aa_buff = malloc(3 * aabw * aabh * sizeof(unsigned short))) == NULL)
      return NULL;
    for (i = 0; i < aabh*aabw; i++){
      z1 = 3*i;
      aa_buff[z1 + 0] = 0;
      aa_buff[z1 + 1] = 0;
      aa_buff[z1 + 2] = 0;
    }
    for (y = y0; y <= y1; y++){
      for (x = x0; x <= x1; x++){
	z1 = 3 * (aabw * ((y-y0)/aa_factor) + (x-x0)/aa_factor);
	z2 = fb->raster * y + 3*x;
	aa_buff[z1 + 0] += fb->data[z2 + 0];
	aa_buff[z1 + 1] += fb->data[z2 + 1];
	aa_buff[z1 + 2] += fb->data[z2 + 2];
      }
    }
    break;

  }

  return aa_buff;
}


Public unsigned char* 
DVI_fb_raster_pointer(DVI_FRAME_BUFFER fb, long y)
{
  unsigned char *p;

  if ((fb == NULL) || (fb->data == NULL))
    return NULL;
  if ((y < 0) || (fb->height <= y))
    return NULL;

  switch (fb->type){
  case DVI_FB_TYPE_MONO:
    p = &fb->data[fb->raster * y];
    break;
  case DVI_FB_TYPE_RGB:    
    p = &fb->data[fb->raster * y];
    break;
  }

  return  p;
}


Public int 
DVI_fb_raster_nonzero(DVI_FRAME_BUFFER fb, long y)
{
  switch (DVI_fb_raster_zero(fb, y)){
  case 0:
    return 1;  /* nonzero */
  case 1:
    return 0;  /* zero */
  case -1:
  default:
    break;
  }
  return -1;
}


Public int 
DVI_fb_raster_zero(DVI_FRAME_BUFFER fb, long y)
{
  long            x, z;
  unsigned char  *fp;

  if ((fb == NULL) || (fb->data == NULL))
    return -1;
  if ((y < 0) || (fb->height <= y))
    return -1;

  switch (fb->type){
  case DVI_FB_TYPE_MONO:
    fp = &fb->data[fb->raster * y];
    for (x = 0; x < fb->w_bytes; x++){
      if (*fp != 0)
	return 0;  /* non-zero */
      fp++;
    }
    break;
  case DVI_FB_TYPE_RGB:    
    z = fb->raster * y;
    for (x = 0; x < fb->width; x++){
      if (fb->data[z + 0] != fb->rgb_bg_r)
	return 0;  /* non-zero */
      if (fb->data[z + 1] != fb->rgb_bg_g)
	return 0;  /* non-zero */
      if (fb->data[z + 2] != fb->rgb_bg_b)
	return 0;  /* non-zero */
      z += 3;
    }
    break;
  }

  return 1;  /* zero */
}


Public int 
DVI_fb_boundingbox(DVI_FRAME_BUFFER fb,
		   long *xminp, long *xmaxp, long *yminp, long *ymaxp)
{
  int  v;


  if ((fb == NULL) || (fb->data == NULL))
    return -1;

  switch (fb->type){
  case DVI_FB_TYPE_MONO:
    v = fb_boundingbox_mono(fb,xminp, xmaxp, yminp, ymaxp);
    break;
  case DVI_FB_TYPE_RGB:
    v = fb_boundingbox_rgb(fb,xminp, xmaxp, yminp, ymaxp);
    break;
  }

  return v;
}


static int 
fb_boundingbox_mono(DVI_FRAME_BUFFER fb,
		    long *xminp, long *xmaxp, long *yminp, long *ymaxp)
{
  long    x, y, wx;
  int     cont, wb, i;    
  unsigned char  *fp;

  if ((fb == NULL) || (fb->data == NULL))
    return -1;
  if (fb->type != DVI_FB_TYPE_MONO)
    abort();

  *xminp = 0; 
  *xmaxp = fb->width - 1; 
  *yminp = 0; 
  *ymaxp = fb->height - 1; 

  /* Find min y */
  *yminp = 0; 
  while ((*yminp < fb->height) && (DVI_fb_raster_zero(fb, *yminp) == 1)){
    (*yminp)++;
  }

  /* If we have no black pixels, just return */
  if (*yminp >= fb->height){
    *xminp = -1;
    *xmaxp = -1;
    *yminp = -1;
    *ymaxp = -1;
    return -1;
  }

  /* Find max y */
  *ymaxp = fb->height - 1;
  while ((*ymaxp >= 0) && (DVI_fb_raster_zero(fb, *ymaxp) == 1)){
    (*ymaxp)--;
  }

  /* Find min x */
  cont = 1;
  *xminp = fb->width - 1;
  /* find non-zero byte */
  for (wx = 0; (cont == 1) && (wx < (fb->width+7)/8); wx++){
    y = *yminp;
    fp = &fb->data[fb->raster * y + wx];
    for ( ; y <= *ymaxp; y++, fp += fb->w_bytes){
      if (*fp == 0)
	continue;
      /* non-zero byte is found. find a 1-value bit in the byte. */
      wb = 8;
      if (wx == (fb->width+7)/8 - 1)
	wb = (fb->width - 1) % 8;
      for (i = 0; i <= wb; i++){
	if ((*fp & ith_bits[i]) != 0){
	  cont = 0;
	  x = 8 * wx + i;
	  if (*xminp > x)
	    *xminp = x;
	}
      }
    }
  }

  /* Find max x */
  cont = 1;
  *xmaxp = 0;
  /* find non-zero byte */
  for (wx = (fb->width+7)/8 - 1; (cont == 1) && (wx >= 0); --wx){
    y = *yminp;
    fp = &fb->data[fb->raster * y + wx];
    for ( ; y <= *ymaxp; y++, fp += fb->w_bytes){
      if (*fp == 0)
	continue;
      /* non-zero byte is found. find a 1-value bit in the byte. */
      wb = 7;
      if (wx == (fb->width+7)/8 - 1)
	wb = (fb->width - 1) % 8;
      for (i = wb; i >= 0; --i){
	if ((*fp & ith_bits[i]) != 0){
	  cont = 0;
	  x = 8 * wx + i;
	  if (*xmaxp < x)
	    *xmaxp = x;
	}
      }
    }
  }

  return 0;
}


static int 
fb_boundingbox_rgb(DVI_FRAME_BUFFER fb,
		   long *xminp, long *xmaxp, long *yminp, long *ymaxp)
{
  long    y, z;

  if ((fb == NULL) || (fb->data == NULL))
    return -1;
  if (fb->type != DVI_FB_TYPE_RGB)
    abort();
    
  *xminp = 0; 
  *xmaxp = fb->width - 1; 
  *yminp = 0; 
  *ymaxp = fb->height - 1; 

  /* Find min y */
  *yminp = 0; 
  while ((*yminp < fb->height) && (DVI_fb_raster_zero(fb, *yminp) == 1)){
    (*yminp)++;
  }

  /* If we have no black pixels, just return */
  if (*yminp >= fb->height){
    *xminp = -1;
    *xmaxp = -1;
    *yminp = -1;
    *ymaxp = -1;
    return -1;
  }

  /* Find max y */
  *ymaxp = fb->height - 1;
  while ((*ymaxp >= 0) && (DVI_fb_raster_zero(fb, *ymaxp) == 1)){
    (*ymaxp)--;
  }

  /* Find min x */
  for (*xminp = 0; *xminp < fb->width; (*xminp)++){
    for (y = *yminp; y <= *ymaxp; y++){
      z = fb->raster * y + 3 * (*xminp);
      if (fb->data[z + 0] != fb->rgb_bg_r)
	break;
      if (fb->data[z + 1] != fb->rgb_bg_g)
	break;
      if (fb->data[z + 2] != fb->rgb_bg_b)
	break;
    }
    if (y <= *ymaxp)
      break;
  }

  /* Find max x */
  for (*xmaxp = fb->width-1; *xmaxp >= 0; (*xmaxp)--){
    for (y = *yminp; y <= *ymaxp; y++){
      z = fb->raster * y + 3 * (*xmaxp);
      if (fb->data[z + 0] != fb->rgb_bg_r)
	break;
      if (fb->data[z + 1] != fb->rgb_bg_g)
	break;
      if (fb->data[z + 2] != fb->rgb_bg_b)
	break;
    }
    if (y <= *ymaxp)
      break;
  }

  return 0;
}


/*EOF*/
