/*
 * spc_ps.c - a PostScript module (GhostScript interface) for 
 *            `special' DVI instruction
 * by Hirotsugu Kakugawa
 *
 *  2 Jun 1997  Delived from special.c.
 *  6 Jun 1997  Improved code. Added dvi_special_PSfile().
 * 11 Jun 1997  Added cilpping feature. 
 * 17 Jun 1997  Added bitmap format conversions (put_eps_XXX2YYY()).
 * 30 Oct 1998  Enhanced error checking for Ghostscript.
 * 19 Dec 1998  Bugs of invoking ghostscript in case of lacking fork()
 *              and pipe() are fixed.
 * 29 Mar 1999  Added asynchronous gs process invocation.
 * 19 Sep 1999  Added incremental display of eps figures
 *  1 Oct 1999  Support for psfig.sty.
 * 10 Dec 1999  Support for \scalebox{} command of graphic{s,x}.sty.
 *  3 Feb 2000  Added code for disabling `showpage' in EPS file.
 *
 */
/*
 * Copyright (C) 1996-2000  Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This file is part of the DVIlib Library.  This library is free
 * software; you can redistribute it and/or modify it under the terms of
 * the GNU Library General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  This library is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#ifdef HAVE_MALLOC_H
#  include <malloc.h>
#endif
#if defined(HAVE_STRING_H)
#  include  <string.h>
#endif
#if defined(HAVE_STRINGS_H)
#  include  <strings.h>
#endif
#include <ctype.h>
#include <sys/errno.h>
#include <sys/param.h>
#include <signal.h>
#ifdef HAVE_FCNTL_H
#  include <fcntl.h>
#endif
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#ifdef HAVE_SELECT
#  include <sys/types.h>
#  include <sys/socket.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#  include <sys/wait.h>
#endif


#undef DEBUG_DISPLAY_EPS_INFO



extern int     errno;

#include "dvi-2_6.h"
#include "defs.h"
#include "cache.h"
#include "spc_ps.h"
#include "spc_grf.h"
#include "private.h"


Private int        gs_invocation_nerr = 0;


struct ps_str_arg {
  char  *name;
  int    namelen;
  char  *val;
  int   *given;
};
struct  ps_float_arg {
  char   *name;
  int     namelen;
  double *val;
  int    *given;
};
struct  ps_bool_arg {
  char   *name;
  int     namelen; 
  int    *val;
  int    *given;
};

#define  N_SPECIAL_ARGS_STR    10
#define  N_SPECIAL_ARGS_FLOAT  30
#define  N_SPECIAL_ARGS_BOOL   10

Private struct ps_str_arg    special_args_str[N_SPECIAL_ARGS_STR];
Private struct ps_float_arg  special_args_float[N_SPECIAL_ARGS_FLOAT];
Private struct ps_bool_arg   special_args_bool[N_SPECIAL_ARGS_BOOL];




#define EPS_OUTPUT_FORMAT_UNKNOWN      -1
#define EPS_OUTPUT_FORMAT_BITMAP       0
#define EPS_OUTPUT_FORMAT_GRAYMAP      1
#define EPS_OUTPUT_FORMAT_PIXMAP_RGB   2


Private void   put_frame(DVI_DEVICE,DVI,long,long,long,long);
Private int    parse_special_postscriptbox(DVI,DVI_DEVICE,char*);
Private double numberstr2pt(DVI,DVI_DEVICE,double,char*);
Private int    parse_special_args(DVI,DVI_DEVICE,char*);
static void    init_ps_info(struct ps *psi);
Private void   put_eps_figure(DVI,DVI_DEVICE,long,long);
Private void   put_eps_bm2gray(DVI,DVI_DEVICE,long,long,DVI_BITMAP);
Private void   put_eps_bm2rgb(DVI,DVI_DEVICE,long,long,DVI_BITMAP);
Private void   put_eps_gray2rgb(DVI,DVI_DEVICE,long,long,DVI_GRAYMAP);
Private int    eps2bitmap(DVI,DVI_DEVICE,int);
Private FILE  *eps2psfile(DVI,DVI_DEVICE,char*,int);
Private int    eps_output_format(DVI_DEVICE dev);

Private int    gs_convert_to_bitmap(DVI,DVI_DEVICE,char*,int,long,long);
Private int    gs_read(DVI,DVI_DEVICE,int,unsigned char*,int);
Private int    gs_read_char(DVI,DVI_DEVICE,int,unsigned char*);
Private void   gs_unread_char(DVI,DVI_DEVICE,int,unsigned char*);
Private int    gs_end_read(DVI,DVI_DEVICE);
Private void   gs_kill(void); 






/**
 ** 'ps: ' special
 **/

#define  PSFIG_MAGIC_NUMBER  65781.76

Public int 
dvi_special_ps(DVI dvi, DVI_DEVICE dev, char *cmd, long x, long y)
{
  int     size_type, n;
  char    s[1024];
  int     type, angle, push_angle;
  double  tmp, llx, lly, urx, ury;
  struct ps  *psi;
  
  push_angle = 0;

  psi = &PD(dvi, ps_info);

  if (strncmp(cmd, "ps::[begin]", 11) == 0){
    init_ps_info(psi);  /* must be here */
    n = sscanf(cmd, "%*s %lf %lf %lf %lf %lf %lf %s",
	       &psi->param_hsize, &psi->param_vsize,
	       &psi->param_llx, &psi->param_lly,
	       &psi->param_urx, &psi->param_ury,
	       s);
    if (n == 7){
      psi->bitmap_hoffset = 0;
      psi->bitmap_voffset = 0;
      if (strcmp(s, "startTexFig") == 0){
	psi->given_filename = 0;
	psi->param_hsize /= PSFIG_MAGIC_NUMBER;
	psi->param_vsize /= PSFIG_MAGIC_NUMBER;
	psi->given_hsize = 1;
	psi->given_vsize = 1;
	psi->param_llx /= PSFIG_MAGIC_NUMBER;
	psi->param_lly /= PSFIG_MAGIC_NUMBER;
	psi->param_urx /= PSFIG_MAGIC_NUMBER;
	psi->param_ury /= PSFIG_MAGIC_NUMBER;
	psi->given_llx = 1;
	psi->given_lly = 1;
	psi->given_urx = 1;
	psi->given_ury = 1;
      }
      psi->given_rwi = 0;
      psi->given_rhi = 0;
    }
  } else if (strncmp(cmd, "ps::[end]", 9) == 0){
    while (push_angle > 0){
      dvi_ps_gstate_pop(dvi, PS_CODE_TYPE_ROTATE_BEGIN, NULL, NULL, NULL);
      push_angle--;
    }
  } else if (strncmp(cmd, "ps:: ", 5) == 0){
    char  s1[256], s2[256];
    switch (sscanf(&cmd[5], "%s%s", s1, s2)){
    default:
      break; 
    case 1:    /* ps:: doclip ... unsupported. Figures are cliped. */
      break;
    case 2:    /* ps:: X rotate. not supported... */
      if (strcmp(s2, "rotate") == 0){
	angle = atoi(s1);
	dvi_ps_gstate_push(dvi, PS_CODE_TYPE_ROTATE_BEGIN, 1, 1, angle);
	PD(dvi,gstate_transformed) 
	  = dvi_ps_gstate_get_current(dvi, NULL, NULL, &angle);
	PD(dvi,gstate_angle)   = angle;
	push_angle++;
      }
      break;
    }
  } else if (strncmp(cmd, "ps: plotfile", 12) == 0){
    n = sscanf(cmd, "%*s %*s %s", psi->param_filename);
    if (n == 1){
      psi->given_filename = 1;
      if ((psi->given_hsize == 1) && (psi->given_vsize == 1)){
	size_type = EPS_SIZE_SPEC_VH_SIZE;
      } else {
	size_type = EPS_SIZE_SPEC_VH_SCALE;
	if (psi->given_hscale == 0)
	  psi->param_hscale = 1.0;
	if (psi->given_vscale == 0)
	  psi->param_vscale = 1.0;
      }
      switch (PD(dvi,gstate_angle)){  /* darty hack */
      default:
      case 0:
	break;
      case 90:
	llx = psi->param_lly;
	lly = -psi->param_llx - psi->param_hsize;
	urx = psi->param_lly + psi->param_vsize;
	ury = -psi->param_llx;
	psi->param_llx = llx;
	psi->param_lly = lly;
	psi->param_urx = urx;
	psi->param_ury = ury;
	tmp = psi->param_hsize;
	psi->param_vsize = psi->param_hsize;
	psi->param_hsize = tmp;
	break;
      }
      if (eps2bitmap(dvi, dev, size_type) < 0)
	return -1;
      if (DEV_CALL(dev,draw_ps_figures)(dev, dvi) == 1){
	put_eps_figure(dvi, dev, x, y); 
      } else {
	put_frame(dev, dvi, x, y,
		  psi->bm_width, psi->bm_height);
      }
      if (psi->bm.bitmap != NULL){
	free(psi->bm.bitmap);
	psi->bm.bitmap = NULL;
      }
    }
  } else if (strncmp(cmd, "ps: ", 4) == 0){
    type = dvi_ps_code_type(dvi, &cmd[4]);
  }

  return 0;
}




/**
 **  'postscriptbox' special
 **/

Public int 
dvi_special_postscriptbox(DVI dvi, DVI_DEVICE dev, char *cmd, long x, long y)
{
  struct ps  *psi;
  
  psi = &PD(dvi, ps_info);
  init_ps_info(psi);

  psi->bitmap_hoffset = 0;
  psi->bitmap_voffset = 0;
  if (parse_special_postscriptbox(dvi, dev, cmd) < 0)
    return -1;

  if (eps2bitmap(dvi, dev, EPS_SIZE_SPEC_WIDTH_HEIGHT) < 0)
    return -1;

  if (DEV_CALL(dev,draw_ps_figures)(dev, dvi) == 1){
    put_eps_figure(dvi, dev, 
		   x + psi->bitmap_hoffset, y + psi->bitmap_voffset); 
  } else {
    put_frame(dev, dvi, x + psi->bitmap_hoffset, y + psi->bitmap_voffset,
	      psi->bm_width, psi->bm_height);
  }

  if (psi->bm.bitmap != NULL){
    free(psi->bm.bitmap);
    psi->bm.bitmap = NULL;
  }

  return 0;
}

Private int
parse_special_postscriptbox(DVI dvi, DVI_DEVICE dev, char *cmd)
{
  struct ps *psi;
  char  *params, *param1, *param2, *param3, *p;

  if ((params = malloc(strlen(cmd)+1)) == NULL){
    DEV_CALL(dev, message_fatal)(dev, dvi, "No memory.");
    return -1;
  }
  strcpy(params, cmd);

  psi = &PD(dvi, ps_info);

  psi->given_hsize = 0;
  psi->given_vsize = 0;
  psi->given_filename = 0;

  /* Parse "postscriptbox{#1}{#2}{#3}" */
  /* #1 */
  if ((p = strchr(params, '{')) == NULL)
    goto Error;
  param1 = p+1;
  if ((p = strchr(param1, '}')) == NULL)
    goto Error;
  *p = '\0';
  /* #2 */
  if ((p = strchr(p+1, '{')) == NULL)
    goto Error;
  param2 = p+1;
  if ((p = strchr(param2, '}')) == NULL)
    goto Error;
  *p = '\0';
  /* #3 */
  if ((p = strchr(p+1, '{')) == NULL)
    goto Error;
  param3 = p+1;
  if ((p = strchr(param3, '}')) == NULL)
    goto Error;
  *p = '\0';

  if ((psi->param_hsize = numberstr2pt(dvi, dev, dev->h_dpi, param1)) < 0)
    goto Error;
  if ((psi->param_vsize = numberstr2pt(dvi, dev, dev->v_dpi, param2)) < 0)
    goto Error;
  strcpy(psi->param_filename, param3);
  psi->given_hsize = 1;
  psi->given_vsize = 1;
  psi->given_filename = 1;

  free(params);
  params = NULL;
  return 0;

Error:
  free(params);
  params = NULL;
  return -1;
}

Private double
numberstr2pt(DVI dvi, DVI_DEVICE dev, double dpi, char *str)
{
  double  val;
  char    *p;

  val = atof(str);
  p = str;
  while (isdigit((int)*p) || (*p == '.'))
    p++;
  if (*p =='\0')
    return val;
  
  if (strcmp(p, "pt") == 0)
    return val;

  DEV_CALL(dev,message_error)(dev, dvi, "DVIlib: Unknown unit %s", p);
  return val;
}




/**
 ** 'epsfile=' special
 **/
Private int eps_read_boundingbox(FILE*,struct ps*);
Private int read_pbm(DVI,DVI_DEVICE,int,DVI_BITMAP,long,long);
Private int read_ppm(DVI,DVI_DEVICE,int,DVI_PIXMAP_RGB,long,long);
Private int read_pgm(DVI,DVI_DEVICE,int,DVI_GRAYMAP,long,long);
Private int read_bitmap_header(DVI,DVI_DEVICE,int,char*,long*,long*,int*);

Public int 
dvi_special_epsfile(DVI dvi, DVI_DEVICE dev, char *cmd, long x, long y)
{
  int         size_type;
  struct ps  *psi;
  
  psi = &PD(dvi, ps_info);
  init_ps_info(psi);

  psi->bitmap_hoffset = 0;
  psi->bitmap_voffset = 0;
  if (parse_special_args(dvi, dev, cmd) < 0)
    return -1;
  if ((psi->given_hsize == 1) && (psi->given_vsize == 1)){
    size_type = EPS_SIZE_SPEC_VH_SIZE;
  } else {
    size_type = EPS_SIZE_SPEC_VH_SCALE;
    if (psi->given_hscale == 0)
      psi->param_hscale = 1.0;
    if (psi->given_vscale == 0)
      psi->param_vscale = 1.0;
  }

  if (eps2bitmap(dvi, dev, size_type) < 0)
    return -1;

  if (DEV_CALL(dev,draw_ps_figures)(dev, dvi) == 1){
    put_eps_figure(dvi, dev, x, y); 
  } else {
    put_frame(dev, dvi, x + psi->bitmap_hoffset, y + psi->bitmap_voffset,
	      psi->bm_width, psi->bm_height);
  }

  if (psi->bm.bitmap != NULL){
    free(psi->bm.bitmap);
    psi->bm.bitmap = NULL;
  }

  return 0;
}




/**
 ** 'PSfile=' special
 **/
Public int 
dvi_special_PSfile(DVI dvi, DVI_DEVICE dev, char *cmd, long x, long y)
{
  struct ps  *psi;
  
  psi = &PD(dvi, ps_info);
  init_ps_info(psi);

  psi->bitmap_hoffset = 0;
  psi->bitmap_voffset = 0;
  if (parse_special_args(dvi, dev, cmd) < 0)
    return -1;

  if (eps2bitmap(dvi, dev, EPS_SIZE_SPEC_BBX) < 0)
    return -1;

  if (DEV_CALL(dev,draw_ps_figures)(dev, dvi) == 1){
    put_eps_figure(dvi, dev,
		   x + psi->bitmap_hoffset, y + psi->bitmap_voffset); 
  } else {
    put_frame(dev, dvi, x + psi->bitmap_hoffset, y + psi->bitmap_voffset,
	      psi->bm_width, psi->bm_height);
  }

  if (psi->bm.bitmap != NULL){
    free(psi->bm.bitmap);
    psi->bm.bitmap = NULL;
  }

  return 0;
}


Private void
put_frame(DVI_DEVICE dev, DVI dvi, long x, long y, long w, long h)
{
  int  t;

  t = 0.3 * dev->h_dpi / 72.27;

  dvi_put_rectangle(dev, dvi, x, y, w, t);
  dvi_put_rectangle(dev, dvi, x, y, t, h);
  dvi_put_rectangle(dev, dvi, x, y + h - t, w, 1);
  dvi_put_rectangle(dev, dvi, x + w - 1, y, t, h);
#if 0
  printf("** (%ld,%ld) w=%ld, h=%ld\n", x, y, w, h);
#endif
}



/*
 * Parse arguments 
 */

Private int
parse_special_args(DVI dvi, DVI_DEVICE dev, char *cmd)
{
  char  *p, *p1, ch, delim;
  int   i;

  for (i = 0; special_args_str[i].name != NULL; i++){
    strcpy(special_args_str[i].val, ""); 
    *special_args_str[i].given = 0;
  }
  for (i = 0; special_args_float[i].name != NULL; i++){
    *special_args_float[i].val = 0;
    *special_args_float[i].given = 0;
  }
  for (i = 0; special_args_bool[i].name != NULL; i++){
    *special_args_bool[i].val = 0;
    *special_args_bool[i].given = 0;
  }

  p = cmd;
  while (*p != '\0'){
    while (isspace((int)*p) && (*p != '\0'))
      p++;
    if (*p == '\0')
      break;
    /* parse string args */
    for (i = 0; special_args_str[i].name != NULL; i++){
      if (special_args_str[i].namelen < 0)
	special_args_str[i].namelen = strlen(special_args_str[i].name);
      if (strncmp(p, special_args_str[i].name, 
		  special_args_str[i].namelen) != 0)
	continue;
      p = p + special_args_str[i].namelen;
      while (isspace((int)*p) && (*p != '\0'))
	p++;
      delim = ' ';
      if (*p == '"'){
	delim = '"';
	p++;
      }
      p1 = p;
      if (delim == '"'){
	while ((*p1 != delim) && (*p1 != '\0'))
	  p1++;
      } else {
	while (!isspace((int)*p1) && (*p1 != '\0'))
	  p1++;
      }
      ch = *p1;
      *p1 = '\0';
      strcpy(special_args_str[i].val, p); 
      *special_args_str[i].given = 1;
      *p1 = ch;
      p = p1;
      if (*p == delim)
	p++;
      goto DoNext;
    }
    /* parse float args */
    for (i = 0; special_args_float[i].name != NULL; i++){
      if (special_args_float[i].namelen < 0)
	special_args_float[i].namelen = strlen(special_args_float[i].name);
      if (strncmp(p, special_args_float[i].name, 
		  special_args_float[i].namelen) != 0)
	continue;
      p = p + special_args_float[i].namelen;
      p1 = p; 
      while (!isspace((int)*p1) && (*p1 != '\0'))
	p1++;
      ch = *p1;
      *p1 = '\0';
      *special_args_float[i].val   = atof(p); 
      *special_args_float[i].given = 1;
      /**printf("%s%f\n", special_args_float[i].name, atof(p));**/
      *p1 = ch;
      p = p1;
      goto DoNext;
    }
    /* parse bool args */
    for (i = 0; special_args_bool[i].name != NULL; i++){
      if (special_args_bool[i].namelen < 0)
	special_args_bool[i].namelen = strlen(special_args_bool[i].name);
      if (strncmp(p, special_args_bool[i].name, 
		  special_args_bool[i].namelen) != 0)
	continue;
      p = p + special_args_bool[i].namelen;
      *special_args_bool[i].val   = 1; 
      *special_args_bool[i].given = 1;
      goto DoNext;
    }
    /* not found */
    DEV_CALL(dev,message_error)
      (dev, dvi, "DVIlib: Unknown argument for special command: %s\n", p);
    return -1;

DoNext:
    ;
  }
  return 0;
}

Private void
init_ps_info(struct ps *psi)
{
  int  i;

  memset(psi, 0, sizeof(struct ps));

  i = 0;
  special_args_str[i].name    = "epsfile=";
  special_args_str[i].namelen = -1;
  special_args_str[i].val     =  psi->param_filename;
  special_args_str[i].given   = &psi->given_filename;
  i++;
  special_args_str[i].name    = "PSfile=";
  special_args_str[i].namelen = -1;
  special_args_str[i].val     =  psi->param_filename;
  special_args_str[i].given   = &psi->given_filename;
  i++;
  special_args_str[i].name    = NULL;
  special_args_str[i].namelen = -1;
  special_args_str[i].val     = NULL;
  special_args_str[i].given   = NULL;

  i = 0;
  special_args_float[i].name    = "hscale=";
  special_args_float[i].namelen = -1;
  special_args_float[i].val     = &psi->param_hscale;
  special_args_float[i].given   = &psi->given_hscale;
  i++;
  special_args_float[i].name    = "vscale=";
  special_args_float[i].namelen = -1;
  special_args_float[i].val     = &psi->param_vscale;
  special_args_float[i].given   = &psi->given_vscale;
  i++;
  special_args_float[i].name    = "hsize=";
  special_args_float[i].namelen = -1;
  special_args_float[i].val     = &psi->param_hsize;
  special_args_float[i].given   = &psi->given_hsize;
  i++;
  special_args_float[i].name    = "vsize=";
  special_args_float[i].namelen = -1;
  special_args_float[i].val     = &psi->param_vsize;
  special_args_float[i].given   = &psi->given_vsize;
  i++;
  special_args_float[i].name    = "hoffset=";
  special_args_float[i].namelen = -1;
  special_args_float[i].val     = &psi->param_hoffset;
  special_args_float[i].given   = &psi->given_hoffset;
  i++;
  special_args_float[i].name    = "voffset";
  special_args_float[i].namelen = -1;
  special_args_float[i].val     = &psi->param_voffset;
  special_args_float[i].given   = &psi->given_voffset;
  i++;
  special_args_float[i].name    = "llx=";
  special_args_float[i].namelen = -1;
  special_args_float[i].val     = &psi->param_llx;
  special_args_float[i].given   = &psi->given_llx;
  i++;
  special_args_float[i].name    = "lly=";
  special_args_float[i].namelen = -1;
  special_args_float[i].val     = &psi->param_lly;
  special_args_float[i].given   = &psi->given_lly;
  i++;
  special_args_float[i].name    = "urx=";
  special_args_float[i].namelen = -1;
  special_args_float[i].val     = &psi->param_urx;
  special_args_float[i].given   = &psi->given_urx;
  i++;
  special_args_float[i].name    = "ury=";
  special_args_float[i].namelen = -1;
  special_args_float[i].val     = &psi->param_ury;
  special_args_float[i].given   = &psi->given_ury;
  i++;
  special_args_float[i].name    = "rwi=";
  special_args_float[i].namelen = -1;
  special_args_float[i].val     = &psi->param_rwi;
  special_args_float[i].given   = &psi->given_rwi;
  i++;
  special_args_float[i].name    = "rhi=";
  special_args_float[i].namelen = -1;
  special_args_float[i].val     = &psi->param_rhi;
  special_args_float[i].given   = &psi->given_rhi;
  i++;
  special_args_float[i].name    = NULL;
  special_args_float[i].namelen = -1;
  special_args_float[i].val     = NULL;
  special_args_float[i].given   = NULL;

  i = 0;
  special_args_bool[i].name    = "clip";
  special_args_bool[i].namelen = -1;
  special_args_bool[i].val     = &psi->param_clip;
  special_args_bool[i].given   = &psi->given_clip;
  i++;
  special_args_bool[i].name    = NULL;
  special_args_bool[i].namelen = -1;
  special_args_bool[i].val     = NULL;
  special_args_bool[i].given   = NULL;
}





/*
 * EPS => Bitmap
 */

Private int
eps2bitmap(DVI dvi, DVI_DEVICE dev, int size_type)
{
  static int  last_format = EPS_OUTPUT_FORMAT_UNKNOWN;
  struct ps  *psi;
  char    tempfile[1024];
  FILE   *fp_psfile;
  int     max_level;
  long    w, h;
  char    format;
  int     fd, res;

  psi = &PD(dvi, ps_info);

  if (psi->given_filename == 0){
    DEV_CALL(dev,message_error)(dev, dvi, "EPS file name is not given."); 
    return -1;
  }

  if ((fp_psfile = eps2psfile(dvi, dev, tempfile, size_type)) == NULL){
    gs_invocation_nerr++;
    return -1;
  }

  if (DEV_CALL(dev,draw_ps_figures)(dev, dvi) == 0){
    DEV_CALL(dev,message_advice)(dev, dvi, 
				 "Skipping drawing a EPS figure: %s",
				 psi->param_filename);
    fclose(fp_psfile); 
    unlink(tempfile);
    return 0;
  }

  psi->output_format = eps_output_format(dev);
  psi->bm_format     = psi->output_format;

  if ((last_format != EPS_OUTPUT_FORMAT_UNKNOWN) 
      && (last_format != psi->output_format)){
    gs_kill();
  }
  
  res = 0;
  if ((fd = gs_convert_to_bitmap(dvi, dev, tempfile, psi->bm_format, 
				 psi->bm_width, psi->bm_height)) < 0){
    fclose(fp_psfile); 
    unlink(tempfile);
    gs_invocation_nerr++;
    return -1;
  }

  if (getenv("DVILIB_DEBUG_EPS") != NULL){
    printf("Temporary PostScript file to Ghostscript: %s\n", tempfile);
  } 

  last_format = psi->output_format;

  if (read_bitmap_header(dvi, dev, fd, &format, &w, &h, &max_level) < 0){
    fclose(fp_psfile); 
    unlink(tempfile);
    gs_invocation_nerr++;
    return -1;
  }

  if ((psi->bm_width != w) || (psi->bm_height != h)){
    DEV_CALL(dev,message_warning)
      (dev, dvi, "Output figure of ghostscript is bogus.");
    fclose(fp_psfile); 
    unlink(tempfile);
    gs_invocation_nerr++;
    if (getenv("DVILIB_DEBUG_EPS") != NULL){
      printf("Expected W and H are (%ld,%ld). GS output is (%ld,%ld)\n",
	     psi->bm_width, psi->bm_height, w, h);
    }
    return -1;
  }

  switch (format){
  case '4': /* PBM */
    psi->bm_format = EPS_OUTPUT_FORMAT_BITMAP;
    if ((res = read_pbm(dvi, dev, fd, &psi->bm, 
			psi->bm_width, psi->bm_height)) < 0)
      gs_kill();
    break;
  case '5': /* PGM */
    psi->bm_format = EPS_OUTPUT_FORMAT_GRAYMAP;
    psi->bm_gray.max = max_level;
    if ((res = read_pgm(dvi, dev, fd, &psi->bm_gray,
			psi->bm_width, psi->bm_height)) < 0)
      gs_kill();
    break;
  case '6': /* PPM */
    psi->bm_format = EPS_OUTPUT_FORMAT_PIXMAP_RGB;
    psi->bm_rgb.max = max_level;
    if ((res = read_ppm(dvi, dev, fd, &psi->bm_rgb, 
			psi->bm_width, psi->bm_height)) < 0)
      gs_kill();
  default:
    break;
  }

  gs_end_read(dvi, dev);
  fclose(fp_psfile); 
  unlink(tempfile);

  gs_invocation_nerr = 0;

  return res;
}

Private int 
eps_output_format(DVI_DEVICE dev)
{
  int  fmt;

  fmt = EPS_OUTPUT_FORMAT_BITMAP;
  if (DEV_METHOD_DEF(dev,put_pixmap_rgb)) 
    fmt = EPS_OUTPUT_FORMAT_PIXMAP_RGB;
  else if (DEV_METHOD_DEF(dev,put_graymap))
    fmt = EPS_OUTPUT_FORMAT_GRAYMAP;
  return fmt;
}


Private FILE*
eps2psfile(DVI dvi, DVI_DEVICE dev, char *tempfile, int size_type)
{
  struct ps *psi;
  char    *dvi_path, ps_path[MAXPATHLEN], buff[BUFSIZ], *p;
  int      fd_eps_w, fd_eps_r;
  FILE    *fp_eps_w, *fp_eps_r, *fp_in;
  int      need_clipping, in_header, ignore, len, i;
  double   w, h, mx, my, w_tmp, h_tmp, r;
  int      r_tmp, r_angle;


  psi = &PD(dvi, ps_info);

  sprintf(tempfile, "%s/dvilib.XXXXXX", DEV_CALL(dev,temp_dir)(dev, dvi));

#ifdef HAVE_MKSTEMP
  if ((fd_eps_w = mkstemp(tempfile)) < 0)
    return NULL;
#else
  mktemp(tempfile);
  if ((fd_eps_w = open(tempfile, O_WRONLY|O_CREAT|O_TRUNC, 0600)) < 0)
    return NULL;
#endif

  fp_eps_w = fdopen(fd_eps_w, FOPEN_WR_MODE_TEXT);
  if (fp_eps_w == NULL){
    if (fd_eps_w >= 0)
      close(fd_eps_w);
    DEV_CALL(dev,message_error)
      (dev, dvi, "Can't create a temporary file for PostScript figures.");
    unlink(tempfile);
    return NULL;
  }

  dvi_path = PD(dvi, file_name);
  if (strrchr(dvi_path, '/') == NULL){
    strcpy(ps_path, psi->param_filename);
  } else {
    strcpy(ps_path, dvi_path);
    p = strrchr(ps_path, '/');
    *p = '\0';
    strcat(ps_path, "/");
    strcat(ps_path, psi->param_filename);
  }
  if ((fp_in = fopen(ps_path, FOPEN_RD_MODE_TEXT)) == NULL){
    if ((fp_in = fopen(psi->param_filename, FOPEN_RD_MODE_TEXT)) == NULL){
      DEV_CALL(dev,message_warning)(dev, dvi, 
				    "Can't find EPS file: %s.", 
				    psi->param_filename);
      fclose(fp_eps_w); 
      unlink(tempfile);
      return NULL;
    }
  }

  if (eps_read_boundingbox(fp_in, psi) < 0){
    fclose(fp_in);
    fclose(fp_eps_w);
    unlink(tempfile);
    return NULL;
  }

  w = (double)(psi->ps_urx - psi->ps_llx);
  h = (double)(psi->ps_ury - psi->ps_lly);

  if ((mx = PD(dvi,gstate_scale_h)) <= 0)
    mx = 0 - mx;
  if ((my = PD(dvi,gstate_scale_v)) <= 0)
    my = 0 - my;

  switch (size_type){

  default:
    fprintf(stderr, "DVIlib internal error. (in spc_ps.c)\n");
    abort();

  case EPS_SIZE_SPEC_VH_SCALE:
    w_tmp = mx * psi->param_hscale * dev->h_dpi/(72.0*PD(dvi,shrink));
    h_tmp = my * psi->param_vscale * dev->v_dpi/(72.0*PD(dvi,shrink));
    psi->param_hscale = w_tmp;
    psi->param_vscale = h_tmp;
    psi->bm_width  = psi->param_hscale * w + 0.5;
    psi->bm_height = psi->param_vscale * h + 0.5;
    psi->bitmap_hoffset = 0;
    psi->bitmap_voffset = -psi->bm_height;
    break;

  case EPS_SIZE_SPEC_VH_SIZE:
    w_tmp = mx * psi->param_hsize * dev->h_dpi/(72.0*PD(dvi,shrink));
    h_tmp = my * psi->param_vsize * dev->v_dpi/(72.0*PD(dvi,shrink));
    psi->param_hscale = w_tmp / w;
    psi->param_vscale = h_tmp / h;
    psi->bm_width  = w_tmp + 0.5;
    psi->bm_height = h_tmp + 0.5;
    psi->bitmap_hoffset = 0;
    psi->bitmap_voffset = -psi->bm_height;
    break;

  case EPS_SIZE_SPEC_WIDTH_HEIGHT:
    w_tmp = mx * psi->param_hsize * dev->h_dpi/(72.0*PD(dvi,shrink));
    h_tmp = my * psi->param_vsize * dev->v_dpi/(72.0*PD(dvi,shrink));
    psi->param_hscale = w_tmp / w;
    psi->param_vscale = h_tmp / h;
    psi->bm_width  = w_tmp + 0.5;
    psi->bm_height = h_tmp + 0.5;
    psi->bitmap_hoffset = 0;
    psi->bitmap_voffset = -psi->bm_height;
    break;

  case EPS_SIZE_SPEC_BBX:
    psi->param_hscale = mx * dev->h_dpi/(72.0*PD(dvi,shrink));
    psi->param_vscale = my * dev->v_dpi/(72.0*PD(dvi,shrink));
    if ((psi->given_rwi == 1) && (psi->given_rhi == 0)){
      r = (psi->param_rwi / 10.0) / (psi->param_urx - psi->param_llx);
      psi->param_hscale *= r;
      psi->param_vscale *= r;
    } else if ((psi->given_rwi == 0) && (psi->given_rhi == 1)){
      r = (psi->param_rhi / 10.0) / (psi->param_ury - psi->param_lly);
      psi->param_hscale *= r;
      psi->param_vscale *= r;
    } else if ((psi->given_rwi == 1) && (psi->given_rhi == 1)){
      psi->param_hscale 
	*= ((psi->param_rwi / 10.0) / (psi->param_urx - psi->param_llx));
      psi->param_vscale 
	*= ((psi->param_rhi / 10.0) / (psi->param_ury - psi->param_lly));
    }
    need_clipping = 0;
    if ((psi->given_clip == 1) && (psi->param_clip == 1))
      need_clipping = 1;
    if (DVI_PROPERTY_TEST(PD(dvi,property), DVI_PROP_LATEX209_EPSF_STY))
      need_clipping = 1;
    if (need_clipping == 1){
      w_tmp = (psi->param_urx - psi->param_llx);
      h_tmp = (psi->param_ury - psi->param_lly);
      psi->bm_width  = w_tmp * psi->param_hscale + 0.5;
      psi->bm_height = h_tmp * psi->param_vscale + 0.5;
      psi->bitmap_hoffset = 0;
      psi->bitmap_voffset = -psi->bm_height;
      psi->ps_llx = psi->param_llx;
      psi->ps_lly = psi->param_lly;
      psi->ps_urx = psi->param_urx;
      psi->ps_ury = psi->param_ury;
    } else {
      w_tmp = psi->ps_llx - psi->param_llx;
      h_tmp = psi->ps_ury - psi->param_lly;
      psi->bm_width  = w * psi->param_hscale + 0.5;
      psi->bm_height = h * psi->param_vscale + 0.5;
      psi->bitmap_hoffset = 0 - (w_tmp * psi->param_hscale + 0.5);
      psi->bitmap_voffset = 0 - (h_tmp * psi->param_vscale + 0.5);
    }
    break;

  }

  r_angle = PD(dvi,gstate_angle);
  while (r_angle < 0)
    r_angle += 360;
  r_angle %= 360;

#ifdef DEBUG_DISPLAY_EPS_INFO
  printf("\nFilename: %s\n", psi->param_filename);
  printf("size_type=%d\n", size_type);
  printf("Bounding Box: %f %f %f %f\n", 
	 psi->ps_llx, psi->ps_lly, psi->ps_urx, psi->ps_ury);
  printf("H_DPI=%f, V_DPI=%f S=%f\n", 
         dev->h_dpi, dev->v_dpi, PD(dvi,shrink));
  printf("W=%ld, H=%ld\n", psi->bm_width, psi->bm_height);
  printf("%f %f scale\n", psi->param_hscale, psi->param_vscale);
  printf("%f %f translate\n", -psi->ps_llx, -psi->ps_lly);
  printf("%d rotate\n", r_angle);
#endif

  rewind(fp_in);


  /* Disable `showpage'. (See "5002.EPSF_Spec" by Adobe for detail.) */
  fprintf(fp_eps_w, "/BEGINEPSFILE{ %%def\n");  
  fprintf(fp_eps_w, "   /EPSFsave save def\n");  
  fprintf(fp_eps_w, "   0 setgray 0 setlinecap 1 setlinewidth\n");  

  fprintf(fp_eps_w, "   0 setlinejoin 10 setmiterlimit [] 0 setdash\n");  
  fprintf(fp_eps_w, "   newpath\n");  
  fprintf(fp_eps_w, "   /showpage {} def\n");  
  fprintf(fp_eps_w, "} bind def\n");  
  fprintf(fp_eps_w, "/ENDEPSFILE { %%def \n");  
  fprintf(fp_eps_w, "   EPSFsave restore\n");  
  fprintf(fp_eps_w, "} bind def\n");  
  fprintf(fp_eps_w, "BEGINEPSFILE\n");  

  fprintf(fp_eps_w, "initgraphics\n");
  fprintf(fp_eps_w, "gsave\n");  

  fprintf(fp_eps_w, "%f %f scale\n", psi->param_hscale, psi->param_vscale);

  switch (PD(dvi,gstate_angle)){
  default:
  case 0:
    r_angle = 0;
    fprintf(fp_eps_w, "%f %f translate\n", -psi->ps_llx, -psi->ps_lly);
    break;
  case 90:
    r_tmp = psi->bm_height;
    psi->bm_height = psi->bm_width;
    psi->bm_width = r_tmp;
    fprintf(fp_eps_w, "%f %f translate\n", psi->ps_ury, -psi->ps_llx);
    psi->bitmap_hoffset -=  psi->bm_width;
    psi->bitmap_voffset += (psi->bm_width - psi->bm_height);
    break;
  case 180:
    fprintf(fp_eps_w, "%f %f translate\n", psi->ps_urx, psi->ps_ury);
    psi->bitmap_hoffset -= psi->bm_width;
    psi->bitmap_voffset += psi->bm_height;
    break;
  case 270:
    r_tmp = psi->bm_height;
    psi->bm_height = psi->bm_width;
    psi->bm_width = r_tmp;
    fprintf(fp_eps_w, "%f %f translate\n", -psi->ps_lly, psi->ps_urx);
    psi->bitmap_voffset += psi->bm_width;
    break;
  }
  fprintf(fp_eps_w, "%d rotate\n", r_angle);

#if 0
  printf("angle %d\n", r_angle);
  printf("w, h = %ld %ld\n", psi->bm_width, psi->bm_height);
#endif


  in_header = 1; 
  if (fgets(buff, sizeof(buff), fp_in) != NULL){
    fprintf(fp_eps_w, "%s", buff);
    if (strncmp(buff, "%!PS-Adobe", 10) == 0){
      len = strlen(buff);
      for (i = 0; i < len; i++){
	if (strncmp(&buff[i], "EPSF-", 5) == 0){
	  break;
	}
      }
    }
  }

  while (in_header == 1){
    if (fgets(buff, sizeof(buff), fp_in) == NULL)
      break;
    if (strncmp(buff, "%%BoundingBox:", 14) == 0){
      fprintf(fp_eps_w, "%%%%BoundingBox: %.2f %.2f %.2f %.2f\n", 
	      psi->ps_llx, psi->ps_lly, psi->ps_urx, psi->ps_ury);
    } else {
      fprintf(fp_eps_w, "%s", buff);
      if (strncmp(buff, "%%EndComments", 13) == 0)
	in_header = 0;
      if (strncmp(buff, "%%EndProlog", 11) == 0)
	in_header = 0;
    }
  }

  ignore = 0;
  for (;;){
    int n;
    char  p1[1024], p2[1024];

    if (fgets(buff, sizeof(buff), fp_in) == NULL)
      break;

    n = sscanf(buff, "%s %s", p1, p2);

    /* ignore size specs */
    if ((n == 2) && (strcmp(p1, "%%PaperSize:") == 0)) {
      ;

    } else if ((n == 2) && strcmp(p1, "%%BeginPaperSize:") == 0) {
      ignore = 1;

    } else if ((n == 1) && (strcmp(p1, "%%EndPaperSize") == 0)) {
      ignore = 0;

    /* ignore some TGIF code */
    } else if ((n == 2) && 
	       (strcmp(p1, "tgifsavedpage") == 0) 
	       && (strcmp(p2, "restore") == 0) ){
      fprintf(fp_eps_w, "%% %s", buff);

    } else if (ignore == 0){
      fprintf(fp_eps_w, "%s", buff);
    }
  }

  fprintf(fp_eps_w, "grestore\n");  
  fprintf(fp_eps_w, "ENDEPSFILE\n");  
  fprintf(fp_eps_w, "showpage\n");  

  fclose(fp_in);
  fclose(fp_eps_w);

  if ((fd_eps_r = open(tempfile, O_RDONLY, 0600)) < 0)
    return NULL;
  if ((fp_eps_r = fdopen(fd_eps_r, FOPEN_RD_MODE_TEXT)) == NULL){
    close(fd_eps_r);
    return NULL;
  }
  
  return fp_eps_r; 
  /* Caller must must close "fp_eps_r" and unlink "tempfile". */
}

Private int
eps_read_boundingbox(FILE *fp, struct ps *psi)
{
  int   n;
  char  buff[BUFSIZ];

  rewind(fp);
  n = -1;
  for (;;){
    if (fgets(buff, sizeof(buff), fp) == NULL)
      break;
    buff[sizeof(buff)-1] = '\0';
    if (strncmp(buff, "%%BoundingBox:", 14) == 0){
      n = sscanf(&(buff[14]), "%lf %lf %lf %lf", 
		 &(psi->ps_llx), &(psi->ps_lly), 
		 &(psi->ps_urx), &(psi->ps_ury));
      break;
    }
    if (strncmp(buff, "%%EndComments:", 14) == 0)
      break;
    if (strncmp(buff, "%%EndProlog:", 12) == 0)
      break;
  }
  rewind(fp);

  if ((n != 0)
      && (psi->given_llx == 1) && (psi->given_lly == 1)
      && (psi->given_urx == 1) && (psi->given_ury == 1)){
    psi->ps_llx = psi->param_llx;
    psi->ps_lly = psi->param_lly;
    psi->ps_urx = psi->param_urx;
    psi->ps_ury = psi->param_ury;
    n = 4;
  }

  return (n==4) ? 0 : -1;
}



/* Put an EPS figure on device */

Private void   put_eps_figure_bitmap(DVI, DVI_DEVICE, DVI_BITMAP,
				     long, long);
Private void   put_eps_figure_graymap(DVI, DVI_DEVICE, DVI_GRAYMAP,
				      long, long);
Private void   put_eps_figure_pixmap_rgb(DVI, DVI_DEVICE, DVI_PIXMAP_RGB,
				       long, long);

Private void
put_eps_figure(DVI dvi, DVI_DEVICE dev, long x, long y)
{
  struct ps *psi;

  psi = &PD(dvi, ps_info);

  switch (psi->output_format){

  case EPS_OUTPUT_FORMAT_BITMAP:
    switch (psi->bm_format){
    case EPS_OUTPUT_FORMAT_BITMAP:
      put_eps_figure_bitmap(dvi, dev, &psi->bm, x, y);
      break;
    case EPS_OUTPUT_FORMAT_PIXMAP_RGB:
    case EPS_OUTPUT_FORMAT_GRAYMAP:
      break;
    }
    break;
  
  case EPS_OUTPUT_FORMAT_GRAYMAP:
    switch (psi->bm_format){
    case EPS_OUTPUT_FORMAT_BITMAP:
      if (DEV_METHOD_DEF(dev,put_bitmap))
	put_eps_figure_bitmap(dvi, dev, &psi->bm, x, y);
      else 
	put_eps_bm2gray(dvi, dev, x, y, &psi->bm);
      break;
    case EPS_OUTPUT_FORMAT_PIXMAP_RGB:
      break;
    case EPS_OUTPUT_FORMAT_GRAYMAP:
      put_eps_figure_graymap(dvi, dev, &psi->bm_gray, x, y);
      break;
    }
    break;

  case EPS_OUTPUT_FORMAT_PIXMAP_RGB:
    switch (psi->bm_format){
    case EPS_OUTPUT_FORMAT_BITMAP:
      if (DEV_METHOD_DEF(dev,put_bitmap))
	put_eps_figure_bitmap(dvi, dev, &psi->bm, x, y);
      else 
	put_eps_bm2rgb(dvi, dev, x, y, &psi->bm);
      break;
    case EPS_OUTPUT_FORMAT_PIXMAP_RGB:
      put_eps_figure_pixmap_rgb(dvi, dev, &psi->bm_rgb, x, y);
      break;
    case EPS_OUTPUT_FORMAT_GRAYMAP:
      put_eps_gray2rgb(dvi, dev, x, y, &psi->bm_gray);
      break;
    }
    break;
  }

  switch (psi->bm_format){
  case EPS_OUTPUT_FORMAT_BITMAP:
    free(psi->bm.bitmap);
    psi->bm.bitmap = NULL;
    break;
  case EPS_OUTPUT_FORMAT_PIXMAP_RGB:
    free(psi->bm_rgb.bitmap);
    psi->bm_rgb.bitmap = NULL;
    break;
  case EPS_OUTPUT_FORMAT_GRAYMAP:
    free(psi->bm_gray.bitmap);
    psi->bm_gray.bitmap = NULL;
    break;
  }
}

Private void
put_eps_bm2gray(DVI dvi, DVI_DEVICE dev, long x, long y, DVI_BITMAP bm)
{
  static unsigned char  bits[] = {
    0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01};
  struct dvi_s_graymap  bm_gray;
  unsigned char         *bp, *gp; 
  int                   xxx, yyy;

  if (DEV_METHOD_UNDEF(dev,put_graymap))
    return;

  bm_gray.width  = bm->width;
  bm_gray.height = bm->height;
  bm_gray.raster = bm->width;
  bm_gray.max    = 255;
  bm_gray.bitmap = (unsigned char*)malloc(bm->width * bm->height);
  if (bm_gray.bitmap == NULL)
    return; 

  for (yyy = 0; yyy < bm->height; yyy++){
    bp = &bm->bitmap[yyy*bm->raster];
    gp = &bm_gray.bitmap[yyy*bm_gray.raster];
    for (xxx = 0; xxx < bm->width; xxx++){
      if ((bp[xxx/8] & bits[xxx%8]) != 0)
	gp[xxx] = 255;
      else
	gp[xxx] = 0;
    }
  }

  put_eps_figure_graymap(dvi, dev, &bm_gray, x, y);

  free(bm_gray.bitmap);
  bm_gray.bitmap = NULL;
}

Private void
put_eps_bm2rgb(DVI dvi, DVI_DEVICE dev, long x, long y, DVI_BITMAP bm)
{
  static unsigned char  bits[] = {
    0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01};
  struct dvi_s_pixmap_rgb  bm_rgb;
  unsigned char            *bp, *rp; 
  int                      xxx, yyy, x_rgb;

  if (DEV_METHOD_UNDEF(dev,put_pixmap_rgb))
    return;

  bm_rgb.width  = bm->width;
  bm_rgb.height = bm->height;
  bm_rgb.raster = 3 * bm->width;
  bm_rgb.max    = 255;
  bm_rgb.bitmap = (unsigned char*)malloc(3 * bm->width * bm->height);
  if (bm_rgb.bitmap == NULL)
    return; 

  for (yyy = 0; yyy < bm->height; yyy++){
    bp = &bm->bitmap[yyy*bm->raster];
    rp = &bm_rgb.bitmap[yyy*bm_rgb.raster];
    for (xxx = 0, x_rgb = 0; xxx < bm->width; xxx++, x_rgb += 3){
      if ((bp[xxx/8] & bits[xxx%8]) != 0)
	rp[x_rgb+0] = rp[x_rgb+1] = rp[x_rgb+2] = 255;
      else 
	rp[x_rgb+0] = rp[x_rgb+1] = rp[x_rgb+2] = 0;
    }
  }

  put_eps_figure_pixmap_rgb(dvi, dev, &bm_rgb, x, y);

  free(bm_rgb.bitmap);
  bm_rgb.bitmap = NULL;
}

Private void
put_eps_gray2rgb(DVI dvi, DVI_DEVICE dev, long x, long y, DVI_GRAYMAP bm_gray)
{
  struct dvi_s_pixmap_rgb  bm_rgb;
  unsigned char            *gp, *rp; 
  int                      xxx, yyy, x_rgb;

  if (DEV_METHOD_UNDEF(dev,put_pixmap_rgb))
    return;

  bm_rgb.width  = bm_gray->width;
  bm_rgb.height = bm_gray->height;
  bm_rgb.raster = 3 * bm_gray->width;
  bm_rgb.max    = bm_gray->max;
  bm_rgb.bitmap = (unsigned char*)malloc(bm_gray->width * bm_gray->height);
  if (bm_rgb.bitmap == NULL)
    return; 

  for (yyy = 0; yyy < bm_gray->height; yyy++){
    gp = &bm_gray->bitmap[yyy*bm_gray->raster];
    rp = &bm_rgb.bitmap[yyy*bm_rgb.raster];
    for (xxx = 0, x_rgb = 0; xxx < bm_gray->width; xxx++, x_rgb += 3)
      rp[x_rgb+0] = rp[x_rgb+1] = rp[x_rgb+2] = gp[xxx];
  }

  put_eps_figure_pixmap_rgb(dvi, dev, &bm_rgb, x, y);

  free(bm_rgb.bitmap);
  bm_rgb.bitmap = NULL;
}


Private void
put_eps_figure_bitmap(DVI dvi, DVI_DEVICE dev, DVI_BITMAP bm, long x, long y)
{
  struct dvi_s_bitmap  b0;
  int  step, i;

  if (!DEV_METHOD_DEF(dev,put_bitmap))
    return;

  step = -1;
  if (DVI_PROPERTY_TEST(PD(dvi,property), 
			DVI_PROP_INCREMENTAL_EPS_DISPLAY)){
    step = DEV_CALL(dev, ps_figure_display_step)(dev, dvi,
						 bm->width, bm->height);
    if (step < 1)
      step = EPS_BITMAP_INCREMENTAL_DISPLAY;
  }
  if (step < 1){
    DEV_CALL(dev, put_bitmap)(dev, dvi, bm, -1, -1, -1, x, y);
  } else {
    b0.width  = bm->width;
    b0.raster = bm->raster;
    for (i = 0; i < bm->height; i += step) {
      b0.bitmap = &bm->bitmap[bm->raster * i];
      b0.height = step;
      if (i + step > bm->height)
	b0.height = bm->height - i;
      DEV_CALL(dev, put_bitmap)(dev, dvi, &b0, -1, -1, -1, x, y+i);
    }
  }
}

Private void
put_eps_figure_graymap(DVI dvi, DVI_DEVICE dev, DVI_GRAYMAP gm, long x, long y)
{
  struct dvi_s_graymap  g0;
  int  step, i;

  if (!DEV_METHOD_DEF(dev,put_graymap))
    return;

  step = -1;
  if (DVI_PROPERTY_TEST(PD(dvi,property), 
			DVI_PROP_INCREMENTAL_EPS_DISPLAY)){
    step = DEV_CALL(dev, ps_figure_display_step)(dev, dvi,
						 gm->width, gm->height);
    if (step < 1)
      step = EPS_BITMAP_INCREMENTAL_DISPLAY;
  }
  if (step < 1){
    DEV_CALL(dev, put_graymap)(dev, dvi, gm, -1, -1, -1, x, y);
  } else {
    g0.width  = gm->width;
    g0.raster = gm->raster;
    g0.max    = gm->max;
    for (i = 0; i < gm->height; i += step){
      g0.bitmap = &gm->bitmap[gm->raster * i];
      g0.height = step;
      if (i + step > gm->height)
	g0.height = gm->height - i;
      DEV_CALL(dev, put_graymap)(dev, dvi, &g0, -1, -1, -1, x, y+i);
    }
  }
}


Private void
put_eps_figure_pixmap_rgb(DVI dvi, DVI_DEVICE dev, DVI_PIXMAP_RGB pm, 
			  long x, long y)
{
  struct dvi_s_pixmap_rgb  p0;
  int  step, i;

  if (!DEV_METHOD_DEF(dev,put_pixmap_rgb))
    return;

  step = -1;
  if (DVI_PROPERTY_TEST(PD(dvi,property), 
			DVI_PROP_INCREMENTAL_EPS_DISPLAY)){
    step = DEV_CALL(dev, ps_figure_display_step)(dev, dvi,
						 pm->width, pm->height);
    if (step < 1)
      step = EPS_BITMAP_INCREMENTAL_DISPLAY;
  }
  if (step < 1){
    DEV_CALL(dev, put_pixmap_rgb)(dev, dvi, pm, -1, -1, -1, x, y);
  } else {
    p0.width  = pm->width;
    p0.raster = pm->raster;
    p0.max    = pm->max;
    for (i = 0; i < pm->height; i += step) {
      p0.bitmap = &pm->bitmap[pm->raster * i];
      p0.height = step;
      if (i + step > pm->height)
	p0.height = pm->height - i;
      DEV_CALL(dev, put_pixmap_rgb)(dev, dvi, &p0, -1, -1, -1, x, y+i);
    }
  }
}




#define FILE_FORMAT_PBM   0
#define FILE_FORMAT_PPM   1
#define FILE_FORMAT_PGM   2

/*
 * Read raw PBM file
 */
Private int
read_pbm(DVI dvi, DVI_DEVICE dev, int fd, DVI_BITMAP bm,
	 long width, long height)
{
  long            size;
  int             n; 

  bm->width  = width;
  bm->height = height;
  bm->raster = (width + 7)/8;
  bm->bitmap = (unsigned char*)malloc(bm->raster * bm->height);
  if (bm->bitmap == NULL){
    DEV_CALL(dev,message_fatal)(dev, dvi, "No memory."); 
    return -1;
  }
  memclr(bm->bitmap, bm->raster * bm->height);

  size = ((width + 7) / 8) * height;
  if ((n = gs_read(dvi, dev, fd, bm->bitmap, size)) <= 0){
    free(bm->bitmap);
    bm->bitmap = NULL;
    return -1;
  }
  return 0;
}

/*
 * Read raw PPM file
 */
Private int
read_ppm(DVI dvi, DVI_DEVICE dev, int fd, DVI_PIXMAP_RGB bm_rgb,
	 long width, long height)
{
  int     n; 

  bm_rgb->width  = width;
  bm_rgb->height = height;
  bm_rgb->raster = 3 * width;
  bm_rgb->bitmap
    = (unsigned char*)malloc(bm_rgb->raster * bm_rgb->height);
  if (bm_rgb->bitmap == NULL){
    DEV_CALL(dev,message_fatal)(dev, dvi, "No memory."); 
    return -1;
  }
  memclr(bm_rgb->bitmap, bm_rgb->raster * bm_rgb->height);

  if ((n = gs_read(dvi, dev, fd, bm_rgb->bitmap, 3*width*height)) <= 0){
    free(bm_rgb->bitmap);
    bm_rgb->bitmap = NULL;
    return -1;
  }
  return 0;
}

/*
 * Read raw PGM file
 */
Private int
read_pgm(DVI dvi, DVI_DEVICE dev, int fd, DVI_GRAYMAP bm_gray,
	 long width, long height)
{
  int             n; 

  bm_gray->width  = width;
  bm_gray->height = height;
  bm_gray->raster = width;
  bm_gray->bitmap
      = (unsigned char*)malloc(bm_gray->raster * bm_gray->height);
  if (bm_gray->bitmap == NULL){
    DEV_CALL(dev,message_fatal)(dev, dvi, "No memory."); 
    return -1;
  }
  memclr(bm_gray->bitmap, bm_gray->raster * bm_gray->height);

  if ((n = gs_read(dvi, dev, fd, bm_gray->bitmap, width*height)) <= 0){
    free(bm_gray->bitmap);
    bm_gray->bitmap = NULL;
    return -1;
  }
  return 0;
}

/* read PBM, PPM, PGM header */
Private int
read_bitmap_header(DVI dvi, DVI_DEVICE dev, int fd, 
		   char *formatp, long *widthp, long *heightp,
		   int *max_levelp)
{
  int   n;
  unsigned char  buff[16];

  *formatp = ' ';
  *widthp  = -1;
  *heightp = -1;

  if ((n = gs_read(dvi, dev, fd, buff, 3)) <= 0)
    return -1;
  *formatp = buff[1];

  if (buff[0] != 'P')
    return -2;
  if  ((buff[1] != '4') && (buff[1] != '5') && (buff[1] != '6'))
    return -3;

  /* skip comment lines */
  for (;;){
    if (gs_read_char(dvi, dev, fd, buff) <= 0)
      return -4;
    if (buff[0] != '#'){
      gs_unread_char(dvi, dev, fd, buff);
      break;
    }
    for (;;){
      if (gs_read_char(dvi, dev, fd, buff) <= 0)
	return -5;
      if (buff[0] == '\n')
	break;
    }
  }

  /* width */
  *widthp = 0;
  for (;;){
    if (gs_read_char(dvi, dev, fd, buff) <= 0)
      return -6;
    if (!isdigit(buff[0]))
      break;
    *widthp = (*widthp)*10L + (buff[0]-'0');
  }
  if (!isspace(buff[0]))
    return -7;

  /* height */
  *heightp = 0;
  for (;;){
    if (gs_read_char(dvi, dev, fd, buff) <= 0)
      return -8;
    if (!isdigit(buff[0]))
      break;
    *heightp = (*heightp)*10L + (buff[0]-'0');
  }

  /* max value */
  switch (buff[1]){
  case '5': /* PGM */
  case '6': /* PPM */
    *max_levelp = 0;
    for (;;){
      if (gs_read_char(dvi, dev, fd, buff) <= 0)
	return -9;
      if (!isdigit(buff[0]))
	break;
      *max_levelp = (*max_levelp)*10L + (buff[0]-'0');
    }
    break;
  case '4': /* PBM */
    *max_levelp = 1;
    break;
  }
  return 0;
}




/*
 * GhostScript Interface 
 */

#define GIVEUP_LIMIT                 5
#define MAX_RETRY_PER_GS_INVOCATION  3

#define GS_STAT_UNINITED  1
#define GS_STAT_RUNNING   2
#define GS_STAT_DIED      3
Private int  GS_ProcessStatus = GS_STAT_UNINITED;


#ifdef HAVE_PIPE

/*
 * GhostScript Interface
 * Copyright (C) 1996-1999 by Hirotsugu Kakugawa (h.kakugawa@computer.org)
 *
 * Becase the cost of process creation is high, GhostScript (gs) is invoked 
 * only once until it dies for errors in a PostScript file or system error.
 * That is, a gs process rasterizes many PS files. As communication links 
 * between driver and gs, I use two pipes for bidirectional communication.
 *
 * PROTOCOL:
 * 1. First, driver process creates a gs process.  Pipes are created at 
 *   this time. The rasterized bitmap is sent to stdout of gs process.
 * 2. When driver needs a rasterized EPS figure, it makes a PS file in which
 *   a EPS file is embedded and then, sends a message 
 *       [ /HWSize [ WIDTH HEIGHT ] currentdevice putdeviceprops pop 
 *       (PS-FILE-NAME) run flush
 *   to the gs process.  Then, gs process starts rasterizing a PS file
 *   and the resulting bitmap is sent to stdout.
 * 3. Driver process receives the bitmap via pipe. 
 * 4. When a gs process dies for some reason, driver try to invoke gs again.
 * 5. If driver failes to read bitmap, it kills gs process and creates new
 *   gs process to reset the protocol.
 */

Private int  GS_ProcessID     = -1;
Private int  GS_Pipe_To_GS    = -1;
Private int  GS_Pipe_From_GS  = -1;
Private int  GS_LastFormat    = -1;

Private int   start_ghostscript_process(DVI,DVI_DEVICE,int,int);
Private int   try_start_ghostscript_process(DVI,DVI_DEVICE,int,int);
Private int   send_to_gs(char*);
Private RETSIGTYPE  gs_when_died(int);


Private int
gs_convert_to_bitmap(DVI dvi, DVI_DEVICE dev, char *psfile, 
		     int bm_format, long bm_width, long bm_height)
{
  char  buff[1024];

  if (start_ghostscript_process(dvi, dev, bm_format, 0) < 0)
    return -1;

  DEV_CALL(dev,message_advice)(dev, dvi, "Drawing a PostScript figure...");

  sprintf(buff, "[ /HWSize [ %ld %ld ] currentdevice putdeviceprops pop\n",
	  bm_width, bm_height);
  if (getenv("DVILIB_DEBUG_EPS") != NULL){
    printf("Send to Ghostscrip: %s", buff);
  }
  if (send_to_gs(buff) < 0)
    return -1;

  sprintf(buff, "(%s) run flush\n", psfile);
  if (getenv("DVILIB_DEBUG_EPS") != NULL){
    printf("Send to Ghostscrip: %s", buff);
  }
  if (send_to_gs(buff) < 0)
    return -1;

  return GS_Pipe_From_GS;
}

Private int
gs_end_read(DVI dvi, DVI_DEVICE dev)
{
  if (GS_ProcessStatus == GS_STAT_RUNNING)
    DEV_CALL(dev,message_advice)(dev, dvi, "done.");
  return 0;
}


Private int 
send_to_gs(char *buff)
{
  int  rest, n;
  char *p;

  p    = buff;
  rest = strlen(buff);
  while (rest > 0){
    if ((n = write(GS_Pipe_To_GS, p, rest)) < 0){
      gs_kill();
      return -1;
    }
    rest -= n;
    p = &(p[n]);
  }
  return 0;
}

Public int
dvi_special_invoke_gs(DVI dvi, DVI_DEVICE dev)
{
  return  start_ghostscript_process(dvi, dev, eps_output_format(dev), 1);
}

Private int
start_ghostscript_process(DVI dvi, DVI_DEVICE dev, int bm_format, int first)
{
  static int  retry  = 0;
  int         retry2 = 0;

  if (GS_LastFormat == -1)
    GS_LastFormat = bm_format;
  if (GS_LastFormat != bm_format)
    gs_kill();

  if (GS_ProcessStatus == GS_STAT_RUNNING)
    return 0;
  if (GS_ProcessStatus == GS_STAT_UNINITED)
    if (try_start_ghostscript_process(dvi, dev, bm_format, 1) < 0)
      return -1;

  if (retry >= GIVEUP_LIMIT)
    return -1;
  retry2 = 0;
  while (GS_ProcessStatus == GS_STAT_DIED){
    retry2++;
    if (retry2 == 1)
      DEV_CALL(dev,message_advice)
	(dev, dvi, "Invoke GhostScript...");
    else 
      DEV_CALL(dev,message_advice)
	(dev, dvi, "Trying to invoke GhostScript again...");
    if (try_start_ghostscript_process(dvi, dev, bm_format, 0) >= 0){
      DEV_CALL(dev,message_advice)(dev, dvi, "done.");
      retry = 0;
      break;
    }
    if (retry2 >= MAX_RETRY_PER_GS_INVOCATION){
      DEV_CALL(dev,message_advice)
	(dev, dvi, "I tried many times to invoke GhostScript but in vain.");
      retry++;
      if (retry >= GIVEUP_LIMIT)
	DEV_CALL(dev,message_advice)
	  (dev, dvi, "I don't try invoking GhostScript any more.");
      return -1;
    }
#ifdef HAVE_SLEEP
    sleep(1);
#endif
  }
  return 0;
}

Private int
try_start_ghostscript_process(DVI dvi, DVI_DEVICE dev, 
			      int bm_format, int first)
{
  int   i;
  int   pipe_dvi2gs[2], pipe_gs2dvi[2];
  char  *gs_output_device, *gs_argv[16];
  char  OptFormat[128];

  if (first == 1)
    DEV_CALL(dev,message_advice)(dev, dvi, "Invoking GhostScript...");

  switch (bm_format){
  default:
  case EPS_OUTPUT_FORMAT_BITMAP:
    gs_output_device = "pbmraw";
    break;
  case EPS_OUTPUT_FORMAT_PIXMAP_RGB: 
    gs_output_device = "ppmraw";
    break;
  case EPS_OUTPUT_FORMAT_GRAYMAP: 
    gs_output_device = "pgmraw";
    break;
  }

  i = 0;
  gs_argv[i++] = DEV_CALL(dev,gs_program_path)(dev, dvi);
  gs_argv[i++] = "-dQUIET";
  gs_argv[i++] = "-dNOPAUSE";
  sprintf(OptFormat, "-sDEVICE=%s", gs_output_device);
  gs_argv[i++] = OptFormat;
  gs_argv[i++] = "-sOutputFile=-"; 
  gs_argv[i++] = "-";
  gs_argv[i++] = NULL;

  if (getenv("DVILIB_DEBUG_EPS") != NULL){
    int  j;
    printf("Invoke Ghostscript: ");
    for (j = 0; gs_argv[j] != NULL; j++)
      printf("%s ", gs_argv[j]);
    printf("\n");
  }

  if (pipe(pipe_gs2dvi) < 0){
    if (gs_invocation_nerr == 0)
      DEV_CALL(dev,message_error)(dev, dvi, "Failed to invoke GhostScript.");
    return -1;
  }
  if (pipe(pipe_dvi2gs) < 0){
    if (gs_invocation_nerr == 0)
      DEV_CALL(dev,message_error)(dev, dvi, "Failed to invoke GhostScript.");
    return -1;
  }

#ifdef SIGCHLD
#if 1
  signal(SIGCHLD,  gs_when_died);
#endif
#endif
#ifdef SIGPIPE
  signal(SIGPIPE,  gs_when_died);
#endif

  GS_Pipe_To_GS    = -1;
  GS_Pipe_From_GS  = -1;
  GS_ProcessStatus = GS_STAT_RUNNING;
  GS_ProcessID     = fork();
  switch (GS_ProcessID){
  case 0:
    /*** GS ***/
#if defined(__linux__)
    (void) setpgid(0,0);
#elif defined(__svr4__)||defined(__SVR4__)
    (void) setpgrp();
#else
    (void) setpgrp(0, getpid());
#endif
    /* gs <- dvi */
    close(0);
    dup2(pipe_dvi2gs[0], 0);
    close(pipe_dvi2gs[0]);
    close(pipe_dvi2gs[1]);
    /* gs -> dvi */
    close(1);
    dup2(pipe_gs2dvi[1], 1);
    close(pipe_gs2dvi[0]);
    close(pipe_gs2dvi[1]);
    /* exec gs */
    execvp(DEV_CALL(dev,gs_program_path)(dev, dvi), gs_argv);
    if (gs_invocation_nerr == 0)
      fprintf(stderr, "Failed to exec Ghostscript: %s\n", 
	      DEV_CALL(dev,gs_program_path)(dev, dvi));
    close(0);
    close(1);
    exit(1);  /* exit if failed to exec gs */
    break;
  default:
    /*** DVI ***/
    /* dvi -> gs */
    close(pipe_dvi2gs[0]);
    GS_Pipe_To_GS = pipe_dvi2gs[1];
    /* dvi <- gs */
    close(pipe_gs2dvi[1]);
    GS_Pipe_From_GS = pipe_gs2dvi[0];
    if (first == 1)
      DEV_CALL(dev,message_advice)(dev, dvi, "done.");
    break;
  case -1:
    GS_ProcessStatus = GS_STAT_DIED;
    close(pipe_dvi2gs[0]); close(pipe_dvi2gs[1]);
    close(pipe_gs2dvi[0]); close(pipe_gs2dvi[1]);
    if (gs_invocation_nerr == 0)
      DEV_CALL(dev,message_error)(dev, dvi, "Failed to invoke GhostScript.");
    return -1;
    break;
  }
  return 0;
}


Private RETSIGTYPE
gs_when_died(int sig)
{
  int  pid, pst;

  if (GS_ProcessID < 0)
    return;

  do {
#ifdef HAVE_WAITPID
    /* we use WNOHANG option, since this routine may be called 
     * by SIGCHLD of a process other than ghostscript */
    pid = waitpid(GS_ProcessID, &pst, WNOHANG);
    if (pid == 0)      /* SIGCHLD by a non-gs process */
      return;
#else
    /* XXX wrong code (incorrect when non-gs process terminates) */
    pid = wait(&pst);
    if (pid != GS_ProcessID)
      return;
#endif
  } while ((pid == -1) && (errno == EINTR));
  
 
#if DEBUG_DISPLAY_EPS_INFO
  if (WIFEXITED(pst))   printf("normal\n");
  if (WIFSIGNALED(pst)) printf("abnormal\n");
  if (WIFSTOPPED(pst))  printf("stop\n");
#endif
  
  GS_ProcessStatus = GS_STAT_DIED;
  GS_ProcessID     = -1;
  if (GS_Pipe_To_GS >= 0)
    close(GS_Pipe_To_GS);
  GS_Pipe_To_GS = -1;
  if (GS_Pipe_From_GS >= 0)
    close(GS_Pipe_From_GS);
  GS_Pipe_From_GS = -1;
}

Private void
gs_kill(void)
{
  gs_invocation_nerr = 0;

  if ((GS_ProcessStatus == GS_STAT_RUNNING)
      && (GS_ProcessID != -1)){
    /* Just send EOF */
    if (GS_Pipe_To_GS >= 0)
      close(GS_Pipe_To_GS);
    GS_Pipe_To_GS = -1;
    if (GS_Pipe_From_GS >= 0)
      close(GS_Pipe_From_GS);
    GS_Pipe_From_GS = -1;

#if 1
    GS_ProcessStatus = GS_STAT_DIED;
    GS_ProcessID     = -1;
#else
    kill(SIGKILL, GS_ProcessID);
    while (GS_ProcessStatus != GS_STAT_DIED){
# ifdef HAVE_USLEEP
      usleep(100000);
# else
#  ifdef HAVE_SLEEP
      sleep(1);      
#  endif
# endif
    }
#endif
  }
}


#else /*!HAVE_PIPE*/


Private char GS_OutputFileName[MAXPATHLEN];
Private int  GS_OutputFileFD;

Private int
gs_convert_to_bitmap(DVI dvi, DVI_DEVICE dev, 
		     char *psfile, 
		     int bm_format, long bm_width, long bm_height)
{
  int    pst;
  char  *gs_dev;
  char   OptFormat[128], OptBitmapSize[256], OptOutput[MAXPATHLEN+16];

  GS_OutputFileFD = -1;

  sprintf(GS_OutputFileName, "%s/dvi-XXXXXX.bm",
	  DEV_CALL(dev,temp_dir)(dev,dvi));
  tmpnam(GS_OutputFileName);
  if (strcmp(GS_OutputFileName, "") == 0)
    return -1;

  switch (bm_format){
  default:
  case EPS_OUTPUT_FORMAT_BITMAP:
    gs_dev = "pbmraw";
    break;
  case EPS_OUTPUT_FORMAT_PIXMAP_RGB: 
    gs_dev = "ppmraw";
    break;
  case EPS_OUTPUT_FORMAT_GRAYMAP: 
    gs_dev = "pgmraw";
    break;
  }

  sprintf(OptFormat,     "-sDEVICE=%s", gs_dev);
  sprintf(OptBitmapSize, "-g%ldx%ld", bm_width, bm_height);
  sprintf(OptOutput,     "-sOutputFile=%s", GS_OutputFileName);

#ifdef HAVE_FORK
  {
    int    pid, i;
    char  *gs_argv[16];

    i = 0;
    gs_argv[i++] = DEV_CALL(dev,gs_program_path)(dev, dvi);
    gs_argv[i++] = "-dQUIET";
    gs_argv[i++] = "-dNOPAUSE";
    gs_argv[i++] = OptFormat;
    gs_argv[i++] = OptBitmapSize;
    gs_argv[i++] = OptOutput; 
    gs_argv[i++] = psfile;
    gs_argv[i++] = "quit.ps";
    gs_argv[i++] = NULL;
    
    if (getenv("DVILIB_DEBUG_EPS") != NULL){
      int  j;
      printf("Invoke Ghostscript: ");
      for (j = 0; gs_argv[j] != NULL; j++)
	printf("%s ", gs_argv[j]);
      printf("\n");
    }

    pid = fork();
    switch (pid){
    case 0:
      execvp(DEV_CALL(dev,gs_program_path)(dev, dvi), gs_argv);
      exit(1);			/* exit if failed to exec gs */
      break;
    case -1:
      pst = -1;
      break;
    default:
      DEV_CALL(dev,message_advice)(dev, dvi, "Drawing a PostScript figure...");
      for (;;){
	int  p;
#if HAVE_WAITPID
	do {
	  p = waitpid(pid, &pst, 0);
	} while ((p == -1) && (errno == EINTR));
#else
	/* XXX wrong code, if case other process dies */
	do {
	  p = wait(&pst);
	} while (p != pid);
#endif
      }
    }
  }
#else
  {
    char  gs_cmd[3*MAXPATHLEN];

    strcpy(gs_cmd, "");
    strcat(gs_cmd, DEV_CALL(dev,gs_program_path)(dev, dvi));
    strcat(gs_cmd, " -dQUIET");
    strcat(gs_cmd, " -dNOPAUSE");
    strcat(gs_cmd, " ");
    strcat(gs_cmd, OptFormat);
    strcat(gs_cmd, " ");
    strcat(gs_cmd, OptBitmapSize);
    strcat(gs_cmd, " ");
    strcat(gs_cmd, OptOutput); 
    strcat(gs_cmd, " ");
    strcat(gs_cmd, psfile);
    strcat(gs_cmd, " ");
    strcat(gs_cmd, "quit.ps");

    if (getenv("DVILIB_DEBUG_EPS") != NULL){
      int  j;
      printf("Invoke Ghostscript: ");
      for (j = 0; gs_argv[j] != NULL; j++)
	printf("%s ", gs_argv[j]);
      printf("\n");
    }

    DEV_CALL(dev,message_advice)(dev, dvi, "Drawing a PostScript figure...");
    pst = system(gs_cmd);
  }
#endif /*HAVE_FORK*/

  if (pst == 0){
    DEV_CALL(dev,message_advice)(dev, dvi, "done.");
  } else {
    DEV_CALL(dev,message_error)(dev, dvi, 
				"Failed to draw a PostScript figure.");
    gs_end_read(dvi, dev);
    return -1;
  }

  GS_OutputFileFD = open(GS_OutputFileName, O_RDONLY);
  return GS_OutputFileFD;
}

Private int
gs_end_read(DVI dvi, DVI_DEVICE dev)
{
  if (GS_OutputFileFD >= 0)
    close(GS_OutputFileFD);
  if (strcmp(GS_OutputFileName, "") != 0)
    unlink(GS_OutputFileName);
  return 0;
}

Private void
gs_kill(void)
{
  gs_invocation_nerr = 0;
}

Public int
dvi_special_invoke_gs(DVI dvi, DVI_DEVICE dev, int bm_format)
{
  return 0;
}

#endif /*HAVE_PIPE*/



Private char  gs_char_buff      = '\0';
Private int   gs_char_buff_full = 0;

Private int
gs_read_char(DVI dvi, DVI_DEVICE dev, int fd, unsigned char *buff)
{
  if (gs_char_buff_full == 1){
    gs_char_buff_full = 0;
    *buff = gs_char_buff;
    return 1;
  }

  gs_char_buff_full = 0;
  if (gs_read(dvi, dev, fd, buff, 1) <= 0)
    return -1;
  return 1;
}

Private void
gs_unread_char(DVI dvi, DVI_DEVICE dev, int fd, unsigned char *buff)
{
  gs_char_buff_full = 1;
  gs_char_buff = *buff;
}

Private int
gs_read(DVI dvi, DVI_DEVICE dev, int fd, unsigned char *buff, int n)
{
  int  i, rest;

  rest = n;

  if (rest <= 0)
    return 0;

  if (gs_char_buff_full == 1){
    gs_char_buff_full = 0;
    *buff = gs_char_buff;
    buff = &buff[1];
    if ((--rest) <= 0)
      return n;
  }

  while (rest > 0){
#ifdef HAVE_SELECT
    fd_set         fdset;
    struct timeval tv;
    int            val;

    for (;;){
      if (GS_ProcessStatus == GS_STAT_DIED)
	return -1;
      FD_ZERO(&fdset);
      FD_SET(fd, &fdset);
      tv.tv_sec  = DEV_CALL(dev,gs_timeout_value)(dev, dvi);
      tv.tv_usec = 0;
      if (tv.tv_sec <= 0)
	tv.tv_sec = DEFAULT_GS_TIMEOUT;
      val = select(fd+1, &fdset, (fd_set*)0, (fd_set*)0, &tv);
      if (errno == EINTR)
	continue;
      if (val < 0){
	return -1;
      } if (val > 0){
	if (FD_ISSET(fd, &fdset) != 0){
	  i = read(fd, buff, rest);
          break;
	}
      } else {
	if (DEV_METHOD_DEF(dev,gs_timeout_give_up)
	    && (DEV_CALL(dev,gs_timeout_give_up)(dev, dvi) == 0)){
	  continue;
	} else {
	  gs_kill();
	  return -1;
	}
      }
    }
#else /*!HAVE_SELECT*/
    i = read(fd, buff, rest);
#endif /*HAVE_SELECT*/

    if (i < 0)
      return -1;
    if (i == 0)
      return n - rest;

    rest -= i;
    buff = &buff[i];
  }

  return n;
}



#if 0
Private int
dump(buff, w, h)
     unsigned char *buff;
     int            w, h;
{
  int  x, y, rast;

  rast = (w+7)/8;
  for (y = 0; y < h; y++){
    for (x = 0; x < rast; x++)
      printf("%02x", buff[rast*y+x]);
    printf("\n");
  }
  return 0;
} 
#endif


/*EOF*/
