/* -*- mode: c; c-file-style: "gnu" -*-
 * worker.c -- Worker support code
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file worker.c
 * This file is the interface towards the "Worker" process. The
 * Worker's aim is to map files into memory, while Thy herself is
 * doing other stuff, like serving requests that are already in core,
 * or processing new ones. This way, the time Thy spends accessing the
 * disc is kept quite low, and should increase the overall performance
 * a little. Especially on NFS.
 */

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#include "compat/compat.h"

#include "config.h"
#include "fabs.h"
#include "misc.h"
#include "thy.h"
#include "types.h"
#include "worker.h"

/** @internal Socket connected to the worker process.
 */
static int _thy_worker_socks[2] = {-1, -1};
/** @internal PID of the worker process.
 */
static pid_t _thy_worker_child = 0;

/** @internal Shake hands with the Worker.
 * This function does the handshaking with the Worker. For now,
 * this consists only of telling it our protocol version, and seeing
 * if it supports it.
 *
 * @returns Zero on success, -1 otherwise.
 */
static int
_thy_worker_handshake (void)
{
  char c[2];
  char *msg = NULL;
  int len;

  len = asprintf (&msg, "VERSION\n1\n%d\n", _THY_WORKER_PROTOCOL_VERSION);

  if (write (_thy_worker_socks[0], msg, len) != len)
    {
      bhc_error ("Error communicating with the Worker: %s",
		 strerror (errno));
      free (msg);
      return -1;
    }
  free (msg);
  if (read (_thy_worker_socks[1], c, 2) != 2)
    {
      bhc_error ("Error communicating with the Worker: %s",
		 strerror (errno));
      return -1;
    }

  return (c[0] == '1' && c[1] == '\n') ? 0 : -1;
}

/** Initialise the Worker engine.
 * This one spawns off the Worker process, and sets up the
 * communication pipes.
 *
 * @returns Zero on success, -1 upon failiure.
 */
int
thy_worker_init (void)
{
  int socks[2][2], i, j, e = 0;
  const config_t *config = config_get ();

  _thy_worker_socks[0] = -1;
  _thy_worker_socks[1] = -1;

  if (config->options.worker != THY_BOOL_TRUE)
    return 0;

  if (pipe (socks[0]) == -1)
    {
      bhc_error ("pipe() failed: %s", strerror (errno));
      return -1;
    }
  if (pipe (socks[1]) == -1)
    {
      bhc_error ("pipe() failed: %s", strerror (errno));
      return -1;
    }

  for (i = 0; i <= 1; i++)
    for (j = 0; j <= 1; j++)
      {
	if (fcntl (socks[i][j], F_SETFD, FD_CLOEXEC) == -1)
	  {
	    bhc_error ("fcntl: %s", strerror (errno));
	    close (socks[i][j]);
	    e = 1;
	  }
      }
  if (e)
    return -1;

  _thy_worker_child = fork ();
  switch (_thy_worker_child)
    {
    case 0:
      /* Child */
      closelog ();
      openlog ("thy/worker", LOG_PID, LOG_DAEMON);
      thy_priv_drop (config->worker.uid);
      dup2 (socks[0][0], STDIN_FILENO);
      dup2 (socks[1][1], STDOUT_FILENO);
      execv (config->worker.path, config->worker.args.argv);
      bhc_exit (1);
      break;
    case -1:
      /* Error */
      bhc_error ("fork(): %s", strerror (errno));
      return -1;
    default:
      /* Parent */
      bhc_log ("Worker (%s) started as PID %d", config->worker.path,
	       _thy_worker_child);
      close (socks[0][0]);
      close (socks[1][1]);
      _thy_worker_socks[0] = socks[0][1];
      _thy_worker_socks[1] = socks[1][0];

      if (_thy_worker_handshake () == -1)
	{
	  bhc_error ("%s", "Worker handshake failed.");
	  kill (_thy_worker_child, SIGTERM);
	  _thy_worker_socks[0] = -1;
	  _thy_worker_socks[1] = -1;
	}
      return 0;
    }
  return -1;
}

/** Terminate the Worker process.
 * Sends a QUIT command to the Worker. We cannot simply kill it,
 * as it might run with higher priveleges then Thy herself.
 */
void
thy_worker_quit (void)
{
  write (_thy_worker_socks[0], "QUIT\n0\n", sizeof ("QUIT\n0\n"));
}

/** Register a file with the Worker.
 * Sends an index and a filename to the Worker - if using one -, and
 * returns the result.
 *
 * @param idx is the index of the entry. It will passwed to
 * thy_worker_check() too.
 * @param fn is the filename we're requesting work for.
 *
 * @returns Zero on success, -1 upon error, or if the Worker engine is
 * disabled.
 */
int
thy_worker_register (int idx, const char *fn)
{
  char *msg;
  ssize_t msglen;
  char res[3];

  /* If the pipe is disabled, we return a failiure. */
  if (_thy_worker_socks[0] == -1)
    return -1;

  if (idx < 0 || !fn)
    return -1;

  asprintf (&msg, "REGISTER\n2\n%d\n%s\n", idx, fn);
  msglen = strlen (msg);
  if (write (_thy_worker_socks[0], msg, msglen) != msglen)
    {
      bhc_error ("Error communicating with the Worker: %s",
		 strerror (errno));
      free (msg);
      return -1;
    }
  free (msg);

  if (read (_thy_worker_socks[1], res, 2) != 2)
    {
      bhc_error ("Error communicating with the Worker: %s",
		 strerror (errno));
      return -1;
    }

  if (res[0] == '1')
    {
      fabs_mmap (idx);
      return 0;
    }
  return -1;
}

/** Unregister a file descriptor.
 * Request the unregistering of a file descriptor.
 *
 * @param idx is the index of the entry to unregister.
 *
 * @returns Zero on success, -1 on error or when the Worker engine is
 * disabled.
 */
int
thy_worker_unregister (int idx)
{
  char *msg;
  ssize_t msglen;
  char res[3];

  /* If the pipe is disabled, we return a failure. */
  if (_thy_worker_socks[0] == -1)
    return -1;

  if (idx < 0)
    return -1;

  fabs_munmap (idx);

  asprintf (&msg, "UNREGISTER\n1\n%d\n", idx);
  msglen = strlen (msg);
  if (write (_thy_worker_socks[0], msg, msglen) != msglen)
    {
      bhc_error ("Error communicating with the Worker: %s",
		 strerror (errno));
      free (msg);
      return -1;
    }
  free (msg);

  if (read (_thy_worker_socks[1], res, 2) != 2)
    {
      bhc_error ("Error communicating with the Worker: %s",
		 strerror (errno));
      return -1;
    }

  if (res[0] == '1')
    return 0;
  return -1;
}

/** Check if a given range is available for serving.
 * Asks the Worker if a given range of a file is ready for serving.
 *
 * @param idx is the entry index of the file we're asking about.
 * @param spos is the starting position.
 * @param epos is the ending position.
 *
 * @returns The available bytes (starting at spos) ready to be served,
 * or -1 on error or when the Worker engine is disabled.
 */
ssize_t
thy_worker_check (int idx, off_t spos, off_t epos)
{
  char *msg;
  ssize_t msglen;
  char res[12];
  ssize_t ires;

  /* If the pipe is disabled, we return a failure. */
  if (_thy_worker_socks[0] == -1)
    return -1;

  if (idx < 0 || spos < 0 || (epos < spos && epos != -1))
    return -1;

  asprintf (&msg, "CHECK\n3\n%d\n%ld\n%ld\n", idx, (long)spos, (long)epos);
  msglen = strlen (msg);
  if (write (_thy_worker_socks[0], msg, msglen) != msglen)
    {
      bhc_error ("Error communicating with the Worker: %s",
		 strerror (errno));
      free (msg);
      return -1;
    }
  free (msg);

  memset (res, 0, sizeof (res));
  if (read (_thy_worker_socks[1], res, 11) != 11)
    {
      bhc_error ("Error communicating with the Worker: %s",
		 strerror (errno));
      return -1;
    }

  ires = atol (res);
  return ires;
}
