#include "caimagehueselector.h"
#include "caimagemanipulation.h"
#include "caresources.h"
#include "catrophy.h"


/** Constructor for an empty image hue selector.
    \param direction Vertical selector? (Use Up/Down arrows to control)
*/
CAImageHueSelector::CAImageHueSelector( Direction direction )
        : CAImageView() {
    if( CA_APP->debug ) std::cout << "CAImageHueSelector() begin" << std::endl;

    this->direction = direction;
    hueImage = 0;
    hue = 0;

    if( CA_APP->debug ) std::cout << "CAImageHueSelector() end" << std::endl;
}



/** Constructor for an image hue selector with an image.
    \param upperText Text above image
    \param lowerText Text under image
    \param image The image itself
    \param type Selector type (Image, Selector, HueSelector)
    \param direction Vertical selector? (Use Up/Down arrows to control)
*/
CAImageHueSelector::CAImageHueSelector( const char* upperText,
                                        const char* lowerText,
                                        CL_Surface* image,
                                        bool autoResize,
                                        Direction direction )
        : CAImageView( upperText,
                       lowerText,
                       image,
               autoResize ) {
    if( CA_APP->debug ) std::cout << "CAImageHueSelector() begin 2" << std::endl;

    this->direction = direction;
    hueImage = CAImageManipulation::changeHSV( image, 0,0,0 );
    hue = 0;
    setImageSize( image->get_width(), image->get_height() );

    if( CA_APP->debug ) std::cout << "CAImageHueSelector() end 2" << std::endl;
}



/** Destructor.
*/
CAImageHueSelector::~CAImageHueSelector() {
    if( hueImage ) {
        delete hueImage;
        hueImage = 0;
    }
}



/** Sets the size of the image view (without top/bottom button).
    Bigger images get cropped (top left part shown).
    \param w Width of image view
    \param h Height of image view
*/
void
CAImageHueSelector::setImageSize( int w, int h ) {
    CAImageView::setImageSize( w,h );

    if( direction==Horizontal ) {
        width += 2*CA_RES->gui_arrow_l->get_width();
    } else {
        //height += 2*CA_RES->gui_arrow_t->get_height();
    }


    right = left + width;
    bottom = top + height;
}



/** Changes the images hue (only for hue selectors with one image).
    The old image gets deleted.
*/
void
CAImageHueSelector::changeImageHue( bool forward ) {
    if( image.image ) {
        int amount = 5 * (forward ? 1 : -1);
        hue += amount;
        if( hue>360 ) hue -= 360;
        if( hue<0   ) hue += 360;

        if( hueImage ) delete hueImage;
        hueImage = CAImageManipulation::changeHSV( image.image, hue,0,0 );
    }
}



/** Displays the image view with its current image/ texts.
*/
void
CAImageHueSelector::display( bool active ) {
    //CAImageView::display( active );

    CL_ClipRect crAll( 0,0, CA_APP->width, CA_APP->height );
    CL_ClipRect crImage( left,top, right, bottom );
    CL_Display::set_clip_rect( crImage );

    CL_Display::fill_rect( left,top, right,bottom, 0.0,0.0,0.0, 0.25 );

    displayTextButtons( active );

    if( hueImage ) hueImage->put_screen( left + (width-hueImage->get_width())/2,
                                             top+barHeight );

    displayArrows( active );

    CL_Display::set_clip_rect( crAll );
}



/** Handles key events for this view / selector.
*/
void
CAImageHueSelector::handleKey( const CL_Key &key ) {
    /*
    ##
    ## Doesn't work under Windoze - so let's do it different...
    ##
    // Hue selectors allow the user to change the images hue:
    //
    // Increase:
    //
    if( (key.id==CL_KEY_LEFT && direction==Horizontal) ||
        (key.id==CL_KEY_DOWN && direction==Vertical)      ) {
      changeImageHue( false );
}

    // Decrease:
    //
    if( (key.id==CL_KEY_RIGHT && direction==Horizontal) ||
        (key.id==CL_KEY_UP    && direction==Vertical)      ) {
      changeImageHue( true );
}
    */

    if( (CL_Keyboard::get_keycode(CL_KEY_LEFT) && direction==Horizontal) ||
            (CL_Keyboard::get_keycode(CL_KEY_DOWN) && direction==Vertical) ) {
        changeImageHue( false );
    }

    if( (CL_Keyboard::get_keycode(CL_KEY_RIGHT) && direction==Horizontal) ||
            (CL_Keyboard::get_keycode(CL_KEY_UP)    && direction==Vertical) ) {
        changeImageHue( true );
    }
}


// EOF
