/*
 * Copyright (C) 2003  Stefan Kleine Stegemann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "AppController.h"
#include "DocumentWindow.h"
#include "DocumentWindowController.h"
#include "ViewPDFDocument.h"
#include "PrefsController.h"
#include "ViewPDF.h"

#include <Foundation/NSString.h>
#include <Foundation/NSException.h>
#include <Foundation/NSDictionary.h>
#include <Foundation/NSEnumerator.h>
#include <Foundation/NSArray.h>
#include <Foundation/NSBundle.h>
#include <Foundation/NSUserDefaults.h>
#include <Foundation/NSValue.h>
#include <Foundation/NSNotification.h>
#include <AppKit/NSApplication.h>
#include <AppKit/NSDocumentController.h>
#include <AppKit/NSMenu.h>
#include <AppKit/NSNibLoading.h>
#include <AppKit/NSFontManager.h>


/* shared instance of AppController */
AppController*  sharedInstance;

/* used to store the OutlinePanel's frame in the user's preferences.  */
static NSString* OutlinePanelFrameName = @"OutlinePanelFrame";

/*
 * Non-Public methods.
 */
@interface AppController(Private)
+ (void) _initDefaults;
- (void) _windowWillCloseNotification: (NSNotification*)aNotification;
- (void) _outlineItemSelected: (NSNotification*)aNotification;
@end

/*
 * The Main Controller of the ViewPDF Application.
 */
@implementation AppController

-(id) init
{
  if (sharedInstance) 
  {
     [self dealloc];
     return sharedInstance;
  }

  if ((self = [super init]))
  {
      prefsController = nil;
      mainWindow      = nil;
      
      [[NSNotificationCenter defaultCenter]
         addObserver: self
            selector: @selector(_windowWillCloseNotification:)
                name: NSWindowWillCloseNotification
              object: nil];

      [[NSNotificationCenter defaultCenter]
         addObserver: self
            selector: @selector(_outlineItemSelected:)
                name: N_OutlineItemSelected
              object: nil];
  }

  return self;
}


-(void) dealloc
{
   NSLog(@"dealloc AppController, retain count is %d", [self retainCount]);

   [prefsController release];
   [[NSNotificationCenter defaultCenter] removeObserver: self];

   [super dealloc];
}


/*
 * Class initialization.
 */
+ (void) initialize
{
   [super initialize];
   
   sharedInstance = nil;

   [self _initDefaults];
}


/*
 * Return a shared instance of the Applications controller.
 * Note that if this instance has been obtained once, someone
 * has to release this instance.
 */
+ (AppController*) sharedController
{
   if (sharedInstance == nil)
   {
      sharedInstance = [[AppController alloc] init];
   }

   return sharedInstance;
}


- (OutlinePanel*) sharedOutlinePanel
{
   if (!outlinePanel)
   {
      [NSBundle loadNibNamed: @"outline.gorm" owner: self];

      // store/restore the outline panel's frame from the
      // user's defaults
      [outlinePanel setFrameAutosaveName: OutlinePanelFrameName];
      [outlinePanel setFrameUsingName: OutlinePanelFrameName];
   }
   
   NSAssert(outlinePanel, @"no outline panel");
   return outlinePanel;
}


- (void) applicationWillFinishLaunching: (NSNotification*)aNotification 
{
   return;
}


- (void) applicationDidFinishLaunching: (NSNotification*)aNotification 
{
   return;
}


/*
 * Create and display a new Image Window together with a controller.
 * The created controller is returned. Both, window and controller
 * are autoreleased. The controller is added the list of window
 * controllers for the specified document.
 */
- (DocumentWindowController*) createAndDisplayWindowForDocument: (ViewPDFDocument*)aDocument
{
   DocumentWindow*           win;
   DocumentWindowController* theController;

   win = [[DocumentWindow alloc] init];
   // win will be released by it's controller
   [[win documentView] displayMessage: @"Loading Document ..."];

   theController = [[[DocumentWindowController alloc] initWithWindow: win] autorelease];
   [aDocument addWindowController: theController];
   [win setDelegate: theController];

   // We have to do this ourself, as there is currently no nib file
   [aDocument windowControllerDidLoadNib: theController];

   // FIXME if possible, don't know whether this is a bug or not.
   // It seems that the isReleasedWhenClosed flag is set to NO
   // when the controller has no document on init. The flag is
   // not toggled when a document is associated to the controller.
   [win setReleasedWhenClosed: YES];
   
   /*
   NSLog(@"CHECKPOINT controllers document %@", [[theController document] displayName]);
   NSLog(@"CHECKPOINT isReleasedWhenClosed %d", [win isReleasedWhenClosed]);
   */
   
   [win center];
   [win orderFrontRegardless];
   [win makeMainWindow];
   [win display];

   //[win makeKeyAndOrderFront: self];

   return theController;
}


/*
 * Display the applications preferences dialog.
 */
- (void) showPrefs: (id)sender
{
   if (!prefsController)
   {
      [NSBundle loadNibNamed: @"preferences.gorm" owner: self];
   }

   [[prefsController window] orderFrontRegardless];
   [[prefsController window] makeKeyWindow];
   [[prefsController window] display];
}


/*
 * Callback for PrefsController to set it's instance
 * when loaded from nib.
 */
- (void) setPrefsController: (id)controller
{
   prefsController = [controller retain];
}


/*
 * Display an info dialog for the ViewPDF application.
 */
- (void)showInfoPanel:(id)sender
{
  [[NSApplication sharedApplication] orderFrontStandardInfoPanel:sender];
}

/*
 * Display the shared outline panel.
 */ 
- (void) showOutlinePanel: (id)sender
{
   [[self sharedOutlinePanel] setTitle: @"Document Outline"];
   [[self sharedOutlinePanel] orderFrontRegardless];
   [[self sharedOutlinePanel] makeKeyWindow];
   [[self sharedOutlinePanel] display];   
}


/*
 * Set the current main document window.
 */
- (void) setMainWindow: (NSWindow*)aWindow
{
   mainWindow = aWindow;

   NSAssert((aWindow == nil) || ([aWindow isKindOfClass: [DocumentWindow class]]),
            @"main window is no document window");

   if (mainWindow)
   {
      // update the outline
      id document = [[mainWindow windowController] document];
      [[self sharedOutlinePanel] showOutlineForDocument: [document pdfDocument]
                                              usingName: [document fileName]];
   }
}


/*
 * Get the current main document window or nil if
 * the user has not opened a document.
 */
- (NSWindow*) mainWindow
{
   return mainWindow;
}

@end


/* ----------------------------------------------------- */
/*  Category Private                                     */
/* ----------------------------------------------------- */

@implementation AppController (Private)

/*
 * Register the standard User-Defaults.
 */
+ (void) _initDefaults
{
   NSMutableDictionary* stdDefs = [NSMutableDictionary dictionary];
   NSUserDefaults*      defaults = [NSUserDefaults standardUserDefaults];

   defaults = [NSUserDefaults standardUserDefaults];
   stdDefs = [NSMutableDictionary dictionary];
   
   [stdDefs setObject: @"YES" forKey: PrefsToolbarVisible];

   [defaults registerDefaults: stdDefs];
}


/*
 * If the window that will close is the current main window,
 * reset the main window.
 */
- (void) _windowWillCloseNotification: (NSNotification*)aNotification
{
   id document;

   // we are only interested in document windows
   if (![[aNotification object] isKindOfClass: [DocumentWindow class]])
   {
      return;
   }

   document = [[[aNotification object] windowController] document];
   [[self sharedOutlinePanel] outlineWillClose: [document fileName]];

   if ([aNotification object] == [self mainWindow])
   {
      [self setMainWindow: nil];
   }
}


/*
 * Forward the item selection the main window's controller.
 */
- (void) _outlineItemSelected: (NSNotification*)aNotification
{
   id theItem;

   if (![self mainWindow])
   {
      return;
   }

   theItem = [[aNotification userInfo] objectForKey: UserInfoKeyOutlineItem];
   [[[self mainWindow] windowController] outlineItemSelected: theItem];
}

@end
