/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2003 Ron Steinke <rsteinke@w-link.net>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "box.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"

wftk::Box::Box(Orientation orient) : orientation_(orient)
{
  setPackingInfo();
}

void wftk::Box::setOrientation(Orientation o)
{
  orientation_ = o;
  if(!children_.empty())
    packingUpdate();
}

void wftk::Box::setPackingInfo()
{
  // reset to defaults
  ScreenArea::setPackingInfo();

  packing_info_.x.expand = packing_info_.y.expand = false;

  weights_ = PackingInfo::Weights();

  for(ChildList::iterator I = children_.begin(); I != children_.end(); ++I) {
    const PackingInfo &child_info = (*I)->getPackingInfo();

    switch(orientation_) {
      case LEFT_TO_RIGHT:
      case RIGHT_TO_LEFT:
        packing_info_.x.extend(child_info.x);
        packing_info_.y.contain(child_info.y);
        weights_.extend(child_info.x);
        break;
      case TOP_TO_BOTTOM:
      case BOTTOM_TO_TOP:
        packing_info_.x.contain(child_info.x);
        packing_info_.y.extend(child_info.y);
        weights_.extend(child_info.y);
        break;
      default:
        assert(false);
        return;
    }
  }
}

void wftk::Box::handleResize(Uint16 w, Uint16 h)
{
  // set shape and opacity based on new size
  ScreenArea::handleResize(w, h);

  const PackingInfo::Expander *box_main, *box_perp;
  unsigned perp_space;

  // figure out how to divide any extra space

  switch(orientation_) {
    case LEFT_TO_RIGHT:
    case RIGHT_TO_LEFT:
      box_main = &packing_info_.x;
      box_perp = &packing_info_.y;
      weights_.setExpand(packing_info_.x.pref, w);
      perp_space = h;
      break;
    case TOP_TO_BOTTOM:
    case BOTTOM_TO_TOP:
      box_main = &packing_info_.y;
      box_perp = &packing_info_.x;
      weights_.setExpand(packing_info_.y.pref, h);
      perp_space = w;
      break;
    default:
      assert(false);
      return;
  }

  // keep track of how much space we've filled as a double,
  // to handle roundoff error in the overage
  double space_filled = 0;
  unsigned pixels_filled = 0;

  for(ChildList::iterator I = children_.begin(); I != children_.end(); ++I) {
    const PackingInfo& child_info = (*I)->getPackingInfo();
    const PackingInfo::Expander *child_main, *child_perp;

    switch(orientation_) {
      case LEFT_TO_RIGHT:
      case RIGHT_TO_LEFT:
        child_main = &child_info.x;
        child_perp = &child_info.y;
        break;
      case TOP_TO_BOTTOM:
      case BOTTOM_TO_TOP:
        child_main = &child_info.y;
        child_perp = &child_info.x;
        break;
      default:
        assert(false);
        return;
    }

    unsigned perp_size;
    if(perp_space < child_perp->pref)
      perp_size = (child_perp->min < perp_space) ? perp_space : child_perp->min;
    else
      perp_size = child_perp->expand ? perp_space : child_perp->pref;

    unsigned perp_offset = (perp_space > perp_size) ? (perp_space - perp_size) / 2 : 0;

    space_filled += child_main->pref + weights_.padding(*child_main);

    unsigned calc_size = (unsigned) (space_filled + 0.5) - pixels_filled;

    switch(orientation_) {
      case LEFT_TO_RIGHT:
        (*I)->resize(Rect(pixels_filled, perp_offset, calc_size, perp_size));
        break;
      case RIGHT_TO_LEFT:
        (*I)->resize(Rect(width() - (pixels_filled + calc_size),
          perp_offset, calc_size, perp_size));
        break;
      case TOP_TO_BOTTOM:
        (*I)->resize(Rect(perp_offset, pixels_filled,
          perp_size, calc_size));
        break;
      case BOTTOM_TO_TOP:
        (*I)->resize(Rect(perp_offset, height()
          - (pixels_filled + calc_size), perp_size, calc_size));
        break;
      default:
        assert(false);
        return;
    }

    pixels_filled += calc_size;
  } 
}

void wftk::Box::pack(ScreenArea* sa, const iterator& iter)
{
  children_.insert(iter.iter_, sa);
  sa->setParent(this);
}

void
wftk::Box::erase(const iterator& iter)
{
  iter->setParent(0);
  children_.erase(iter.iter_);
  packingUpdate();
}

void
wftk::Box::clear()
{
  children_.clear();
  packingUpdate();
}

wftk::Box::iterator
wftk::Box::find(ScreenArea& sa)
{
  ChildList::iterator I = children_.begin();

  while(I != children_.end()) {
    if(*I == &sa)
      break;
    ++I;
  }

  return I;
}
