/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
// written by Karsten Laux, June 1999  

#ifndef _PIXELFORMAT_H_
#define _PIXELFORMAT_H_

#include <wftk/color.h>

//just for typedefs ...
#include <SDL/SDL.h>
#include <iosfwd>
#include <string>

namespace wftk {

/**Format of pixels in memory.
 * The format RGB0888 means that the display runs at 24bpp, but the
 * pixel values are 4-byte-aligned in memory for faster access on a 
 * 32bit system (Matrox cards do this).
 *
 * This class is essentially a const reference to an SDL_PixelFormat.
 * You can create one by passing it an SDL_Surface (in which case
 * it increments the refcount of the surface, to keep the surface's
 * pixelformat in memory until it's done with it), or you can
 * take a reference to certain formats defined by an enum.
 **/ 
class Pixelformat
{
 public:
  /**
   * A list of predefined formats
   **/
  enum Format {ABGR8888 /*!< . */, RGBA8888 /*!< . */, ARGB8888 /*!< . */,
	BGRA8888 /*!< . */, RGB888 /*!< . */, BGR888 /*!< . */,
	RGB0888 /*!< . */, BGR0888 /*!< . */, RGB565 /*!< . */,
	RGB555 /*!< . */, IND8 /*!< an 8-bit indexed format */, NUM_FORMATS};
  /// Create an invalid Pixelformat
  Pixelformat() : surface_(0), format_(0) {}
  /// Get a predefined Pixelformat
  Pixelformat(Format format);
  /// Copy a Pixelformat
  Pixelformat(const Pixelformat& other);
  /// Get a Pixelformat which acts as a reference to a surface's internal SDL_Pixelformat
  /**
   * Uses SDL_Surface's internal reference counting to keep the
   * SDL_Pixelformat reference valid, regardless of what is done to
   * the surface.
   **/
  Pixelformat(SDL_Surface* surface);
  ///
  ~Pixelformat();

  /// Get a reference to the internal SDL_PixelFormat
  operator const SDL_PixelFormat&() const;

  /// Copy another Pixelformat into this one
  Pixelformat& operator=(const Pixelformat& pf);
  /// Copy a predefined format into this one
  Pixelformat& operator=(Format);

  /// Get a string 'name' for the format
  std::string asString() const;
  /// Print the format's 'name' to the stream
  friend std::ostream& operator<<(std::ostream&, const Pixelformat&);
  /// Compare two Pixelformats
  bool operator==(const Pixelformat& pf) const
	{return pf.format_ ? operator==(*pf.format_) : !format_;}
  /// Compare a Pixelformat with an SDL_PixelFormat
  bool operator==(const SDL_PixelFormat&) const;
  /// Compare a Pixelformat with an SDL_PixelFormat
  friend bool operator==(const SDL_PixelFormat& f1, const Pixelformat& f2)
	{return f2 == f1;}
  /// Compare two Pixelformats
  bool operator!=(const Pixelformat& pf) const {return !operator==(pf);}
  /// Compare a Pixelformat with an SDL_PixelFormat
  bool operator!=(const SDL_PixelFormat& format) const {return !operator==(format);}
  /// Compare a Pixelformat with an SDL_PixelFormat
  friend bool operator!=(const SDL_PixelFormat& f1, const Pixelformat& f2)
	{return !(f2 == f1);}
  /// Map a Color to its pixel representation in this format
  Uint32 mapToPixel(const Color& col) const;
  /// Map a pixel in this format to a color
  Color mapToColor(Uint32 pixel) const;
  /// Returns true if this is a valid Pixelformat
  bool valid() const {return format_ != 0;}
  /// Get the palette, if this surface is indexed
  const SDL_Palette* getPalette() const {return format_ ? format_->palette : 0;}
  /// Get the color_num color in the palette
  Color getColor(unsigned color_num) const;

  /// Get bytes per pixel
  int bpp() const {return format_ ? format_->BytesPerPixel : 0;}
  /// Get bits per pixel
  int bitspp() const {return format_ ? format_->BitsPerPixel : 0;}
  /// Get the red mask
  Uint32 rMask() const {return format_ ? format_->Rmask : 0;}
  /// Get the green mask
  Uint32 gMask() const {return format_ ? format_->Gmask : 0;}
  /// Get the blue mask
  Uint32 bMask() const {return format_ ? format_->Bmask : 0;}
  /// Get the alpha mask
  Uint32 aMask() const {return format_ ? format_->Amask : 0;}
  
  /// The Pixelformat of the display (once it's been initialized)
  static Pixelformat displayFormat;

 private:

  ///
  SDL_Surface* surface_;
  ///
  const SDL_PixelFormat* format_;
};

}


#endif
