#include <stdio.h>

#include "constant.h"
  
/**** x include files ****/
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <X11/keysymdef.h>

#if USE_LONGJMP
#include <setjmp.h>
#endif

#include "extern.h"


/******************************************************************************
  dump_board (filename, use_brief)

  Dump all relevant information about the current state of the game to
  the file <filename>.  If <use_brief>, just dump minimal terrain and
  side info (as generated by edit, for example). 
******************************************************************************/

dump_board (filename, use_brief)
  char filename[];
  int use_brief;
{
  int i, j;

  FILE *fp,
       *fopen();

  cell_type cell;

  if ((fp = fopen (filename, "w")) == NULL)
    throw_error ("Unable to open map file: %s", filename);

  /** Save the crucial parameters which determine compatibility **/

  fwrite (&Config->board_x_size,	sizeof(short), 1, fp);
  fwrite (&Config->board_y_size,	sizeof(short), 1, fp);
  fwrite (&Config->level_min,		sizeof(int), 1, fp);
  fwrite (&Config->level_max,		sizeof(int), 1, fp);
  fwrite (&Config->tile_type,		sizeof(int), 1, fp);
  fwrite (&Config->value_int[OPTION_CELL][0], sizeof(int), 1, fp);
  fwrite (&Config->side_count,		sizeof(short), 1, fp);
  if (use_brief)
    fprintf (fp, "B");
  else
    fprintf (fp, "F");

  /** Save each cell **/

  for (i=0; i<Board->cell_count; i++)
    dump_cell (CELL(i), Config->side_count,
			Config->direction_count, fp, use_brief);

  fclose (fp);
}



/******************************************************************************
  dump_cell (cell, side_count, direction_count, fp, use_brief)

  Dump all relevant information about <cell> to the dump file pointed to by
  <fp>, making sure to save only relevant information.  If <use_brief>,
  just dump essential terrain and side information.
******************************************************************************/

dump_cell (cell, side_count, direction_count, fp, use_brief)
  cell_type *cell;
  int side_count,
      direction_count;
  FILE *fp;
{
  int i,
      fixed_size;

  s_char dummy;

  if (use_brief)
  {
    fwrite (&cell->level,			sizeof(s_char), 1, fp);

    if (cell->side == SIDE_NONE && cell->angle == 0)
    {
      dummy = SIDE_VOID;
      fwrite (&dummy,				sizeof(s_char), 1, fp);
    }
    else
    {
      fwrite (&cell->side,			sizeof(s_char), 1, fp);
      if (cell->side != SIDE_NONE)
        fwrite (&(cell->value[cell->side]),	sizeof(s_char), 1, fp);

      fwrite (&cell->angle,			sizeof(short), 1, fp);
      if (cell->angle == ANGLE_FULL)
        fwrite (&cell->growth,			sizeof(s_char), 1, fp);
      else if (cell->angle > 0)
        fwrite (&cell->old_growth,		sizeof(s_char), 1, fp);
    }
  }
  else
  {
    /** Save the first part of the cell_type structure, which consists	**/
    /** of statically defined variables.				**/

    fixed_size = &(cell->old_value) - &(cell->level) + sizeof(char);
    fwrite (cell, fixed_size, 1, fp);

    /** Save the second part of the cell_type structure, which contains	**/
    /** dynamically allocated arrays.					**/

    fwrite (cell->dir,		sizeof(s_char), direction_count, fp);
    fwrite (cell->march,	sizeof(s_char), side_count, fp);
    fwrite (cell->march_dir,	sizeof(s_char), side_count, fp);
    fwrite (cell->march_type,	sizeof(s_char), side_count, fp);
    fwrite (cell->value,	sizeof(s_char), side_count, fp);
    fwrite (cell->seen,		sizeof(s_char), side_count, fp);
    fwrite (cell->draw_level,	sizeof(s_char), side_count, fp);
  }
}



/******************************************************************************
  load_board_header ()

  Load all relevant information about the current state of the game from
  the pointer <Config->fp>.
******************************************************************************/

load_board_header ()
{
  int i, j,
      side, player,
      cell_size;
  short side_count;
  char dummy;

  /** Load the set of header information **/

  fread (&Config->board_x_size,	sizeof(short), 1, Config->fp);
  fread (&Config->board_y_size,	sizeof(short), 1, Config->fp);
  fread (&Config->level_min,	sizeof(int), 1, Config->fp);
  fread (&Config->level_max,	sizeof(int), 1, Config->fp);
  fread (&Config->tile_type,	sizeof(int), 1, Config->fp);
  fread (&cell_size,		sizeof(int), 1, Config->fp);

  fread (&Config->load_side_count, sizeof(short), 1, Config->fp);

  fread (&dummy, sizeof(char), 1, Config->fp);
  if (dummy == 'B')
    Config->use_brief_load = TRUE;
  else
    Config->use_brief_load = FALSE;

  /** Correct any conflicting options **/

  if (Config->level_min < 0)
  {
    /** Set sea values if not set in command line **/

    if (!Config->enable_all[OPTION_SEA])
    {
      for (side=0; side<Config->side_count; side++)
        Config->value_double[OPTION_SEA][side] = 5;
    }

    Config->enable_all[OPTION_SEA] = TRUE;
    Config->enable_terrain = TRUE;
  }

  Config->value_int_all[OPTION_HILL_TONES] = Config->level_max+1;
  Config->value_int_all[OPTION_FOREST_TONES] = Config->level_max+1;
  Config->value_int_all[OPTION_SEA_TONES] = -Config->level_min;

  if (Config->level_max > 0)
  {
    /** Set default hill values if not set in command line **/

    if (!Config->enable_all[OPTION_HILLS])
    {
      for (side=0; side<Config->side_count; side++)
        Config->value_double[OPTION_HILLS][side] = 5.0*HILL_FACTOR;
    }

    if (!Config->enable_all[OPTION_HILLS] &&
                        !Config->enable_all[OPTION_FOREST])
      Config->enable_all[OPTION_HILLS] = TRUE;

    Config->enable_terrain = TRUE;
  }

  Config->fill_number = -Config->level_min;

  /** Set cell sizes for any side which didn't explicitly set it **/

  for (player=0; player<Config->player_count; player++)
  {
    side = Config->player_to_side[player];
    if (!Config->enable[OPTION_CELL][side])
    {
      Config->value_int[OPTION_CELL][side] = cell_size;
      Config->cell_size[side] = cell_size;
    }
  }  

  /** If side counts in file and game don't match, set OVERWRITE so	**/
  /** that troops are not loaded.					**/

  if (Config->side_count != Config->load_side_count)
    Config->enable_all[OPTION_OVERWRITE] = TRUE;
}



/******************************************************************************
  load_board (use_brief)

  Load each cell of the board from the already open file (from which the
  header has already been loaded).  If <use_brief>, load a succinct version
  of the board.
******************************************************************************/

load_board (use_brief)
  int use_brief;
{
  int i, j;

  cell_type *cell;

  /** Load each cell **/

  for (i=0; i<Board->cell_count; i++)
  {
    cell = CELL(i);
    load_cell (cell, Config->load_side_count,
		Config->direction_count, Config->fp, use_brief);

    j = 0;
  }

  fclose (Config->fp);
}



/******************************************************************************
  load_cell (cell, side_count, direction_count, fp, use_brief)

  Load all relevant information about <cell> from the dump file pointed to by
  <fp>, making sure to load only relevant information.  If <use_brief>, load
  a succinct version of the board.
******************************************************************************/

load_cell (cell, load_side_count, direction_count, fp, use_brief)
  cell_type *cell;
  int load_side_count,
      direction_count,
      use_brief;
  FILE *fp;
{
  int i,
      fixed_size,
      full_fixed_size;
  char buffer[1000];

  if (use_brief)
  {
    fread (&cell->level,			sizeof(s_char), 1, fp);
    fread (&cell->side,				sizeof(s_char), 1, fp);

    if (cell->side == SIDE_VOID)
      cell->side = SIDE_NONE;
    else
    {
      if (cell->side != SIDE_NONE)
        fread (&(cell->value[cell->side]),	sizeof(s_char), 1, fp);

      fread (&cell->angle,			sizeof(short), 1, fp);
      if (cell->angle == ANGLE_FULL)
      {
        fread (&cell->growth,			sizeof(s_char), 1, fp);
        cell->old_growth = cell->growth;
      }
      else if (cell->angle > 0)
      {
        fread (&cell->old_growth,		sizeof(s_char), 1, fp);
        cell->growth = 0;
      }
    }
  }
  else
  {
    /** Load the first part of the cell_type structure, which consists	**/
    /** of statically defined variables.				**/

    fixed_size = &(cell->side) - &(cell->level);
    fread (&(cell->level), fixed_size, 1, fp);

    fixed_size = &(cell->old_value) - &(cell->side) + sizeof(s_char);
    if (Config->enable_all[OPTION_OVERWRITE])
      fread (buffer, fixed_size, 1, fp);
    else
      fread (&(cell->side), fixed_size, 1, fp);

    /** Load the second part of the cell_type structure, which contains	**/
    /** dynamically allocated arrays.  Assume arrays already allocated.	**/

    if (Config->enable_all[OPTION_OVERWRITE])
    {
      fread (buffer, sizeof(s_char), direction_count, fp);
      fread (buffer, sizeof(s_char), load_side_count, fp);
      fread (buffer, sizeof(s_char), load_side_count, fp);
      fread (buffer, sizeof(s_char), load_side_count, fp);
      fread (buffer, sizeof(s_char), load_side_count, fp);
      fread (buffer, sizeof(s_char), load_side_count, fp);
      fread (buffer, sizeof(s_char), load_side_count, fp);
    }
    else
    {
      fread (cell->dir,		sizeof(s_char), direction_count, fp);
      fread (cell->march,	sizeof(s_char), load_side_count, fp);
      fread (cell->march_dir,	sizeof(s_char), load_side_count, fp);
      fread (cell->march_type,	sizeof(s_char), load_side_count, fp);
      fread (cell->value,	sizeof(s_char), load_side_count, fp);
      fread (cell->seen,	sizeof(s_char), load_side_count, fp);
      fread (cell->draw_level,	sizeof(s_char), load_side_count, fp);
    }
  }
}
