/* $Id: utils.c,v 1.4 1999/07/25 10:57:23 stano Exp $

   Utilities

   (C) 1999 Stanislav Meduna <stano@eunet.sk>
*/

#include <utils.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <malloc.h>
#include <limits.h>
#include <stdlib.h>

#include <libintl.h>

#define _(s) dgettext(PACKAGE, s)

/* === Forward declarations === */

static int sort_cmp(const void *a1, const void *a2);

/* === Public interface === */

/* Create all missing path elements and try to make a file there.
   Report denied access.
*/
int make_path_for(const char *fn, int *denied)
{
	char full[PATH_MAX+1];
	const char *start;

	*denied = 0;

	if (strlen(fn) >= PATH_MAX)
	{
		fprintf(stderr, _("file name too long: %s\n"), fn);
		return -1;
	}

	*full = 0;
	start = fn;

	/* Break path into elements, checking each of them */
	if (*start == '/')
	{
		strcpy(full, "/");
		start++;
	}

	for (;;)
	{
		char *p;

		if (! *start)
			break;

		p = strchr(start, '/');
		if (p == NULL)
			break;

		if (p == start)
		{
			start++;
			continue;
		}


		strncat(full, start, p-start);

		/* If exists, check whether it is a dir; create otherwise */
		if (access(full, F_OK) < 0)
		{
			if (errno != ENOENT)
			{
				fprintf(stderr, _("cannot access %s: %s\n"), full, strerror(errno));
				return -1;
			}

			if (mkdir(full, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH) < 0)
			{
				if (errno == EACCES)
				{
					*denied = 1;
					return 0;
				}

				fprintf(stderr, _("Cannot create directory %s: %s\n"), full, strerror(errno));
				return -1;
			}
		}
		else
		{
			struct stat st;

			if (stat(full, &st) < 0)
			{
				fprintf(stderr, _("cannot stat %s: %s\n"), full, strerror(errno));
				return -1;
			}

			if (! S_ISDIR(st.st_mode))
			{
				fprintf(stderr, _("%s not a directory in path to %s\n"), full, fn);
				return -1;
			}
		}

		strcat(full, "/");
		start = p+1;
	}

	/* Check the dir for writeability */
	if (full[strlen(full) - 1] == '/')
		full[strlen(full) - 1] = 0;

	if (*full && access(full, W_OK) < 0)
	{
		if (errno == EACCES)
		{
			*denied = 1;
			return 0;
		}

		fprintf(stderr, _("cannot access %s: %s\n"), full, strerror(errno));
		return -1;
	}

	unlink(fn);

	if (access(fn, F_OK) == 0)
	{
		fprintf(stderr, _("couldn't unlink %s\n"), fn);
		return -1;
	}

	*denied = 0;

	return 0;
}

/* Create a directory, reporting possible problems */
int create_dir(const char *name)
{
	/* The directory must not exist */
	if (access(name, F_OK) >= 0)
		return 0;

	if (errno != ENOENT)
	{
		fprintf(stderr, _("Problem checking access of %s: %s\n"), name, strerror(errno));
		return -1;
	}

	if (flag_verbose)
		fprintf(stderr, _("Creating %s\n"), name);

	if (mkdir(name, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH) < 0)
	{
		fprintf(stderr, _("Cannot create directory %s: %s\n"), name, strerror(errno));
		return -1;
	}

	return 0;
}

/* Start a new list */
void list_start(char ***strings, int *nstrings, int *nalloc)
{
	*strings = 0;
	*nstrings = 0;
	*nalloc = 0;
}

/* Append a string to the list */
int list_append(char ***strings, int *nstrings, int *nalloc, const char *str)
{
	char *s = strdup(str);

	if (s == NULL)
	{
        	fprintf(stderr, _("Out of memory\n"));
        	return -1;
	}

	return list_append_ptr(strings, nstrings, nalloc, s);
}

/* Append a dynamically allocated string to the list (will be freed by list_free)*/
int list_append_ptr(char ***strings, int *nstrings, int *nalloc, char *str)
{
	const int CHUNK = 1024;

	if (*nstrings >= *nalloc)
	{
        	if (*nalloc == 0)
                	*strings = (char **) malloc(CHUNK * sizeof(char *));
        	else
                	*strings = (char **) realloc(*strings, (*nalloc+CHUNK)*sizeof(char *));
        	*nalloc += CHUNK;
	}

	if (*strings == NULL)
	{
        	fprintf(stderr, _("Out of memory\n"));
        	return -1;
	}


	(*strings)[*nstrings] = str;

	(*nstrings)++;

	return 0;
}

/* Free the list and all items */
void list_free(char ***strings, int *nstrings, int *nalloc)
{
	int i;

	for (i=0; i < *nstrings; i++)
		free((*strings)[i]);

	if (*strings)
		free(*strings);
}

/* Sort the list */
void list_sort(char ***strings, int *nstrings, int *nalloc)
{
	qsort(*strings, *nstrings, sizeof(char *), sort_cmp);
}

/* Print a formatted help line */
void help_line(const char *short_opt, const char *long_opt, const char *descr)
{
	if (short_opt && *short_opt)
		printf("  -%s", short_opt);
	else
		printf("    ");

	if (long_opt && *long_opt)
		printf("%c --%-16s ", (short_opt && *short_opt) ? ',' : ' ', long_opt);
	else
		printf("%-20s ", "");
	printf("%s", descr);
	printf("\n");

	return;
}



/* === Private functions === */

static int sort_cmp(const void *a1, const void *a2)
{
	char **s1 = (char **) a1;
	char **s2 = (char **) a2;

	if (*s1 == NULL && *s2 == NULL)
		return 0;

	if (*s1 == NULL)
		return -1;

	if (*s2 == NULL)
		return 1;

	return strcmp(*s1, *s2);
}

