
#include "CEgFileSpec.h"

#include "CEgIOFile.h"

#ifdef EG_WIN
#include <stdio.h>

#ifndef GetCurrentDirectory
#include <windows.h>
#endif

#endif

#include "EgOSUtils.h"

#ifdef EG_MAC
#include <Resources.h>
#include <Files.h>
#endif

#ifdef EG_POSIX
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#endif


CEgFileSpec::CEgFileSpec() {

	Assign( nil, 0 );
}

CEgFileSpec::CEgFileSpec( const char* inFileName, long inType  ) {

	Assign( nil, 0 );
	AssignPathName( inFileName );
	SetType( inType );

	mFileName = mFileNameNE.getCStr();
}

CEgFileSpec::CEgFileSpec( const CEgFileSpec& inSpec ) {

	Assign( inSpec );
}

void CEgFileSpec::Assign( const CEgFileSpec& inSpec ) {

	Assign( inSpec.OSSpec(), inSpec.GetType() );
}

void CEgFileSpec::Assign( const void* inOSSpecPtr, long inType ) {

	mFileType = inType;

#ifdef EG_MAC
	mSpecData.Assign( (char*) inOSSpecPtr, sizeof( FSSpec ) + 32 );

	FSSpec* spec = (FSSpec*) OSSpec();
	DirInfo	pb;
	long err;

	// If we have a folder (w/ no name), turn it into a parent + folder name scheme
	// This is the case, for example, if the MacOS spec refers to a bundled folder (ie, Mac OS X app-folder)
	if ( inOSSpecPtr ) {

		if ( spec -> name[ 0 ] == 0 && spec -> parID ) {

			// See if we have a file or dir w/ the given name...
			pb.ioCompletion	= nil;
			pb.ioVRefNum	= spec -> vRefNum;
			pb.ioNamePtr 	= spec -> name;	// Name will receive the file name
			pb.ioDrDirID 	= spec -> parID;
			pb.ioFDirIndex	= -1;
			err = ::PBGetCatInfoSync( (CInfoPBRec*) &pb );

			// Assign the spec (name is already assigned)
			spec -> vRefNum = pb.ioVRefNum;
			spec -> parID = pb.ioDrParID;
		}
	}
#endif

#if EG_WIN || EG_POSIX
	mSpecData.Assign( (char*) inOSSpecPtr );
#endif

	AssignFileName();
}

void CEgFileSpec::AssignPathName( const char* inPathName, const CEgFileSpec* inFromFolder ) {

	if ( ! inFromFolder )
		inFromFolder = &EgOSUtils::sAppSpec;

	if ( inPathName ) {
#ifdef EG_MAC
		FSSpec		spec;
		UtilStr		path;

		// Get spec info
		long vRefNum 	= ( inFromFolder ) ? ( (FSSpec*) inFromFolder -> OSSpec() ) -> vRefNum : 0;
		long dirID 	= ( inFromFolder ) ? ( (FSSpec*) inFromFolder -> OSSpec() ) -> parID : 0;

		// Make a partial pathname
		if ( ! ( vRefNum == 0 && dirID == 0 ) )	// no vref num and dirID means root
			path.Append( FILE_SYS_SEP_CHAR );
		path.Append( inFromFolder -> GetFileName() );
		if ( inPathName[ 0 ] != FILE_SYS_SEP_CHAR )
			path.Append( FILE_SYS_SEP_CHAR );
		path.Append( inPathName );

		::FSMakeFSSpec( vRefNum, dirID, path.getPasStr(), &spec );
		Assign( &spec, mFileType );
#elif EG_WIN || EG_POSIX
		mSpecData.Assign( (char*) inFromFolder -> OSSpec() );
		char endChar = mSpecData.getChar( mSpecData.length() );
#ifdef EG_POSIX
		if ( inPathName[ 0 ] == '~') {
			mSpecData.Append( getenv( "HOME" ) );
			mSpecData.Append( &inPathName[1] );
		}
		else
#endif
		{
			if ( inPathName[ 0 ] != FILE_SYS_SEP_CHAR && endChar != FILE_SYS_SEP_CHAR )
				mSpecData.Append( FILE_SYS_SEP_CHAR );
			else if ( inPathName[ 0 ] == FILE_SYS_SEP_CHAR && endChar == FILE_SYS_SEP_CHAR )
				mSpecData.Trunc( 1 );
			mSpecData.Append( inPathName );
		}

		AssignFileName();
#endif
	}
}

void CEgFileSpec::AssignDefaultVol() {

#if EG_MAC
	FSSpec spec;
	WDPBRec pb;

	pb.ioCompletion = nil;
	pb.ioNamePtr = spec.name;

	long err = PBHGetVolSync( &pb );

	spec.parID = 0;
	spec.vRefNum = 0; //pb.ioVRefNum;
	Assign( &spec );
#elif EG_WIN
	Assign( "C:" );
	Append( FILE_SYS_SEP_CHAR );
#elif EG_POSIX
//	Append( FILE_SYS_SEP_CHAR );
#endif
}

void CEgFileSpec::AssignParent( CEgFileSpec& inSpec ) {

	if ( ! inSpec.OSSpec() )
		goto bail;

#if EG_MAC
	OSErr				err;
	DirInfo				pb;
	FSSpec* 			spec = (FSSpec*) inSpec.OSSpec(), newSpec;
	Str255				fname;

	// See if we have a file or dir w/ the given name...
	pb.ioCompletion	= nil;
	pb.ioVRefNum	= spec -> vRefNum;
	pb.ioNamePtr 	= fname;
	pb.ioDrDirID 	= spec -> parID;
	pb.ioFDirIndex	= -1;
	err = ::PBGetCatInfoSync( (CInfoPBRec*) &pb );

/*
	pb.ioCompletion	= nil;
	pb.ioVRefNum	= spec -> vRefNum;
	pb.ioNamePtr 	= fname;
	pb.ioDrDirID 	= pb.ioDrParID;
	pb.ioFDirIndex	= -1;
	err = ::PBGetCatInfoSync( (CInfoPBRec*) &pb );
	*/
	::FSMakeFSSpec( pb.ioVRefNum, pb.ioDrParID, fname, &newSpec );
	Assign( &newSpec );
#endif

bail: ;

}

int CEgFileSpec::CreateFolder( bool inReplace ) const {

	long result = CEGFILESPEC_NOT_IMPLEMENTED;

	if ( inReplace )
		Delete();

#if EG_MAC
	long newID;
	FSSpec* spec = (FSSpec*) OSSpec();
	result = ::FSpDirCreate( spec, 0, &newID );
	switch ( result ) {
		case noErr:
			result = CEGFILESPEC_SUCCESS;
			break;
		case dupFNErr:
			result = ( inReplace ) ?
				CEGFILESPEC_SUCCESS : CEGFILESPEC_ALREADY_EXISTS;
			break;
		case dirFulErr:
			result = CEGFILESPEC_DISK_FULL;
			break;
		case dirNFErr:
		case nsvErr:
		case fnfErr:
			result = CEGFILESPEC_FNF_ERR;
			break;
		case -5000:
		case -5031:
		case vLckdErr:
		case wPrErr:
			result = CEGFILESPEC_LOCKED_ERR;
			break;
		default:
			result = CEGFILESPEC_CREATE_ERR;
			break;
	}
#else
	result = CEGFILESPEC_NOT_IMPLEMENTED;
#endif

	return result;
}

long CEgFileSpec::GetType() const {

#if (defined EG_MAC || defined EG_POSIX)
	return mFileType;
#endif

#ifdef EG_WIN
	return EgOSUtils::FindExtension( &mFileName );
#endif
}

void CEgFileSpec::TypeToExt( UtilStr& ioStr, long inType ) {

	long c, i, pos = ioStr.length();

	for ( i = 0; i < 4; i++ ) {
		c = (inType >> i * 8) & 0xFF;
		if ( c > 32 )
			ioStr.Insert( pos, c, 1 );
	}
	if ( ioStr.length() > pos ) {
		if ( ioStr.getChar( pos + 1 ) != '.' )
			ioStr.Insert( pos, '.', 1 );
	}
}

void CEgFileSpec::SetType( long inType ) {

	mFileType = inType;
}

int CEgFileSpec::Delete() const {

	int result = CEGFILESPEC_DELETE_ERR;

	if ( OSSpec() ) {

#ifdef EG_MAC
		result = ::FSpDelete( (FSSpec*) mSpecData.getCStr() );
		switch ( result ) {
		case noErr:
			result = CEGFILESPEC_SUCCESS;
			break;
		case dirNFErr:
		case nsvErr:
		case fnfErr:
			result = CEGFILESPEC_FNF_ERR;
			break;
		case fLckdErr:
		case wPrErr:
		case fBsyErr:
			result = CEGFILESPEC_LOCKED_ERR;
			break;
		default:
			result = CEGFILESPEC_DELETE_ERR;
			break;
		}
#elif EG_WIN
		remove( mSpecData.getCStr() );
		result = CEGFILESPEC_SUCCESS;
#endif
	}

	return result;
}

void CEgFileSpec::Rename( const UtilStr& inNewName ) {

#if EG_MAC
	inNewName.copyTo( ((FSSpec*) OSSpec()) -> name, 32 );
#elif EG_WIN || EG_POSIX
	int pos;
	pos = mSpecData.FindLastInstanceOf( FILE_SYS_SEP_CHAR );
	mSpecData.Trunc( mSpecData.length() - pos );
	mSpecData.Append( inNewName );
#endif
	AssignFileName();
}

void CEgFileSpec::MakeUnique() {

	UtilStr origName, newName;
	long i;

	// Make a copy of the original file name
	GetFileName( origName, true );
#if EG_MAC
	origName.Keep( 29 );
#endif
	origName.Append( " " );

	// Once this spec isn't taken/used, then we're done
	for ( i = 1; i < 10000 && Exists(); i++ ) {
		newName.Assign( origName );
		newName.Append( i );
		Rename( newName );
	}
}

void CEgFileSpec::AssignFileName() {

	mFileNameNE.Wipe();

	if ( OSSpec() ) {
#ifdef EG_MAC
		unsigned char* name = ((FSSpec*) OSSpec()) -> name;
		mFileName.Assign( name );
#elif EG_WIN || EG_POSIX
		// If it's a folder, temporarily remove the end FILE_SYS_SEP_CHAR
		long pos, len = mSpecData.length();
		bool isFolder = (mSpecData.getChar( len ) == FILE_SYS_SEP_CHAR);
		if ( isFolder ) {
			pos = mSpecData.FindPrevInstanceOf( len - 1, FILE_SYS_SEP_CHAR );
			mFileName.Assign( mSpecData.getCStr() + pos, len - pos - 1 );
		}
		else {
			pos = mSpecData.FindLastInstanceOf( FILE_SYS_SEP_CHAR );
			mFileName.Assign( mSpecData.getCStr() + pos );
		}
#endif
		// Strip the extension
		EgOSUtils::FindExtension( mFileName, &mFileNameNE, nil );
	}
}


/*

void CEgFileSpec::GetFileNameNE( UtilStr& outFileName ) const	{
	long pos = mFileName.FindLastInstanceOf( '.' );	// Find where ext begins

	if ( pos > 0 )
		pos--;
	else
		pos = mFileName.length();

	outFileName.Assign( mFileName.getCStr(), pos );
}


const char* CEgFileSpec::GetFileName( bool inExtension ) const {
	if ( inExtension )
		return mFileName;
	else
		return mFileNameNE.getCStr();
}


void CEgFileSpec::GetFileName( UtilStr& outName, bool inExtension ) const {

	if ( inExtension )
		outName.Assign( mFileName );
	else
		outName.Assign( mFileNameNE );
}

*/


void CEgFileSpec::GetExtension( UtilStr& outExt ) const	{

	outExt.Assign( mFileName.getCStr() + mFileNameNE.length() );
	outExt.Capitalize();
}


void CEgFileSpec::GetPathname( UtilStr& outPathname ) const {

	if ( ! OSSpec() )
		return;
#if EG_MAC
	DirInfo  	info;
	Str255		dirName;
	OSErr		err;
	FSSpec*		spec = (FSSpec*) OSSpec();

	info.ioDrParID = spec -> parID;

	outPathname.Assign( mFileName );

	do {
		info.ioNamePtr = dirName;
		info.ioVRefNum = spec -> vRefNum;
		info.ioFDirIndex = -1;
		info.ioDrDirID = info.ioDrParID;
		err = PBGetCatInfo( (CInfoPBRec*) &info, 0 );

		// OS 9 returns the folder name in ioNamePtr,
		// but OS X doesn't so we use FSMakeFSSpec to get folder names

		// Get file name (use dirName as a buffer)
		spec = (FSSpec*) dirName;
		::FSMakeFSSpec( info.ioVRefNum, info.ioDrDirID, "\p", spec );

		spec -> name[0]++;
		spec -> name[ spec -> name[0]     ] = ':';
		spec -> name[ spec -> name[0] + 1 ] = 0;
		outPathname.Prepend( (char*) &spec -> name[1] );
	}
	while ( info.ioDrDirID != 2 );
#elif EG_WIN
	char pathName[ 501 ];
	char* filePart;
	::GetFullPathName( (char*) OSSpec(), 500, pathName, &filePart );
	outPathname.Assign( pathName );
#elif EG_POSIX
	if ((mFileName.getCStr() != NULL) && ((mFileName.getCStr())[0] == '~')) {
		outPathname.Assign( getenv( "HOME" ) );
		outPathname.Append( &(mFileName.getCStr()[1]) );
	}
	else { outPathname.Assign( mFileName ); }
#endif
}

void CEgFileSpec::SaveAs( const CEgFileSpec& inDestSpec ) const {

	if ( OSSpec() && inDestSpec.OSSpec() ) {
#ifdef EG_MAC
		::FSpExchangeFiles ( (FSSpec*) OSSpec(), (FSSpec*) inDestSpec.OSSpec() );
		Delete();
#elif EG_WIN
		inDestSpec.Delete();
		rename( (char*) OSSpec(), (char*) inDestSpec.OSSpec() );
#endif
	}
}

int CEgFileSpec::Exists() const {

	int 		result = 0;
	CEgIFile	iFile;

#if EG_MAC
	OSErr				err;
	HFileInfo			pb;
	FSSpec* 			spec = (FSSpec*) OSSpec();
	UtilStr				fname;

	// Prep a str to receive un updated name change
	fname.Dim( 40 );
	fname.Assign( spec -> name );

	// See if we have a file or dir w/ the given name...
	pb.ioCompletion	= nil;
	pb.ioDirID 		= spec -> parID;
	pb.ioVRefNum	= spec -> vRefNum;
	pb.ioNamePtr 	= fname.getPasStr();
	pb.ioFDirIndex	= 0;
	err = ::PBGetCatInfoSync( (CInfoPBRec*) &pb );
	if ( err == noErr ) {
		if ( pb.ioFlAttrib & 0x10 )	// Do we have a dir?
			result = 2;
		else
			result = 1;
	}
#elif EG_WIN
	long attribs = ::GetFileAttributes( (char*) OSSpec() );

	if ( attribs == 0xFFFFFFFF )
		result = 0;
	else if ( attribs & FILE_ATTRIBUTE_DIRECTORY )
		result = 2;
	else
		result = 1;
#elif EG_POSIX
	struct stat statdata;

	if (stat((char *) OSSpec(), &statdata) != 0)
		result = 0;
	if (S_ISDIR(statdata.st_mode))
		result = 2;
	else if (S_ISREG(statdata.st_mode))
		result = 1;
#endif

	return result;
}

CEgErr CEgFileSpec::Duplicate( const CEgFileSpec& inDestSpec ) const {

	CEgIOFile	oFile;
	CEgIFile	iFile;

	// Open the source and destination files
	iFile.open( this );
	iFile.seek( 0 );
	if ( ! iFile.noErr() )
		oFile.open( &inDestSpec );

	long	numBytes;
	long	pos	= 0;
	long	size	= iFile.size();
	CEgErr	err;

	// Copy the file in 50k chunks
	while ( pos < size && oFile.noErr() && iFile.noErr() ) {
		numBytes = 50000;
		if ( numBytes + pos > size )
			numBytes = size - pos;
		oFile.CEgOStream::PutBlock( iFile, numBytes );
		pos += numBytes;
	}

	if ( iFile.noErr() )
		err = oFile;
	else
		err = iFile;

	return err;
}

void CEgFileSpec::ChangeExt( const char* inExt ) {

	UtilStr	fileName;
	long		len;

	GetFileName( fileName, true );
	fileName.Append( '.' );
	len = fileName.length();
	fileName.Append( inExt );
	fileName.Keep( len + 3 );
	Rename( fileName );
}

bool CEgFileSpec::CompareExtn( const char* inExtension ) {

	// Proceed only if we have a valid spec
	if ( mSpecData.length() > 0 ) {

		// Check the extension...
		if ( UtilStr::StrCmp( mFileName.getCStr() + mFileNameNE.length(), inExtension, false ) == 0 )
			return true;
	}

	return false;
}

bool CEgFileSpec::IsTextFile() {

	int extLen = mFileName.length() - mFileNameNE.length();

	// Exit if we don't have a valid spec
	if ( mSpecData.length() == 0 )
		return false;

	// Is there an TXT extension?
	if ( CompareExtn( ".TXT" ) )

		return true;

	else if ( extLen == 0 ) {

		// If there's no extension and we're in Wintelville, then assume it's a text file
#if EG_WIN || EG_POSIX
		return true;
#endif

		// Does MacOS see it as a TEXT type?
#if EG_MAC
		FInfo fileInfo;
		int err = ::FSpGetFInfo( (FSSpec*) OSSpec(), &fileInfo );
		if ( err == noErr && fileInfo.fdType == 'TEXT' )
			return true;
#endif
	}

	return false;
}

const void* CEgFileSpec::OSSpec() const {

	if ( mSpecData.length() > 0 )
		return mSpecData.getCStr();
	else
		return nil;
}

bool CEgFileSpec::Equals( const Hashable* inComp, long inParam ) const {

	return mFileName.compareTo( &( (CEgFileSpec*) inComp ) -> mFileName, inParam ) == 0;
}

long CEgFileSpec::Hash( long inParam ) const {

	return mFileName.Hash( inParam );
}
