/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <ctype.h>
#include <sys/stat.h>
#include <errno.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>

#include "dialogs.h"

#include "singit_debug.h"
#include "singit/song.h"
#include "singit_song_private.h"
#include "singit/macros.h"
#include "singit_main_config.h"

#include "dlg_messagebox.h"
#include "dlg_input_number.h"
#include "dlg_input_time.h"
#include "editor_singit_main.h"
#include "singit_tools.h"
#include "singit_main.h"

#include "pixmaps.xpm.h"

#include "editor_clist_timestamps.h"
#include "editor_plain_text.h"
#include "editor_synced_lyrics.h"
#include "editor_config.h"
#include "editor_status.h"

#ifdef HAVE_QUERY_DEPS
#include "editor_query.h"
#endif

static GtkWidget *singit_editor_win = NULL;
static GtkWidget
		*editor_vbox,
			*editor_menubar_handle_box, *editor_menubar,
			*editor_toolbar_handle_box, *editor_toolbar,
				*new_file_button,
				*load_file_button,
				*get_current_song_button,
				*reload_lyrics_button,

				*guess_filename_button,

				*save_file_button,
				*save_file_as_button,

#ifdef HAVE_TAG_MANAGEMENT_SUPPORT
				*save_file_as_id3_button,
#endif
				*export_as_html_button,

//				*sync_mode_toggle_button,

				*strip_html_button,
				*strip_timetags_button,
				*nice_text_button,
				*auto_change_time_button,
				*change_time_button,
				*check_sync_consistency_button,

#ifdef HAVE_TAG_MANAGEMENT_SUPPORT
				*id3_button,
#endif
				*config_button,
				*help_button,
				*close_button,

			*editor_type_notebook,
				*editor_clist_timestamps, *editor_clist_timestamps_label,
				*editor_plain_text, *editor_plain_text_label,
				*editor_synced_lyrics, *editor_synced_lyrics_label,
#ifdef HAVE_QUERY_DEPS
				*editor_query, *editor_query_label,
#endif

			*editor_status_hbox,
				*editor_status_time_bar,
				*editor_mode_gtk_event_box, *editor_status_mode_bar,
				*editor_status_consistency_bar,
				*editor_status_modified_bar,
			*editor_song_gtk_event_box, *editor_status_filename_bar;

static GtkTooltips *editor_tooltips = NULL;
static GtkWidget *save_question_win = NULL;
static GtkWidget *ok_cancel_win = NULL;
static GtkWidget *input_time_win = NULL;

static GtkAccelGroup *editor_menubar_accel_group = NULL;
static GtkItemFactory *editor_menubar_item_factory = NULL;

extern VisPlugin singit_vp;

static void singit_editor_cleanup(void);
static void new_lyrics_event(GtkWidget *widget, gpointer data);
static void save_lyric_file_event(GtkWidget *widget, gpointer data);
static void show_save_error_message_box(gchar *filename);
static gint check_sync_consistency(gboolean invalidate);

static void toggle_modified_state (gboolean state);
static gboolean check_user_continue (void);

typedef enum _HTMLBreaks {

	HBS_NORM = 1 << 0,
	HBS_BR =   1 << 1,
	HBS_P =    1 << 2
}
HTMLBreaks;

enum {
	ACT_FILE_NEW,
	ACT_STRIP_LINEFEED,
	ACT_STRIP_BR,
	ACT_STRIP_P,
	ACT_TRANS_DIS,
	ACT_TRANS_UW,
	ACT_TRANS_WU,
};

enum {
	TAB_SYNC = 0,
	TAB_TAGS,
	TAB_TEXT,
	TAB_QUERY,
};

#define USE_EXT_TIMETAGS \
	(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item \
	(editor_menubar_item_factory, "/Options/Ext. timetags"))->active ? \
	2 : 1)

gboolean singit_editor_is_realized()
{
	return ((singit_editor_win) && (singit_editor_win->window) &&
		GTK_WIDGET_REALIZED(singit_editor_win));
}

static gchar* get_base_path()
{
        gchar *buffer, *fileDir = NULL;
	gchar **directories;
	gint i = 0;
	SingitSong *my_song;

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [get_base_path]\n"));
#endif

	directories = g_strsplit(GET_SCD->basePath, ",", 0);

	while ((directories[i] != NULL) && (fileDir == NULL)) {
		if (directories[i][0] == '~') {
			if (directories[i][1] == '~') {
				my_song = singit_song_attach(STATUS->song);
				if ((my_song != NULL) && (my_song->song_filename != NULL)) {
					buffer = &directories[i][2];
					fileDir = g_strconcat(g_dirname(my_song->song_filename), buffer, NULL);
					singit_song_detach(&my_song);
				}
			}
			else {
				buffer = &directories[i][1];
				fileDir = g_strconcat(g_get_home_dir(), buffer, NULL);
			}
		}
		else { fileDir = g_strdup(g_strconcat(directories[i], NULL)); }
		i++;
	}

	// If we cant't find a direcory get home
	if (fileDir == NULL)
		{ fileDir = g_strdup(g_get_home_dir()); }

	g_strfreev(directories);
	return fileDir;
}

static gchar* get_first_path(const gchar *filename)
{
        gchar *base_path, *fileDir;
	gchar **extensions;

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [get_first_path]\n"));
#endif

	extensions = g_strsplit(GET_SCD->lyricExtension, ",", 0);
	base_path = get_base_path();
	fileDir = g_strconcat(base_path, (filename == NULL) ? "*" : filename, ".", extensions[0], NULL);
	g_free(base_path);
	g_strfreev(extensions);

	return fileDir;
}

static gchar* get_filename_base(const gchar *filename)
{
	gchar *dot_pos, *base, *result;

	g_return_val_if_fail(filename != NULL, NULL);

	base = g_basename(filename);
	dot_pos = strrchr(base, '.');
	if ((dot_pos == NULL) || (dot_pos == base))
		{ return strdup(base); }

	result = g_strndup(base, (dot_pos - base));
	result[dot_pos - base] = '\0';

	return result;
}

static gchar* guess_new_lyrics_filename(SingitSong *song)
{
	gchar *file_base, *lyrics_path = NULL;

	g_return_val_if_fail(song != NULL, NULL);

	file_base = get_filename_base(song->song_filename);

	if (file_base != NULL) {
		lyrics_path = get_first_path(file_base);
		g_free(file_base);
	}

	if (lyrics_path != NULL)
		return lyrics_path;
	else
		return g_strdup(song->lyric_filename);
}

static void set_status_filename(gchar *full_name, gboolean guessed)
{
	gchar *statusbar_text, *tooltip_text;

	if (!singit_editor_is_realized())
		{ return; }

// Same as: g_return_if_fail(!((full_name == NULL) && (guessed == TRUE)));
	g_return_if_fail((full_name != NULL) || (guessed == FALSE));

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [set_status_filename]\n"));
#endif

	EDT_STATUS->is_guessed_filename = guessed;

	if (EDT_STATUS->filename != NULL)
		{ g_free(EDT_STATUS->filename); }
	if (full_name != NULL) {
		EDT_STATUS->filename = g_strdup(full_name);
		if (guessed == TRUE)
			tooltip_text = g_strconcat("(", full_name, ")", NULL);
		else
			tooltip_text = g_strdup(full_name);

		statusbar_text = g_strconcat(" ", _("Lyrics"), ": ", tooltip_text, " ", NULL);
	}
	else {
		EDT_STATUS->filename = full_name;
		tooltip_text = g_strconcat("(", _("unknown"), ")", NULL);
	}
	statusbar_text = g_strconcat(" ", _("Lyrics"), ": ", tooltip_text, " ", NULL);

	if (EDT_STATUS->filename_context_id) {
		gtk_statusbar_pop
			(GTK_STATUSBAR(editor_status_filename_bar),
			EDT_STATUS->filename_context_id);
	}
	else {
		EDT_STATUS->filename_context_id =
			gtk_statusbar_get_context_id
				(GTK_STATUSBAR(editor_status_filename_bar), "filename");
	}

	gtk_statusbar_push(GTK_STATUSBAR(editor_status_filename_bar),
		EDT_STATUS->filename_context_id, statusbar_text);

	gtk_tooltips_set_tip(editor_tooltips,
		editor_song_gtk_event_box, tooltip_text, NULL);

	g_free(statusbar_text);
	g_free(tooltip_text);
}

void singit_editor_set_time(guint time)
{
	gchar timeStamp[8];
	static guint last_time = 1;

	if (!(singit_editor_is_realized() && editor_status_time_bar)) { return; }
	if (time > 5999000) { return; }

	if (last_time != time / 1000) {
		last_time = time / 1000;
		timeStamp[7] = '\0';
		sprintf(timeStamp, " %.2i:%.2i ", last_time / 60, last_time % 60);

		GDK_THREADS_ENTER();
		gtk_statusbar_pop(GTK_STATUSBAR(editor_status_time_bar),
			EDT_STATUS->time_context_id);
		gtk_statusbar_push(GTK_STATUSBAR(editor_status_time_bar),
			EDT_STATUS->time_context_id, timeStamp);
		GDK_THREADS_LEAVE();
	}
}

static gboolean save_question_message_box_event(GtkWidget *widget, gint buttonNr, gpointer data)
{
	gboolean close_dialog = TRUE;

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [save_question_message_box_event] : %i\n", buttonNr));
#endif

	switch (buttonNr) {
	case 1: // Cancel (Default)
		EDT_STATUS->continue_func = FALSE;
		break;
	case 2: // Yes
		EDT_STATUS->continue_func = TRUE;
		break;
	case 3: // No
		EDT_STATUS->continue_func = TRUE;
		break;
	}

	return close_dialog;
}

static gboolean ok_cancel_win_button_pressed_event
	(GtkWidget *widget, gint buttonNr, gpointer user_data)
{
#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [inconsistence_question_button_pressed_event] : %i\n", buttonNr));
#endif

	switch (buttonNr) {
	case 1:
		EDT_STATUS->continue_func = FALSE;
		break;
	case 2:
		EDT_STATUS->continue_func = TRUE;
		break;
	}

	return TRUE;
}

static void toggle_modified_state (gboolean state)
{
	if (state && (check_sync_consistency(2) != -2)) {
		check_sync_consistency(1);
		gtk_statusbar_pop(GTK_STATUSBAR(editor_status_consistency_bar),
			EDT_STATUS->valid_consistency_id);
	}
	if (state == EDT_STATUS->text_changed)
		{ return; }

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_singit_main.c [toggle_modified_state]\n"));
#endif
	if (state == TRUE) {
		gtk_statusbar_pop(GTK_STATUSBAR(editor_status_modified_bar),
			EDT_STATUS->modified_context_id);
	}
	else {
		gtk_statusbar_push(GTK_STATUSBAR(editor_status_modified_bar),
			EDT_STATUS->modified_context_id, EDT_STATUS->changed_spaces);
		singit_editor_view_modify
			(SINGIT_EDITOR_VIEW(editor_clist_timestamps), FALSE);
		singit_editor_view_modify
			(SINGIT_EDITOR_VIEW(editor_plain_text), FALSE);
		singit_editor_view_modify
			(SINGIT_EDITOR_VIEW(editor_synced_lyrics), FALSE);
	}
	EDT_STATUS->text_changed = state;
}

static gchar* get_text_from_editor (gint *carret, gint page)
{
	gchar *text = NULL;
	GtkWidget *editor;

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [get_text_from_editor]\n"));
#endif

	if (page < 0) {
		editor = gtk_notebook_get_nth_page
			(GTK_NOTEBOOK(editor_type_notebook),
			 gtk_notebook_get_current_page
				(GTK_NOTEBOOK(editor_type_notebook)));
	}
	else {
		editor = gtk_notebook_get_nth_page
			(GTK_NOTEBOOK(editor_type_notebook), page);
	}

	if (!IS_SINGIT_EDITOR_VIEW(editor))
		{ return NULL; }

	text = singit_editor_view_get_text
		(SINGIT_EDITOR_VIEW(editor), carret);

	return text;
}

static gboolean set_editor_from_text (gchar *text, gint carret, gint page)
{
	GtkWidget *editor;

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [set_editor_from_text]\n"));
#endif

	if (page < 0) {
		editor = gtk_notebook_get_nth_page
			(GTK_NOTEBOOK(editor_type_notebook),
			 gtk_notebook_get_current_page
				(GTK_NOTEBOOK(editor_type_notebook)));
	}
	else {
		editor = gtk_notebook_get_nth_page
			(GTK_NOTEBOOK(editor_type_notebook), page);
	}

	if (!IS_SINGIT_EDITOR_VIEW(editor))
		{ return FALSE; }

	singit_editor_view_set_text
		(SINGIT_EDITOR_VIEW(editor), text, carret);

	return TRUE;
}

// 0 - calc / 1 - invalidate / 2 - get last result
static gint check_sync_consistency(gint mode)
{
	static gint valid_consistency = -2;
	gchar *text;
	SingitSong *song;

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [check_sync_consistency]\n"));
#endif

	switch (mode) {
	case 1:
		valid_consistency = -2;
	case 2:
		break;
	default:
		if (valid_consistency > -2) { return valid_consistency; }
		text = get_text_from_editor(NULL, -1);
		if (!text) { return -1; }

		song = SINGIT_SONG(singit_song_new(NULL));
		singit_song_read_text_stream(song, text);
		valid_consistency = singit_song_check_sync_lyric_consistency(song);
		singit_song_free(song);
		break;
	}
	return valid_consistency;
}

static void check_sync_consistency_event (GtkWidget *widget, gpointer data)
{
	gchar *output;
	gint result;

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [check_sync_consistency_event]\n"));
#endif

	result = check_sync_consistency(FALSE);

	if (result == -1) {
		output = g_strdup_printf(" %s ", _("Ok"));
		gtk_statusbar_push(GTK_STATUSBAR(editor_status_consistency_bar),
			EDT_STATUS->valid_consistency_id, output);
	}
	else {
		output = g_strdup_printf(" %s : %i ", _("Error"), result);
		gtk_statusbar_push(GTK_STATUSBAR(editor_status_consistency_bar),
			EDT_STATUS->valid_consistency_id, output);
	}

	g_free(output);
}

static gboolean skip_html_comment(gchar** source_pos)
{
	gchar *end_comment;

	if (strncmp((*source_pos), "<!--", 4) == 0) {
		end_comment = strstr((*source_pos), "-->");
		if (end_comment == NULL) {
			end_comment += strlen(end_comment);
		}
		else
			{ end_comment += 4; }

		(*source_pos) = end_comment;

		return TRUE;
	}

	return FALSE;
}

static gboolean render_simple_html_tags(gchar** source_pos, gchar** copy_pos)
{
	// * Keep it sorted by length !!! *

	typedef struct _TagMap {
		const gchar* tag;
		const gchar* render;
	}
	TagMap;

#define MAX_TAG_LEN 6
	static TagMap translate_tag_table[] =
		{ { "<P>", "\n\n" },	// 3
		  { "<BR>", "\n" },	// 4
		  { "<P/>", "\n\n" },
		  { "<BR/>", "\n" },	// 5
		  { "<P />", "\n\n" },
		  { "</BR>", "\n" },
		  { "<BR />", "\n" },	// 6
		  { NULL, NULL } };

	gint i;
	guint tag_length = 0;

	while (((*source_pos)[tag_length] != '\0') &&
		((*source_pos)[tag_length] != '>') &&
		(tag_length < MAX_TAG_LEN))
	{
		tag_length++;
	}
	tag_length++;

	if (tag_length >= MAX_TAG_LEN)
		{ return FALSE; }

	i = 0;
	while (translate_tag_table[i].tag != NULL) {
		if ((tag_length == strlen(translate_tag_table[i].tag)) &&
			(g_strncasecmp((*source_pos), translate_tag_table[i].tag, tag_length) == 0))
		{
			memcpy((*copy_pos), translate_tag_table[i].render,
				strlen(translate_tag_table[i].render));
			(*source_pos) += tag_length - 1;
			(*copy_pos) += strlen(translate_tag_table[i].render);
			return TRUE;
		}
		i++;
	}

	return FALSE;
}

static void strip_html_tags_event (GtkWidget *widget, gpointer data)
{
	gchar *new_text = NULL, *pos_begin, *pos_end, *pos_copy, *text;
	gint len, freelines, tag_level;
//	gint cp_line_nr, pos_line_nr;
//	gchar **lines;

	text = get_text_from_editor(NULL, -1);
	if (!text) { return; }
	if (!check_user_continue()) { return; }

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [strip_html_tags_event]\n"));
#endif

	new_text = g_new(gchar, strlen(text) + 1);
	pos_copy = new_text;
	pos_end = pos_begin = text;
	tag_level = freelines = 0;
	while (pos_end != NULL) {
		switch (pos_end[0]) {
		case '<':
			if (tag_level == 0) {
				len = pos_end - pos_begin;
				if (len > 0) {
					strncpy(pos_copy, pos_begin, len);
					pos_copy += len;
				}
				if (skip_html_comment(&pos_end) == TRUE) {
					pos_begin = pos_end;
				}
				if (render_simple_html_tags
						(&pos_end, &pos_copy) == FALSE)
				{
					pos_begin = pos_end;
					tag_level++;
				}
				else { pos_begin = (pos_end + 1); }
			}
			else { tag_level++; }
			break;
		case '>':
			tag_level--;
			if (tag_level == 0)
				{ pos_begin = (pos_end + 1); }
			else if (tag_level < 0)
				{ tag_level = 0; }
			break;
		case '\0':
			if (tag_level == 0)
				{ strcpy(pos_copy, pos_begin); }
			pos_end = NULL;
			continue;
		case '&':
			if (strncmp(pos_end, "&nbsp;", 6) == 0) {
				len = pos_end - pos_begin;
				strncpy(pos_copy, pos_begin, len);
				pos_copy[len] = ' ';
				pos_copy += len;
				pos_end += 6;
			}
			break;
		case '\n':
			freelines++;
			break;
		}
		pos_end++;
	}
	g_free(text);
	text = new_text;
/*
	lines = g_strsplit(new_text, "\n", 0);
	g_free(new_text);
	cp_line_nr = pos_line_nr = 0;
	while (lines[i] != NULL
	text = g_strjoinv("\n", lines);
*/
	set_editor_from_text(text, 0, -1);
	g_free(text);

	toggle_modified_state(TRUE);
}

static void nice_text_event (GtkWidget *widget, gpointer data)
{
	gchar *text;
	gint i = 0;
	gchar up_str[2] = " \0";
	gchar **lines;
	gchar *temp;

	text = get_text_from_editor(NULL, -1);
	if (!text) { return; }

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [nice_text_event]\n"));
#endif

	lines = g_strsplit(text, "\n", 0);
	g_free(text);

	while (lines[i]) {
		temp = g_strdup(g_strstrip(lines[i]));
		up_str[0] = temp[0];
		g_strup(&up_str[0]);
		temp[0] = up_str[0];
		g_free(lines[i]);
		lines[i] = temp;
		i++;
	}

	text = g_strjoinv("\n", lines);
	g_strfreev(lines);

	set_editor_from_text(text, 0, -1);
	g_free(text);

	toggle_modified_state(TRUE);
}

static void strip_time_tags_event (GtkWidget *widget, gpointer data)
{
	gchar *new_text = NULL, *pos_begin, *pos_end, *pos_copy, *text;
	gint len, tag_type;

	text = get_text_from_editor(NULL, -1);
	if (!text) { return; }
	if (!check_user_continue()) { return; }

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [strip_time_tags_event]\n"));
#endif

	new_text = g_new(gchar, strlen(text) + 1);
	pos_copy = new_text;
	pos_end = pos_begin = text;
	while (pos_begin) {
		pos_end = strstr(pos_end, "[");
		if (!pos_end) {
			strcpy(pos_copy, pos_begin);
			pos_begin = NULL;
		}
		else {
			tag_type = singit_song_get_timetag_type(pos_end);
			switch (tag_type) {
			case 0:
				pos_end++;
				break;
			case 1:
				len = pos_end - pos_begin;
				strncpy (pos_copy, pos_begin, len);
				pos_copy += len;
				pos_end += 7;
				pos_begin = pos_end;
				break;
			case 2:
				len = pos_end - pos_begin;
				strncpy (pos_copy, pos_begin, len);
				pos_copy += len;
				pos_end += 11;
				pos_begin = pos_end;
				break;
			}
		}
	}
	g_free(text);

	set_editor_from_text(new_text, 0, -1);
	g_free(new_text);

	toggle_modified_state(TRUE);
}

static void ok_button_export_as_html_event (GtkWidget *widget, GtkWidget *fileSelector)
{
	FILE *file;
	gchar *new_text = NULL, *tmp = NULL, *text;
	gchar **stream, *connected = NULL;
	gchar timeToken[42];
	gint offset = 0, last_line = -1;
	guint i = 0;
	SingitSong *song = NULL;
	GList *item = NULL;
	gchar *filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileSelector));

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [ok_button_export_as_html_event] : %s\n", filename));
#endif

	if (!(file = fopen(filename, "w"))) {
		show_save_error_message_box(filename);
		gtk_widget_destroy(fileSelector);
		return;
	}

	song = SINGIT_SONG(singit_song_new(NULL));
	text = get_text_from_editor(NULL, -1);
	singit_song_read_text_stream(song, text);
	timeToken[37] = '\0';
	timeToken[41] = '\0';

	stream = g_new(gchar*, song->lyric_lines+1);
	stream[song->lyric_lines] = NULL;
	for (i = 0; i < song->lyric_lines; i++)
		{ stream[i] = g_strdup(song->lyrics[i]); }
	item = song->first_token;
	while (item) {
		if (USE_EXT_TIMETAGS) {
			sprintf(timeToken, "<FONT CLASS=\"lyrictag\">[%.2i:%.2i:%.3i]</FONT>", tTime(item) / 60000, (tTime(item) / 1000) % 60, tTime(item) % 1000);
			if ((gint) tLine(item) != last_line) { offset = 0; }
			else { offset += 41; }
		}
		else {
			sprintf(timeToken, "<FONT CLASS=\"lyrictag\">[%.2i:%.2i]</FONT>", tTime(item) / 60000, (tTime(item) / 1000) % 60);
			if ((gint) tLine(item) != last_line) { offset = 0; }
			else { offset += 37; }
		}
		last_line = tLine(item);
		tmp = stream[tLine(item)];
		connected = tools_insert_string(stream[tLine(item)], &timeToken[0], tPos(item) + offset);
		if (connected) {
			stream[tLine(item)] = connected;
			g_free(tmp);
		}
		item = g_list_next(item);
	}
	for (i = 0; i < song->lyric_lines; i++) {
		tmp = stream[i];
		stream[i] = g_strconcat(stream[i], "<BR />", NULL);
		g_free(tmp);
	}
	tmp = new_text = g_strjoinv("\n", stream);
	new_text = g_strconcat(GET_SCD->htmlHeader, new_text, GET_SCD->htmlTail, NULL);
	g_free(tmp);

	if (fwrite(new_text, 1, strlen(new_text), file) != strlen(new_text))
       	{
		show_save_error_message_box(filename);
	}

	if (new_text) { g_free(new_text); }
	fclose(file);

	singit_song_free(song);
	g_free(text);

	gtk_widget_destroy(fileSelector);
}

static void export_as_html_event (GtkWidget *widget, gpointer data)
{
	GtkWidget *fileSelector;

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [export_as_html_event]\n"));
#endif

	fileSelector = gtk_file_selection_new(_("Export lyric file as HTML"));

	gtk_signal_connect(GTK_OBJECT(fileSelector), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(fileSelector));

	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileSelector)->ok_button),
		"clicked", GTK_SIGNAL_FUNC(ok_button_export_as_html_event), fileSelector);

	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fileSelector)->cancel_button),
		"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(fileSelector));

	gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileSelector), "");

	gtk_widget_show(fileSelector);
}

static gboolean check_user_continue(void)
{
#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [check_user_continue]\n"));
#endif

	if (EDT_STATUS->text_changed) {
		message_box_dialog_show(MESSAGE_BOX_DIALOG(save_question_win), TRUE,
			GTK_WINDOW(singit_editor_win));
		return EDT_STATUS->continue_func;
	}
	return TRUE;
}

static void new_lyrics_event(GtkWidget *widget, gpointer data)
{
#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [new_lyrics_event]\n"));
#endif

	if (check_user_continue()) {
		set_status_filename(NULL, FALSE);
		set_editor_from_text(NULL, 0, -1);
		toggle_modified_state(FALSE);
		gtk_notebook_set_page
			(GTK_NOTEBOOK(editor_type_notebook), TAB_TEXT);
	}
}

static void editor_main_move_cursor_event (GtkEditable *editable, gint x, gint y, gpointer user_data)
{
	gboolean sensitive;
//	g_print("Test\n");

	g_return_if_fail(GTK_IS_TEXT(editable));
	g_return_if_fail(IS_EDITOR_PLAIN_TEXT(GTK_WIDGET(editable)->parent));

	sensitive = editor_plain_text_is_selected(EDITOR_PLAIN_TEXT(GTK_WIDGET(editable)->parent));

	gtk_widget_set_sensitive(auto_change_time_button, sensitive);
	gtk_widget_set_sensitive(change_time_button, sensitive);
	gtk_widget_set_sensitive(gtk_item_factory_get_item(editor_menubar_item_factory, "/Edit/Change time..."), sensitive);
}

static void change_time_dlg_button_event(InputTimeDialog *mldlg, gint number, gpointer data)
{
	SingitSong *song;
	gchar *text;

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [change_time_dlg_button_event]\n"));
#endif

	if (number != 0) {
		text = editor_plain_text_get_selection
			(EDITOR_PLAIN_TEXT(editor_plain_text));
		if (text) {
			song = SINGIT_SONG(singit_song_new(NULL));
			singit_song_read_text_stream(song, text);
			g_free(text);
			singit_song_modify_overall_time(song, number);
			singit_song_write_text_stream(song, &text, USE_EXT_TIMETAGS);
			singit_song_free(song);

			editor_plain_text_xchg_selection
				(EDITOR_PLAIN_TEXT(editor_plain_text), text);
			g_free(text);
		}
	}
	input_time_dialog_hide(mldlg);
}

static void change_time_event(GtkWidget *widget, gpointer data)
{
	SingitSong *song;
	gchar *text;
	gboolean show_dialog = FALSE;

	if (editor_plain_text_is_selected(EDITOR_PLAIN_TEXT(editor_plain_text))) {
		text = editor_plain_text_get_selection
			(EDITOR_PLAIN_TEXT(editor_plain_text));
		if (text != NULL) {
			song = SINGIT_SONG(singit_song_new(NULL));
			singit_song_read_text_stream(song, text);
			if (song->first_token != NULL)  {
//				input_time_dialog_set_min_time
//					(INPUT_TIME_DIALOG(input_time_win), -1 * tTime(song->first_token));
				input_time_dialog_set_time_offset
					(INPUT_TIME_DIALOG(input_time_win), tTime(song->first_token));
				show_dialog = TRUE;
			}
			g_free(text);
			singit_song_free(song);
		}

		if (show_dialog == TRUE) {
			input_time_dialog_show(INPUT_TIME_DIALOG(input_time_win),
				GTK_WINDOW(singit_editor_win), TRUE, TRUE);
		}
	}
}

static inline gint singit_song_get_first_time_gap(gchar *lyric_text)
{
	gint result_time = 0, tag_time;
	gchar *token_pos = lyric_text;
	gboolean token_found = FALSE;

	g_return_val_if_fail(lyric_text != NULL, -1);

	while ((token_pos != NULL) && (token_found == FALSE))
	{
		token_pos = strstr(token_pos, "[");
		if (token_pos != NULL) {
			token_found = extrakt_timetag_information(token_pos, &tag_time);
			token_pos++;
		}
	}

	if ((token_found == TRUE) && (token_pos != NULL))
	{
		token_found = FALSE;
		while ((token_pos != NULL) && (token_found == FALSE))
		{
			token_pos = strstr(token_pos, "[");
			if (token_pos != NULL) {
				token_found = extrakt_timetag_information(token_pos, &result_time);
				token_pos++;
			}
		}

		if (token_found == TRUE)
			{ result_time = tag_time - result_time; }
	}

	return result_time;
}

static void auto_change_time_event(GtkWidget *widget, gpointer data)
{
	SingitSong *song;
	gchar *text, *tmp_text;
	gint gap_time;

	if (editor_plain_text_is_selected(EDITOR_PLAIN_TEXT(editor_plain_text))) {
		text = editor_plain_text_get_selection
			(EDITOR_PLAIN_TEXT(editor_plain_text));
		if (text != NULL) {
			gap_time = singit_song_get_first_time_gap(text);
			if (gap_time != 0)
			{
				tmp_text = singit_song_remove_token(text, 0);
				g_free(text);
				text = tmp_text;

				song = SINGIT_SONG(singit_song_new(NULL));
				singit_song_read_text_stream(song, text);

				// Change the overall time to the difference between first and second
				singit_song_modify_overall_time(song, gap_time);

				// Exchange the text
				g_free(text);
				singit_song_write_text_stream(song, &text, USE_EXT_TIMETAGS);

				editor_plain_text_xchg_selection
					(EDITOR_PLAIN_TEXT(editor_plain_text), text);

				singit_song_free(song);
			}
			g_free(text);
		}
	}
}

static void lyrics_from_song_event(GtkWidget *widget, gpointer data)
{
	SingitSong *cur_song = NULL;
	gchar *text, *file_name;

	cur_song = singit_song_attach(STATUS->song);

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [lyrics_from_song_event]\n"));
#endif

	if (!check_user_continue()) {
		singit_song_detach(&cur_song);
		return;
	}

	if (!singit_song_text_found(cur_song)) {
		if ((EDT_STATUS->filename != NULL) &&
			(EDT_STATUS->is_guessed_filename == FALSE))
		{
			set_editor_from_text(NULL, 0, -1);
			toggle_modified_state(FALSE);
		}
		file_name = guess_new_lyrics_filename(cur_song);
		set_status_filename(file_name, TRUE);
		g_free(file_name);
		singit_song_detach(&cur_song);
		return;
	}

	singit_song_write_text_stream(cur_song, &text, USE_EXT_TIMETAGS);
	set_status_filename(cur_song->lyric_filename, FALSE);
	singit_song_detach(&cur_song);
	set_editor_from_text(text, 0, -1);
	g_free(text);

	toggle_modified_state(FALSE);
}

static void reload_lyrics_event(GtkWidget *widget, gpointer data)
{
	SingitSong *song;
	gchar *text = NULL;

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [reload_lyrics_event]\n"));
#endif

	if (EDT_STATUS->filename != NULL)
	{
		if (STATUS != NULL) {
			song = singit_song_attach(STATUS->song);
			if ((song != NULL) && (song->lyric_filename != NULL)) {
				if (strcmp(song->lyric_filename, EDT_STATUS->filename) == 0) {
					singit_song_write_text_stream(song, &text, USE_EXT_TIMETAGS);
				}
				singit_song_detach(&song);
			}
		}
		if (text == NULL) {
			song = SINGIT_SONG(singit_song_new(NULL));
			if (singit_song_load_lyrics(song, EDT_STATUS->filename)) {
				singit_song_write_text_stream(song, &text, USE_EXT_TIMETAGS);
			}
			singit_song_detach(&song);
		}
		if (text != NULL) {
			if (check_user_continue()) {
				set_editor_from_text(text, 0, -1);
				toggle_modified_state(FALSE);
			}
			g_free(text);
		}
	}
}

static void guess_filename_event(GtkWidget *widget, gpointer data)
{
	SingitSong *cur_song = NULL;
	gchar *file_name;

	cur_song = singit_song_attach(STATUS->song);

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [guess_filename_event]\n"));
#endif

	if (!singit_song_text_found(cur_song) || check_user_continue()) {
		file_name = guess_new_lyrics_filename(cur_song);
		set_status_filename(file_name, TRUE);
		g_free(file_name);
	}

	singit_song_detach(&cur_song);
}

static void ok_button_open_file_selector_event(GtkWidget *widget, GtkWidget *fileSelector)
{
	SingitSong *song = NULL;
	gchar *text, *dir;
	gchar *filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileSelector));

	if (EDT_STATUS->last_directory != NULL)
		{ g_free(EDT_STATUS->last_directory); }

	dir = strrchr(filename, '/');
	EDT_STATUS->last_directory = ((dir == NULL) ?
		g_strdup(filename) : g_strndup(filename, dir - filename + 1));

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [ok_button_open_file_selector_event] : %s\n", filename));
#endif

	song = SINGIT_SONG(singit_song_new(NULL));
	if (singit_song_load_lyrics(song, filename) && check_user_continue()) {
		singit_song_write_text_stream(song, &text, USE_EXT_TIMETAGS);
		set_editor_from_text(text, 0, -1);
		g_free(text);
		set_status_filename(song->lyric_filename, FALSE);
		toggle_modified_state(FALSE);
		message_box_dialog_hide(MESSAGE_BOX_DIALOG(save_question_win));
	}

	singit_song_free(song);
	if (!GET_ECD->multiSelect)
		{ gtk_widget_destroy(fileSelector); }
}

static void open_lyric_file_event(GtkWidget *widget, gpointer data)
{
	GtkWidget *fileSelector;
	gchar *file = NULL;
        struct stat stats;

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [open_lyric_file_event]\n"));
#endif

	fileSelector = gtk_file_selection_new(_("Lyrics selection"));

	gtk_signal_connect(GTK_OBJECT(fileSelector), "destroy",
		(GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT(fileSelector));

	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileSelector)->ok_button),
		"clicked", (GtkSignalFunc) ok_button_open_file_selector_event, fileSelector);

	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fileSelector)->cancel_button),
		"clicked", (GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT(fileSelector));

	if (EDT_STATUS->last_directory != NULL) {
		if (stat(EDT_STATUS->last_directory, &stats) == 0)
			{ file = g_strdup(EDT_STATUS->last_directory); }
	}

	if (file == NULL) {
		file = get_first_path(NULL);
		if (stat(file, &stats) == -1) {
			g_free(file);
			file = g_strdup("*");
		}
	}

	gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileSelector), file);
	g_free(file);

	gtk_widget_show(fileSelector);
}

static void show_save_error_message_box(gchar* fileName)
{
	GtkWidget *message_box = NULL;
	GtkWidget *message_table, *message_label, *message_button_ok;
	gchar *text;

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [show_save_error_message_box]\n"));
#endif

	message_box = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(message_box), _("Error"));
	gtk_window_set_modal(GTK_WINDOW(message_box), TRUE);
	gtk_signal_connect_object(GTK_OBJECT(message_box), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(message_box));

	message_table = gtk_table_new (1, 1, TRUE);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(message_box)->vbox), message_table);
	gtk_container_border_width(GTK_CONTAINER(message_table), 7);
	gtk_widget_show(message_table);

	if (fileName != NULL)
		{ text = g_strconcat(_("File save failed"), "\n\n", fileName, NULL); }
	else
		{ text = _("File save failed");	}
	message_label = gtk_label_new (text);
	g_free(text);
	gtk_table_attach_defaults (GTK_TABLE(message_table), message_label, 0, 1, 0, 1);
	gtk_widget_show(message_label);

	message_button_ok = gtk_button_new_with_label(_("Ok"));
	gtk_signal_connect_object(GTK_OBJECT(message_button_ok), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(message_box));
	GTK_WIDGET_SET_FLAGS(message_button_ok, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(message_box)->action_area), message_button_ok, TRUE, TRUE, 0);
	gtk_widget_show(message_button_ok);

	gtk_widget_show(message_box);
}

static void ok_button_save_as_file_selector_event(GtkWidget *widget, GtkWidget *fileSelector)
{
	SingitSong *song;
	gchar *text;
	gchar *filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileSelector));

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [ok_button_save_as_file_selector_event] : %s\n", filename));
#endif

	song = SINGIT_SONG(singit_song_new(NULL));
	text = get_text_from_editor(NULL, -1);
	singit_song_read_text_stream(song, text);

	pthread_mutex_lock(&STATUS->lyrics_rw_mutex);
	if (!singit_song_save_to_text_file(song, filename, USE_EXT_TIMETAGS))
	{
		pthread_mutex_unlock(&STATUS->lyrics_rw_mutex);
		show_save_error_message_box(filename);
	}
	else {
		pthread_mutex_unlock(&STATUS->lyrics_rw_mutex);
		set_status_filename(song->lyric_filename, FALSE);
		toggle_modified_state(FALSE);
		STATUS->text_update = TRUE;
	}
	singit_song_free(song);
	g_free(text);

	gtk_widget_destroy(fileSelector);
}

static void save_as_lyric_file_event(GtkWidget *widget, gpointer data)
{
	GtkWidget *fileSelector;
	gchar *file = NULL;
	gchar *selected_text = NULL, *temp_text;
	GtkWidget *editor;
	SingitSong *cur_song = NULL;

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [save_as_lyric_file_event]\n"));
#endif

	editor = gtk_notebook_get_nth_page
		(GTK_NOTEBOOK(editor_type_notebook),
		 gtk_notebook_get_current_page
			(GTK_NOTEBOOK(editor_type_notebook)));

	if (editor == editor_plain_text) {
		selected_text = editor_plain_text_get_selection
			(EDITOR_PLAIN_TEXT(editor_plain_text));

		if (selected_text != NULL) {
			if ((strlen(selected_text) > 0) && (strstr(selected_text, "\n") == NULL)) {
				if (GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Convert/To spaces"))->active == TRUE) {
					temp_text = tools_replace_string(selected_text, "_", " ");
					g_free(selected_text);
					selected_text = temp_text;
				}
				else if (GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Convert/To underscores"))->active == TRUE) {
					temp_text = tools_replace_string(selected_text, " ", "_");
					g_free(selected_text);
					selected_text = temp_text;
				}
				file = get_first_path(selected_text);
			}
			g_free(selected_text);
			selected_text = NULL;
		}
	}

	if (file == NULL) {
		if ((EDT_STATUS->filename != NULL) &&
			(EDT_STATUS->is_guessed_filename == TRUE))
		{
			file = g_strdup(EDT_STATUS->filename);
		}

		if (file == NULL) {
			cur_song = singit_song_attach(STATUS->song);
			if (cur_song != NULL) {
				file = guess_new_lyrics_filename(cur_song);
				singit_song_detach(&cur_song);
			}
			if (file == NULL)
				file = get_first_path(NULL);
		}
	}

	fileSelector = gtk_file_selection_new(_("Lyric file save"));
//	gtk_widget_realize(fileSelector);

	gtk_signal_connect(GTK_OBJECT(fileSelector), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroy), &fileSelector);

	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileSelector)->ok_button),
		"clicked", GTK_SIGNAL_FUNC(ok_button_save_as_file_selector_event), fileSelector);

	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fileSelector)->cancel_button),
		"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(fileSelector));

	gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileSelector), file);
	g_free(file);

	gtk_window_set_transient_for(GTK_WINDOW(fileSelector), GTK_WINDOW(singit_editor_win));
	gtk_window_set_modal(GTK_WINDOW(fileSelector), TRUE);
	gtk_window_set_position(GTK_WINDOW(fileSelector), GTK_WIN_POS_MOUSE);
	gtk_widget_show(fileSelector);
}

static void save_lyric_file_event(GtkWidget *widget, gpointer data)
{
	SingitSong *song = NULL;
	gchar *text;

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [save_lyric_file_event]\n"));
#endif

//	if (!check_user_save_sync()) { return; }

//	if (EDT_STATUS->filename == NULL) ||
//		(EDT_STATUS->is_guessed_filename == TRUE))
	if (EDT_STATUS->filename == NULL)
	{
		save_as_lyric_file_event(NULL, NULL);
	}
	else {
		text = get_text_from_editor(NULL, -1);
		song = SINGIT_SONG(singit_song_new(NULL));
		singit_song_read_text_stream(song, text);

		pthread_mutex_lock(&STATUS->lyrics_rw_mutex);
		if (!singit_song_save_to_text_file(song, EDT_STATUS->filename, USE_EXT_TIMETAGS))
		{
			pthread_mutex_unlock(&STATUS->lyrics_rw_mutex);
			save_as_lyric_file_event(NULL, NULL);
		}
		else {
			pthread_mutex_unlock(&STATUS->lyrics_rw_mutex);
			if (EDT_STATUS->is_guessed_filename == TRUE)
				set_status_filename(song->lyric_filename, FALSE);
			toggle_modified_state(FALSE);
			STATUS->text_update = TRUE;
		}
		g_free(text);
		singit_song_free(song);
	}
}

static gboolean singit_editor_main_keypress(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	gint page_nr;
	GtkWidget *editor;

	switch(event->keyval) {
		case GDK_Escape:
			if (check_user_continue())
				singit_editor_hide();
			break;
		case GDK_l:
			if (event->state & GDK_CONTROL_MASK) {
				lyrics_from_song_event(NULL, NULL);
				return TRUE;
			}
			break;
		case GDK_n:
			if (event->state & GDK_CONTROL_MASK) {
				new_lyrics_event(NULL, NULL);
				return TRUE;
			}
			break;
		case GDK_o:
			if (event->state & GDK_CONTROL_MASK) {
				open_lyric_file_event(NULL, NULL);
				return TRUE;
			}
			break;
		case GDK_s:
			if (event->state & GDK_CONTROL_MASK) {
				save_lyric_file_event(NULL, NULL);
				return TRUE;
			}
			break;
		case GDK_t:
			if (event->state & GDK_CONTROL_MASK) {
				page_nr = gtk_notebook_get_current_page
					(GTK_NOTEBOOK(editor_type_notebook));
				if (event->state & GDK_SHIFT_MASK) { page_nr--; }
				else { page_nr++; }
				editor = gtk_notebook_get_nth_page
					(GTK_NOTEBOOK(editor_type_notebook), page_nr);

				if (!editor) {
					if (page_nr > 0) { page_nr = 0; }
					else {
						page_nr = 0;
						while (gtk_notebook_get_nth_page
							(GTK_NOTEBOOK(editor_type_notebook), page_nr))
						{
							page_nr++;
						}
						page_nr--;
					}
				}
				gtk_notebook_set_page
					(GTK_NOTEBOOK(editor_type_notebook), page_nr);
			}
			return TRUE;
		default:
			return FALSE;
	}
	return FALSE;
}

static void singit_editor_set_editor_mode_status_bar(gint page_num)
{
	gchar *mode_str, *tstring = NULL;
	gboolean set_mode = TRUE;

	if (!singit_editor_is_realized())
		{ return; }

	switch (page_num) {
	case 0: tstring = _("synced"); break;
	case 1: tstring = _("timestamps"); break;
	case 2: tstring = _("text"); break;
	default: set_mode = FALSE; break;
	}

	if (GTK_STATUSBAR(editor_status_mode_bar)->messages != NULL) {
		gtk_statusbar_pop(GTK_STATUSBAR(editor_status_mode_bar),
			EDT_STATUS->mode_context_id);
	}

	if (set_mode == TRUE) {
		mode_str = g_strconcat(" ", _("Editor mode"), ": ", tstring, NULL);
	}
	else {
		mode_str = g_strconcat(" ", _("Submitting author:"), " ",
			((GET_SCD->submittingAuthor != NULL) ?
			GET_SCD->submittingAuthor : ""), " ",
			_("(double click)"), NULL);
	}

	gtk_statusbar_push(GTK_STATUSBAR(editor_status_mode_bar),
		EDT_STATUS->mode_context_id, mode_str);
	g_free(mode_str);
}

static void singit_editor_switch_notebook_page (GtkNotebook *notebook,
		GtkNotebookPage *page, gint page_num, gpointer user_data)
{
	gboolean sensitive;

	if (gtk_notebook_get_current_page
		(GTK_NOTEBOOK(user_data)) < 0)
	{
		return;
	}

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_singit_main.c [singit_editor_switch_notebook_page]\n"));
	DEBUG(DLV_ALL, ("   %i (new - page_num) / %i (old - gtk_notebook_get_current_page)\n", page_num,
		gtk_notebook_get_current_page(GTK_NOTEBOOK(user_data))));
#endif

	if (EDT_STATUS->text == NULL)
		EDT_STATUS->text = get_text_from_editor(&(EDT_STATUS->text_carret),
			gtk_notebook_get_current_page(GTK_NOTEBOOK(user_data)));

	sensitive = (page_num == TAB_TEXT);
	gtk_widget_set_sensitive(auto_change_time_button, sensitive);
	gtk_widget_set_sensitive(change_time_button, sensitive);
	gtk_widget_set_sensitive(gtk_item_factory_get_item
		(editor_menubar_item_factory, "/Edit/Change time..."), sensitive);
/*
	gtk_widget_set_sensitive(new_file_button,
				*load_file_button,
				*get_current_song_button,
				*reload_lyrics_button,

				*guess_filename_button,

				*save_file_button,
				*save_file_as_button,

#ifdef HAVE_TAG_MANAGEMENT_SUPPORT
				*save_file_as_id3_button,
#endif
				*export_as_html_button,

//				*sync_mode_toggle_button,

				*strip_html_button,
				*strip_timetags_button,
				*nice_text_button,
				*auto_change_time_button,
				*change_time_button,
				*check_sync_consistency_button,

#ifdef HAVE_TAG_MANAGEMENT_SUPPORT
				*id3_button,
#endif
				*config_button,
				*help_button,
				*close_button,
	}
*/
	singit_editor_set_editor_mode_status_bar(page_num);
}

static void singit_editor_switch_notebook_page_after (GtkNotebook *notebook,
		GtkNotebookPage *page, gint page_num, gpointer user_data)
{
	if (gtk_notebook_get_current_page(GTK_NOTEBOOK(user_data)) < 0) { return; }

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_singit_main.c [singit_editor_switch_notebook_page]\n"));
	DEBUG(DLV_ALL, ("   %i (new - page_num) / %i (old - gtk_notebook_get_current_page)\n", page_num,
		gtk_notebook_get_current_page(GTK_NOTEBOOK(user_data))));
#endif

	if (set_editor_from_text(EDT_STATUS->text, EDT_STATUS->text_carret, page_num) == TRUE) {
		g_free(EDT_STATUS->text);
		EDT_STATUS->text = NULL;
	}
}

static void singit_editor_cleanup(void)
{
	g_return_if_fail(EDT_STATUS->cleanedUp == FALSE);

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [singit_editor_cleanup]\n"));
#endif

	EDT_STATUS->cleanedUp = TRUE;

	if (save_question_win) { gtk_widget_destroy(save_question_win); }
	if (input_time_win) { gtk_widget_destroy(input_time_win); }
	if (ok_cancel_win) { gtk_widget_destroy(ok_cancel_win); }

	singit_config_save_editor();

	gtk_widget_destroy(singit_editor_win);
	gtk_object_destroy(GTK_OBJECT(editor_menubar_item_factory));

	editor_status_unref(NULL);
	singit_status_unref(NULL);

	singit_main_finish(FALSE);
}

void singit_editor_hide(void)
{
#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [singit_editor_hide]\n"));
#endif

	if (singit_editor_is_realized())
		{ singit_editor_cleanup(); }
}

static gint singit_editor_delete_event( GtkWidget *widget, GdkEvent  *event, gpointer data )
{
	if (check_user_continue()) {
		singit_editor_cleanup();
		return (FALSE);
	}
	return (TRUE);
}

static void close_editor_event(GtkWidget *widget, gpointer user_data)
{
	if (check_user_continue())
		{ singit_editor_cleanup(); }
}

static void build_check_dialogs()
{
	save_question_win = message_box_dialog_new (_("Question"),
		_("Do you want to save your changes ?"), _("Cancel"));
	message_box_dialog_prepend_button
		(MESSAGE_BOX_DIALOG(save_question_win), _("Yes"),
		 GTK_SIGNAL_FUNC(save_lyric_file_event));
//	message_box_dialog_prepend_button
//		(MESSAGE_BOX_DIALOG(save_question_win), _("Save as"),
//		GTK_SIGNAL_FUNC(save_as_lyric_file_event));
	message_box_dialog_prepend_button
		(MESSAGE_BOX_DIALOG(save_question_win), _("No"), NULL);
	gtk_signal_connect(GTK_OBJECT(save_question_win), "button_pressed",
		GTK_SIGNAL_FUNC(save_question_message_box_event), NULL);
	gtk_signal_connect(GTK_OBJECT(save_question_win), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroyed), &save_question_win);

	ok_cancel_win = message_box_dialog_new (_("Question"),
		_("Your lyrics are inconsistent!\n"
		"This may cause errors in the final file\n"
		"Do you want to proceed ?"),
		_("  Cancel  "));
	message_box_dialog_prepend_button(MESSAGE_BOX_DIALOG(ok_cancel_win),
		_(" Proceed "), NULL);
	gtk_signal_connect(GTK_OBJECT(ok_cancel_win), "button_pressed",
		GTK_SIGNAL_FUNC(ok_cancel_win_button_pressed_event), NULL);
	gtk_signal_connect(GTK_OBJECT(ok_cancel_win), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroyed), &ok_cancel_win);

	input_time_win = input_time_dialog_new(_("Set time offset"), 200, 100);
	gtk_signal_connect(GTK_OBJECT(input_time_win), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroyed), &input_time_win);
	gtk_signal_connect(GTK_OBJECT(input_time_win), "new_time",
		GTK_SIGNAL_FUNC(change_time_dlg_button_event), NULL);
}

static void tooltips_check_event (GtkWidget *widget, gpointer data)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_singit_main.c [tooltips_check_event]\n"));
#endif

	GET_ECD->showHints =
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
			(editor_menubar_item_factory,
			"/Options/Show tooltips"))->active;
	gtk_toolbar_set_tooltips
		(GTK_TOOLBAR(editor_toolbar), GET_ECD->showHints);
}

static void extended_timetags_check_event (GtkWidget *widget, gpointer data)
{
#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [extended_timetags_check_event]\n"));
#endif

	GET_ECD->extendedLyrics =
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
			(editor_menubar_item_factory, "/Options/Ext. timetags"))->active;
}

static void editor_main_show_about_event (GtkWidget *widget, gpointer data)
{
#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [editor_main_show_about_event]\n"));
#endif

	singit_about_show_page((guint) data);
}

static void editor_main_show_config_event (GtkWidget *widget, gpointer data)
{
#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [editor_main_show_config_event]\n"));
#endif

	singit_config_show_page(2);
}

static void editor_main_save_config_event (GtkWidget *widget, gpointer data)
{
#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [editor_main_save_config_event]\n"));
#endif

	singit_config_gen_emit_update(EDT_STATUS->config);
	singit_config_gen_save(EDT_STATUS->config);
}

static void editor_main_set_strip_html_event (GtkWidget *widget, gpointer data)
{
#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [editor_main_set_strip_html_status_event]\n"));
#endif

	widget = gtk_item_factory_get_widget_by_action(editor_menubar_item_factory, (gint) data);

	if (GTK_IS_CHECK_MENU_ITEM(widget)) {
		if (GTK_CHECK_MENU_ITEM(widget)->active == TRUE) {
			if (!(((gint) data) & EDT_STATUS->strip_html_state))
				EDT_STATUS->strip_html_state =
					(EDT_STATUS->strip_html_state | (gint) data);
		}
		else {
			if ((((gint) data) & EDT_STATUS->strip_html_state))
				EDT_STATUS->strip_html_state =
					(EDT_STATUS->strip_html_state ^ ! (gint) data);
		}
	}
//	g_print("%i / %s\n", EDT_STATUS->strip_html_state, gtk_type_name(GTK_OBJECT(widget)->klass->type));
}

#ifdef HAVE_QUERY_DEPS
static void editor_main_set_artist_match
	(gpointer callback_data, guint callback_action, GtkWidget *widget)
{
	GET_ECD->artistMatch = callback_action;

	if (editor_query != NULL)
		editor_query_widget_set_search_matches
			(EDITOR_QUERY_WIDGET(editor_query),
			GET_ECD->artistMatch, GET_ECD->titleMatch);
}

static void editor_main_set_title_match
	(gpointer callback_data, guint callback_action, GtkWidget *widget)
{
	GET_ECD->titleMatch = callback_action;

	if (editor_query != NULL)
		editor_query_widget_set_search_matches
			(EDITOR_QUERY_WIDGET(editor_query),
			GET_ECD->artistMatch, GET_ECD->titleMatch);
}
#endif

static gchar* editor_main_menu_translate_func(const gchar *path, gpointer func_data)
{
	gchar **items = g_strsplit(path, "/", 0);
	gchar *tmp = NULL;
	gint i = 0;
//	g_print("%s\n", path);

	i = 0;
	while (items[i] != NULL) { i++; }
	if (i > 0) {
		i--;
		tmp = items[i];
		items[i] = g_strdup(_(items[i]));
		g_free(tmp);
	}

	tmp = g_strjoinv("/", items);
	g_strfreev(items);
	return tmp;
}

#if 0
// Seems to be the only way for a simpler translation of item factory items
// You could also translate the whole strings "/../../.." but
// translating the leaf item is sufficient and easier to read
static gchar *menu_items[] = {
	N_("_File"),
		N_("_New"),
		N_("_Open..."),
		N_("_Revert"),
		N_("_Save"),
		N_("Save _As..."),
		N_("_Export HTML..."),
		N_("_Quit"),
	N_("_Edit"),
		N_("Strip _HTML"),
		N_("Strip _timtags"),
		N_("_Nice text"),
		N_("_Change time..."),
	N_("_Options"),
		N_("_Ext. timetags"),
		N_("_Show tooltips"),
		N_("_Multi open"),
		N_("_Auto sync check"),
		N_("Strip _HTML"),
		N_("_Convert"),
			N_("_Disabled"),
			N_("To _spaces"),
			N_("To _underscores"),
		N_("_Artist match"), N_("_Title match"),
			N_("_Perfect"),
			N_("_First words"),
			N_("_All words"),
			N_("An_y words"),
		N_("_More options..."),
		N_("_Save options"),
	N_("_Help"),
		N_("_About..."),
		N_("_Keybindings..."),
};
#endif

// Menu attach / keystoke / event handler / user data / item type (or radio button reference)
// Macro to calm the compiler warnings
#define MI(item, accel, callb, action, type) \
	{ (gchar*) item, (gchar*) accel, callb, action, (gchar*) type },
static const GtkItemFactoryEntry editor_menubar_items[] = {
	MI("/_File",                               NULL,                              NULL,                  0,                       "<Branch>")
	MI("/File/_New",                   "<control>N",                  new_lyrics_event,                  0,                             NULL)
	MI("/File/_Open...",               "<control>O",             open_lyric_file_event,                  0,                             NULL)
	MI("/File/_Revert",                        NULL,               reload_lyrics_event,                  0,                             NULL)
	MI("/File/_Save",                  "<control>S",             save_lyric_file_event,                  0,                             NULL)
	MI("/File/Save _As...",                    NULL,          save_as_lyric_file_event,                  0,                             NULL)
	MI("/File/sep1",                           NULL,                              NULL,                  0,                    "<Separator>")
	MI("/File/_Export HTML...",                NULL,              export_as_html_event,                  0,                             NULL)
	MI("/File/sep2",                           NULL,                              NULL,                  0,                    "<Separator>")
	MI("/File/_Quit",                  "<control>Q",                close_editor_event,                  0,                             NULL)
	MI("/_Edit",                               NULL,                              NULL,                  0,                       "<Branch>")
	MI("/Edit/Strip _HTML",                    NULL,             strip_html_tags_event,                  0,                             NULL)
	MI("/Edit/Strip _timtags",                 NULL,             strip_time_tags_event,                  0,                             NULL)
	MI("/Edit/_Nice text",                     NULL,                   nice_text_event,                  0,                             NULL)
	MI("/Edit/sep1",                           NULL,                              NULL,                  0,                    "<Separator>")
	MI("/Edit/_Change time...",                NULL,                 change_time_event,                  0,                         "<Item>")
	MI("/_Options",                            NULL,                              NULL,                  0,                       "<Branch>")
	MI("/Options/_Ext. timetags",              NULL,     extended_timetags_check_event,                  0,                    "<CheckItem>")
	MI("/Options/_Show tooltips",              NULL,              tooltips_check_event,                  0,                    "<CheckItem>")
	MI("/Options/_Multi open",                 NULL,                              NULL,                  0,                    "<CheckItem>")
	MI("/Options/_Auto sync check",            NULL,                              NULL,                  0,                    "<CheckItem>")
	MI("/Options/sep1",                        NULL,                              NULL,                  0,                    "<Separator>")
	MI("/Options/Strip _HTML",                 NULL,                              NULL,                  0,                       "<Branch>")
	MI("/Options/Strip HTML/\\n",              NULL,  editor_main_set_strip_html_event, ACT_STRIP_LINEFEED,                    "<CheckItem>")
	MI("/Options/Strip HTML/<BR>",             NULL,  editor_main_set_strip_html_event,       ACT_STRIP_BR,                    "<CheckItem>")
	MI("/Options/Strip HTML/<P>",              NULL,  editor_main_set_strip_html_event,        ACT_STRIP_P,                    "<CheckItem>")
	MI("/Options/_Convert",                    NULL,                              NULL,                  0,                       "<Branch>")
	MI("/Options/Convert/_Disabled",           NULL,                              NULL,      ACT_TRANS_DIS,                    "<RadioItem>")
	MI("/Options/Convert/To _spaces",          NULL,                              NULL,       ACT_TRANS_UW,      "/Options/Convert/Disabled")
	MI("/Options/Convert/To _underscores",     NULL,                              NULL,       ACT_TRANS_WU,      "/Options/Convert/Disabled")
	MI("/Options/sep2",                        NULL,                              NULL,                  0,                    "<Separator>")
#ifdef HAVE_QUERY_DEPS
	MI("/Options/_Artist match",               NULL,                              NULL,                  0,                       "<Branch>")
	MI("/Options/Artist match/_Perfect",       NULL,      editor_main_set_artist_match,      MATCH_PERFECT,                    "<RadioItem>")
	MI("/Options/Artist match/_First words",   NULL,      editor_main_set_artist_match,        MATCH_FIRST,  "/Options/Artist match/Perfect")
	MI("/Options/Artist match/_All words",     NULL,      editor_main_set_artist_match,          MATCH_ALL,  "/Options/Artist match/Perfect")
	MI("/Options/Artist match/An_y words",     NULL,      editor_main_set_artist_match,          MATCH_ANY,  "/Options/Artist match/Perfect")
	MI("/Options/_Title match",                NULL,                              NULL,                  0,                       "<Branch>")
	MI("/Options/Title match/_Perfect",        NULL,       editor_main_set_title_match,      MATCH_PERFECT,                    "<RadioItem>")
	MI("/Options/Title match/_First words",    NULL,       editor_main_set_title_match,        MATCH_FIRST,   "/Options/Title match/Perfect")
	MI("/Options/Title match/_All words",      NULL,       editor_main_set_title_match,          MATCH_ALL,   "/Options/Title match/Perfect")
	MI("/Options/Title match/An_y words",      NULL,       editor_main_set_title_match,          MATCH_ANY,   "/Options/Title match/Perfect")
	MI("/Options/sep3",                        NULL,                              NULL,                  0,                    "<Separator>")
#endif
	MI("/Options/_More options...",            NULL,     editor_main_show_config_event,                  0,                             NULL)
	MI("/Options/_Save options",               NULL,     editor_main_save_config_event,                  0,                             NULL)
	MI("/_Help",                               NULL,                              NULL,                  0,                   "<LastBranch>")
	MI("/Help/_About...",              "<control>A",      editor_main_show_about_event,                  0,                             NULL)
	MI("/Help/sep1",                           NULL,                              NULL,                  0,                    "<Separator>")
	MI("/Help/_Keybindings...",                "F1",      editor_main_show_about_event,                  1,                             NULL)
};
#undef MI

static GtkWidget* get_main_menu(GtkWidget *window)
{
	gint nmenu_items = sizeof (editor_menubar_items) / sizeof (editor_menubar_items[0]);

	editor_menubar_accel_group = gtk_accel_group_new ();

	editor_menubar_item_factory = gtk_item_factory_new
		(GTK_TYPE_MENU_BAR, "<main>", editor_menubar_accel_group);
	gtk_item_factory_set_translate_func(editor_menubar_item_factory,
		editor_main_menu_translate_func, NULL, NULL);

	gtk_item_factory_create_items (editor_menubar_item_factory,
		nmenu_items, (GtkItemFactoryEntry*) editor_menubar_items, NULL);

	gtk_check_menu_item_set_active
		(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
		(editor_menubar_item_factory, "/Options/Ext. timetags")),
		GET_ECD->extendedLyrics);
	gtk_check_menu_item_set_active
		(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
		(editor_menubar_item_factory, "/Options/Show tooltips")),
		GET_ECD->showHints);

	switch (GET_ECD->underscoreWhiteTransform) {
	case 1:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
			(editor_menubar_item_factory,
			"/Options/Convert/To spaces")), TRUE);
		break;
	case 2:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
			(editor_menubar_item_factory,
			"/Options/Convert/To underscores")), TRUE);
		break;
	default:
		break;
	}

#ifdef HAVE_QUERY_DEPS
	switch (GET_ECD->artistMatch) {
	case MATCH_FIRST:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
			(editor_menubar_item_factory,
			"/Options/Artist match/First words")), TRUE);
		break;
	case MATCH_ALL:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
			(editor_menubar_item_factory,
			"/Options/Artist match/All words")), TRUE);
		break;
	case MATCH_ANY:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
			(editor_menubar_item_factory,
			"/Options/Artist match/Any words")), TRUE);
		break;
	default:
		break;
	}

	switch (GET_ECD->titleMatch) {
	case MATCH_FIRST:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
			(editor_menubar_item_factory,
			"/Options/Title match/First words")), TRUE);
		break;
	case MATCH_ALL:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
			(editor_menubar_item_factory,
			"/Options/Title match/All words")), TRUE);
		break;
	case MATCH_ANY:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
			(editor_menubar_item_factory,
			"/Options/Title match/Any words")), TRUE);
		break;
	default:
		break;
	}
#endif

	gtk_check_menu_item_set_active
		(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
		(editor_menubar_item_factory, "/Options/Strip HTML/\\n")),
		(GET_ECD->stripHtml & HBS_NORM));
	gtk_check_menu_item_set_active
		(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
		(editor_menubar_item_factory, "/Options/Strip HTML/<BR>")),
		(GET_ECD->stripHtml & HBS_BR));
	gtk_check_menu_item_set_active
		(GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
		(editor_menubar_item_factory, "/Options/Strip HTML/<P>")),
		(GET_ECD->stripHtml & HBS_P));

	gtk_widget_set_sensitive(gtk_item_factory_get_item
		(editor_menubar_item_factory, "/Edit/Change time..."), FALSE);

	gtk_window_add_accel_group (GTK_WINDOW (window), editor_menubar_accel_group);

	return gtk_item_factory_get_widget (editor_menubar_item_factory, "<main>");
}

static void singit_editor_view_modified(SingitEditorView *sev, gboolean modified)
{
	if (modified == TRUE)
		toggle_modified_state(modified);
}

static gboolean singit_editor_view_keypress(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	gboolean result = TRUE;

	switch (event->keyval) {
		case GDK_Escape:
			if (check_user_continue) {
				singit_editor_hide();
				return TRUE;
			}
			break;
		default:
			result = FALSE;
			break;
        }
        return result;
}

#ifdef HAVE_QUERY_DEPS
static void singit_editor_query_set_text(GtkWidget *widget, gchar *text, gpointer user_data)
{
	if (check_user_continue()) {
		if (EDT_STATUS->text != NULL)
			g_free(EDT_STATUS->text);
		EDT_STATUS->text = g_strdup(text);
		toggle_modified_state(FALSE);
		set_status_filename(NULL, FALSE);
		gtk_notebook_set_page
			(GTK_NOTEBOOK(editor_type_notebook), TAB_TEXT);
	}
}
#endif

static void singit_editor_config_update_event
	(SingitConfigGen *scg, EditorConfigData *cfg_data, gpointer user_data)
{
#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [singit_editor_config_update_event]\n"));
#endif

	cfg_data->multiSelect = GTK_CHECK_MENU_ITEM
		(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Multi open"))->active;
	cfg_data->showHints = GTK_CHECK_MENU_ITEM
		(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Show tooltips"))->active;
	cfg_data->extendedLyrics = GTK_CHECK_MENU_ITEM
		(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Ext. timetags"))->active;
	cfg_data->checkSyncBeforeSave = GTK_CHECK_MENU_ITEM
		(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Auto sync check"))->active;

	if (GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
		(editor_menubar_item_factory, "/Options/Convert/To spaces"))->active == TRUE)
	{
		cfg_data->underscoreWhiteTransform = 1;
	}
	else if (GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item
		(editor_menubar_item_factory, "/Options/Convert/To underscores"))->active == TRUE)
	{
		cfg_data->underscoreWhiteTransform = 2;
	}
	else { cfg_data->underscoreWhiteTransform = 0; }

#ifdef HAVE_QUERY_DEPS
	if (GET_SCD->httpProxyUsage == TRUE) {
		HTTPProxy proxy	= {
			GET_SCD->httpProxyServer,
			GET_SCD->httpProxyPort,
			GET_SCD->httpProxyUser,
			GET_SCD->httpProxyPasswd
		};
		editor_query_widget_set_default_proxy
			(EDITOR_QUERY_WIDGET(editor_query), &proxy);
	}
	else {
		editor_query_widget_set_default_proxy
			(EDITOR_QUERY_WIDGET(editor_query), NULL);
	}

	if (gtk_notebook_get_current_page
		(GTK_NOTEBOOK(editor_type_notebook)) == TAB_QUERY)
	{
		if (GTK_STATUSBAR(editor_status_mode_bar)->messages != NULL) {
			gtk_statusbar_pop(GTK_STATUSBAR(editor_status_mode_bar),
				EDT_STATUS->mode_context_id);
		}

		gchar *author = g_strconcat(" ", _("Submitting author:"), " ",
			((GET_SCD->submittingAuthor != NULL) ?
			GET_SCD->submittingAuthor : ""), " ",
			_("(double click)"), NULL);

		gtk_statusbar_push(GTK_STATUSBAR(editor_status_mode_bar),
			EDT_STATUS->mode_context_id, author);

		g_free(author);
	}
#endif
}

static gboolean on_mode_status_bar_event_box_button_press
	(GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
	g_return_val_if_fail(widget != NULL, FALSE);
	g_return_val_if_fail(event != NULL, FALSE);

	switch (event->type) {
	case GDK_2BUTTON_PRESS:
		if (gtk_notebook_get_current_page
			(GTK_NOTEBOOK(editor_type_notebook)) == TAB_QUERY)
		{
			singit_config_show_author();
			return TRUE;
		}
	default:
		break;
	}

	return FALSE;
}

void singit_editor_show(void)
{
	GdkPixmap * icon = NULL;
	GdkBitmap * mask = NULL;
	GtkWidget * iconw = NULL;
	gchar *title;

	if (singit_editor_win) {
		if (singit_editor_is_realized())
			gdk_window_raise(singit_editor_win->window);
		return;
	}

	if (editor_status_ref() == NULL)
		{ return; }

	if (singit_status_ref() == NULL) {
		editor_status_unref(NULL);
		return;
	}

#ifdef CODEDEBUG
	DEBUG(DLV_EDITOR, ("editor_singit_main.c [singit_editor_show]\n"));
#endif

	gtk_signal_connect(GTK_OBJECT(EDT_STATUS->config), "update",
		GTK_SIGNAL_FUNC(singit_editor_config_update_event), NULL);

	build_check_dialogs();

	editor_tooltips = gtk_tooltips_new ();

	singit_editor_win = gtk_window_new(GTK_WINDOW_DIALOG);
	title = g_strconcat(SINGIT_VERSION_STRING, " - ", _("Lyrics editor"), NULL);
	gtk_window_set_title(GTK_WINDOW(singit_editor_win), title);
	g_free(title);
	gtk_window_set_policy(GTK_WINDOW(singit_editor_win), FALSE, TRUE, FALSE);
	gtk_signal_connect(GTK_OBJECT(singit_editor_win), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroyed), &singit_editor_win);
	gtk_signal_connect(GTK_OBJECT(singit_editor_win), "delete_event",
		GTK_SIGNAL_FUNC(singit_editor_delete_event), NULL);
	gtk_signal_connect(GTK_OBJECT(singit_editor_win), "key-press-event",
		GTK_SIGNAL_FUNC(singit_editor_main_keypress), NULL);
	gtk_container_border_width(GTK_CONTAINER(singit_editor_win), 2);
	gtk_widget_realize(singit_editor_win);

	editor_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(singit_editor_win), editor_vbox);
	gtk_widget_show(editor_vbox);

// *	Menu-Bar *
	editor_menubar_handle_box = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(editor_vbox), editor_menubar_handle_box, FALSE, FALSE, 2 );
	gtk_widget_show(editor_menubar_handle_box);

// *	Tool-Bar *
	editor_toolbar_handle_box = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(editor_vbox), editor_toolbar_handle_box, FALSE, FALSE, 2 );
	gtk_widget_show(editor_toolbar_handle_box);

	editor_toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
	gtk_container_set_border_width(GTK_CONTAINER(editor_toolbar), 1);
	gtk_toolbar_set_space_size(GTK_TOOLBAR(editor_toolbar), 5);
	gtk_container_add(GTK_CONTAINER(editor_toolbar_handle_box), editor_toolbar);
	gtk_widget_show(editor_toolbar);

// *	Menu-Bar
	editor_menubar = get_main_menu(singit_editor_win);
	gtk_container_add(GTK_CONTAINER(editor_menubar_handle_box), editor_menubar);
	gtk_widget_show(editor_menubar);

// *	Tool-Bar
	gtk_toolbar_set_tooltips(GTK_TOOLBAR(editor_toolbar),
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_item(editor_menubar_item_factory, "/Options/Show tooltips"))->active);

	icon = gdk_pixmap_create_from_xpm_d
		(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) filenew_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	new_file_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("New lyrics"),
		_("Creates an empty lyrics file"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(new_lyrics_event),
		NULL);

        gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) fileopen_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	load_file_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Load lyrics"),
		_("Loads a lyric file"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(open_lyric_file_event),
		NULL);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) openlyrics_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	get_current_song_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Load song lyrics"),
		_("Loads the lyric file form current song"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(lyrics_from_song_event),
		NULL);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) reload_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	reload_lyrics_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Reload lyrics"),
		_("Reloads the lyrics from source"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(reload_lyrics_event),
		NULL);

	gtk_toolbar_append_space(GTK_TOOLBAR(editor_toolbar));

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) guessfilename_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	guess_filename_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Guess filename"),
		_("Guesses a filename from current song"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(guess_filename_event),
		NULL);

	gtk_toolbar_append_space(GTK_TOOLBAR(editor_toolbar));

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) filesave_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	save_file_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Save lyrics"),
		_("Save the lyric file"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(save_lyric_file_event),
		NULL);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) filesaveas_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	save_file_as_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Save lyrics as"),
		_("Save the lyric file with new filename"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(save_as_lyric_file_event),
		NULL);

#ifdef HAVE_TAG_MANAGEMENT_SUPPORT
	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) saveasid3_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	save_file_as_id3_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Save lyrics as id3"),
		_("Save the lyric inside an id3v2.x.x tag"),
		"Private",
		iconw,
		NULL,
		NULL);
	gtk_widget_set_sensitive(save_file_as_id3_button, FALSE);
#endif

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) exporthtml_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	export_as_html_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Export HTML"),
		_("Export lyrics as HTML file"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(export_as_html_event),
		NULL);

	gtk_toolbar_append_space(GTK_TOOLBAR(editor_toolbar));
/*
	iconw = gtk_pixmap_new(icon, mask);
	sync_mode_toggle_button = gtk_toolbar_append_element(
		GTK_TOOLBAR(editor_toolbar),
		GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
		NULL,
		_("Sync lyric mode"),
		_("Toggle the editor mode to sync lyric"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC (toggle_sync_mode_event),
		editor_toolbar);
	gtk_widget_set_sensitive(sync_mode_toggle_button, FALSE);

	gtk_toolbar_append_space(GTK_TOOLBAR(editor_toolbar));
*/
	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) striphtml_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	strip_html_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Strip HTML"),
		_("Removes all HTML tags from the text"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(strip_html_tags_event),
		NULL);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) striptags_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	strip_timetags_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Strip timetags"),
		_("Removes all timetags from the text"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(strip_time_tags_event),
		NULL);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) align_left_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	nice_text_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Nice text"),
		_("Strips whitespaces / Upper starting letter"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(nice_text_event),
		NULL);

	gtk_toolbar_append_space(GTK_TOOLBAR(editor_toolbar));

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) autotime_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	auto_change_time_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Auto-Change Time"),
		_("Auto-Change Time of Selection"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(auto_change_time_event),
		NULL);
	gtk_widget_set_sensitive(auto_change_time_button, FALSE);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) time_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	change_time_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Change Time"),
		_("Change Time of Selection"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(change_time_event),
		NULL);
	gtk_widget_set_sensitive(change_time_button, FALSE);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) search_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	check_sync_consistency_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Check consistency"),
		_("Check consistency of syncronized lyrics"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(check_sync_consistency_event),
		NULL);

	gtk_toolbar_append_space(GTK_TOOLBAR(editor_toolbar));

#ifdef HAVE_TAG_MANAGEMENT_SUPPORT
	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) id3info_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	id3_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("ID3"),
		_("Show the id3 tag window"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(singit_tag_manager_show),
		NULL);
#endif

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) preferences_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	config_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Preferences"),
		_("Show the configuration window"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(singit_config_show),
		NULL);

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) help_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	help_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),
		_("Help"),
		_("Show the keybindings window"),
		"Private",
		iconw,
		GTK_SIGNAL_FUNC(editor_main_show_about_event),
		(gpointer) 1);

	gtk_toolbar_append_space(GTK_TOOLBAR(editor_toolbar));

	gdk_pixmap_unref(icon);
	icon = gdk_pixmap_create_from_xpm_d(singit_editor_win->window, &mask, &singit_editor_win->style->white, (gchar**) quit_xpm);
	iconw = gtk_pixmap_new(icon, mask);
	close_button = gtk_toolbar_append_item(
		GTK_TOOLBAR(editor_toolbar),   // * our toolbar *
		_("Close"),                    // * button label *
		_("Closes the editor window"), // * this button's tooltip *
		"Private",                     // * tooltip private info *
		iconw,                         // * icon widget *
		GTK_SIGNAL_FUNC(close_editor_event),  // * a signal *
		NULL);

	gdk_pixmap_unref(icon);

// * editor type notebook *
	editor_type_notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(editor_type_notebook), GTK_POS_TOP);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(editor_type_notebook), FALSE);
	gtk_notebook_set_homogeneous_tabs(GTK_NOTEBOOK(editor_type_notebook), TRUE);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(editor_type_notebook), TRUE);
	gtk_box_pack_start(GTK_BOX(editor_vbox), editor_type_notebook, TRUE, TRUE, 0);
	gtk_widget_show(editor_type_notebook);

	gtk_signal_connect_object(GTK_OBJECT(editor_type_notebook),
		"switch-page", GTK_SIGNAL_FUNC(singit_editor_switch_notebook_page),
		(gpointer) editor_type_notebook);
	gtk_signal_connect_after(GTK_OBJECT(editor_type_notebook),
		"switch-page", GTK_SIGNAL_FUNC(singit_editor_switch_notebook_page_after),
		(gpointer) editor_type_notebook);

	editor_synced_lyrics = editor_synced_lyrics_new();
	gtk_container_set_border_width(GTK_CONTAINER(editor_synced_lyrics), 2);
	gtk_widget_show(editor_synced_lyrics);
	editor_synced_lyrics_label = gtk_label_new(_(" Synced "));
	gtk_widget_show(editor_synced_lyrics_label);
	gtk_signal_connect(GTK_OBJECT(editor_synced_lyrics),
		"modified", GTK_SIGNAL_FUNC(singit_editor_view_modified), NULL);
	gtk_signal_connect(GTK_OBJECT(editor_synced_lyrics),
		"key-press-event",
		GTK_SIGNAL_FUNC(singit_editor_view_keypress), NULL);
	gtk_notebook_append_page(GTK_NOTEBOOK(editor_type_notebook),
		editor_synced_lyrics, editor_synced_lyrics_label);

	editor_clist_timestamps = editor_clist_timestamps_new();
	gtk_container_set_border_width(GTK_CONTAINER(editor_clist_timestamps), 2);
	gtk_widget_show(editor_clist_timestamps);
	editor_clist_timestamps_label = gtk_label_new(_(" Timestamps "));
	gtk_widget_show(editor_clist_timestamps_label);
	gtk_signal_connect(GTK_OBJECT(editor_clist_timestamps),
		"modified", GTK_SIGNAL_FUNC(singit_editor_view_modified), NULL);
	gtk_signal_connect(GTK_OBJECT(editor_clist_timestamps),
		"key-press-event",
		GTK_SIGNAL_FUNC(singit_editor_view_keypress), NULL);
	gtk_notebook_append_page(GTK_NOTEBOOK(editor_type_notebook),
		editor_clist_timestamps, editor_clist_timestamps_label);

	editor_plain_text = editor_plain_text_new();
	gtk_container_set_border_width(GTK_CONTAINER(editor_plain_text), 2);
	gtk_signal_connect_after(GTK_OBJECT(EDITOR_PLAIN_TEXT(editor_plain_text)->text),
		"move-cursor", GTK_SIGNAL_FUNC(editor_main_move_cursor_event), NULL);
	gtk_widget_show(editor_plain_text);
	editor_plain_text_label = gtk_label_new(_(" Plain text "));
	gtk_widget_show(editor_plain_text_label);
	gtk_signal_connect(GTK_OBJECT(editor_plain_text),
		"modified", GTK_SIGNAL_FUNC(singit_editor_view_modified), NULL);
	gtk_signal_connect(GTK_OBJECT(editor_plain_text),
		"key-press-event",
		GTK_SIGNAL_FUNC(singit_editor_view_keypress), NULL);
	gtk_notebook_append_page(GTK_NOTEBOOK(editor_type_notebook),
		editor_plain_text, editor_plain_text_label);

#ifdef HAVE_QUERY_DEPS
// * Query tab - just one query - we don't want to stress the servers *
	editor_query = editor_query_widget_new();
	gtk_container_set_border_width(GTK_CONTAINER(editor_query), 2);
	editor_query_widget_set_search_matches
		(EDITOR_QUERY_WIDGET(editor_query),
		GET_ECD->artistMatch, GET_ECD->titleMatch);
	gtk_signal_connect(GTK_OBJECT(editor_query), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroyed), &editor_query);
	gtk_widget_show(editor_query);

	// Create a server object
	LyrixRequestHTTP *http_data = lyrix_request_data_new();
	lyrix_request_data_set_server
		(http_data, "localhost", 80, "composant/search.php",
		"composant/getLyrics.php", "composant/add.php");
	lyrix_request_data_set_agent(http_data, PACKAGE, VERSION);
	if (editor_query_widget_add_server
		(EDITOR_QUERY_WIDGET(editor_query), http_data) == FALSE)
	{
		lyrix_request_data_free(http_data);
	}

	http_data = lyrix_request_data_new();
	lyrix_request_data_set_server
		(http_data, "lyrix.planet-work.com", 80, "composant/search.php",
		"composant/getLyrics.php", "composant/add.html");
	lyrix_request_data_set_agent(http_data, PACKAGE, VERSION);
	if (editor_query_widget_add_server
		(EDITOR_QUERY_WIDGET(editor_query), http_data) == FALSE)
	{
		lyrix_request_data_free(http_data);
	}

	editor_query_label = gtk_label_new(_(" Query "));
	gtk_widget_show(editor_query_label);
	gtk_signal_connect(GTK_OBJECT(editor_query),
		"key-press-event",
		GTK_SIGNAL_FUNC(singit_editor_view_keypress), NULL);
	gtk_signal_connect(GTK_OBJECT(editor_query),
		"set_text",
		GTK_SIGNAL_FUNC(singit_editor_query_set_text), NULL);
	gtk_notebook_append_page(GTK_NOTEBOOK(editor_type_notebook),
		editor_query, editor_query_label);
#endif

// * status bars *
	editor_status_hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(editor_vbox), editor_status_hbox, FALSE, FALSE, 0);
	gtk_widget_show(editor_status_hbox);

	editor_status_time_bar = gtk_statusbar_new();
	gtk_box_pack_start(GTK_BOX(editor_status_hbox), editor_status_time_bar, FALSE, FALSE, 0);
	gtk_widget_show(editor_status_time_bar);

	editor_mode_gtk_event_box = gtk_event_box_new ();
	gtk_box_pack_start(GTK_BOX(editor_status_hbox),
		editor_mode_gtk_event_box, TRUE, TRUE, 0);
	gtk_widget_show(editor_mode_gtk_event_box);

	gtk_signal_connect(GTK_OBJECT(editor_mode_gtk_event_box),
		"button_press_event",
		GTK_SIGNAL_FUNC(on_mode_status_bar_event_box_button_press),
		NULL);

	editor_status_mode_bar = gtk_statusbar_new();
	gtk_container_add (GTK_CONTAINER(editor_mode_gtk_event_box),
		editor_status_mode_bar);
	gtk_widget_show(editor_status_mode_bar);

	editor_status_modified_bar = gtk_statusbar_new();
	gtk_box_pack_start(GTK_BOX(editor_status_hbox), editor_status_modified_bar, FALSE, FALSE, 0);
	EDT_STATUS->modified_context_id =
		gtk_statusbar_get_context_id(GTK_STATUSBAR(editor_status_modified_bar), "modified");
	gtk_statusbar_push(GTK_STATUSBAR(editor_status_modified_bar), EDT_STATUS->modified_context_id,
		_(" modified "));
	EDT_STATUS->changed_spaces = g_strnfill(strlen(_(" modified ")), ' ');
	gtk_statusbar_push(GTK_STATUSBAR(editor_status_modified_bar), EDT_STATUS->modified_context_id,
		EDT_STATUS->changed_spaces);
	gtk_widget_show(editor_status_modified_bar);

	editor_status_consistency_bar = gtk_statusbar_new();
	gtk_box_pack_start(GTK_BOX(editor_status_hbox), editor_status_consistency_bar, FALSE, FALSE, 0);
	EDT_STATUS->valid_consistency_id =
		gtk_statusbar_get_context_id(GTK_STATUSBAR(editor_status_consistency_bar), "validation");
	gtk_statusbar_push(GTK_STATUSBAR(editor_status_consistency_bar),
		EDT_STATUS->valid_consistency_id, _(" Not checked "));
	gtk_widget_show(editor_status_consistency_bar);

	editor_song_gtk_event_box = gtk_event_box_new ();
	gtk_box_pack_start(GTK_BOX(editor_vbox), editor_song_gtk_event_box, FALSE, FALSE, 0);
	gtk_widget_show (editor_song_gtk_event_box);

	editor_status_filename_bar = gtk_statusbar_new();
	gtk_container_add (GTK_CONTAINER (editor_song_gtk_event_box), editor_status_filename_bar);
	gtk_widget_show(editor_status_filename_bar);

	set_status_filename(NULL, FALSE);

	EDT_STATUS->time_context_id =
		gtk_statusbar_get_context_id(GTK_STATUSBAR(editor_status_time_bar), "timestamps");
	gtk_statusbar_push(GTK_STATUSBAR(editor_status_time_bar), EDT_STATUS->time_context_id, " 00:00 ");

	EDT_STATUS->mode_context_id =
		gtk_statusbar_get_context_id(GTK_STATUSBAR(editor_status_mode_bar), "modes");

	singit_editor_set_editor_mode_status_bar(0);

	gtk_object_set_data (GTK_OBJECT (singit_editor_win), "tooltips", editor_tooltips);

	gtk_widget_show(singit_editor_win);

	gtk_notebook_set_page(GTK_NOTEBOOK(editor_type_notebook), TAB_TAGS);

	if ((GET_ECD->winPosX < 0) || (GET_ECD->winPosY < 0)) {
		gtk_window_set_position
			(GTK_WINDOW(singit_editor_win), GTK_WIN_POS_MOUSE);
	}
	else {
		gtk_widget_set_uposition(singit_editor_win,
			GET_ECD->winPosX, GET_ECD->winPosY);
	}

	singit_main_init(FALSE);
}
