#charset "us-ascii"

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library - extras: special-purpose object classes
 *   
 *   This module defines classes for specialized simulation objects.
 */

/* include the library header */
#include "adv3.h"


/* ------------------------------------------------------------------------ */
/*
 *   A "complex" container is an object that can have multiple kinds of
 *   contents simultaneously.  For example, a complex container could act
 *   as both a surface, so that some objects are sitting on top of it, and
 *   simultaneously as a container, with objects inside.
 *   
 *   The standard containment model only allows one kind of containment
 *   per container, because the nature of the containment is a feature of
 *   the container itself.  The complex container handles multiple
 *   simultaneous containment types by using one or more sub-containers:
 *   for example, if we want to be able to act as both a surface and a
 *   regular container, we use two sub-containers, one of class Surface
 *   and one of class Container, to hold the different types of contents.
 *   When we need to perform an operation specific to a certain
 *   containment type, we delegate the operation to the sub-container of
 *   the appropriate type.
 *   
 *   Note that the complex container itself treats its direct contents as
 *   components, so any component parts can be made direct contents of the
 *   complex container object.
 */
class ComplexContainer: Thing
    /*
     *   Our inner container, if any.  This is a "secret" object (in other
     *   words, it doesn't appear to players as a separate named object)
     *   that we use to store the contents that are meant to be within the
     *   complex container.  If this is to be used, it should be set to a
     *   Container object - the most convenient way to do this is by using
     *   the nested object syntax to define a ComplexComponent Container
     *   instance, like so:
     *   
     *   washingMachine: ComplexContainer
     *.    subContainer: ComplexComponent, Container { etc }
     *.  ;
     *   
     *   Note that we use the ComplexComponent class (as well as
     *   Container) for the sub-container object.  This makes the
     *   sub-container automatically use the name of its enclosing object
     *   in messages (in this case, the sub-container will use the same
     *   name as the washing machine).
     *   
     *   If this property is left as nil, then we don't have an inner
     *   container.  
     */
    subContainer = nil

    /*
     *   Our inner surface, if any.  This is a secret object like the
     *   inner container; this object acts as our surface. 
     */
    subSurface = nil

    /* 
     *   Show our status.  We'll show the status for each of our
     *   sub-objects, so that we list any contents of our sub-container or
     *   sub-surface along with our description. 
     */
    examineStatus()
    {
        /* inherit any default handling first */
        inherited();

        /* if we have a sub-container, show its status */
        if (subContainer != nil)
            subContainer.examineStatus();

        /* if we have a sub-surface, show its status */
        if (subSurface != nil)
            subSurface.examineStatus();
    }

    /* 
     *   route all commands that treat us as a container to our
     *   sub-container object 
     */
    dobjFor(Open) maybeRemapTo(subContainer != nil, Open, subContainer)
    dobjFor(Close) maybeRemapTo(subContainer != nil, Close, subContainer)
    dobjFor(LookIn) maybeRemapTo(subContainer != nil, LookIn, subContainer)
    iobjFor(PutIn) maybeRemapTo(subContainer != nil,
                                PutIn, DirectObject, subContainer)

    /* route commands that treat us as a surface to our sub-surface */
    iobjFor(PutOn) maybeRemapTo(subSurface != nil,
                                PutOn, DirectObject, subSurface)
;

/*
 *   A component object of a complex container.  This class can be used as
 *   a mix-in for sub-objects of a complex container (the subContainer or
 *   subSurface) defined as nested objects.
 *   
 *   This class is based on Component, which is suitable for complex
 *   container sub-objects because it makes them inseparable from the
 *   complex container.  It's also based on NameAsParent, which makes the
 *   object automatically use the same name (in messages) as the lexical
 *   parent object.  This is usually what one wants for a sub-object of a
 *   complex container, because it makes the sub-object essentially
 *   invisible to the user by referring to the sub-object in messages as
 *   though it were the complex container itself: "The washing machine
 *   contains...".
 *   
 *   This class also automatically initializes our location to our lexical
 *   parent, during the pre-initialization process.  Any of these that are
 *   dynamically created at run-time (using 'new') must have their
 *   locations set manually, because initializeLocation() won't be called
 *   automatically in those cases.  
 */
class ComplexComponent: Component, NameAsParent
    initializeLocation()
    {
        /* set our location to our lexical parent */
        location = lexicalParent;

        /* inherit default so we initialize our container's 'contents' list */
        inherited();
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   A "stretchy container."  This is a simple container subclass whose
 *   external bulk changes according to the bulks of the contents.  
 */
class StretchyContainer: Container
    /* 
     *   Our minimum bulk.  This is the minimum bulk we'll report, even
     *   when the aggregate bulks of our contents are below this limit. 
     */
    minBulk = 0

    /* get my total external bulk */
    getBulk()
    {
        local tot;

        /* start with my own intrinsic bulk */
        tot = bulk;

        /* add the bulk contribution from my contents */
        tot += getBulkForContents();

        /* return the total, but never less than the minimum */
        return tot >= minBulk ? tot : minBulk;
    }

    /*
     *   Calculate the contribution to my external bulk of my contents.
     *   The default for a stretchy container is to conform exactly to the
     *   contents, as though the container weren't present at all, hence
     *   we simply sum the bulks of our contents.  Subclasses can override
     *   this to define other aggregate bulk effects as needed.  
     */
    getBulkForContents()
    {
        local tot;
        
        /* sum the bulks of the items in our contents */
        tot = 0;
        foreach (local cur in contents)
            tot += cur.getBulk();

        /* return the total */
        return tot;
    }

    /*
     *   Check what happens when a new object is inserted into my
     *   contents.  This is called with the new object already tentatively
     *   added to my contents, so we can examine our current status to see
     *   if everything works.
     *   
     *   Since we can change our own size when a new item is added to our
     *   contents, we'll trigger a full bulk change check. 
     */
    checkBulkInserted(insertedObj)
    {
        /* 
         *   inherit the normal handling to ensure that the new object
         *   fits within this container 
         */
        inherited(insertedObj);

        /* 
         *   since we can change our own shape when items are added to our
         *   contents, trigger a full bulk check on myself 
         */
        checkBulkChange();
    }

    /* 
     *   Check a bulk change of one of my direct contents.  Since my own
     *   bulk changes whenever the bulk of one of my contents changes, we
     *   must propagate the bulk change of our contents as a change in our
     *   own bulk. 
     */
    checkBulkChangeWithin(changingObj)
    {
        /* 
         *   This might cause a change in my own bulk, since my bulk
         *   depends on the bulks of my contents.  When this is called,
         *   obj is already set to indicate its new bulk; since we
         *   calculate our own bulk by looking at our contents' bulks,
         *   this means that our own getBulk will now report the latest
         *   value including obj's new bulk. 
         */
        checkBulkChange();
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   "Bag of holding."  This is a mix-in that actively moves items from
 *   the holding actor's direct inventory into itself when space in the
 *   actor's hands becomes overcommitted.
 *   
 *   The bag of holding as a concept offers a solution to the conflict
 *   between the desire many authors have for at least a degree of realism
 *   in inventory management and the desire not to bother the player with
 *   the details thereof.  Manually juggling inventory is simply not very
 *   interesting, so many players are justifiably irritated if the game
 *   forces them to do very much of this.  It has become so widely agreed
 *   that inventory management is bad, in fact, that many games dispense
 *   with the problem entirely by making the player character's carrying
 *   capacity unlimited.  Many authors (and some players) dislike this
 *   approach, though, because of the ridiculously large inventories that
 *   players tend to accumulate in large games.  Enter the bag of holding:
 *   this is an item that can plausibly contain lots of stuff and that the
 *   player character would plausibly take everywhere, such as a large
 *   backpack or a trenchcoat with lots of pockets.  As the player needs
 *   space to hold items picked up, the game automatically moves objects
 *   into the bag of holding.  
 */
class BagOfHolding: object
    /*
     *   Get my bags of holding.  Since we are a bag of holding, we'll add
     *   ourselves to the vector, then we'll inherit the normal handling
     *   to pick up our contents. 
     */
    getBagsOfHolding(vec)
    {
        /* we're a bag of holding */
        vec.append(self);

        /* inherit the normal handling */
        inherited(vec);
    }

    /*
     *   Get my "affinity" for the given object.  This is an indication of
     *   how strongly this bag wants to contain the object.  The affinity
     *   is a number in arbitrary units; higher numbers indicate stronger
     *   affinities.  An affinity of zero means that the bag does not want
     *   to contain the object at all.
     *   
     *   The purpose of the affinity is to support specialized holders
     *   that are designed to hold only specific types of objects, and
     *   allow these specialized holders to implicitly gather their
     *   specific objects.  For example, a key ring might only hold keys,
     *   so it would have a high affinity for keys and a zero affinity for
     *   everything else.  A lunchbox might have a higher affinity for
     *   things like sandwiches than for anything else, but might be
     *   willing to serve as a general container for other small items as
     *   well.
     *   
     *   The units of affinity are arbitrary, but the library uses the
     *   following values for its own classes:
     *   
     *   0 - no affinity at all; the bag cannot hold the object
     *   
     *   50 - willing to hold the object, but not of the preferred type
     *   
     *   100 - default affinity; willing and able to hold the object, but
     *   just as willing to hold most other things
     *   
     *   200 - special affinity; this object is of a type that we
     *   especially want to hold
     *   
     *   We intentionally space these loosely so that games can use
     *   intermediate levels if desired.
     *   
     *   When we are looking for bags of holding to consolidate an actor's
     *   directly-held inventory, note that we always move the object with
     *   the highest bag-to-object affinity out of all of the objects
     *   under consideration.  So, if you want to give a particular kind
     *   of bag priority so that the library uses that bag before any
     *   other bag, make this routine return a higher affinity for the
     *   bag's objects than any other bags do.
     *   
     *   By default, we'll return the default affinity of 100.
     *   Specialized bags that don't hold all types of objects must
     *   override this to return zero for objects they can't hold.  
     */
    affinityFor(obj)
    {
        /* 
         *   my affinity for myself is zero, for obvious reasons; for
         *   everything else, use the default affinity 
         */
        return (obj == self ? 0 : 100);
    }
;


/*
 *   Bag Affinity Info - this class keeps track of the affinity of a bag
 *   of holding for an object it might contain.  We use this class in
 *   building bag affinity lists.  
 */
class BagAffinityInfo: object
    construct(obj, bulk, aff, bag)
    {
        /* save our parameters */
        obj_ = obj;
        bulk_ = bulk;
        aff_ = aff;
        bag_ = bag;
    }

    /*
     *   Compare this item to another item, for affinity ranking purposes.
     *   Returns positive if I should rank higher than the other item,
     *   zero if we have equal ranking, negative if I rank lower than the
     *   other item.  
     */
    compareAffinityTo(other)
    {
        /* 
         *   if this object is the indirect object of 'take from', treat
         *   it as having the lowest ranking 
         */
        if (gActionIs(TakeFrom) && gIobj == obj_)
            return -1;

        /* if we have different affinities, sort according to affinity */
        if (aff_ != other.aff_)
            return aff_ - other.aff_;

        /* we have the same affinity, so put the higher bulk item first */
        if (bulk_ != other.bulk_)
            return bulk_ - other.bulk_;

        /* 
         *   We have the same affinity and same bulk; rank according to
         *   how recently the items were picked up.  Put away the oldest
         *   items first, so the lower holding (older) index has the
         *   higher ranking.  (Note that because lower holding index is
         *   the higher ranking, we return the negative of the holding
         *   index comparison.)  
         */
        return other.obj_.holdingIndex - obj_.holdingIndex;
    }

    /* the object the bag wants to contain */
    obj_ = nil

    /* the object's bulk */
    bulk_ = nil

    /* the bag that wants to contain the object */
    bag_ = nil

    /* affinity of the bag for the object */
    aff_ = nil
;


/* ------------------------------------------------------------------------ */
/*
 *   Keyring - a place to stash keys
 *   
 *   Keyrings have some special properties:
 *   
 *   - A keyring is a bag of holding with special affinity for keys.
 *   
 *   - A keyring can only contain keys.
 *   
 *   - Keys are considered to be on the outside of the ring, so a key can
 *   be used even if attached to the keyring (in other words, if the ring
 *   itself is held, a key attached to the ring is also considered held).
 *   
 *   - If an actor in possession of a keyring executes an "unlock" command
 *   without specifying what key to use, we will automatically test each
 *   key on the ring to find the one that works.
 *   
 *   - When an actor takes one of our keys, and the actor is in possession
 *   of this keyring, we'll automatically attach the key to the keyring
 *   immediately.  
 */
class Keyring: BagOfHolding, Thing
    /* lister for showing our contents in-line as part of a list entry */
    inlineContentsLister = keyringListingContentsLister

    /* lister for showing our contents as part of "examine" */
    examineLister = keyringExamineContentsLister

    /* 
     *   Determine if a key fits our keyring.  By default, we will accept
     *   any object of class Key.  However, subclasses might want to
     *   override this to associate particular keys with particular
     *   keyrings rather than having a single generic keyring.  To allow
     *   only particular keys onto this keyring, override this routine to
     *   return true only for the desired keys.  
     */
    isMyKey(key)
    {
        /* accept any object of class Key */
        return key.ofKind(Key);
    }

    /* we have high affinity for our keys */
    affinityFor(obj)
    {
        /* 
         *   if the object is one of my keys, we have high affinity;
         *   otherwise we don't accept it at all 
         */
        if (isMyKey(obj))
            return 200;
        else
            return 0;
    }

    /* implicitly put a key on the keyring */
    tryPuttingObjInBag(target)
    {
        /* we're a container, so use "put in" to get the object */
        return tryImplicitActionMsg(&announceMoveToBag, PutOn, target, self);
    }

    /* on examining the keyring, show the keys */
    desc()
    {
        /* list our keys through our "examine" lister */
        examineLister.showList(
            gActor, self, contents, 0, 0, gActor.visibleInfoTable(), nil);
    }

    /* allow putting a key on the keyring */
    iobjFor(PutOn)
    {
        /* we can only put keys on keyrings */
        verify()
        {
            /* we'll only allow our own keys to be attached */
            if (gDobj == nil)
            {
                /* 
                 *   we don't know the actual direct object yet, but we
                 *   can at least check to see if any of the possible
                 *   dobj's is my kind of key 
                 */
                if (gTentativeDobj.indexWhich({x: isMyKey(x.obj_)}) == nil)
                    illogical(&objNotForKeyring);
            }
            else if (!isMyKey(gDobj))
            {
                /* the dobj isn't a valid key for this keyring */
                illogical(&objNotForKeyring);
            }
        }
        
        /* put a key on me */
        action()
        {
            /* move the key into me */
            gDobj.moveInto(self);
            
            /* show the default "put on" response */
            defaultReport(&okayPutOn);
        }
    }

    /* treat "attach x to keyring" as "put x on keyring" */
    iobjFor(AttachTo) remapTo(PutOn, DirectObject, self)

    /* treat "detach x from keyring" as "take x from keyring" */
    iobjFor(DetachFrom) remapTo(TakeFrom, DirectObject, self)

    /* receive notification before an action */
    beforeAction()
    {
        /*
         *   Note whether or not we want to consider moving the direct
         *   object to the keyring after a "take" command.  We will
         *   consider doing so only if the direct object isn't already on
         *   the keyring - if it is, we don't want to move it back right
         *   after removing it, obviously.
         *   
         *   Skip the implicit keyring attachment if the current command
         *   is implicit, because they must be doing something that
         *   requires holding the object, in which case taking it is
         *   incidental.  It could be actively annoying to attach the
         *   object to the keyring in such cases - for example, if the
         *   command is "put key on keyring," attaching it as part of the
         *   implicit action would render the explicit command redundant
         *   and cause it to fail.  
         */
        moveAfterTake = (!gAction.isImplicit
                         && gDobj != nil
                         && !gDobj.isDirectlyIn(self));
    }

    /* flag: consider moving to keyring after this "take" action */
    moveAfterTake = nil

    /* receive notification after an action */
    afterAction()
    {
        /*
         *   If the command was "take", and the direct object was a key,
         *   and the actor involved is holding the keyring and can touch
         *   it, and the command succeeded in moving the key to the
         *   actor's direct inventory, then move the key onto the keyring.
         *   Only consider this if we decided to during the "before"
         *   notification.  
         */
        if (moveAfterTake
            && gActionIs(Take)
            && isMyKey(gDobj)
            && isIn(gActor)
            && gActor.canTouch(self)
            && gDobj.isDirectlyIn(gActor))
        {
            /* move the key to me */
            gDobj.moveInto(self);

            /* 
             *   Mention what we did.  If the only report for this action
             *   so far is the default 'take' response, then use the
             *   combined taken-and-attached message.  Otherwise, append
             *   our 'attached' message, which is suitable to use after
             *   other messages. 
             */
            if (gTranscript.currentActionHasReport(
                {x: (x.ofKind(CommandReportMessage)
                     && x.messageProp_ != &okayTake)}))
            {
                /* 
                 *   we have a non-default message already, so add our
                 *   message indicating that we added the key to the
                 *   keyring 
                 */
                reportAfter(&movedKeyToKeyring, self);
            }
            else
            {
                /* use the combination taken-and-attached message */
                mainReport(&takenAndMovedToKeyring, self);
            }
        }
    }

    /* find among our keys a key that works the direct object */
    findWorkingKey(lock)
    {
        /* try each key on the keyring */
        foreach (local key in contents)
        {
            /* 
             *   if this is the key that unlocks the lock, replace the
             *   command with 'unlock lock with key' 
             */
            if (lock.keyFitsLock(key))
            {
                /* note that we tried keys and found the right one */
                extraReport(foundKeyMsg, self, key);
                
                /* return the key */
                return key;
            }
        }

        /* we didn't find the right key - indicate failure */
        reportFailure(foundNoKeyMsg, self);
        return nil;
    }

    /*
     *   Append my directly-held contents to a vector when I'm directly
     *   held.  We consider all of the keys on the keyring to be
     *   effectively at the same containment level as the keyring, so if
     *   the keyring is held, so are its attached keys.  
     */
    appendHeldContents(vec)
    {
        /* append all of our contents, since they're held when we are */
        vec.appendUnique(contents);
    }

    /* 
     *   message to display to indicate that we found/failed to find a
     *   working key on the keyring 
     */
    foundKeyMsg = &foundKeyOnKeyring
    foundNoKeyMsg = &foundNoKeyOnKeyring

    /*
     *   Announce myself as a default object for an action.
     *   
     *   Do not announce a keyring as a default for "lock with" or "unlock
     *   with".  Although we can use a keyring as the indirect object of a
     *   lock/unlock command, we don't actually do the unlocking with the
     *   keyring; so, when we're chosen as the default, suppress the
     *   announcement, since it would imply that we're being used to lock
     *   or unlock something.  
     */
    announceDefaultObject(whichObj, action, resolvedAllObjects)
    {
        /* if it's not a lock-with or unlock-with, use the default message */
        if (!action.ofKind(LockWithAction)
            && !action.ofKind(UnlockWithAction))
        {
            /* for anything but our special cases, use the default handling */
            return inherited(whichObj, action, resolvedAllObjects);
        }

        /* use no announcement */
        return '';
    }
    
    /* 
     *   Allow locking or unlocking an object with a keyring.  This will
     *   automatically try each key on the keyring to see if it fits the
     *   lock. 
     */
    iobjFor(LockWith)
    {
        verify()
        {
            /* if we don't have any keys, we're not locking anything */
            if (contents.length() == 0)
                illogical(&cannotLockWith);

            /* 
             *   if we know the direct object, and we don't have any keys
             *   that are plausible for the direct object, we're an
             *   unlikely match 
             */
            if (gDobj != nil)
            {
                local foundPlausibleKey;
                
                /* 
                 *   try each of my keys to see if it's plausible for the
                 *   direct object 
                 */
                foundPlausibleKey = nil;
                foreach (local cur in contents)
                {
                    /* 
                     *   if this is a plausible key, note that we have at
                     *   least one plausible key 
                     */
                    if (gDobj.keyIsPlausible(cur))
                    {
                        /* note that we found a plausible key */
                        foundPlausibleKey = true;

                        /* no need to look any further - one is good enough */
                        break;
                    }
                }

                /* 
                 *   If we didn't find a plausible key, we're an unlikely
                 *   match.
                 *   
                 *   If we did find a plausible key, increase the
                 *   likelihood that this is the indirect object so that
                 *   it's greater than the likelihood for any random key
                 *   that's plausible for the lock (which has the default
                 *   likelihood of 100), but less than the likelihood of
                 *   the known good key (which is 150).  This will cause a
                 *   keyring to be taken as a default over any ordinary
                 *   key, but will cause the correct key to override the
                 *   keyring as the default if the correct key is known to
                 *   the player already.  
                 */
                if (foundPlausibleKey)
                    logicalRank(140, 'keyring with plausible key');
                else
                    logicalRank(50, 'no plausible key');
            }
        }

        action()
        {
            local key;

            /* 
             *   Try finding a working key.  If we find one, replace the
             *   command with 'lock <lock> with <key>, so that we have the
             *   full effect of the 'lock with' command using the key
             *   itself. 
             */
            if ((key = findWorkingKey(gDobj)) != nil)
                replaceAction(LockWith, gDobj, key);
        }
    }

    iobjFor(UnlockWith)
    {
        /* verify the same as for LockWith */
        verify()
        {
            /* if we don't have any keys, we're not unlocking anything */
            if (contents.length() == 0)
                illogical(&cannotUnlockWith);
            else
                verifyIobjLockWith();
        }

        action()
        {
            local key;

            /* 
             *   if we can find a working key, run an 'unlock with' action
             *   using the key 
             */
            if ((key = findWorkingKey(gDobj)) != nil)
                replaceAction(UnlockWith, gDobj, key);
        }
    }
;

/*
 *   Key - this is an object that can be used to unlock things, and which
 *   can be stored on a keyring.  The key that unlocks a lock is
 *   identified with a property on the lock, not on the key.
 */
class Key: Thing
    /*
     *   A key on a keyring that is being held by an actor is considered
     *   to be held by the actor, since the key does not have to be
     *   removed from the keyring in order to be manipulated as though it
     *   were directly held.  
     */
    isHeldBy(actor)
    {
        /* 
         *   if I'm on a keyring, I'm being held if the keyring is being
         *   held; otherwise, use the default definition 
         */
        if (location != nil && location.ofKind(Keyring))
            return location.isHeldBy(actor);
        else
            return inherited(actor);
    }

    /*
     *   Try making the current command's actor hold me.  If we're on a
     *   keyring, we'll simply try to make the keyring itself held, rather
     *   than taking the key off the keyring; otherwise, we'll inherit the
     *   default behavior to make ourselves held.  
     */
    tryHolding()
    {
        if (location != nil && location.ofKind(Keyring))
            return location.tryHolding();
        else
            return inherited();
    }

    /* -------------------------------------------------------------------- */
    /*
     *   Action processing 
     */

    /* treat "detach key" as "take key" if it's on a keyring */
    dobjFor(Detach)
    {
        verify()
        {
            /* if I'm not on a keyring, there's nothing to detach from */
            if (location == nil || !location.ofKind(Keyring))
                illogical(&keyNotDetachable);
        }
        remap()
        {
            /* if I'm on a keyring, remap to "take self" */
            if (location != nil && location.ofKind(Keyring))
                return [TakeAction, self];
            else
                return inherited();
        }
    }

    /* "lock with" */
    iobjFor(LockWith)
    {
        verify()
        {
            /* 
             *   if we know the direct object is a LockableWithKey, we can
             *   perform some additional checks on the likelihood of this
             *   key being the intended key for the lock 
             */
            if (gDobj != nil
                && gDobj.ofKind(LockableWithKey))
            {
                /*
                 *   If the player should know that we're the key for the
                 *   lock, boost our likelihood so that we'll be picked
                 *   out automatically from an ambiguous set of keys.  
                 */
                if (gDobj.isKeyKnown(self))
                    logicalRank(150, 'known key');

                /* 
                 *   if this isn't a plausible key for the lockable, it's
                 *   unlikely that this is a match 
                 */
                if (!gDobj.keyIsPlausible(self))
                    illogical(keyNotPlausibleMsg);
            }
        }
    }

    /* 
     *   the message to use when the key is obviously not plausible for a
     *   given lock 
     */
    keyNotPlausibleMsg = &keyDoesNotFitLock

    /* "unlock with" */
    iobjFor(UnlockWith)
    {
        verify()
        {
            /* use the same key selection we use for "lock with" */
            verifyIobjLockWith();
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   A Dispenser is a container for a special type of item, such as a book
 *   of matches or a box of candy.
 */
class Dispenser: Container
    /* 
     *   Can we return one of our items to the dispenser once the item is
     *   dispensed?  Books of matches wouldn't generally allow this, since
     *   a match must be torn out to be removed, but simple box dispensers
     *   probably would.  By default, we won't allow returning an item
     *   once dispensed.  
     */
    canReturnItem = nil

    /* 
     *   Is the item one of the types of items we dispense?  Normally, we
     *   dispense identical items, so our default implementation simply
     *   determines if the item is an instance of our dispensable class.
     *   If the dispenser can hand out items of multiple, unrelated
     *   classes, this can be overridden to use a different means of
     *   identifying the dispensed items.  
     */
    isMyItem(obj) { return obj.ofKind(myItemClass); }

    /*
     *   The class of items we dispense.  This is used by the default
     *   implementation of isMyItem(), so subclasses that inherit that
     *   implementation should provide the appropriate base class here.  
     */
    myItemClass = Dispensable

    /* "put in" indirect object handler */
    iobjFor(PutIn)
    {
        verify()
        {
            /* if we know the direct object, consider it further */
            if (gDobj != nil)
            {
                /* if we don't allow returning our items, don't allow it */
                if (!canReturnItem && isMyItem(gDobj))
                    illogical(&cannotReturnToDispenser);

                /* if it's not my dispensed item, it can't go in here */
                if (!isMyItem(gDobj))
                    illogical(&cannotPutInDispenser);
            }

            /* inherit default handling */
            inherited();
        }
    }
;

/*
 *   A Dispensable is an item that comes from a Dispenser.  This is in
 *   most respects an ordinary item; the only special thing about it is
 *   that if we're still in our dispenser, we're an unlikely match for any
 *   command except "take" and the like.  
 */
class Dispensable: Thing
    /* 
     *   My dispenser.  This is usually my initial location, so by default
     *   we'll pre-initialize this to our location. 
     */
    myDispenser = nil

    /* pre-initialization */
    initializeThing()
    {
        /* inherit the default initialization */
        inherited();

        /* 
         *   We're usually in our dispenser initially, so assume that our
         *   dispenser is simply our initial location.  If myDispenser is
         *   overridden in a subclass, don't overwrite the inherited
         *   value.  
         */
        if (propType(&myDispenser) == TypeNil)
            myDispenser = location;
    }

    dobjFor(All)
    {
        verify()
        {
            /* 
             *   If we're in our dispenser, and the command isn't "take"
             *   or "take from", reduce our disambiguation likelihood -
             *   it's more likely that the actor is referring to another
             *   equivalent item that they've already removed from the
             *   dispenser.  
             */
            if (isIn(myDispenser)
                && !gActionIs(Take) && !gActionIs(TakeFrom))
            {
                /* we're in our dispenser - reduce the likelihood */
                logicalRank(60, 'in dispenser');
            }
        }
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   A Matchbook is a special dispenser for matches. 
 */
class Matchbook: Openable, Dispenser
    /* we cannot return a match to a matchbook */
    canReturnItem = nil

    /* 
     *   we dispense matches (subclasses can override this if they want to
     *   dispense a specialized match subclass) 
     */
    myItemClass = Matchstick

    /*
     *   Act as a collective for any items within me.  This will have no
     *   effect unless we also have a plural name that matches that of the
     *   contained items.
     *   
     *   It is usually desirable for a matchbook to act as a collective
     *   for the contained items, so that a command like "take matches"
     *   will be taken to apply to the matchbook rather than the
     *   individual matches.  
     */
    isCollectiveFor(obj) { return obj.isIn(self); }

    /*
     *   Append my directly-held contents to a vector when I'm directly
     *   held.  When the matchbook is open, append our matches, because we
     *   consider the matches to be effectively attached to the matchbook
     *   (rather than contained within it).  
     */
    appendHeldContents(vec)
    {
        /* if we're open, append our contents */
        if (isOpen)
            vec.appendUnique(contents);
    }
;

/*
 *   A FireSource is an object that can set another object on fire.  This
 *   is a mix-in class that can be used with other classes.  
 */
class FireSource: object
    /* 
     *   We can use a fire source to light another object, provided the
     *   fire source is itself burning.  We don't provide any action
     *   handling - we leave that to the direct object.  
     */
    iobjFor(BurnWith)
    {
        preCond = [objHeld, objBurning]
        verify() { }
    }
;

/*
 *   A Matchstick is a self-igniting match from a matchbook.  (We use this
 *   lengthy name rather than simply "Match" because the latter is too
 *   generic, and could be taken by a casual reader for an object
 *   representing a successful search result or the like.)  
 */
class Matchstick: FireSource, LightSource
    /* matches have fairly feeble light */
    brightnessOn = 2

    /* not lit initially */
    isLit = nil

    /* amount of time we burn, in turns */
    burnLength = 2

    /* default long description describes burning status */
    desc()
    {
        if (isLit)
            libMessages.litMatchDesc(self);
        else
            libMessages.unlitMatchDesc(self);
    }

    /* get our state */
    getState = (isLit ? matchStateLit : matchStateUnlit)

    /* get a list of all states */
    allStates = [matchStateLit, matchStateUnlit]
    
    /* "burn" action */
    dobjFor(Burn)
    {
        preCond = [objHeld]
        verify()
        {
            /* can't light a match that's already burning */
            if (isLit)
                illogicalNow(&alreadyBurning);
        }
        action()
        {
            local t;
            
            /* describe it */
            defaultReport(&okayBurnMatch);

            /* make myself lit */
            makeLit(true);

            /* get our default burn length */
            t = burnLength;

            /* 
             *   if this is an implicit command, reduce the burn length by
             *   one turn - this ensures that the player can't
             *   artificially extend the match's useful life by doing
             *   something that implicitly lights the match 
             */
            if (gAction.isImplicit)
                --t;

            /* start our burn-out timer going */
            new SenseFuse(self, &matchBurnedOut, t, self, sight);
        }
    }

    /* "extinguish" */
    dobjFor(Extinguish)
    {
        verify()
        {
            /* can't extinguish a match that isn't burning */
            if (!isLit)
                illogicalNow(&matchNotLit);
        }
        action()
        {
            /* describe the match going out */
            defaultReport(&okayExtinguishMatch);

            /* no longer lit */
            makeLit(nil);

            /* remove the match from the game */
            moveInto(nil);
        }
    }

    /* fuse handler for burning out */
    matchBurnedOut()
    {
        /* 
         *   if I'm not still burning, I must have been extinguished
         *   explicitly already, so there's nothing to do 
         */
        if (!isLit)
            return;
        
        /* make sure we separate any output from other commands */
        "<.p>";

        /* report that we're done burning */
        libMessages.matchBurnedOut(self);

        /* 
         *   remove myself from the game (for simplicity, a match simply
         *   disappears when it's done burning) 
         */
        moveInto(nil);
    }

    /* matches usually come in bunches of equivalents */
    isEquivalent = true
;

/*
 *   A candle is an item that can be set on fire for a controlled burn.
 *   Although we call this a candle, this class can be used for other
 *   types of fuel burners, such as torches and oil lanterns.  
 */
class Candle: LightSource
    /* provide a bright light by default */
    brightnessOn = 3

    /* not lit initially */
    isLit = nil

    /* 
     *   Our "fuel" level - this is the number of turns that we can
     *   continue to burn.  On each turn while we're lit, we'll reduce the
     *   fuel level by one, and we'll automatically extinguish ourselves
     *   when this reaches zero.
     *   
     *   For a candle that burns forever, simply set this to nil, which
     *   indicates that the candle doesn't use fuel at all but can simply
     *   burn indefinitely.  
     */
    fuelLevel = 20

    /* 
     *   The message we display when we try to light the candle and we're
     *   out of fuel.  This message can be overridden by subclasses that
     *   don't fit the default message.  
     */
    outOfFuelMsg = &candleOutOfFuel

    /* the message we display when we successfully light the candle */
    okayBurnMsg = &okayBurnCandle

    /* show a message when the candle runs out fuel while burning */
    sayBurnedOut()
    {
        /* by default, show our standard library message */
        libMessages.candleBurnedOut(self);
    }

    /* 
     *   Determine if I can be lit with the specific indirect object.  By
     *   default, we'll allow any object to light us if the object passes
     *   the normal checks applied by its own iobjFor(BurnWith) handlers.
     *   This can be overridden if we can only be lit with specific
     *   sources of fire; for example, a furnace with a deeply-recessed
     *   burner could refuse to be lit by anything but particular long
     *   matches, or a particular type of fuel could refuse to be lit
     *   except by certain especially hot flames.  
     */
    canLightWith(obj) { return true; }

    /* 
     *   Default long description describes burning status.  In most
     *   cases, this should be overridden to provide more details, such as
     *   information on our fuel level. 
     */
    desc()
    {
        if (isLit)
            libMessages.litCandleDesc(self);
        else
            inherited();
    }

    /* light or extinguish */
    makeLit(lit)
    {
        /* inherit the default handling */
        inherited(lit);

        /* if we're lit, activate our daemon; otherwise, stop our daemon */
        if (isLit)
        {
            /* start our burn daemon going */
            burnDaemonObj =
                new SenseDaemon(self, &burnDaemon, 1, self, sight);
        }
        else
        {
            /* stop our daemon */
            eventManager.removeEvent(burnDaemonObj);

            /* forget out daemon */
            burnDaemonObj = nil;
        }
    }

    /* burn daemon - this is called on each turn while we're burning */
    burnDaemon()
    {
        /* if we use fuel, consume one increment of fuel for this turn */
        if (fuelLevel != nil)
        {
            /* 
             *   If our fuel level has reached zero, stop burning.  Note
             *   that the daemon is called on the first turn after we
             *   start burning, so we must go through a turn with the fuel
             *   level at zero before we stop burning.  
             */
            if (fuelLevel == 0)
            {
                /* extinguish the candle */
                makeLit(nil);

                /* make sure we separate any output from other commands */
                "<.p>";

                /* mention that the candle goes out */
                sayBurnedOut();
            }
            else
            {
                /* reduce our fuel level by one */
                --fuelLevel;
            }
        }
    }

    /* our daemon object, valid while we're burning */
    burnDaemonObj = nil

    /* "burn with" action */
    dobjFor(BurnWith)
    {
        preCond = [touchObj]
        verify()
        {
            /* can't light it if it's already lit */
            if (isLit)
                illogicalNow(&alreadyBurning);
        }
        check()
        {
            /* 
             *   make sure the object being used to light us is a valid
             *   source of fire for us 
             */
            if (!canLightWith(obj))
            {
                reportFailure(&cannotBurnDobjWith);
                exit;
            }
        }
        action()
        {
            /* if the fuel level is zero, we can't light */
            if (fuelLevel == 0)
            {
                /* we can't light a candle with no fuel */
                defaultReport(outOfFuelMsg);
            }
            else
            {
                /* describe it */
                defaultReport(okayBurnMsg);
                
                /* make myself lit */
                makeLit(true);
            }
        }
    }

    /* "extinguish" */
    dobjFor(Extinguish)
    {
        verify()
        {
            /* can't extinguish a match that isn't burning */
            if (!isLit)
                illogicalNow(&candleNotLit);
        }
        action()
        {
            /* describe the match going out */
            defaultReport(&okayExtinguishCandle);

            /* no longer lit */
            makeLit(nil);
        }
    }
;


