/* $Header: d:/cvsroot/tads/tads3/TCMAIN.H,v 1.3 1999/07/11 00:46:53 MJRoberts Exp $ */

/* Copyright (c) 1999, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
Name
  tcmain.h - TADS 3 Compiler - main compiler driver
Function
  
Notes
  
Modified
  04/22/99 MJRoberts  - Creation
*/

#ifndef TCMAIN_H
#define TCMAIN_H

#include <stdarg.h>

#include "t3std.h"
#include "tcerr.h"

class CTcMain
{
public:
    /* Initialize the compiler.  Creates all global objects. */
    static void init(class CTcHostIfc *hostifc,
                     class CResLoader *res_loader,
                     const char *default_charset);

    /* Terminate the compiler.  Deletes all global objects. */
    static void terminate();

    /* initialize/terminate the error subsystem */
    static void tc_err_init(size_t param_stack_size,
                            class CResLoader *res_loader);
    static void tc_err_term();
    
    /* log an error - varargs arguments - static routine */
    static void S_log_error(class CTcTokFileDesc *linedesc, long linenum,
                            int *err_counter, int *warn_counter,
                            int verbose, int show_warnings, int pedantic,
                            int test_report_mode,
                            tc_severity_t severity, int err, ...)
    {
        va_list args;

        /* pass through to our va_list-style handler */
        va_start(args, err);
        S_v_log_error(linedesc, linenum, err_counter, warn_counter, 0, 0,
                      verbose, show_warnings, pedantic, test_report_mode,
                      severity, err, 0, args);
        va_end(args);
    }

    /* log an error - arguments from a CVmException object */
    static void S_log_error(class CTcTokFileDesc *linedesc, long linenum,
                            int *err_counter, int *warn_counter,
                            int verbose, int show_warnings, int pedantic,
                            int test_report_mode,
                            tc_severity_t severity,
                            struct CVmException *exc);

    /* set verbosity for error messages */
    void set_verbosity(int verbose) { verbose_ = verbose; }

    /* turn all warnings on or off */
    void set_warnings(int show) { show_warnings_ = show; }

    /* turn pedantic warnings on or off */
    void set_pedantic(int show) { pedantic_ = show; }

    /* get/set test reporting mode */
    int get_test_report_mode() const { return test_report_mode_; }
    void set_test_report_mode(int flag) { test_report_mode_ = flag; }

    /* log an error - va_list-style arguments - static routine */
    static void S_v_log_error(class CTcTokFileDesc *linedesc, long linenum,
                              int *err_counter, int *warn_counter,
                              int *first_error, int *first_warning,
                              int verbose, int show_warnings, int pedantic,
                              int test_report_mode,
                              tc_severity_t severity, int err,
                              struct CVmException *exc, va_list args);

    /* log an error - varargs */
    void log_error(class CTcTokFileDesc *linedesc, long linenum,
                   tc_severity_t severity, int err, ...)
    {
        va_list args;

        /* pass through to our va_list-style handler */
        va_start(args, err);
        S_v_log_error(linedesc, linenum, &error_count_, &warning_count_,
                      &first_error_, &first_warning_,
                      verbose_, show_warnings_, pedantic_, test_report_mode_,
                      severity, err, 0, args);
        va_end(args);

        /* if we've exceeded the maximum error limit, throw a fatal error */
        check_error_limit();
    }

    void v_log_error(class CTcTokFileDesc *linedesc, long linenum,
                     tc_severity_t severity, int err, va_list args)
    {
        /* call our static routine */
        S_v_log_error(linedesc, linenum, &error_count_, &warning_count_,
                      &first_error_, &first_warning_,
                      verbose_, show_warnings_, pedantic_, test_report_mode_,
                      severity, err, 0, args);

        /* if we've exceeded the maximum error limit, throw a fatal error */
        check_error_limit();
    }

    /* get the error/warning count */
    int get_error_count() const { return error_count_; }
    int get_warning_count() const { return warning_count_; }

    /* reset the error and warning counters */
    void reset_error_counts()
    {
        /* clear the counters */
        error_count_ = 0;
        warning_count_ = 0;

        /* clear the error/warning memories */
        first_error_ = 0;
        first_warning_ = 0;
    }

    /* 
     *   get the first compilation error/warning code - we keep track of
     *   these for times when we have limited error reporting capabilities
     *   and can only report a single error, such as when we're compiling
     *   an expression in the debugger 
     */
    int get_first_error() const { return first_error_; }
    int get_first_warning() const { return first_warning_; }

    /* 
     *   check the error count against the error limit, and throw a fatal
     *   error if we've exceeded it 
     */
    void check_error_limit();

    /* get the console output character mapper */
    static class CCharmapToLocal *get_console_mapper()
        { return console_mapper_; }

private:
    CTcMain(class CResLoader *res_loader, const char *default_charset);
    ~CTcMain();

    /* error and warning count */
    int error_count_;
    int warning_count_;

    /* first error/warning code we've encountered */
    int first_error_;
    int first_warning_;

    /*
     *   Maximum error limit - if we encounter more than this many errors
     *   in a single compilation unit, we'll abort the compilation with a
     *   fatal error.  This at least limits the amount of garbage we'll
     *   display if we run into a really bad cascading parsing error
     *   situation where we just can't resynchronize.  
     */
    int max_error_count_;

    /* 
     *   error message verbosity - if this is true, we'll display verbose
     *   error messages, otherwise terse messages 
     */
    uint verbose_ : 1;

    /*
     *   Warning flag - if this is true, we'll display warnings; we'll
     *   suppress all warnings if this is false. 
     */
    uint show_warnings_ : 1;

    /* 
     *   Pedantic flag - if this is true, we'll display all warnings,
     *   including "pedantic" warnings.  If this is false, we'll suppress
     *   pedantic warnings. 
     */
    uint pedantic_ : 1;

    /* 
     *   Test report mode - if this is true, we'll show only root filenames
     *   for source locations in error reports.  Otherwise, we'll show the
     *   full path as given.  
     */
    uint test_report_mode_ : 1;

    /* resource loader */
    class CResLoader *res_loader_;

    /* default character set name */
    char *default_charset_;

    /* 
     *   flag: we have tried loading an external compiler error message
     *   file and failed; if we fail once during a session, we won't try
     *   again, to avoid repeated searches for a message file during
     *   compilations of multiple files 
     */
    static int err_no_extern_messages_;

    /* count of references to the error subsystem */
    static int err_refs_;

    /* 
     *   The console character mapper.  We use this to map error messages
     *   to the console character set.  This is a static so that we can
     *   access it from the static error message printer routines.  
     */
    static class CCharmapToLocal *console_mapper_;
};

#endif /* TCMAIN_H */

