/*
		       Endeavour Mark II API Context
 */

#ifndef EDV_CONTEXT_H
#define EDV_CONTEXT_H

#include <glib.h>
#include "cfg.h"
#include "edv_device.h"
#include "edv_mime_type.h"


typedef struct _edv_context_struct	edv_context_struct;
#define EDV_CONTEXT(p)			((edv_context_struct *)(p))


/*
 *	Endeavour 2 Context:
 *
 *	All members in this structure are private, do not access them
 *	publically.
 *
 *	This structure is the fundimental "connection" to Endeavour 2,
 *	it is used by all Endeavour 2 API  functions that interact with
 *	Endeavour 2 or to perform Endeavour 2 related procedures.
 */
struct _edv_context_struct {

	/* Configuration list */
	cfg_item_struct	*cfg_list;

        /* Devices list */
        edv_device_struct	**device;
        gint			total_devices;

	/* MIME Types list */
	edv_mime_type_struct	**mimetype;
	gint			total_mimetypes;

	/* List of queued interprocess commands to be sent, these
	 * will be sent when EDVContextFlush() is called
	 */
	gchar		**queued_command;
	gint		total_queued_commands;

	/* Quick references */
	gchar		*recycled_index_file,
			*prog_file,
			*prog_full_path;

};


/*
 *	Creates a new Endeavour 2 Context.
 *
 *	The function EDVContextLoadConfigurationFile() should be
 *	called after this call in order to set up the Endeavour 2
 *	Context.
 *
 *	Returns the Endeavour 2 Context or NULL on error. To delete
 *	the returned Endeavour 2 Context, call EDVContextDelete().
 */
extern edv_context_struct *EDVContextNew(void);

/*
 *	Opens the configuration and sets up the Endeavour 2 Context.
 *
 *	The ctx specifies the Endeavour 2 Context. This should be
 *	called right after EDVContextNew() to ensure that the context
 *	ctx is set up properly before passing to any other function.
 *
 *	The path specifies the full path to the alternate configuration
 *	file. If path is NULL then the default configuration file found
 *	in the user's home directory will be used:
 *
 *		$HOME/.endeavour2/endeavour2.ini
 */
extern void EDVContextLoadConfigurationFile(
	edv_context_struct *ctx,
	const gchar *path		/* Can be NULL for default file */
);

/*
 *	Returns the number of queued commands.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
extern gint EDVContextCommandsPending(edv_context_struct *ctx);

/*
 *	Appends a command to the queued commands list.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The cmd specifies the command. This command will be queued, to
 *	actually execute it, call EDVContextFlush() or EDVContextSync().
 */
extern void EDVContextQueueCommand(
	edv_context_struct *ctx,
	const gchar *cmd
);

/*
 *	Processes and flushes all pending operations and resources.
 *
 *	All queued commands will be sent (regardless if Endeavour is
 *	running or not).
 *
 *	This call will not block/wait for Endeavour to acknowlage
 *	the request, for that use EDVContextWait();
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
extern void EDVContextFlush(edv_context_struct *ctx);

/*
 *	Waits for a response from Endeavour 2 (if it is running) to
 *	indicate that it has processed all the commands sent to it
 *	from a prior call to EDVContextFlush() (if any).
 *
 *	If Endeavour 2 is not running then this call returns
 *	immediately.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
extern void EDVContextWait(edv_context_struct *ctx);

/*
 *	Macro to flush all pending operations and wait for them to
 *	be completed.
 */
#define EDVContextSync(_ctx_) {		\
 EDVContextFlush(_ctx_);		\
 EDVContextWait(_ctx_);			\
}

/*
 *	Deletes the Endeavour 2 Context and closes to "connection"
 *	to Endeavour 2 (if it was running). Any pending operations
 *	will not be performed and will be discarded.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
extern void EDVContextDelete(edv_context_struct *ctx);


#endif	/* EDV_CONTEXT_H */
