#ifndef KADU_USERGROUP_H
#define KADU_USERGROUP_H

#include <qmap.h>
#include <qobject.h>
#include <qshared.h>
#include <qstring.h>
#include <qvaluelist.h>
#include <qvariant.h>

#include "userlistelement.h"

class UserGroupData;
class UserListElements;

enum BehaviourForAnonymous {TrueForAnonymous, FalseForAnonymous};

/**
	\class UserGroup
	Klasa odpowiedzialna za przechowywanie grup kontaktw. Daje moliwo
	bardzo szybkiego wyszukiwania oraz informowania o zdarzeniach zwizanych
	z kontaktami (zmiana statusu, dodanie, usunicie, zmiana danych).
	Patrz: UserList, GroupsManager, UsersWithDescription, OnlineUsers,
	       BlockedUsers, BlockingUsers, AnonymousUsers
	Jeeli nie potrzebujesz informacji o zdarzeniach, lepiej uy jest klasy
	UserListElements.

	Parametr "last" w sygnaach dostpnych w tej klasie, powinien by uywany
	tylko gdy mamy absolutn pewno, e zmiana danych nastpia dla wszystkich
	kontaktw z tej grupy (i ani jednego wicej). Dlatego te bezpieczne jest
	uywanie tego parametru tylko w slotach podczonych do klasy UserList.
	Jeeli mimo to potrzebujemy skorzysta z parametru massively w celu wykonania
	jakiej czynnoci dopiero po zakoczeniu zmian, moemy posuy si QTimerem
	z czasem 0 (czyli po oddaniu sterowania ptli zdarze Qt) i jednokrotnym
	uruchomieniem. Przykad:
	<code>
	class MojaKlasa
	{
		...
		QTimer refreshTimer;
		...
	};
	MojaKlasa::MojaKlasa()
	{
		...
		connect(&refreshTimer, SIGNAL(timeout()), this, SLOT(refresh()));
		...
	}
	inline void MojaKlasa::refreshLater()
	{
		refreshTimer.start(0, true);
	}

	void MojaKlasa::refresh()
	{
		...//bardzo kosztowne odwieanie
	}

	void MojaKlasa::userAdded(UserListElement elem, bool massively, bool last)
	{
		if (massively)
		{
			akcja1; //np jakie buforowanie
			...
			refreshLater();
		}
		else
		{
			akcja2;
			...
			refresh();
		}
	}
	</code>
**/
class UserGroup : public QObject
{
	Q_OBJECT

	public:
		typedef QValueListConstIterator<UserListElement> const_iterator;
		typedef QValueListIterator<UserListElement> iterator;
		typedef size_t size_type;

		/**
			\fn UserGroup(int size, const char *name = 0)
			\param size spodziewana ilo elementw * 2
			\param name nazwa grupy
		**/
		UserGroup(int size, const char *name = 0);

		/**
			\fn UserGroup(const QValueList<UserListElement> &group, const char *name = 0)
			\param group
			\param name nazwa grupy
		**/
		UserGroup(const QValueList<UserListElement> &group, const char *name = 0);

		/**
			\fn virtual ~UserGroup()
			Zwalnia pami zwizan z grup
		**/
		virtual ~UserGroup();

		/**
			\fn bool equals(UserListElements users) const
			\param users
			Zoono O(users.count()).
		**/
		bool equals(UserListElements users) const;

		/**
			\fn bool equals(UserListElements users) const
			\param users
			Zoono O(users.count()).
		**/
		bool equals(const UserGroup *group) const;

		/**
			\fn UserListElement byID(const QString &protocolName, const QString &id)
			\param protocolName identyfikator protokou
			\param id identyfikator kontaktu
			Zwraca element listy kontaktw identyfikowany przez nazw
			protokou i identyfikator w tyme protokole.
			Jeeli nie znajdzie si taki kontakt, to tworzony jest nowy
			anonimowy kontakt i dodawany do listy.
			Zoono O(count()), ale w przyszoci bdzie optymalizowana.
		**/
		UserListElement byID(const QString &protocolName, const QString &id);

		/**
			\fn UserListElement byAltNick(const QString &altnick)
			\param altnick wywietlany pseudonim wg. ktrego nastpuje wyszukiwanie.
			\return obiekt reprezentujcy kontakt o podanym pseudonimie wywietlanym.
			Wyszukuje kontakt po wywietlanym pseudonimie.
			Jeeli nie znajdzie si taki kontakt, to tworzony jest nowy
			anonimowy kontakt i dodawany do listy.
			Zoono O(count()), ale w przyszoci bdzie optymalizowana do O(1).
		**/
		UserListElement byAltNick(const QString &altnick);

		/**
			\fn bool contains(const QString &protocolName, const QString &id, BehaviourForAnonymous beh = TrueForAnonymous) const
			\param protocolName identyfikator protokou
			\param id identyfikator kontaktu
			\param beh sposb zachowania dla anonimw
			Sprawdza czy grupa zawiera kontakt o danym identyfikatorze we wskazanym protokole.
			W przypadku kontaktw anonimowych zachowanie tej funkcji okrela parametr beh. Gdy jest utawiony na:
				TrueForAnonymous (domylnie), to zwraca prawd
				FalseForAnonymous, to zwraca fasz
			Zoono O(count()), ale w przyszoci bdzie optymalizowana.
		**/
		bool contains(const QString &protocolName, const QString &id, BehaviourForAnonymous beh = TrueForAnonymous) const;

		/**
			\fn bool contains(UserListElement elem, BehaviourForAnonymous beh = TrueForAnonymous) const
			\param elem sprawdzany kontakt
			\param beh sposb zachowania dla anonimw
			Sprawdza czy grupa zawiera wskazany kontakt.
			W przypadku kontaktw anonimowych zachowanie tej funkcji okrela parametr beh. Gdy jest utawiony na:
				TrueForAnonymous (domylnie), to  zwraca prawd
				FalseForAnonymous, to zwraca fasz
			Zoono O(1).
		**/
		bool contains(UserListElement elem, BehaviourForAnonymous beh = TrueForAnonymous) const;

		/**
			\fn bool containsAltNick(const QString &altnick, BehaviourForAnonymous beh = TrueForAnonymous) const
			\param altnick wywietlany pseudonim
			\param beh sposb zachowania dla anonimw
			Sprawdza czy grupa zawiera kontakt o danym pseudonimie wywietlanym.
			W przypadku kontaktw anonimowych zachowanie tej funkcji okrela parametr beh. Gdy jest utawiony na:
				TrueForAnonymous (domylnie), to  zwraca prawd
				FalseForAnonymous, to zwraca fasz
			Zoono O(count()), ale w przyszoci bdzie optymalizowana do O(1).
		**/
		bool containsAltNick(const QString &altnick, BehaviourForAnonymous beh = TrueForAnonymous) const;

		/**
			\fn size_type count() const
			Zwraca ilo kontaktw.
		**/
		size_type count() const;

		/**
			\fn const_iterator constBegin () const
			Zwraca iterator pocztkowy.
		**/
		const_iterator constBegin () const;

		/**
			\fn const_iterator constEnd () const
			Zwraca iterator kocowy.
		**/
		const_iterator constEnd () const;

		/**
			\fn const_iterator begin () const
			Zwraca iterator pocztkowy.
		**/
		iterator begin () const;

		/**
			\fn const_iterator end () const
			Zwraca iterator kocowy.
		**/
		iterator end () const;

		/**
			\fn UserListElements toUserListElements() const
			Zwraca list kontaktw w postaci obiektu klasy UserListElements
		**/
		UserListElements toUserListElements() const;

		/**
			\fn void resize(int size)
			\param size rozmiar bufora * 2
			Zmienia wewntrzny rozmiar buforw dla klas sownikowych (Q*Dict).
		**/
		void resize(int size);

		QStringList altNicks() const;
		
		void clear();
	public slots:
		/**
			\fn void addUser(UserListElement ule, bool massively = false, bool last = false)
			\param ule obiekt reprezentujcy kontakt, ktry chcemy doda.
			\param massively true, gdy jest to cze wikszego dodawania
			\param last true, gdy massively == true i jest to ostatnie dodanie
			Dodaje do listy podany kontakt.
		**/
		void addUser(UserListElement ule, bool massively = false, bool last = false);

		/**
			\fn void removeUser(UserListElement ule, bool massively = false, bool last = false)
			\param ule obiekt reprezentujcy kontakt, ktry chcemy usun.
			\param massively true, gdy jest to cze wikszego usuwania
			\param last true, gdy massively == true i jest to ostatnie usunicie
			Usuwa podany kontakt z listy.
		**/
		void removeUser(UserListElement ule, bool massively = false, bool last = false);

		/**
			\fn UserListElement addAnonymous(const QString &protocolName, const QString &id, bool massively = false, bool last = false)
			\param protocolName identyfikator protokou
			\param id identyfikator kontaktu
			\param massively true, gdy jest to cze wikszego dodawania
			\param last true, gdy massively == true i jest to ostatnie dodanie
			Dodaje do listy anonimowy kontakt i zwraca go.
		**/
		UserListElement addAnonymous(const QString &protocolName, const QString &id, bool massively = false, bool last = false);

		/**
			\fn void addUsers(const UserGroup *group)
			Dodaje do listy wszystkie kontakty nalece do grupy group
			\param group grupa kontaktw
		**/
		void addUsers(const UserGroup *group);

		/**
			\fn void addUsers(QValueList<UserListElement> users)
			Dodaje do listy wszystkie kontakty nalece do listy users
			\param users kontakty
		**/
		void addUsers(QValueList<UserListElement> users);

		/**
			\fn void removeUsers(const UserGroup *group)
			Usuwa z listy wszystkie kontakty nalece do grupy group
			\param group grupa kontaktw
		**/
		void removeUsers(const UserGroup *group);

		/**
			\fn void removeUsers(QValueList<UserListElement> users)
			Usuwa z listy wszystkie kontakty nalece do listy users
			\param users kontakty
		**/
		void removeUsers(QValueList<UserListElement> users);

	signals:
		/**
			\fn void modified()
			Sygna generowany po zakoczeniu zmian w licie	kontaktw.
			\todo sprawdzi czy zawsze dzieje si to po zakoczeniu zmian
		**/
		void modified();

		/**
			\fn void userDataChanged(UserListElement elem, QString name, QVariant oldValue, QVariant currentValue, bool massively, bool last)
			\param elem kontakt, ktrego dane si zmieniy
			\param name nazwa wasnoci
			\param oldValue stara warto
			\param currentValue nowa (bieca) warto
			\param massively true, gdy jest to cze wikszych zmian
			\param last true, gdy massively == true i jest to ostatnia zmiana
			Sygna generowany gdy dla kontaktu elem zmieni si wasno o nazwie name.
			Przekazuje take star i now warto.
		**/
		void userDataChanged(UserListElement elem, QString name, QVariant oldValue,
							QVariant currentValue, bool massively, bool last);

		void usersDataChanged(QString name);

		/**
			\fn void protocolUserDataChanged(QString protocolName, UserListElement elem, QString name, QVariant oldValue, QVariant currentValue, bool massively, bool last)
			\param protocolName identyfikator protokou
			\param elem kontakt, ktrego dane si zmieniy
			\param name nazwa wasnoci
			\param oldValue stara warto
			\param currentValue nowa (bieca) warto
			\param massively true, gdy jest to cze wikszych zmian
			\param last true, gdy massively == true i jest to ostatnia zmiana
			Sygna generowany gdy dla kontaktu elem zmieni si wasno zwizana z protokoem
			protocolName o nazwie name. Przekazuje take star i now warto.
		**/
		void protocolUserDataChanged(QString protocolName, UserListElement elem,
							QString name, QVariant oldValue, QVariant currentValue,
							bool massively, bool last);

		void protocolUsersDataChanged(QString protocolName, QString name);

		/**
			\fn void userAdded(UserListElement elem, bool massively, bool last)
			\param elem dodany kontakt
			\param massively czy jest to fragment wikszych zmian
			\param last jeeli massively == true, to last == true dla ostatniego kontaktu
			Sygna generowany po dodaniu kontaktu elem do listy
		**/
		void userAdded(UserListElement elem, bool massively, bool last);

		/**
			\fn void addingUser(UserListElement elem, bool massively, bool last)
			\param elem dodany kontakt
			\param massively czy jest to fragment wikszych zmian
			\param last jeeli massively == true, to last == true dla ostatniego kontaktu
			Sygna generowany przed dodaniem kontaktu elem do listy
		**/
		void addingUser(UserListElement elem, bool massively, bool last);

		void usersAdded();

		/**
			\fn void removingUser(UserListElement elem, bool massively, bool last)
			\param elem usuwany kontakt
			\param massively czy jest to fragment wikszych zmian
			\param last jeeli massively == true, to last == true dla ostatniego kontaktu
			Sygna generowany przed usuniciem kontaktu z listy.
		**/
		void removingUser(UserListElement elem, bool massively, bool last);

		/**
			\fn void userRemoved(UserListElement elem, bool massively, bool last)
			\param elem usuwany kontakt
			\param massively czy jest to fragment wikszych zmian
			\param last jeeli massively == true, to last == true dla ostatniego kontaktu
			Sygna generowany po usuniciu kontaktu z listy.
		**/
		void userRemoved(UserListElement elem, bool massively, bool last);

		void usersRemoved();

		/**
			\fn void protocolAdded(UserListElement elem, QString protocolName, bool massively, bool last)
			\param elem kontakt, dla ktrego dodany zosta protok
			\param protocolName identyfikator protokou
			\param massively czy jest to fragment wikszych zmian
			\param last jeeli massively == true, to last == true dla ostatniego kontaktu
			Sygna generowany po dodaniu protokou protocolName kontaktowi elem.
		**/
		void protocolAdded(UserListElement elem, QString protocolName, bool massively, bool last);

		/**
			\fn void removingProtocol(UserListElement elem, QString protocolName, bool massively, bool last)
			\param elem kontakt, dla ktrego usunity zosta protok
			\param protocolName identyfikator protokou
			\param massively czy jest to fragment wikszych zmian
			\param last jeeli massively == true, to last == true dla ostatniego kontaktu
			Sygna generowany przed usuniciem protokou protocolName kontaktu elem.
		**/
		void removingProtocol(UserListElement elem, QString protocolName, bool massively, bool last);

		/**
			\fn void statusChanged(UserListElement elem, QString protocolName, const UserStatus &oldStatus, bool massively, bool last)
			\param elem kontakt, dla ktrego zmieni si status
			\param protocolName identyfikator protokou
			\param oldStatus poprzedni status
			\param massively czy jest to fragment wikszych zmian
			\param last jeeli massively == true, to last == true dla ostatniego kontaktu
		**/
		void statusChanged(UserListElement elem, QString protocolName,
							const UserStatus &oldStatus, bool massively, bool last);

		void usersStatusChanged(QString protocolName);

	protected:
		UserGroupData *d;
		friend class UserListElement;
		friend class ULEPrivate;

		/**
			\fn UserListElement byKey(UserListKey key)
			\param key klucz
			Zwraca element listy kontaktw identyfikowany przez klucz.
		**/
		UserListElement byKey(UserListKey key);

	private:
		/* standardowy konstruktor kopiujcy zepsuby obiekt rdowy */
		UserGroup(const UserGroup &) : QObject(0,0), d(0) {}

		/* standardowy operator kopiowania zepsuby obiekt rdowy */
 		virtual UserGroup &operator = (const UserGroup &) {return *this;}

		/* nie da si porwnywa obiektw tej klasy */
		bool operator==(const UserGroup &) const {return false;}

};

/**
	\class UserListElements
	Prosta lista uytkownikw z kilkoma uatwiaczami.
**/
class UserListElements : public QValueList<UserListElement>
{
	public:
		/**
			\fn UserListElements(UserListElement)
			Konstruktor dodajcy od razu wskazany kontakt
		**/
		UserListElements(UserListElement);

		/**
			\fn UserListElements(const UserListElements &)
			Konstruktor kopiujcy
		**/
		UserListElements(const UserListElements &);

		/**
			\fn UserListElements(const QValueList<UserListElement> &)
			Konstruktor inicjujcy si wskazan list
		**/
		UserListElements(const QValueList<UserListElement> &);

		/**
			\fn UserListElements()
			Standardowy konstruktor.
		**/
		UserListElements();

		/**
			\fn bool equals(const UserListElements &elems) const
			Sprawdza czy bieca lista zaiwera te same elemnty co wskazana.
			\attention {Zoono O(count()*elems.count()).}
		**/
		bool equals(const UserListElements &elems) const;

		/**
			\fn bool equals(const UserGroup *group) const
			Sprawdza czy bieca lista zaiwera te same elemnty co wskazana.
			\attention {Zoono O(count()).}
		**/
		bool equals(const UserGroup *group) const;

		/**
			\fn void sort()
			Sortuje list w sposb jednoznaczny.
		**/
		void sort();

		/**
			\fn bool contains(QString protocol, QString id) const
			Zwraca informacj o tym czy lista zawiera kontakt o wskazanym protokole i identyfiaktorze.
			(Zoono O(n))
		**/
		bool contains(QString protocol, QString id) const;

		/**
			\fn bool contains(UserListElement e) const
			Zwraca informacj o tym czy lista zawiera wskazany kontakt.
			(Zoono O(n))
		**/
		bool contains(UserListElement e) const { return QValueList<UserListElement>::contains(e) > 0;}

		QStringList altNicks() const;
};

#endif
