/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
-----------------------------------------
file  :   SAMLAction.cpp
project :   Policy Server
date :   
author    :   andy
rights    :   (c) Parthenon Computing
notes  :    extension to openSAML C++ library (based on Java class)
--------------------------------------------
*/

#include "internal.h"

using namespace saml;
using namespace std;

SAMLAction::SAMLAction(const XMLCh* data, const XMLCh* nsURI) : m_namespace(XML::assign(nsURI)), m_data(XML::assign(data))
{
    RTTI(SAMLAction);
}

SAMLAction::SAMLAction(DOMElement* e) : m_namespace(NULL), m_data(NULL)
{
    RTTI(SAMLAction);
    fromDOM(e);
}

SAMLAction::SAMLAction(std::istream& in) : SAMLObject(in), m_namespace(NULL), m_data(NULL)
{
    RTTI(SAMLAction);
    fromDOM(m_document->getDocumentElement());
}

SAMLAction::~SAMLAction()
{
    if (m_bOwnStrings) {
        XMLString::release(&m_data);
        XMLString::release(&m_namespace);
    }
}

void SAMLAction::ownStrings()
{
    if (!m_bOwnStrings) {
        m_data=XML::assign(m_data);
        m_namespace=XML::assign(m_namespace);
        m_bOwnStrings=true;
    }
}

void SAMLAction::fromDOM(DOMElement *e)
{
    SAMLObject::fromDOM(e);

    if (SAMLConfig::getConfig().strict_dom_checking && !XML::isElementNamed(e,XML::SAML_NS,L(Action)))
        throw MalformedException("SAMLAction() requires saml:Action at root");

    m_namespace = const_cast<XMLCh*>(e->getAttributeNS(NULL,L(Namespace)));
    if (e->hasChildNodes())
        m_data = const_cast<XMLCh*>(e->getFirstChild()->getNodeValue());

    checkValidity();
}

void SAMLAction::setNamespace(const XMLCh* ns)
{
    if (m_bOwnStrings)
        XMLString::release(&m_namespace);
    else {
        m_namespace=NULL;
        ownStrings();
    }
    m_namespace=XML::assign(ns);
    setDirty();
}

void SAMLAction::setData(const XMLCh* data)
{
    if (XML::isEmpty(data))
        throw SAMLException("data cannot be null or empty");

    if (m_bOwnStrings)
        XMLString::release(&m_data);
    else {
        m_data=NULL;
        ownStrings();
    }
    m_data=XML::assign(data);
    setDirty();
}

DOMElement* SAMLAction::buildRoot(DOMDocument* doc, bool xmlns) const
{
    DOMElement* a = doc->createElementNS(XML::SAML_NS, L(Action));
    if (xmlns)
        a->setAttributeNS(XML::XMLNS_NS, L(xmlns), XML::SAML_NS);
    return a;
}

DOMNode* SAMLAction::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLObject::toDOM(doc, xmlns);
    if (m_bDirty) {
        DOMElement* a=static_cast<DOMElement*>(m_root);
        if(!XML::isEmpty(m_namespace))
            a->setAttributeNS(NULL, L(Namespace), m_namespace);
        a->appendChild(m_root->getOwnerDocument()->createTextNode(m_data));
        setClean();
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(static_cast<DOMElement*>(m_root),XML::SAML_NS);
    }

    return m_root;
}

void SAMLAction::checkValidity() const
{
    if (XML::isEmpty(m_data))
        throw MalformedException("Action is invalid, data must have a value");
}

SAMLObject* SAMLAction::clone() const
{
    return new SAMLAction(m_data, m_namespace);
}

const XMLCh SAMLAction::SAML_ACTION_NAMESPACE_RWEDC_NEG[] = // "urn:oasis:names:tc:SAML:1.0:action:rwedc-negation"
{
    chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
    chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_0, chColon,
    chLatin_a, chLatin_c, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chColon,
    chLatin_r, chLatin_w, chLatin_e, chLatin_d, chLatin_c, chDash,
    chLatin_n, chLatin_e, chLatin_g, chLatin_a, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chNull
};

const XMLCh SAMLAction::SAML_ACTION_NAMESPACE_RWEDC[] = // "urn:oasis:names:tc:SAML:1.0:action:rwedc"
{
    chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
    chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_0, chColon,
    chLatin_a, chLatin_c, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chColon,
    chLatin_r, chLatin_w, chLatin_e, chLatin_d, chLatin_c, chNull
};

const XMLCh SAMLAction::SAML_ACTION_NAMESPACE_GHPP[] = // "urn:oasis:names:tc:SAML:1.0:action:ghpp"
{
    chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
    chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_0, chColon,
    chLatin_a, chLatin_c, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chColon,
    chLatin_g, chLatin_h, chLatin_p, chLatin_p, chNull
};

const XMLCh SAMLAction::SAML_ACTION_NAMESPACE_UNIX[] = // "urn:oasis:names:tc:SAML:1.0:action:unix"
{
    chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
    chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
    chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_0, chColon,
    chLatin_a, chLatin_c, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chColon,
    chLatin_u, chLatin_n, chLatin_i, chLatin_x, chNull
};
