// Berkeley Open Infrastructure for Network Computing
// http://boinc.berkeley.edu
// Copyright (C) 2005 University of California
//
// This is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation;
// either version 2.1 of the License, or (at your option) any later version.
//
// This software is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU Lesser General Public License for more details.
//
// To view the GNU Lesser General Public License visit
// http://www.gnu.org/copyleft/lesser.html
// or write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef _SCHEDULER_OP_
#define _SCHEDULER_OP_

#include <vector>

#include "client_types.h"
#include "http_curl.h"
#include "prefs.h"

// SCHEDULER_OP encapsulates the mechanism for
// 1) fetching master files
// 2) communicating with scheduling servers
// Only one such operation can be in progress at once.

#define SCHEDULER_OP_STATE_IDLE         0
    // invariant: in this state, our HTTP_OP is not in the HTTP_OP_SET
#define SCHEDULER_OP_STATE_GET_MASTER   1
#define SCHEDULER_OP_STATE_RPC          2

// reasons for making a scheduler RPC:
//
typedef enum {
    REASON_USER_REQ,
    REASON_RESULTS_DUE,
    REASON_NEED_WORK,
    REASON_TRICKLE_UP
} SCHEDULER_OP_REASON ;

// defaults related to scheduler RPC policy
// See client_state.h for definitions

#define MASTER_FETCH_PERIOD     10
    // fetch and parse master URL if nrpc_failures is a multiple of this
#define RETRY_CAP               10
    // cap on nrpc_failures
#define MASTER_FETCH_RETRY_CAP 3
    // after this many master-fetch failures, 
    // move into a state in which we retry master fetch
    // at the frequency below
#define MASTER_FETCH_INTERVAL (86400*7)    // 1 week
    // See above

// constants used to bound RPC backoff
#define SCHED_RETRY_DELAY_MIN    60                // 1 minute
#define SCHED_RETRY_DELAY_MAX    (60*60*4)         // 4 hours


class SCHEDULER_OP {
private:
    int scheduler_op_retval;
    HTTP_OP http_op;
    HTTP_OP_SET* http_ops;
    char scheduler_url[256];
    int url_index;                  // index within project's URL list
public:
    PROJECT* cur_proj;               // project we're currently contacting
    int state;
    SCHEDULER_OP_REASON reason;
    double url_random;              // used to randomize order

public:
    SCHEDULER_OP(HTTP_OP_SET*);
    bool poll();
    int init_get_work();
    int init_op_project(PROJECT*, SCHEDULER_OP_REASON);
    int init_master_fetch(PROJECT*);
    bool check_master_fetch_start();
    void backoff(PROJECT* p, const char *error_msg);
    void abort(PROJECT*);
        // if we're doing an op to this project, abort it
private:
    bool update_urls(PROJECT*, std::vector<std::string> &urls);
    int start_op(PROJECT*);
    int start_rpc(PROJECT*);
    int parse_master_file(PROJECT*, std::vector<std::string>&);
};

struct USER_MESSAGE {
    std::string message;
    std::string priority;
    USER_MESSAGE(char*, char*);
};

struct SCHEDULER_REPLY {
    int hostid;
    double request_delay;
    std::vector<USER_MESSAGE> messages;
    char* global_prefs_xml;
        // not including <global_preferences> tags;
        // may include <venue> elements
    char* project_prefs_xml;
        // not including <project_preferences> tags
        // may include <venue> elements
    char master_url[256];
    char host_venue[256];
    unsigned int user_create_time;
    std::vector<APP> apps;
    std::vector<FILE_INFO> file_infos;
    std::vector<std::string> file_deletes;
    std::vector<APP_VERSION> app_versions;
    std::vector<WORKUNIT> workunits;
    std::vector<RESULT> results;
    std::vector<RESULT> result_acks;
    char* code_sign_key;
    char* code_sign_key_signature;
    bool message_ack;
    bool project_is_down;
    bool send_file_list;      

    SCHEDULER_REPLY();
    ~SCHEDULER_REPLY();
    int parse(FILE*, PROJECT*);
};

#endif
