/*
 * rsa_crypt_lib.c - RSA-crypt-library for iTALC
 *
 * iTALC
 * Copyright (c) 2004-2005 Tobias Doerffel <tobias@doerffel.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#include "rsa_crypt_lib.h"

#ifdef RSA_CRYPT_AVAILABLE

#include <openssl/evp.h>
#include <openssl/pem.h>

#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <malloc.h>


RSA * createNewRSA( void )
{
	RSA * rsa = RSA_new();
	if( rsa == NULL )
	{
		printf( "createNewRSA: RSA_new failed\n" );
		exit( -1 );
	}
	if( ( rsa->n = BN_new() ) == NULL )
	{
		printf( "createNewRSA: BN_new failed\n" );
		exit( -1 );
	}
	if( ( rsa->e = BN_new() ) == NULL )
	{
		printf( "createNewRSA: BN_new failed\n" );
		exit( -1 );
	}
	return( rsa );
}




int rsaGenerateChallenge( char * * _cb )
{
	int bytes;
	BIGNUM * challenge_bn = BN_new();

	if( challenge_bn == NULL )
	{
		printf( "rsaGenerateChallenge: BN_new() failed\n" );
	}

	/* Generate a random challenge. */
	BN_rand( challenge_bn, DEFAULT_CHALLENGE_SIZE*8, 0, 0 );
	*_cb = malloc( BN_num_bytes( challenge_bn ) );
	bytes = BN_bn2bin( challenge_bn, (unsigned char * ) *_cb );
	BN_free( challenge_bn );
	return( bytes  );
}




void rsaFreeChallenge( char * _cb )
{
	free( _cb );
}




RSA * readPrivateKey( const char * _fname )
{
	int fd;
	FILE * f;
	EVP_PKEY * pk;
	RSA * rsa;
	fd = open( _fname, O_RDONLY, 0400 );
	if( fd == -1 )
	{
		printf( "readPrivateKey: could not read your private key in %s\n", _fname );
		return( NULL );
	}
	f = fdopen( fd, "r" );
	if (f == NULL) {
		/*printf( "readPrivateKey: fdopen failed: %s\n", strerror( errno ) );*/
		close( fd );
		return( NULL );
	}
	pk = PEM_read_PrivateKey( f, NULL, NULL, NULL );
	fclose( f );
	rsa = NULL;
	if( pk == NULL )
	{
		printf( "readPrivateKey: PEM_read_PrivateKey failed\n" );
	}
	else if( pk->type == EVP_PKEY_RSA )
	{
		rsa = EVP_PKEY_get1_RSA( pk );
		if( RSA_blinding_on( rsa, NULL ) != 1 )
		{
			printf("readPrivateKey: RSA_blinding_on failed\n");
			rsa = NULL;
		}
	}
	return( rsa );
}




RSA * readPublicKey( const char * _fname )
{
	FILE * f;
	RSA * rsa;
	int fd = open( _fname, O_RDONLY, 0400 );
	if( fd == -1 )
	{
		printf( "readPublicKey: could not read your public key in %s\n", _fname );
		return( NULL );
	}
	f = fdopen( fd, "r" );
	if (f == NULL) {
		/*printf( "readPublicKey: fdopen failed: %s\n", strerror( errno ) );*/
		close( fd );
		return( NULL );
	}
	rsa = PEM_read_RSAPublicKey( f, NULL, NULL, NULL );
	fclose( f );

	if( rsa == NULL )
	{
		printf( "readPublicKey: PEM_read_RSAPublicKey failed\n" );
	}
	return( rsa );
}




void writePrivateKey( RSA * _priv_key, const char * _fname )
{
	FILE * f;
	int fd = open( _fname, O_WRONLY | O_CREAT | O_TRUNC, 0600 );
	if( fd == -1 )
	{
		printf( "Could not save your public key in %s\n", _fname );
		exit( -1 );
	}
	f = fdopen( fd, "w" );
	if( f == NULL )
	{
		printf ("fdopen %s failed", _fname );
		exit( -1 );
	}
	PEM_write_RSAPrivateKey( f, _priv_key, NULL, NULL, 0, NULL, NULL );
	fclose( f );
}




void writePublicKey( RSA * _pub_key, const char * _fname )
{
	FILE * f;
	int fd = open( _fname, O_WRONLY | O_CREAT | O_TRUNC, 0644 );
	if( fd == -1 )
	{
		printf( "Could not save your public key in %s\n", _fname );
		exit( -1 );
	}
	f = fdopen( fd, "w" );
	if( f == NULL )
	{
		printf( "fdopen %s failed", _fname );
		exit( -1 );
	}
	PEM_write_RSAPublicKey( f, _pub_key );
	fclose( f );
}




RSA * rsaGeneratePrivateKey( unsigned int _bits )
{
	RSA * priv_key;
	priv_key = RSA_generate_key( _bits, 35, NULL, NULL );
	if( priv_key == NULL )
	{
		printf( "rsaGeneratePrivateKey: key generation failed." );
		exit( -1 );
	}
	return( priv_key );
}




RSA * publicKeyFromPrivateKey( RSA * _src_dsa)
{
	RSA * rsa = createNewRSA();
	BN_copy( rsa->n, _src_dsa->n );
	BN_copy( rsa->e, _src_dsa->e );
	return( rsa );
}


#endif
