#ifndef NOTIFY_H
#define NOTIFY_H
#include <qobject.h>
#include <qstring.h>
#include <qmap.h>
#include <qvariant.h>
#include <qpair.h>
#include <qvaluelist.h>
#include <qstringlist.h>

#include <time.h>

#include "notify_slots.h"
#include "protocol.h"
#include "userlist.h"

/**
 * @defgroup notify Notify
 * @{
 */
class Notifier : public QObject {
	public:
		Notifier(QObject *parent = 0, const char *name = 0) : QObject(parent, name) {};
		virtual ~Notifier() {};
		virtual void externalEvent(const QString &notifyType, const QString &msg, const UserListElements &ules) = 0;
};

class Notify : public QObject
{
	Q_OBJECT
	private:
		class NotifierSlots
		{
			public:
			NotifierSlots();
			NotifierSlots(Notifier *notifier, const QMap<QString, QString> &notifierSlots);

			Notifier *notifier; //obiekt "nasuchujcy" na zdarzenia
			QMap<QString, QPair<QString, bool> > notifierSlots; //nazwa sygnau("NewMessage") -> para<slot (SLOT(mjSlotMessage)), czy poczony>
		};
		QMap<QString, NotifierSlots> notifiers; //nazwa powiadamiacza("Hints") -> obiekt powiadomienia
		QMap<QString, QString> notifySignals; //nazwa sygnau("NewMessage") -> sygna (SIGNAL(newMessage(...)))
		QStringList eventNames;

		class NotifyEvent
		{
			public:
				QString name;
				QCString wname;
				const char *description;
				NotifyEvent() : name(), wname(), description(0){}
		};
		QValueList<NotifyEvent> notifyEvents;
		QMap<QString, QValueList<QCString> > strs;

		/*
		 * dodaje kolumn checkboksw w konfiguracji,
		 * na podstawie notifierSlots decyduje czy doda checkboksa aktywnego czy nie
		 */
		void addConfigColumn(const QString &name, const QMap<QString, QString> &notifierSlots);

		/* usuwa kolumn checkboksw z konfiguracji */
		void removeConfigColumn(const QString &name, const QMap<QString, QPair<QString, bool> > &notifierSlots);

		void addConfigRow(const QString &name, const char *description);
		void removeConfigRow(const QString &name);

	private slots:
		/* pomocniczy slot */
		void newChatSlot(Protocol *protocol, UserListElements senders, const QString &msg, time_t t);
		/* pomocniczy slot */
		void probablyNewMessage(Protocol *protocol, UserListElements senders, const QString &msg, time_t t, bool &grab);

		/*
		 * czy odpowiedni sygna z notifierName (np.: Window, Hint, Sound)
		 * ze slotem slotName (np.: NewChat, ConnError)
		 */
		void connectSlot(const QString &notifierName, const QString &slotName);

		/*
		 * rozcza odpowiedni sygna z notifierName (np.: Window, Hint, Sound)
		 * od slotu slotName (np.: NewChat, ConnError)
		 */
		void disconnectSlot(const QString &notifierName, const QString &slotName);

	public:
		Notify(QObject *parent=0, const char *name=0);
		~Notify();

		/*
		 * rejestruje obiekt, ktry chce otrzymywa informacje o zdarzeniach
		 * name bdzie uyte w oknie konfiguracji (naley podawa angielsk nazw)
		 * slots to mapa odwzorujca nazwy sygnaw na sloty w obiekcie notifier
		 * nazwy sygnaw:
			"NewChat" - nowa rozmowa
			"NewMessage" - nowa wiadomo
			"ConnError" - bd poczenia
			"StatusChanged" - zmiana statusu oglna
			"toAvailable" - uytkownik zmieni status na "Dostpny"
			"toBusy" - uytkownik zmieni status na "Zaraz wracam"
			"toInvisible" - uytkownik zmieni status na "Ukryty"
			"toNotAvailable" - uytkownik zmieni status na "Niewidoczny lub Niedostpny"
			"UserBoxChangeToolTip"
			"Message" - inna wiadomo
		 *
		 * nie trzeba definiowa wszystkich slotw
		 */
		void registerNotifier(const QString &name, Notifier *notifier,
							const QMap<QString, QString> &notifierSlots);
		/*
		 * wyrejestrowuje obiekt
		 */
		void unregisterNotifier(const QString &name);

		void registerEvent(const QString &name, const char *description);
		void unregisterEvent(const QString &name);

		QStringList notifiersList() const;
		const QValueList<Notify::NotifyEvent> &externalNotifyTypes();

	public slots:
		/*
		 * aktualizuje wszystkie poczenia
		 * wywoywany po zmianie konfiguracji dotyczcej obsugi
		 */
		void updateConnections();

		/*
		 * emituje sygna message
		 *  jeeli to==QString::null, to wysya do wszystkich, ktrzy s zaznaczeni w konfiguracji
		 *  jeeli to!=QString::null, to wysya tylko do jednego
		 * Notifier decyduje, ktrych argumentw uy
		 */
		void emitMessage(const QString &from, const QString &to, const QString &message=QString(), const QMap<QString, QVariant> *parameters=NULL, const UserListElement *ule=NULL);

		/* obsuga zmian statusw */
		void statusChanged(UserListElement elem, QString protocolName,
							const UserStatus &oldStatus, bool massively, bool last);

		void notify(const QString &notifyType, const QString &msg, const UserListElements &ules = UserListElements());

	signals:
	//UWAGA: razem ze zmianami nazw/parametrw tych sygnaw naley aktualizowa wpisy w konstruktorze Notify

		/* nowa rozmowa */
		void newChat(Protocol *protocol, UserListElements senders, const QString &msg, time_t t);

		/* nowa wiadomo w oknie chat */
		void newMessage(Protocol *protocol, UserListElements senders, const QString &msg, time_t t, bool &grab);

		/* bd poczenia */
		void connectionError(Protocol *protocol, const QString &message);

		/* uytkownik zmieni status */
		void userStatusChanged(UserListElement elem, QString protocolName, const UserStatus &oldStatus);

		/* uytkownik zmieni status na "Dostpny" */
		void userChangedStatusToAvailable(const QString &protocolName, UserListElement);

		/* uytkownik zmieni status na "Zaraz wracam" */
		void userChangedStatusToBusy(const QString &protocolName, UserListElement);

		/* uytkownik zmieni status na "Niewidoczny" */
		void userChangedStatusToInvisible(const QString &protocolName, UserListElement);

		/* uytkownik zmieni status na "Niewidoczny" lub "Niedostpny" */
		void userChangedStatusToNotAvailable(const QString &protocolName, UserListElement);

		void userBoxChangeToolTip(const QPoint &point, UserListElement, bool show);

		/* inna informacja do powiadomienia */
		void message(const QString &from, const QString &msg, const QMap<QString, QVariant> *parameters, const UserListElement *ule);

	signals:
		/* do uytku wewntrznego */
		void privateMessage(const QString &from, const QString &message, const QMap<QString, QVariant> *parameters, const UserListElement *ule);
};

extern Notify *notify;
extern NotifySlots *notify_slots;

/** @} */

#endif
