/*
 * Copyright (c) 2003-2007 by FlashCode <flashcode@flashtux.org>
 * See README for License detail, AUTHORS for developers list.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/* gui-window.c: window functions, used by all GUI */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <ctype.h>

#include "../common/weechat.h"
#include "gui.h"
#include "../common/command.h"
#include "../common/weeconfig.h"
#include "../common/history.h"
#include "../common/hotlist.h"
#include "../common/log.h"
#include "../common/utf8.h"
#include "../common/util.h"
#include "../irc/irc.h"

#ifdef PLUGINS
#include "../plugins/plugins.h"
#endif


t_gui_buffer *gui_buffers = NULL;           /* pointer to first buffer      */
t_gui_buffer *last_gui_buffer = NULL;       /* pointer to last buffer       */
t_gui_buffer *gui_previous_buffer = NULL;   /* pointer to previous buffer   */
t_gui_buffer *gui_buffer_before_dcc = NULL; /* buffer before dcc switch     */
t_gui_buffer *gui_buffer_raw_data = NULL;   /* buffer with raw IRC data     */
t_gui_buffer *gui_buffer_before_raw_data = NULL; /* buffer before raw switch*/


/*
 * gui_buffer_servers_search: search servers buffer
 *                            (when same buffer is used for all servers)
 */

t_gui_buffer *
gui_buffer_servers_search ()
{
    t_gui_buffer *ptr_buffer;
    
    for (ptr_buffer = gui_buffers; ptr_buffer;
         ptr_buffer = ptr_buffer->next_buffer)
    {
        if (ptr_buffer->all_servers)
            return ptr_buffer;
    }
    
    /* buffer not found */
    return NULL;
}

/*
 * gui_buffer_new: create a new buffer in current window
 */

t_gui_buffer *
gui_buffer_new (t_gui_window *window, void *server, void *channel, int type,
                int switch_to_buffer)
{
    t_gui_buffer *new_buffer, *ptr_buffer;
#ifdef PLUGINS
    char buffer_str[16], *argv[1] = { NULL };
#endif
    
#ifdef DEBUG
    weechat_log_printf ("Creating new buffer\n");
#endif
    
    /* use first buffer if no server is assigned to this buffer */
    if ((type == GUI_BUFFER_TYPE_STANDARD) && gui_buffers
        && (!GUI_SERVER(gui_buffers)) && (!channel))
    {
        if (server)
            ((t_irc_server *)(server))->buffer = gui_buffers;
        if (channel)
            ((t_irc_channel *)(channel))->buffer = gui_buffers;
        gui_buffers->server = server;
        gui_buffers->channel = channel;
        if (cfg_look_one_server_buffer && server && !channel)
            gui_buffers->all_servers = 1;
        if (cfg_log_auto_server)
            gui_log_start (gui_buffers);
        gui_buffers->completion.server = server;
        return gui_buffers;
    }

    /* use "all servers" buffer if found */
    if (cfg_look_one_server_buffer && (type == GUI_BUFFER_TYPE_STANDARD) &&
        gui_buffers && server && !channel)
    {
        ptr_buffer = gui_buffer_servers_search ();
        if (ptr_buffer)
        {
            ((t_irc_server *)(server))->buffer = ptr_buffer;
            ptr_buffer->server = server;
            if (switch_to_buffer)
                gui_window_switch_to_buffer (window, ptr_buffer);
            gui_window_redraw_buffer (ptr_buffer);
            return ptr_buffer;
        }
    }
    
    if ((new_buffer = (t_gui_buffer *)(malloc (sizeof (t_gui_buffer)))))
    {
        new_buffer->num_displayed = 0;
        new_buffer->number = (last_gui_buffer) ? last_gui_buffer->number + 1 : 1;
        
        /* assign server and channel to buffer */
        new_buffer->server = server;
        new_buffer->all_servers = 0;
        new_buffer->channel = channel;
        new_buffer->type = type;
        if (new_buffer->type == GUI_BUFFER_TYPE_RAW_DATA)
            gui_buffer_raw_data = new_buffer;
        /* assign buffer to server and channel */
        if (server && !channel)
        {
            GUI_SERVER(new_buffer)->buffer = new_buffer;
            new_buffer->all_servers = (cfg_look_one_server_buffer) ? 1 : 0;
        }
        if (!gui_buffers && cfg_look_one_server_buffer)
            new_buffer->all_servers = 1;
        if (channel)
            GUI_CHANNEL(new_buffer)->buffer = new_buffer;
        
        if (!window->buffer)
        {
            window->buffer = new_buffer;
            window->first_line_displayed = 1;
            window->start_line = NULL;
            window->start_line_pos = 0;
            gui_window_calculate_pos_size (window, 1);
        }
        
        /* init lines */
        new_buffer->lines = NULL;
        new_buffer->last_line = NULL;
        new_buffer->last_read_line = NULL;
        new_buffer->num_lines = 0;
        new_buffer->line_complete = 1;
        
        /* notify level */
        new_buffer->notify_level = irc_channel_get_notify_level (server, channel);
        
        /* create/append to log file */
        new_buffer->log_filename = NULL;
        new_buffer->log_file = NULL;
        if ((cfg_log_auto_server && GUI_BUFFER_IS_SERVER(new_buffer))
            || (cfg_log_auto_channel && GUI_BUFFER_IS_CHANNEL(new_buffer))
            || (cfg_log_auto_private && GUI_BUFFER_IS_PRIVATE(new_buffer)))
            gui_log_start (new_buffer);
        
        /* init input buffer */
        new_buffer->has_input = (new_buffer->type == GUI_BUFFER_TYPE_STANDARD) ? 1 : 0;
        if (new_buffer->has_input)
        {
            new_buffer->input_buffer_alloc = GUI_INPUT_BUFFER_BLOCK_SIZE;
            new_buffer->input_buffer = (char *) malloc (GUI_INPUT_BUFFER_BLOCK_SIZE);
            new_buffer->input_buffer_color_mask = (char *) malloc (GUI_INPUT_BUFFER_BLOCK_SIZE);
            new_buffer->input_buffer[0] = '\0';
            new_buffer->input_buffer_color_mask[0] = '\0';
        }
        else
        {
            new_buffer->input_buffer = NULL;
            new_buffer->input_buffer_color_mask = NULL;
        }
        new_buffer->input_buffer_size = 0;
        new_buffer->input_buffer_length = 0;
        new_buffer->input_buffer_pos = 0;
        new_buffer->input_buffer_1st_display = 0;
        
        /* init completion */
        completion_init (&(new_buffer->completion), server, channel);
        
        /* init history */
        new_buffer->history = NULL;
        new_buffer->last_history = NULL;
        new_buffer->ptr_history = NULL;
        new_buffer->num_history = 0;
        
        /* text search */
        new_buffer->text_search = GUI_TEXT_SEARCH_DISABLED;
        new_buffer->text_search_exact = 0;
        new_buffer->text_search_found = 0;
        new_buffer->text_search_input = NULL;
        
        /* add buffer to buffers queue */
        new_buffer->prev_buffer = last_gui_buffer;
        if (gui_buffers)
            last_gui_buffer->next_buffer = new_buffer;
        else
            gui_buffers = new_buffer;
        last_gui_buffer = new_buffer;
        new_buffer->next_buffer = NULL;
        
        /* move buffer next to server */
        if (server && cfg_look_open_near_server && (!cfg_look_one_server_buffer))
        {
            ptr_buffer = GUI_SERVER(new_buffer)->buffer;
            while (ptr_buffer && (ptr_buffer->server == server))
            {
                ptr_buffer = ptr_buffer->next_buffer;
            }
            if (ptr_buffer)
                gui_buffer_move_to_number (new_buffer, ptr_buffer->number);
        }
        
        /* switch to new buffer */
        if (switch_to_buffer)
            gui_window_switch_to_buffer (window, new_buffer);
        
        /* redraw buffer */
        gui_window_redraw_buffer (new_buffer);

#ifdef PLUGINS
        snprintf (buffer_str, sizeof (buffer_str) - 1, "%d", new_buffer->number);
        argv[0] = buffer_str;
        (void) plugin_event_handler_exec ("buffer_open", 1, argv);
#endif
    }
    else
        return NULL;
    
    return new_buffer;
}

/*
 * gui_buffer_search: search a buffer by server and channel name
 */

t_gui_buffer *
gui_buffer_search (char *server, char *channel)
{
    t_irc_server *ptr_server, *ptr_srv;
    t_irc_channel *ptr_channel, *ptr_chan;
    t_gui_buffer *ptr_buffer;
    
    ptr_server = NULL;
    ptr_channel = NULL;
    ptr_buffer = NULL;
    
    /* nothing given => print on current buffer */
    if ((!server || !server[0]) && (!channel || !channel[0]))
        ptr_buffer = gui_current_window->buffer;
    else
    {
        if (server && server[0])
        {
            ptr_server = irc_server_search (server);
            if (!ptr_server)
                return NULL;
        }
        
        if (channel && channel[0])
        {
            if (ptr_server)
            {
                ptr_channel = irc_channel_search_any (ptr_server, channel);
                if (ptr_channel)
                    ptr_buffer = ptr_channel->buffer;
            }
            else
            {
                for (ptr_srv = irc_servers; ptr_srv;
                     ptr_srv = ptr_srv->next_server)
                {
                    for (ptr_chan = ptr_srv->channels; ptr_chan;
                         ptr_chan = ptr_chan->next_channel)
                    {
                        if (ascii_strcasecmp (ptr_chan->name, channel) == 0)
                        {
                            ptr_channel = ptr_chan;
                            break;
                        }
                    }
                    if (ptr_channel)
                        break;
                }
                if (ptr_channel)
                    ptr_buffer = ptr_channel->buffer;
            }
        }
        else
        {
            if (ptr_server)
                ptr_buffer = ptr_server->buffer;
            else
                ptr_buffer = gui_current_window->buffer;
        }
    }
    
    if (!ptr_buffer)
        return NULL;
    
    return (ptr_buffer->type != GUI_BUFFER_TYPE_STANDARD) ?
        gui_buffers : ptr_buffer;
}

/*
 * gui_buffer_search_by_number: search a buffer by number
 */

t_gui_buffer *
gui_buffer_search_by_number (int number)
{
    t_gui_buffer *ptr_buffer;
    
    for (ptr_buffer = gui_buffers; ptr_buffer;
         ptr_buffer = ptr_buffer->next_buffer)
    {
        if (ptr_buffer->number == number)
            return ptr_buffer;
    }
    
    /* buffer not found */
    return NULL;
}

/*
 * gui_buffer_find_window: find a window displaying buffer
 */

t_gui_window *
gui_buffer_find_window (t_gui_buffer *buffer)
{
    t_gui_window *ptr_win;
    
    if (gui_current_window->buffer == buffer)
        return gui_current_window;
    
    for (ptr_win = gui_windows; ptr_win;
         ptr_win = ptr_win->next_window)
    {
        if (ptr_win->buffer == buffer)
            return ptr_win;
    }
    
    /* no window found */
    return NULL;
}

/*
 * gui_buffer_find_context: find window/buffer for a server/channel
 */

void
gui_buffer_find_context (void *server, void *channel,
                         t_gui_window **window, t_gui_buffer **buffer)
{
    t_gui_window *ptr_win;
    
    if (!buffer)
        return;
    
    /* first find buffer */
    *buffer = NULL;
    if ((t_irc_channel *)channel && ((t_irc_channel *)channel)->buffer)
        *buffer = ((t_irc_channel *)channel)->buffer;
    else
    {
        if ((t_irc_server *)server && ((t_irc_server *)server)->buffer)
            *buffer = ((t_irc_server *)server)->buffer;
        else
            *buffer = gui_current_window->buffer;
    }
    
    /* then find first window displaying this buffer */
    if (window)
    {
        *window = NULL;
        if (gui_current_window->buffer == *buffer)
            *window = gui_current_window;
        else
        {
            for (ptr_win = gui_windows; ptr_win;
                 ptr_win = ptr_win->next_window)
            {
                if (ptr_win->buffer == *buffer)
                {
                    *window = ptr_win;
                    break;
                }
            }
            if (!*window)
                *window = gui_current_window;
        }
    }
}

/*
 * gui_buffer_is_scrolled: return 1 if all windows displaying buffer are scrolled
 *                         (user doesn't see end of buffer)
 *                         return 0 if at least one window is NOT scrolled
 */

int
gui_buffer_is_scrolled (t_gui_buffer *buffer)
{
    t_gui_window *ptr_win;
    int buffer_found;

    if (!buffer)
        return 0;
    
    buffer_found = 0;
    for (ptr_win = gui_windows; ptr_win; ptr_win = ptr_win->next_window)
    {
        if (ptr_win->buffer == buffer)
        {
            buffer_found = 1;
            /* buffer found and not scrolled, exit immediately */
            if (!ptr_win->scroll)
                return 0;
        }
    }
    
    /* buffer found, and all windows were scrolled */
    if (buffer_found)
        return 1;
    
    /* buffer not found */
    return 0;
}

/*
 * gui_buffer_get_dcc: get pointer to DCC buffer (DCC buffer created if not existing)
 */

t_gui_buffer *
gui_buffer_get_dcc (t_gui_window *window)
{
    t_gui_buffer *ptr_buffer;
    
    /* check if dcc buffer exists */
    for (ptr_buffer = gui_buffers; ptr_buffer; ptr_buffer = ptr_buffer->next_buffer)
    {
        if (ptr_buffer->type == GUI_BUFFER_TYPE_DCC)
            break;
    }
    if (ptr_buffer)
        return ptr_buffer;
    else
        return gui_buffer_new (window, NULL, NULL, GUI_BUFFER_TYPE_DCC, 0);
}

/*
 * gui_buffer_clear: clear buffer content
 */

void
gui_buffer_clear (t_gui_buffer *buffer)
{
    t_gui_window *ptr_win;
    t_gui_line *ptr_line;

    if (!buffer)
        return;

    if (buffer->type == GUI_BUFFER_TYPE_DCC)
        return;
    
    /* remove buffer from hotlist */
    hotlist_remove_buffer (buffer);
    
    /* remove lines from buffer */
    while (buffer->lines)
    {
        ptr_line = buffer->lines->next_line;
        if (buffer->lines->data)
            free (buffer->lines->data);
        free (buffer->lines);
        buffer->lines = ptr_line;
    }
    
    buffer->lines = NULL;
    buffer->last_line = NULL;
    buffer->num_lines = 0;
    buffer->line_complete = 1;

    /* remove any scroll for buffer */
    for (ptr_win = gui_windows; ptr_win; ptr_win = ptr_win->next_window)
    {
        if (ptr_win->buffer == buffer)
        {
            ptr_win->first_line_displayed = 1;
            ptr_win->start_line = NULL;
            ptr_win->start_line_pos = 0;
        }
    }
    
    gui_chat_draw (buffer, 1);
    gui_status_draw (buffer, 1);
}

/*
 * gui_buffer_clear_all: clear all buffers content
 */

void
gui_buffer_clear_all ()
{
    t_gui_buffer *ptr_buffer;
    
    for (ptr_buffer = gui_buffers; ptr_buffer; ptr_buffer = ptr_buffer->next_buffer)
        gui_buffer_clear (ptr_buffer);
}

/*
 * gui_buffer_line_free: delete a line from a buffer
 */

void
gui_buffer_line_free (t_gui_line *line)
{
    t_gui_window *ptr_win;
    
    for (ptr_win = gui_windows; ptr_win; ptr_win = ptr_win->next_window)
    {
        if (ptr_win->start_line == line)
        {
            ptr_win->start_line = ptr_win->start_line->next_line;
            ptr_win->start_line_pos = 0;
            gui_chat_draw (ptr_win->buffer, 0);
            gui_status_draw (ptr_win->buffer, 0);
        }
    }
    if (line->nick)
        free (line->nick);
    if (line->data)
        free (line->data);
    free (line);
}

/*
 * gui_buffer_free: delete a buffer
 */

void
gui_buffer_free (t_gui_buffer *buffer, int switch_to_another)
{
    t_gui_window *ptr_win;
    t_gui_buffer *ptr_buffer;
    t_gui_line *ptr_line;
    t_irc_server *ptr_server;
    int create_new;
#ifdef PLUGINS
    char buffer_str[16], *argv[1] = { NULL };
#endif
    
#ifdef PLUGINS
    snprintf (buffer_str, sizeof (buffer_str) - 1, "%d", buffer->number);
    argv[0] = buffer_str;
    (void) plugin_event_handler_exec ("buffer_close", 1, argv);
#endif
    
    create_new = (buffer->server || buffer->channel);
    
    if (switch_to_another)
    {
        for (ptr_win = gui_windows; ptr_win; ptr_win = ptr_win->next_window)
        {
            if ((buffer == ptr_win->buffer) &&
                ((buffer->next_buffer) || (buffer->prev_buffer)))
                gui_buffer_switch_previous (ptr_win);
        }
    }
    
    hotlist_remove_buffer (buffer);
    if (hotlist_initial_buffer == buffer)
        hotlist_initial_buffer = NULL;

    if (gui_previous_buffer == buffer)
        gui_previous_buffer = NULL;
    
    if (gui_buffer_before_dcc == buffer)
        gui_buffer_before_dcc = NULL;
    
    if (gui_buffer_before_raw_data == buffer)
        gui_buffer_before_raw_data = NULL;
    
    if (buffer->type == GUI_BUFFER_TYPE_RAW_DATA)
        gui_buffer_raw_data = NULL;
    
    for (ptr_server = irc_servers; ptr_server;
         ptr_server = ptr_server->next_server)
    {
        if (ptr_server->saved_buffer == buffer)
            ptr_server->saved_buffer = NULL;
    }
    
    /* decrease buffer number for all next buffers */
    for (ptr_buffer = buffer->next_buffer; ptr_buffer; ptr_buffer = ptr_buffer->next_buffer)
    {
        ptr_buffer->number--;
    }
    
    /* free lines and messages */
    while (buffer->lines)
    {
        ptr_line = buffer->lines->next_line;
        gui_buffer_line_free (buffer->lines);
        buffer->lines = ptr_line;
    }
    
    /* close log if opened */
    if (buffer->log_file)
        gui_log_end (buffer);
    
    if (buffer->input_buffer)
        free (buffer->input_buffer);
    if (buffer->input_buffer_color_mask)
        free (buffer->input_buffer_color_mask);
    
    completion_free (&(buffer->completion));
    
    history_buffer_free (buffer);
    
    if (buffer->text_search_input)
        free (buffer->text_search_input);
    
    /* remove buffer from buffers list */
    if (buffer->prev_buffer)
        buffer->prev_buffer->next_buffer = buffer->next_buffer;
    if (buffer->next_buffer)
        buffer->next_buffer->prev_buffer = buffer->prev_buffer;
    if (gui_buffers == buffer)
        gui_buffers = buffer->next_buffer;
    if (last_gui_buffer == buffer)
        last_gui_buffer = buffer->prev_buffer;
    
    for (ptr_win = gui_windows; ptr_win; ptr_win = ptr_win->next_window)
    {
        if (ptr_win->buffer == buffer)
            ptr_win->buffer = NULL;
    }
    
    free (buffer);
    
    /* always at least one buffer */
    if (!gui_buffers && create_new && switch_to_another)
        (void) gui_buffer_new (gui_windows, NULL, NULL,
                               GUI_BUFFER_TYPE_STANDARD, 1);
    
    if (gui_windows && gui_current_window && gui_current_window->buffer)
        gui_status_draw (gui_current_window->buffer, 1);
}

/*
 * gui_buffer_line_new: create new line for a buffer
 */

t_gui_line *
gui_buffer_line_new (t_gui_buffer *buffer, time_t date)
{
    t_gui_line *new_line, *ptr_line;
    
    if ((new_line = (t_gui_line *) malloc (sizeof (struct t_gui_line))))
    {
        new_line->length = 0;
        new_line->length_align = 0;
        new_line->log_write = 1;
        new_line->line_with_message = 0;
        new_line->line_with_highlight = 0;
        new_line->date = date;
        new_line->nick = NULL;
        new_line->data = NULL;
        new_line->ofs_after_date = -1;
        new_line->ofs_start_message = -1;
        if (!buffer->lines)
            buffer->lines = new_line;
        else
            buffer->last_line->next_line = new_line;
        new_line->prev_line = buffer->last_line;
        new_line->next_line = NULL;
        buffer->last_line = new_line;
        buffer->num_lines++;
    }
    else
    {
        weechat_log_printf (_("Not enough memory for new line\n"));
        return NULL;
    }
    
    /* remove one line if necessary */
    if ((cfg_history_max_lines > 0)
        && (buffer->num_lines > cfg_history_max_lines))
    {
        if (buffer->last_line == buffer->lines)
            buffer->last_line = NULL;
        ptr_line = buffer->lines->next_line;
        gui_buffer_line_free (buffer->lines);
        buffer->lines = ptr_line;
        ptr_line->prev_line = NULL;
        buffer->num_lines--;
    }
    
    return new_line;
}

/*
 * gui_buffer_line_search: search for text in a line
 */

int
gui_buffer_line_search (t_gui_line *line, char *text, int case_sensitive)
{
    char *data;
    int rc;
    
    if (!line || !line->data || !text || !text[0])
        return 0;
    
    rc = 0;
    data = (char *)gui_color_decode ((unsigned char *)line->data, 0, 0);
    if (data)
    {
        if ((case_sensitive && (strstr (data, text)))
            || (!case_sensitive && (ascii_strcasestr (data, text))))
            rc = 1;
        free (data);
    }
    return rc;
}

/*
 * gui_buffer_merge_servers: merge server buffers in one buffer
 */

void
gui_buffer_merge_servers (t_gui_window *window)
{
    t_gui_buffer *ptr_buffer_server, *ptr_buffer, *new_ptr_buffer;
    t_irc_server *ptr_server;
    
    /* new server buffer is the first server buffer found */
    for (ptr_buffer_server = gui_buffers; ptr_buffer_server;
         ptr_buffer_server = ptr_buffer_server->next_buffer)
    {
        if (GUI_BUFFER_IS_SERVER(ptr_buffer_server))
            break;
    }
    
    /* no server buffer found */
    if (!ptr_buffer_server)
        return;

    ptr_buffer = gui_buffers;
    while (ptr_buffer)
    {
        if ((ptr_buffer != ptr_buffer_server)
            && (GUI_BUFFER_IS_SERVER(ptr_buffer)))
        {
            ptr_server = GUI_SERVER(ptr_buffer);
            
            /* add (by pointer artefact) lines from buffer found to server buffer */
            if (ptr_buffer->lines)
            {
                if (ptr_buffer_server->lines)
                {
                    ptr_buffer->lines->prev_line =
                        ptr_buffer_server->last_line;
                    ptr_buffer_server->last_line->next_line =
                        ptr_buffer->lines;
                    ptr_buffer_server->last_line =
                        ptr_buffer->last_line;
                }
                else
                {
                    ptr_buffer_server->lines = ptr_buffer->lines;
                    ptr_buffer_server->last_line = ptr_buffer->last_line;
                }
            }
            
            /* free buffer but not lines, because they're now used by 
               our unique server buffer */
            new_ptr_buffer = ptr_buffer->next_buffer;
            ptr_buffer->lines = NULL;
            gui_buffer_free (ptr_buffer, 1);
            ptr_buffer = new_ptr_buffer;
            
            /* asociate server with new server buffer */
            ptr_server->buffer = ptr_buffer_server;
        }
        else
            ptr_buffer = ptr_buffer->next_buffer;
    }
    
    ptr_buffer_server->all_servers = 1;
    gui_window_redraw_buffer (window->buffer);
}

/*
 * gui_buffer_split_server: split the server buffer into many buffers (one by server)
 */

void
gui_buffer_split_server (t_gui_window *window)
{
    t_gui_buffer *ptr_buffer;
    t_irc_server *ptr_server;
    
    ptr_buffer = gui_buffer_servers_search ();
    
    if (ptr_buffer)
    {
        if (GUI_SERVER(ptr_buffer))
        {
            for (ptr_server = irc_servers; ptr_server;
                 ptr_server = ptr_server->next_server)
            {
                if (ptr_server->buffer
                    && (ptr_server != GUI_SERVER(ptr_buffer))
                    && (ptr_server->buffer == ptr_buffer))
                {
                    ptr_server->buffer = NULL;
                    gui_buffer_new (window, ptr_server, NULL,
                                    GUI_BUFFER_TYPE_STANDARD, 0);
                }
            }
        }
        ptr_buffer->all_servers = 0;
        gui_status_draw (window->buffer, 1);
        gui_input_draw (window->buffer, 1);
    }
}

/*
 * gui_buffer_switch_previous: switch to previous buffer
 */

void
gui_buffer_switch_previous (t_gui_window *window)
{
    if (!gui_ok)
        return;
    
    /* if only one buffer then return */
    if (gui_buffers == last_gui_buffer)
        return;
    
    if (window->buffer->prev_buffer)
        gui_window_switch_to_buffer (window, window->buffer->prev_buffer);
    else
        gui_window_switch_to_buffer (window, last_gui_buffer);
    
    gui_window_redraw_buffer (window->buffer);
}

/*
 * gui_buffer_switch_next: switch to next buffer
 */

void
gui_buffer_switch_next (t_gui_window *window)
{
    if (!gui_ok)
        return;
    
    /* if only one buffer then return */
    if (gui_buffers == last_gui_buffer)
        return;
    
    if (window->buffer->next_buffer)
        gui_window_switch_to_buffer (window, window->buffer->next_buffer);
    else
        gui_window_switch_to_buffer (window, gui_buffers);
    
    gui_window_redraw_buffer (window->buffer);
}

/*
 * gui_buffer_switch_dcc: switch to dcc buffer (create it if it does not exist)
 */

void
gui_buffer_switch_dcc (t_gui_window *window)
{
    t_gui_buffer *ptr_buffer;
    
    /* check if dcc buffer exists */
    for (ptr_buffer = gui_buffers; ptr_buffer; ptr_buffer = ptr_buffer->next_buffer)
    {
        if (ptr_buffer->type == GUI_BUFFER_TYPE_DCC)
            break;
    }
    if (ptr_buffer)
    {
        gui_window_switch_to_buffer (window, ptr_buffer);
        gui_window_redraw_buffer (ptr_buffer);
    }
    else
        gui_buffer_new (window, NULL, NULL, GUI_BUFFER_TYPE_DCC, 1);
}

/*
 * gui_buffer_switch_raw_data: switch to rax IRC data buffer (create it if it does not exist)
 */

void
gui_buffer_switch_raw_data (t_gui_window *window)
{
    t_gui_buffer *ptr_buffer;
    
    /* check if raw IRC data buffer exists */
    for (ptr_buffer = gui_buffers; ptr_buffer; ptr_buffer = ptr_buffer->next_buffer)
    {
        if (ptr_buffer->type == GUI_BUFFER_TYPE_RAW_DATA)
            break;
    }
    if (ptr_buffer)
    {
        gui_window_switch_to_buffer (window, ptr_buffer);
        gui_window_redraw_buffer (ptr_buffer);
    }
    else
        gui_buffer_new (window, NULL, NULL, GUI_BUFFER_TYPE_RAW_DATA, 1);
}

/*
 * gui_buffer_switch_by_number: switch to another buffer with number
 */

t_gui_buffer *
gui_buffer_switch_by_number (t_gui_window *window, int number)
{
    t_gui_buffer *ptr_buffer;
    
    /* invalid buffer */
    if (number < 0)
        return NULL;
    
    /* buffer is currently displayed ? */
    if (number == window->buffer->number)
        return window->buffer;
    
    /* search for buffer in the list */
    for (ptr_buffer = gui_buffers; ptr_buffer; ptr_buffer = ptr_buffer->next_buffer)
    {
        if ((ptr_buffer != window->buffer) && (number == ptr_buffer->number))
        {
            gui_window_switch_to_buffer (window, ptr_buffer);
            gui_window_redraw_buffer (window->buffer);
            return ptr_buffer;
        }
    }
    
    /* buffer not found */
    return NULL;
}

/*
 * gui_buffer_move_to_number: move a buffer to another number
 */

void
gui_buffer_move_to_number (t_gui_buffer *buffer, int number)
{
    t_gui_buffer *ptr_buffer;
    int i;
#ifdef PLUGINS
    char buf1_str[16], buf2_str[16], *argv[2] = { NULL, NULL };
#endif
    
    /* if only one buffer then return */
    if (gui_buffers == last_gui_buffer)
        return;
    
    /* buffer number is already ok ? */
    if (number == buffer->number)
        return;
    
    if (number < 1)
        number = 1;

#ifdef PLUGINS
    snprintf (buf2_str, sizeof (buf2_str) - 1, "%d", buffer->number);
#endif
    
    /* remove buffer from list */
    if (buffer == gui_buffers)
    {
        gui_buffers = buffer->next_buffer;
        gui_buffers->prev_buffer = NULL;
    }
    if (buffer == last_gui_buffer)
    {
        last_gui_buffer = buffer->prev_buffer;
        last_gui_buffer->next_buffer = NULL;
    }
    if (buffer->prev_buffer)
        (buffer->prev_buffer)->next_buffer = buffer->next_buffer;
    if (buffer->next_buffer)
        (buffer->next_buffer)->prev_buffer = buffer->prev_buffer;
    
    if (number == 1)
    {
        gui_buffers->prev_buffer = buffer;
        buffer->prev_buffer = NULL;
        buffer->next_buffer = gui_buffers;
        gui_buffers = buffer;
    }
    else
    {
        /* assign new number to all buffers */
        i = 1;
        for (ptr_buffer = gui_buffers; ptr_buffer; ptr_buffer = ptr_buffer->next_buffer)
        {
            ptr_buffer->number = i++;
        }
        
        /* search for new position in the list */
        for (ptr_buffer = gui_buffers; ptr_buffer; ptr_buffer = ptr_buffer->next_buffer)
        {
            if (ptr_buffer->number == number)
                break;
        }
        if (ptr_buffer)
        {
            /* insert before buffer found */
            buffer->prev_buffer = ptr_buffer->prev_buffer;
            buffer->next_buffer = ptr_buffer;
            if (ptr_buffer->prev_buffer)
                (ptr_buffer->prev_buffer)->next_buffer = buffer;
            ptr_buffer->prev_buffer = buffer;
        }
        else
        {
            /* number not found (too big)? => add to end */
            buffer->prev_buffer = last_gui_buffer;
            buffer->next_buffer = NULL;
            last_gui_buffer->next_buffer = buffer;
            last_gui_buffer = buffer;
        }
        
    }
    
    /* assign new number to all buffers */
    i = 1;
    for (ptr_buffer = gui_buffers; ptr_buffer; ptr_buffer = ptr_buffer->next_buffer)
    {
        ptr_buffer->number = i++;
    }
    
    gui_window_redraw_buffer (buffer);
    
#ifdef PLUGINS
    snprintf (buf1_str, sizeof (buf1_str) - 1, "%d", buffer->number);
    argv[0] = buf1_str;
    argv[1] = buf2_str;
    (void) plugin_event_handler_exec ("buffer_move", 2, argv);
#endif
}

/*
 * gui_buffer_search_text: search text in a buffer
 */

int
gui_buffer_search_text (t_gui_window *window)
{
    t_gui_line *ptr_line;
    
    if (window->buffer->text_search == GUI_TEXT_SEARCH_BACKWARD)
    {
        if (window->buffer->lines
            && window->buffer->input_buffer && window->buffer->input_buffer[0])
        {
            ptr_line = (window->start_line) ?
                window->start_line->prev_line : window->buffer->last_line;
            while (ptr_line)
            {
                if (gui_buffer_line_search (ptr_line, window->buffer->input_buffer,
                                            window->buffer->text_search_exact))
                {
                    window->start_line = ptr_line;
                    window->start_line_pos = 0;
                    window->first_line_displayed =
                        (window->start_line == window->buffer->lines);
                    gui_chat_draw (window->buffer, 1);
                    gui_status_draw (window->buffer, 1);
                    return 1;
                }
                ptr_line = ptr_line->prev_line;
            }
        }
    }
    else if (window->buffer->text_search == GUI_TEXT_SEARCH_FORWARD)
    {
        if (window->buffer->lines
            && window->buffer->input_buffer && window->buffer->input_buffer[0])
        {
            ptr_line = (window->start_line) ?
                window->start_line->next_line : window->buffer->lines->next_line;
            while (ptr_line)
            {
                if (gui_buffer_line_search (ptr_line, window->buffer->input_buffer,
                                            window->buffer->text_search_exact))
                {
                    window->start_line = ptr_line;
                    window->start_line_pos = 0;
                    window->first_line_displayed =
                        (window->start_line == window->buffer->lines);
                    gui_chat_draw (window->buffer, 1);
                    gui_status_draw (window->buffer, 1);
                    return 1;
                }
                ptr_line = ptr_line->next_line;
            }
        }
    }
    return 0;
}

/*
 * gui_buffer_search_start: start search in a buffer
 */

void
gui_buffer_search_start (t_gui_window *window)
{
    window->buffer->text_search = GUI_TEXT_SEARCH_BACKWARD;
    window->buffer->text_search_exact = 0;
    window->buffer->text_search_found = 0;
    if (window->buffer->text_search_input)
    {
        free (window->buffer->text_search_input);
        window->buffer->text_search_input = NULL;
    }
    if (window->buffer->input_buffer && window->buffer->input_buffer[0])
        window->buffer->text_search_input =
            strdup (window->buffer->input_buffer);
    gui_action_delete_line (window, NULL);
    gui_status_draw (window->buffer, 1);
    gui_input_draw (window->buffer, 1);
}

/*
 * gui_buffer_search_restart: restart search (after input changes or exact
 *                            flag (un)set)
 */

void
gui_buffer_search_restart (t_gui_window *window)
{
    window->start_line = NULL;
    window->start_line_pos = 0;
    window->buffer->text_search = GUI_TEXT_SEARCH_BACKWARD;
    window->buffer->text_search_found = 0;
    if (gui_buffer_search_text (window))
        window->buffer->text_search_found = 1;
    else
    {
        gui_chat_draw (window->buffer, 1);
        gui_status_draw (window->buffer, 1);
    }
}

/*
 * gui_buffer_search_stop: stop search in a buffer
 */

void
gui_buffer_search_stop (t_gui_window *window)
{
    window->buffer->text_search = GUI_TEXT_SEARCH_DISABLED;
    window->buffer->text_search = 0;
    gui_action_delete_line (window, NULL);
    if (window->buffer->text_search_input)
    {
        gui_insert_string_input (window, window->buffer->text_search_input, -1);
        free (window->buffer->text_search_input);
        window->buffer->text_search_input = NULL;
    }
    window->start_line = NULL;
    window->start_line_pos = 0;
    hotlist_remove_buffer (window->buffer);
    gui_chat_draw (window->buffer, 0);
    gui_status_draw (window->buffer, 1);
    gui_input_draw (window->buffer, 1);
}

/*
 * gui_buffer_scroll: scroll buffer by # messages or time
 */

void
gui_buffer_scroll (t_gui_window *window, char *scroll)
{
    int direction, stop, count_msg;
    char time_letter, saved_char;
    time_t old_date, diff_date;
    char *error;
    long number;
    t_gui_line *ptr_line;
    struct tm *date_tmp, line_date, old_line_date;
    
    if (window->buffer->lines)
    {
        direction = -1;
        number = 0;
        time_letter = ' ';
        
        // search direction
        if (scroll[0] == '-')
        {
            direction = -1;
            scroll++;
        }
        else if (scroll[0] == '+')
        {
            direction = +1;
            scroll++;
        }
        
        // search number and letter
        char *pos = scroll;
        while (pos && pos[0] && isdigit (pos[0]))
        {
            pos++;
        }
        if (pos)
        {
            if (pos == scroll)
            {
                if (pos[0])
                    time_letter = scroll[0];
            }
            else
            {
                if (pos[0])
                    time_letter = pos[0];
                saved_char = pos[0];
                pos[0] = '\0';
                error = NULL;
                number = strtol (scroll, &error, 10);
                if (!error || (error[0] != '\0'))
                    number = 0;
                pos[0] = saved_char;
            }
        }
        
        /* at least number or letter has to he given */
        if ((number == 0) && (time_letter == ' '))
            return;
        
        // do the scroll!
        stop = 0;
        count_msg = 0;
        if (direction < 0)
            ptr_line = (window->start_line) ?
                window->start_line : window->buffer->last_line;
        else
            ptr_line = (window->start_line) ?
                window->start_line : window->buffer->lines;
        
        old_date = ptr_line->date;
        date_tmp = localtime (&old_date);
        memcpy (&old_line_date, date_tmp, sizeof (struct tm));
        
        while (ptr_line)
        {
            ptr_line = (direction < 0) ? ptr_line->prev_line : ptr_line->next_line;
            
            if (ptr_line)
            {
                if (time_letter == ' ')
                {
                    count_msg++;
                    if (count_msg >= number)
                        stop = 1;
                }
                else
                {
                    date_tmp = localtime (&(ptr_line->date));
                    memcpy (&line_date, date_tmp, sizeof (struct tm));
                    if (old_date > ptr_line->date)
                        diff_date = old_date - ptr_line->date;
                    else
                        diff_date = ptr_line->date - old_date;
                    switch (time_letter)
                    {
                        case 's': /* seconds */
                            if (number == 0)
                            {
                                /* stop if line has different second */
                                if ((line_date.tm_sec != old_line_date.tm_sec)
                                    || (line_date.tm_min != old_line_date.tm_min)
                                    || (line_date.tm_hour != old_line_date.tm_hour)
                                    || (line_date.tm_mday != old_line_date.tm_mday)
                                    || (line_date.tm_mon != old_line_date.tm_mon)
                                    || (line_date.tm_year != old_line_date.tm_year))
                                    if (line_date.tm_sec != old_line_date.tm_sec)
                                        stop = 1;
                            }
                            else if (diff_date >= number)
                                stop = 1;
                            break;
                        case 'm': /* minutes */
                            if (number == 0)
                            {
                                /* stop if line has different minute */
                                if ((line_date.tm_min != old_line_date.tm_min)
                                    || (line_date.tm_hour != old_line_date.tm_hour)
                                    || (line_date.tm_mday != old_line_date.tm_mday)
                                    || (line_date.tm_mon != old_line_date.tm_mon)
                                    || (line_date.tm_year != old_line_date.tm_year))
                                    stop = 1;
                            }
                            else if (diff_date >= number * 60)
                                stop = 1;
                            break;
                        case 'h': /* hours */
                            if (number == 0)
                            {
                                /* stop if line has different hour */
                                if ((line_date.tm_hour != old_line_date.tm_hour)
                                    || (line_date.tm_mday != old_line_date.tm_mday)
                                    || (line_date.tm_mon != old_line_date.tm_mon)
                                    || (line_date.tm_year != old_line_date.tm_year))
                                    stop = 1;
                            }
                            else if (diff_date >= number * 60 * 60)
                                stop = 1;
                            break;
                        case 'd': /* days */
                            if (number == 0)
                            {
                                /* stop if line has different day */
                                if ((line_date.tm_mday != old_line_date.tm_mday)
                                    || (line_date.tm_mon != old_line_date.tm_mon)
                                    || (line_date.tm_year != old_line_date.tm_year))
                                    stop = 1;
                            }
                            else if (diff_date >= number * 60 * 60 * 24)
                                stop = 1;
                            break;
                        case 'M': /* months */
                            if (number == 0)
                            {
                                /* stop if line has different month */
                                if ((line_date.tm_mon != old_line_date.tm_mon)
                                    || (line_date.tm_year != old_line_date.tm_year))
                                    stop = 1;
                            }
                            /* we consider month is 30 days, who will find I'm too
                               lazy to code exact date diff ? ;) */
                            else if (diff_date >= number * 60 * 60 * 24 * 30)
                                stop = 1;
                            break;
                        case 'y': /* years */
                            if (number == 0)
                            {
                                /* stop if line has different year */
                                if (line_date.tm_year != old_line_date.tm_year)
                                    stop = 1;
                            }
                            /* we consider year is 365 days, who will find I'm too
                               lazy to code exact date diff ? ;) */
                            else if (diff_date >= number * 60 * 60 * 24 * 365)
                                stop = 1;
                            break;
                    }
                }
                if (stop)
                {
                    window->start_line = ptr_line;
                    window->start_line_pos = 0;
                    window->first_line_displayed =
                        (window->start_line == window->buffer->lines);
                    gui_chat_draw (window->buffer, 1);
                    gui_status_draw (window->buffer, 0);
                    return;
                }
            }
        }
        if (direction < 0)
            gui_window_scroll_top (window);
        else
            gui_window_scroll_bottom (window);
    }
}

/*
 * gui_buffer_dump_hexa: dump content of buffer as hexa data in log file
 */

void
gui_buffer_dump_hexa (t_gui_buffer *buffer)
{
    t_gui_line *ptr_line;
    int num_line, data_pos;
    char *data_without_colors;
    char hexa[(16 * 3) + 1], ascii[(16 * 2) + 1];
    int hexa_pos, ascii_pos;
    
    weechat_log_printf ("[buffer dump hexa (addr:0x%X)]\n", buffer);
    num_line = 1;
    for (ptr_line = buffer->lines; ptr_line; ptr_line = ptr_line->next_line)
    {
        /* display line without colors */
        data_without_colors = (ptr_line->data) ?
            (char *)gui_color_decode ((unsigned char *)ptr_line->data, 0, 0) : NULL;
        weechat_log_printf ("\n");
        weechat_log_printf ("  line %d: %s\n",
                            num_line,
                            (data_without_colors) ? data_without_colors : "(null)");
        if (data_without_colors)
            free (data_without_colors);

        /* display raw data for line */
        if (ptr_line->data)
        {
            weechat_log_printf ("\n");
            weechat_log_printf ("  raw data for line %d (with color codes):\n",
                                num_line);
            data_pos = 0;
            hexa_pos = 0;
            ascii_pos = 0;
            while (ptr_line->data[data_pos])
            {
                snprintf (hexa + hexa_pos, 4, "%02X ",
                          (unsigned char)(ptr_line->data[data_pos]));
                hexa_pos += 3;
                snprintf (ascii + ascii_pos, 3, "%c ",
                          ((((unsigned char)ptr_line->data[data_pos]) < 32)
                           || (((unsigned char)ptr_line->data[data_pos]) > 127)) ?
                          '.' : (unsigned char)(ptr_line->data[data_pos]));
                ascii_pos += 2;
                if (ascii_pos == 32)
                {
                    weechat_log_printf ("    %-48s  %s\n", hexa, ascii);
                    hexa_pos = 0;
                    ascii_pos = 0;
                }
                data_pos++;
            }
            if (ascii_pos > 0)
                weechat_log_printf ("    %-48s  %s\n", hexa, ascii);
        }
        
        num_line++;
    }
}

/*
 * gui_buffer_print_log: print buffer infos in log (usually for crash dump)
 */

void
gui_buffer_print_log (t_gui_buffer *buffer)
{
    t_gui_line *ptr_line;
    int num;
    
    weechat_log_printf ("[buffer (addr:0x%X)]\n", buffer);
    weechat_log_printf ("  num_displayed. . . . . : %d\n",   buffer->num_displayed);
    weechat_log_printf ("  number . . . . . . . . : %d\n",   buffer->number);
    weechat_log_printf ("  server . . . . . . . . : 0x%X\n", buffer->server);
    weechat_log_printf ("  all_servers. . . . . . : %d\n",   buffer->all_servers);
    weechat_log_printf ("  channel. . . . . . . . : 0x%X\n", buffer->channel);
    weechat_log_printf ("  type . . . . . . . . . : %d\n",   buffer->type);
    weechat_log_printf ("  lines. . . . . . . . . : 0x%X\n", buffer->lines);
    weechat_log_printf ("  last_line. . . . . . . : 0x%X\n", buffer->last_line);
    weechat_log_printf ("  last_read_line . . . . : 0x%X\n", buffer->last_read_line);
    weechat_log_printf ("  num_lines. . . . . . . : %d\n",   buffer->num_lines);
    weechat_log_printf ("  line_complete. . . . . : %d\n",   buffer->line_complete);
    weechat_log_printf ("  notify_level . . . . . : %d\n",   buffer->notify_level);
    weechat_log_printf ("  log_filename . . . . . : '%s'\n", buffer->log_filename);
    weechat_log_printf ("  log_file . . . . . . . : 0x%X\n", buffer->log_file);
    weechat_log_printf ("  has_input. . . . . . . : %d\n",   buffer->has_input);
    weechat_log_printf ("  input_buffer . . . . . : '%s'\n", buffer->input_buffer);
    weechat_log_printf ("  input_buffer_color_mask: '%s'\n", buffer->input_buffer_color_mask);
    weechat_log_printf ("  input_buffer_alloc . . : %d\n",   buffer->input_buffer_alloc);
    weechat_log_printf ("  input_buffer_size. . . : %d\n",   buffer->input_buffer_size);
    weechat_log_printf ("  input_buffer_length. . : %d\n",   buffer->input_buffer_length);
    weechat_log_printf ("  input_buffer_pos . . . : %d\n",   buffer->input_buffer_pos);
    weechat_log_printf ("  input_buffer_1st_disp. : %d\n",   buffer->input_buffer_1st_display);
    weechat_log_printf ("  completion . . . . . . : 0x%X\n", &(buffer->completion));
    weechat_log_printf ("  history. . . . . . . . : 0x%X\n", buffer->history);
    weechat_log_printf ("  last_history . . . . . : 0x%X\n", buffer->last_history);
    weechat_log_printf ("  ptr_history. . . . . . : 0x%X\n", buffer->ptr_history);
    weechat_log_printf ("  num_history. . . . . . : %d\n",   buffer->num_history);
    weechat_log_printf ("  text_search. . . . . . : %d\n",   buffer->text_search);
    weechat_log_printf ("  text_search_exact. . . : %d\n",   buffer->text_search_exact);
    weechat_log_printf ("  text_search_input. . . : '%s'\n", buffer->text_search_input);
    weechat_log_printf ("  prev_buffer. . . . . . : 0x%X\n", buffer->prev_buffer);
    weechat_log_printf ("  next_buffer. . . . . . : 0x%X\n", buffer->next_buffer);
    weechat_log_printf ("\n");
    weechat_log_printf ("  => last 100 lines:\n");
    
    num = 0;
    ptr_line = buffer->last_line;
    while (ptr_line && (num < 100))
    {
        num++;
        ptr_line = ptr_line->prev_line;
    }
    if (!ptr_line)
        ptr_line = buffer->lines;
    else
        ptr_line = ptr_line->next_line;
    
    while (ptr_line)
    {
        num--;
        weechat_log_printf ("       line N-%05d: %s\n",
                            num,
                            (ptr_line->data) ? ptr_line->data : "(empty)");
        
        ptr_line = ptr_line->next_line;
    }
    
    weechat_log_printf ("\n");
    completion_print_log (&(buffer->completion));
}
