#!/usr/bin/perl -w

# apt-cacher-import.pl
# Script to import .deb packages into the Apt-cacher package caching system.
# This script does not need to be run when setting up Apt-cacher for the first
# time: its purpose is to initialise .deb packages that have been copied in
# from some other source, such as a local mirror. Apt-cacher doesn't store
# it's cached .debs in plain format, it prepends HTTP headers to them to send
# out to clients when a package is requested. It also keeps track of which
# packages are fully downloaded by touching a '.complete' file in the 'private'
# directory in the cache. If .debs are just copied straight into the cache
# dir Apt-cacher won't use them because it thinks they are both corrupt (no
# headers) and incomplete (no .complete file). This script allows you to
# copy a bunch of .debs into an import dir, then run this script to prepend
# the HTTP headers and touch the .complete file after moving them to the cache
# dir.
#
# Usage:
# 1. Place your plain debs into /var/cache/apt-cacher/import (or where-ever
#    you set the cache dir to be)
# 2. Run this script: /usr/share/apt-cacher-import.pl
#
# For more information visit www.apt-cacher.org
#
# Copyright (C) 2004, Jonathan Oxer <jon@debian.org>
# Distributed under the terms of the GNU Public Licence (GPL).

#use strict;
#############################################################################
### configuration ###########################################################
# Include the library for the config file parser
require '/usr/share/apt-cacher/apt-cacher-lib.pl';

# Read in the config file and set the necessary variables
my $configfile = '/etc/apt-cacher/apt-cacher.conf';

my $configref;
eval {
        $configref = read_config($configfile);
};
my %config = %$configref;

# not sure what to do if we can't read the config file...
die "Could not read config file: $@" if $@;

my $private_dir = "$config{cache_dir}/private";
my $import_dir = "$config{cache_dir}/import";
my $target_dir = "$config{cache_dir}/packages";
my $header_dir = "$config{cache_dir}/headers";

my $packagesimported = 0;

#############################################################################

if(!$ARGV[0]) {
   syswrite(STDOUT, "No import directory specified as the first argument, using $import_dir\n");
   sleep 2;
}
else {
   $import_dir=$ARGV[0];
}

die "Cannot write to $target_dir" if !-w $target_dir;
die "Cannot write to $header_dir" if !-w $header_dir;

@info = stat($private_dir);

# Move to the import dir to chomp on the actual .deb packages
chdir("$import_dir") || die "apt-cacher-import.pl: can't open the import directory ($import_dir)";

### Loop through all the .debs in the import dir
foreach $packagefile ( <*.deb>, <*.gz>, <*.bz2> ) {
	
	
	# Get some things we need to insert into the header
	$headerdate = `date +"%a, %d %b %Y %X %Z"`;
	$headerdate =~ s/^\s*(.*?)\s*$/$1/;
	($jk,$jk,$jk,$jk,$jk,$jk,$jk,$headerlength,$jk,$jk,$jk,$jk,$jk) = stat($packagefile);
	$headeretag = `date +%H%m%s%N`;
	$headeretag =~ s/^\s*(.*?)\s*$/$1/;
	#print "d: $headerdate, l: $headerlength, e: $headeretag\n";
  $frompackagefile=$packagefile; # backup of the original name
  $packagefile=~s/_\d+%3a/_/;
		
	# Generate a header
	$httpheader = "HTTP/1.1 200 OK
Date: ".$headerdate."
Server: Apache \(Unix\) apt-cacher
Last-Modified: ".$headerdate."
ETag: \"".$headeretag."\"
Accept-Ranges: bytes
Content-Length: ".$headerlength."
Keep-Alive: timeout=10, max=128
Connection: Keep-Alive
Content-Type: application/x-debian-package

"; # there are TWO new lines
		
	# Then cat the header to a temp file
	print "Importing: $packagefile\n";
  unlink "$header_dir/$packagefile", "$target_dir/$packagefile",  "$private_dir/$packagefile.complete"; # just to be sure
  rename($frompackagefile, "$target_dir/$packagefile");
  open($headfile, ">$header_dir/$packagefile");
  print $headfile $httpheader;
  close $headfile;
		
	$completefile = "$private_dir/$packagefile.complete";
	open(MF, ">$completefile");
  close(MF);
  # copy the ownership of the private directory
  chown $info[4], $info[5], "$header_dir/$packagefile", "$target_dir/$packagefile",  "$private_dir/$packagefile.complete";

	$packagesimported++;
}

print "Done.\n";
print "Packages imported: $packagesimported\n";

# Woohoo, all done!
exit 0;

