/* monoalphabetic.key-controls.c - controls for manipulating an a-z key
 * 
 * This program is part of Crank, a cryptanalysis tool
 * Copyright (C) 2000 Matthew Russell
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License (LICENSE) for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include "crank-interface.h"
#include "common.monoalphabetic.h"

/* Global data */
key global_key;
GtkWidget *global_key_labels[26];  /* Pointers to the labels for the global key */

static char tochar = 'a', fromchar = 'A'; /* Stores the current values of the from and to selection lists */

/* Function declares */
static void addButton(GtkWidget *table, const char *text, void callback(GtkWidget *widget, gpointer), int la, int ra, int ta, int ba);
static GtkWidget *make_key_controls(void);
static void cb_invert (GtkWidget *widget, gpointer gdata);
static void cb_shift_L (GtkWidget *widget, gpointer gdata);
static void cb_shift_R (GtkWidget *widget, gpointer gdata);
static void cb_reverse (GtkWidget *widget, gpointer gdata);
static void cb_complete (GtkWidget *widget, gpointer gdata);
static void cb_clear (GtkWidget *widget, gpointer gdata);
static void cb_change (GtkWidget *widget, gpointer gdata);
static void cb_change_from_select(GtkWidget *widget, gpointer gdata);
static void cb_change_to_select(GtkWidget *widget, gpointer gdata);

/* Plugin Interface */
/* ---------------- */

const char name[] = "monoalphabetic.key-controls";
const char version[] = VERSION;
const char author[] = "Matthew Russell";
const char description[] = "Some tools for manually manipulating a monoalphabetic key over 'a'-'z'.";
const int interface_version = 1;
const int plugin_type = PLUGIN_MODE;
const char menu_string[] = "/Monoalphabetic/Key Controls";

int boot(void) {
    key_identity(&global_key);
    return PLUGIN_BOOT_OK;
}

void init_plugin(void) {;}

void deactivate_plugin(void) {;}

GtkWidget *make_widget(char *text) {
    GtkWidget *plugin_controls;
    GtkWidget *key_display;
    GtkWidget *key_controls;
    printf("Entering make_widget()\n");
    fflush(stdout);

    plugin_controls = gtk_hbox_new(FALSE, 0); 
    
    key_display = make_key_display(&global_key, global_key_labels);
    printf("Successfully made key display\n");
    fflush(stdout);
    
    
    gtk_box_pack_start(GTK_BOX(plugin_controls), key_display, FALSE, TRUE, 0);
    gtk_widget_show(key_display);

    key_controls = make_key_controls();
    gtk_box_pack_start(GTK_BOX(plugin_controls), key_controls, FALSE, TRUE, 0);
    gtk_widget_show(key_controls);
    printf("Leaving make_widget()\n");
    fflush(stdout);

    return plugin_controls;
}

char *transform(char *text) {
    update_key_labels(&global_key, global_key_labels);
    return apply_key_text(&global_key, text);
}

void reset(void) {
    key_identity(&global_key);
}

/* Plugin implementation */
/* --------------------- */

static void addButton(GtkWidget *table, const char *text, void callback(GtkWidget *widget, gpointer), int la, int ra, int ta, int ba) {
    GtkWidget *btn;
    btn = gtk_button_new_with_label(text);
    gtk_signal_connect(GTK_OBJECT(btn), "pressed", GTK_SIGNAL_FUNC(callback), NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), btn, la, ra, ta, ba);
    gtk_widget_show(btn);
}

static GtkWidget *make_key_controls(void) {
    GtkWidget *table;
    GtkWidget *option_menu;
    GtkWidget *o_menu;
    GtkWidget *menu_item;
    GSList *group = NULL;
    int i;
    char buf[BUFFER_SIZE];

    table = gtk_table_new(3, 3, FALSE);
    
    addButton(table, "Invert", cb_invert,  0, 1, 0, 1);
    addButton(table, "Shift <<", cb_shift_L, 1, 2, 0, 1);
    addButton(table, "Shift >>", cb_shift_R, 2, 3, 0, 1);
    addButton(table, "Reverse", cb_reverse, 0, 1, 1, 2);
    addButton(table, "Complete", cb_complete, 1, 2, 1, 2);
    addButton(table, "Clear", cb_clear, 2, 3, 1, 2);
    addButton(table, "Change: ", cb_change, 0, 1, 2, 3);

    /* Change FROM menu */
    option_menu = gtk_option_menu_new();
    o_menu = gtk_menu_new();
    for (i = 0; i < 26; i++) {
	sprintf(buf, "%c", i + 'A');
        menu_item = gtk_radio_menu_item_new_with_label(group, buf);
	group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_item));
        gtk_menu_append(GTK_MENU(o_menu), menu_item);
        gtk_signal_connect(GTK_OBJECT(menu_item), "activate", GTK_SIGNAL_FUNC(cb_change_from_select), GINT_TO_POINTER((i + 'A')));
        gtk_widget_show(menu_item);
    }
    gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), o_menu);
    gtk_widget_show(o_menu);
    gtk_widget_show(option_menu);
    gtk_table_attach_defaults(GTK_TABLE(table), option_menu, 1, 2, 2, 3);
       
    /* Change TO menu */
    group = NULL;
    option_menu = gtk_option_menu_new();
    o_menu = gtk_menu_new();
    for (i = 0; i < 26; i++) {
	sprintf(buf, "%c", i + 'a');
        menu_item = gtk_radio_menu_item_new_with_label(group, buf);
	group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_item));
        gtk_menu_append(GTK_MENU(o_menu), menu_item);
        gtk_signal_connect(GTK_OBJECT(menu_item), "activate", GTK_SIGNAL_FUNC(cb_change_to_select), GINT_TO_POINTER((i + 'a')));
        gtk_widget_show(menu_item);
    }
    sprintf(buf, "Blank");
    menu_item = gtk_radio_menu_item_new_with_label(group, buf);
    group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(menu_item));
    gtk_menu_append(GTK_MENU(o_menu), menu_item);
    gtk_signal_connect(GTK_OBJECT(menu_item), "activate", GTK_SIGNAL_FUNC(cb_change_to_select), GINT_TO_POINTER((0)));
    gtk_widget_show(menu_item);
    gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), o_menu);
    gtk_widget_show(o_menu);
    gtk_widget_show(option_menu);
    gtk_table_attach_defaults(GTK_TABLE(table), option_menu, 2, 3, 2, 3);
   
    return table;
}

/* Callbacks */
/* --------- */

static void cb_invert (GtkWidget *widget, gpointer gdata) {
    key_invert(&global_key);
    update_key_labels(&global_key, global_key_labels);
    display_text();
}

static void cb_shift_L (GtkWidget *widget, gpointer gdata) {
    key_shift_L(&global_key);
    update_key_labels(&global_key, global_key_labels);
    display_text();
}

static void cb_shift_R (GtkWidget *widget, gpointer gdata) {
    key_shift_R(&global_key);
    update_key_labels(&global_key, global_key_labels);
    display_text();
}

static void cb_reverse (GtkWidget *widget, gpointer gdata) {
    key_reverse(&global_key);
    update_key_labels(&global_key, global_key_labels);
    display_text();
}

static void cb_complete (GtkWidget *widget, gpointer gdata) {
    key_complete(&global_key);
    update_key_labels(&global_key, global_key_labels);
    display_text();
}

static void cb_clear (GtkWidget *widget, gpointer gdata) {
    key_clear(&global_key);
    update_key_labels(&global_key, global_key_labels);
    display_text();
}

static void cb_change (GtkWidget *widget, gpointer gdata) {
    global_key[(int)fromchar] = tochar;
    update_key_labels(&global_key, global_key_labels);
    display_text();
}

static void cb_change_from_select(GtkWidget *widget, gpointer gdata) {
    fromchar = GPOINTER_TO_INT(gdata); 
}

static void cb_change_to_select(GtkWidget *widget, gpointer gdata) {
    tochar = GPOINTER_TO_INT(gdata);
}
