/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM2 Plugin
 * File: evms2/engine/plugins/lvm2/options.c
 *
 * Routines for managing options.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <plugin.h>
#include "lvm2.h"


/**
 * Create-container routines.
 **/


/**
 * create_container_init_task
 *
 * Initialize the acceptable-objects list and option-descriptor array for a
 * create-container task.
 **/
int create_container_init_task(task_context_t *context)
{
	option_desc_array_t *od = context->option_descriptors;
	list_anchor_t objects = NULL;
	int i, rc;

	LOG_ENTRY();

	/* Get a list of acceptable objects for creating a new container. */
	rc = get_available_objects(NULL, &objects);
	if (rc) {
		goto out;
	}
	EngFncs->merge_lists(context->acceptable_objects, objects, NULL, NULL);

	/* Initialize the "Name" option. Allocate space for the string value
	 * now so we can just strcpy during set_option().
	 */
	i = LVM2_OPTION_CREATE_CONTAINER_NAME_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_CREATE_CONTAINER_NAME_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Name for the new LVM2 container."));
	od->option[i].type = EVMS_Type_String;
	od->option[i].min_len = 1;
	od->option[i].max_len = EVMS_NAME_SIZE;
	od->option[i].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
	od->option[i].value.s = EngFncs->engine_alloc(EVMS_NAME_SIZE + 1);
	if (!od->option[i].value.s) {
		rc = ENOMEM;
		goto out;
	}

	/* Initialize the "Extent-Size" option. Wait until the objects have
	 * been selected during set_objects() to initialize the constraint-list.
	 */
	i = LVM2_OPTION_CREATE_CONTAINER_EXTENT_SIZE_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_CREATE_CONTAINER_EXTENT_SIZE_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Extent-size for the new LVM2 container."));
	od->option[i].tip = EngFncs->engine_strdup(_("Extent-size must be a power-of-2 and at least 8kB."));
	od->option[i].type = EVMS_Type_Unsigned_Int64;
	od->option[i].unit = EVMS_Unit_Sectors;
	od->option[i].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED | EVMS_OPTION_FLAGS_AUTOMATIC;
	od->option[i].constraint_type = EVMS_Collection_List;

	od->count = LVM2_OPTION_CREATE_CONTAINER_COUNT;
	context->min_selected_objects = 1;
	context->max_selected_objects = -1;

out:
	EngFncs->destroy_list(objects);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * create_container_set_objects
 *
 * Validate that the objects in the selected-objects list are valid for creating
 * a new container. Make adjustments to the option-descriptor as appropriate.
 **/
int create_container_set_objects(task_context_t *context,
				 task_effect_t *effect)
{
	option_desc_array_t *od = context->option_descriptors;
	u_int64_t mask = 1, size = (u_int64_t) -1;
	storage_object_t *object;
	list_element_t iter;
	int i;

	LOG_ENTRY();

	/* Determine the max allowed extent-size for this list of objects.
	 * Each object must be big enough for one extent plus metadata.
	 */
	LIST_FOR_EACH(context->selected_objects, iter, object) {
		size = min(object->size, size);
	}

	/* Subtract space for the metadata. */
	size -= (LVM2_LABEL_SCAN_SECTORS + LVM2_DEFAULT_MDA_SIZE);

	/* Round down to a power-of-2. */
	while (size & (size - 1)) {
		size &= ~mask;
		mask <<= 1;
	}
	LOG_DETAILS("Maximum allowed extent size is %"PRIu64".\n", size);

	/* Update the extent-size entry in the option-descriptor. */
	i = LVM2_OPTION_CREATE_CONTAINER_EXTENT_SIZE_IDX;
	EngFncs->engine_free(od->option[i].constraint.list);
	SET_POWER2_LIST64(od->option[i].constraint.list,
			  LVM2_MIN_EXTENT_SIZE, size);
	od->option[i].value.ui64 = min(LVM2_DEFAULT_EXTENT_SIZE, size);

	*effect |= EVMS_Effect_Reload_Options;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * validate_vg_name
 *
 * Make sure the proposed container name is not already in use.
 **/
static int validate_vg_name(char *vg_name, storage_container_t *disk_group)
{
	char container_name[EVMS_NAME_SIZE+1];
	int rc;

	LOG_ENTRY();

	if (strchr(vg_name, ' ')) {
		LOG_WARNING("Container name (%s) cannot contain spaces.\n",
			    vg_name);
		rc = EINVAL;
		goto out;
	}

	vg_name_to_container_name(vg_name, container_name, disk_group);
	rc = EngFncs->register_name(container_name);
	if (rc) {
		LOG_ERROR("Container name \"%s\" is already in use or "
			  "too long.\n", container_name);
		goto out;
	}

	EngFncs->unregister_name(container_name);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * create_container_set_option
 *
 * Validate the specified option-value for a create-container task. If the
 * option-value is valid, update the appropriate entry in the option-descriptor.
 **/
int create_container_set_option(task_context_t *context,
				u_int32_t index,
				value_t *value,
				task_effect_t *effect)
{
	option_desc_array_t *od = context->option_descriptors;
	storage_object_t *object;
	value_list_t *list;
	u_int i;
	int rc = EINVAL;

	LOG_ENTRY();

	switch (index) {

	case LVM2_OPTION_CREATE_CONTAINER_NAME_IDX:
		/* Make sure this name is not in use and isn't too long. */
		object = EngFncs->first_thing(context->selected_objects, NULL);
		rc = validate_vg_name(value->s, object->disk_group);
		if (!rc) {
			strncpy(od->option[index].value.s, value->s, EVMS_NAME_SIZE);
			od->option[index].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		}
		break;

	case LVM2_OPTION_CREATE_CONTAINER_EXTENT_SIZE_IDX:
		/* Make sure the extent-size is a valid entry from the
		 * constraint-list. This will verify that the value is a
		 * power-of-2 and that the value is valid for all the
		 * selected-objects.
		 */
		list = od->option[index].constraint.list;
		if (!list) {
			break;
		}

		if (value->ui64 < list->value[0].ui64) {
			value->ui64 = list->value[0].ui64;
			*effect |= EVMS_Effect_Inexact;
		} else if (value->ui64 > list->value[list->count - 1].ui64) {
			value->ui64 = list->value[list->count - 1].ui64;
			*effect |= EVMS_Effect_Inexact;
		} else {
			for (i = 0; i < list->count; i++) {
				if (value->ui64 == list->value[i].ui64) {
					break;
				} else if (value->ui64 < list->value[i+1].ui64) {
					value->ui64 = list->value[i].ui64;
					*effect |= EVMS_Effect_Inexact;
					break;
				}
			}
		}

		od->option[index].value.ui64 = value->ui64;
		rc = 0;
		break;

	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * create_container_parse_options
 *
 * Parse the option-values from the option-array.
 **/
void create_container_parse_options(option_array_t *options,
				    u_int64_t *extent_size,
				    char **vg_name)
{
	u_int i;

	LOG_ENTRY();

	/* Default values. */
	*extent_size = LVM2_DEFAULT_EXTENT_SIZE;
	*vg_name = NULL;

	for (i = 0; i < options->count; i++) {
		/* If only the name-based index is specified, get the number. */
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name,
				    LVM2_OPTION_CREATE_CONTAINER_NAME_STR)) {
				options->option[i].number = LVM2_OPTION_CREATE_CONTAINER_NAME_IDX;
			} else if (!strcmp(options->option[i].name,
					   LVM2_OPTION_CREATE_CONTAINER_EXTENT_SIZE_STR)) {
				options->option[i].number = LVM2_OPTION_CREATE_CONTAINER_EXTENT_SIZE_IDX;
			} else {
				continue;
			}
		}

		/* Use the number-based index to record the option. */
		switch (options->option[i].number) {

		case LVM2_OPTION_CREATE_CONTAINER_NAME_IDX:
			*vg_name = options->option[i].value.s;
			LOG_DEBUG("Name option: %s\n", *vg_name);
			break;

		case LVM2_OPTION_CREATE_CONTAINER_EXTENT_SIZE_IDX:
			*extent_size = options->option[i].value.ui64;
			LOG_DEBUG("Extent-size option: %"PRIu64"\n", *extent_size);
			break;

		default:
			break;
		}
	}

	LOG_EXIT_VOID();
}

/**
 * create_container_validate_options
 *
 * Verify that these options are valid for creating a new container.
 **/
int create_container_validate_options(u_int64_t *extent_size,
				      char *vg_name,
				      list_anchor_t objects)
{
	storage_object_t *object;
	list_element_t iter;
	u_int64_t mask = 1;
	int rc;

	LOG_ENTRY();

	/* Check that the proposed name is valid. Get the disk-group
	 * pointer from the first object in the list.
	 */
	object = EngFncs->first_thing(objects, NULL);
	rc = validate_vg_name(vg_name, object->disk_group);
	if (rc) {
		goto out;
	}

	/* Check that the extent-size is a power-of-2 and above the minimum
	 * allowed size. If it isn't, round up or down as appropriate.
	 */
	if (*extent_size & (*extent_size - 1)) {
		while (*extent_size & (*extent_size - 1)) {
			*extent_size &= ~mask;
			mask <<= 1;
		}
		LOG_WARNING("Rounded extent-size down to %"PRIu64" sectors.\n",
			    *extent_size);
	}

	if (*extent_size < LVM2_MIN_EXTENT_SIZE) {
		*extent_size = LVM2_MIN_EXTENT_SIZE;
		LOG_WARNING("Rounded extent-size up to minimum allowed size of "
			    "%u sectors.\n", LVM2_MIN_EXTENT_SIZE);
	}

	/* Make sure all selected objects are large enough for the selected
	 * extent-size plus space for metadata.
	 */
	LIST_FOR_EACH(objects, iter, object) {
		if (object->size < LVM2_MIN_PV_SIZE(*extent_size)) {
			LOG_ERROR("Object %s is not large enough for extent-"
				  "size of %"PRIu64" sectors.\n",
				  object->name, *extent_size);
			rc = ENOSPC;
		}
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}


/**
 * Expand-container routines.
 **/


/**
 * expand_container_init_task
 *
 * Initialize the acceptable-objects list for an expand-container task. There
 * are no options for expand-container, so no option-descriptor to initialize.
 **/
int expand_container_init_task(task_context_t *context)
{
	container_data_t *c_data = context->container->private_data;
	list_anchor_t objects = NULL;
	storage_object_t *object;
	list_element_t iter;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Initializing an expand task for container %s.\n",
		  context->container->name);

	/* Get a list of acceptable objects that can be added to the selected
	 * container. Each object must be big enough for at least one extent.
	 */
	rc = get_available_objects(context->container, &objects);
	if (rc) {
		goto out;
	}

	LIST_FOR_EACH(objects, iter, object) {
		if (object->size >= LVM2_MIN_PV_SIZE(c_data->pe_size)) {
			EngFncs->insert_thing(context->acceptable_objects,
					      object, INSERT_AFTER, NULL);
		}
	}

	context->option_descriptors->count = 0;
	context->min_selected_objects = 1;
	context->max_selected_objects = -1;

out:
	EngFncs->destroy_list(objects);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * expand_container_set_objects
 *
 * Validate that the objects in the selected-objects list can be added to the
 * selected container.
 **/
int expand_container_set_objects(task_context_t *context,
				 task_effect_t *effect)
{
	container_data_t *c_data = context->container->private_data;
	storage_object_t *object;
	list_element_t iter;
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Setting objects for an expand of container %s.\n",
		  context->container->name);

	LIST_FOR_EACH(context->selected_objects, iter, object){
		if (object->size < LVM2_MIN_PV_SIZE(c_data->pe_size)) {
			LOG_ERROR("Object %s is too small to be added to "
				  "container %s.\n", object->name,
				  context->container->name);
			rc = ENOSPC;
			break;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/**
 * Shrink-container routines.
 **/


/**
 * shrink_container_init_task
 *
 * Initialize the acceptable-objects list for a shrink-container task. There
 * are no options for shrink-container, so no option-descriptor to initialize.
 **/
int shrink_container_init_task(task_context_t *context)
{
	storage_container_t *container = context->container;
	storage_object_t *object;
	list_element_t iter;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Initializing a shrink task for container %s.\n",
		  container->name);

	/* If the container only has one PV, it cannot be shrunk. */
	if (EngFncs->list_count(container->objects_consumed) <= 1) {
		LOG_DEBUG("Container %s only has one object. Cannot be "
			  "shrunk.\n", container->name);
		rc = EBUSY;
		goto out;
	}

	/* Get a list of acceptable objects that can be removed from the
	 * selected container. All objects that have no data regions
	 * mapping to them are valid.
	 */
	LIST_FOR_EACH(container->objects_consumed, iter, object) {
		rc = can_remove_object(object);
		if (!rc) {
			EngFncs->insert_thing(context->acceptable_objects,
					      object, INSERT_AFTER, NULL);
		}
	}

	context->option_descriptors->count = 0;
	context->min_selected_objects = 1;
	context->max_selected_objects = EngFncs->list_count(container->objects_consumed) - 1;
	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * shrink_container_set_objects
 *
 * Validate that the objects in the selected-objects list can all be removed
 * from the selected container.
 **/
int shrink_container_set_objects(task_context_t *context,
				 task_effect_t *effect)
{
	storage_container_t *container = context->container;
	storage_object_t *object;
	list_element_t iter;
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Setting objects for a shrink of container %s.\n",
		  container->name);

	/* Cannot select all the objects in the container. */
	if (EngFncs->list_count(context->selected_objects) >=
	    EngFncs->list_count(container->objects_consumed)) {
		LOG_ERROR("Cannot remove all objects from container %s.\n",
			  container->name);
		rc = EBUSY;
		goto out;
	}

	/* Check each object in the selected-objects list. */
	LIST_FOR_EACH(context->selected_objects, iter, object){
		rc = can_remove_object(object);
		if (rc) {
			LOG_ERROR("Object %s cannot be removed from container "
				  "%s.\n", object->name, container->name);
			goto out;
		}
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}


/**
 * Create-region routines.
 **/


/**
 * create_region_init_task
 *
 * Initialize the acceptable-objects list and option-descriptor array for a
 * create-region task.
 **/
int create_region_init_task(task_context_t *context)
{
	option_desc_array_t *od = context->option_descriptors;
	storage_container_t *container;
	storage_object_t *freespace;
	list_element_t iter;
	int i, rc = 0;

	LOG_ENTRY();

	/* Add all non-zero-sized freespace regions
	 * to the acceptable-objects list.
	 */
	LIST_FOR_EACH(lvm2_containers, iter, container) {
		freespace = get_freespace_region(container->objects_produced);
		if (freespace && freespace->size > 0) {
			EngFncs->insert_thing(context->acceptable_objects,
					      freespace, INSERT_AFTER, NULL);
		}
	}

	/* No creates are possible if there's no freespace
	 * in any of the containers.
	 */
	if (EngFncs->list_empty(context->acceptable_objects)) {
		rc = ENOSPC;
		goto out;
	}

	/* Initialize the "Name" option. Allocate space for the string value
	 * now so we can just strcpy during set_option().
	 */
	i = LVM2_OPTION_CREATE_REGION_NAME_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_CREATE_REGION_NAME_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Name for the new LVM2 region."));
	od->option[i].type = EVMS_Type_String;
	od->option[i].min_len = 1;
	od->option[i].max_len = EVMS_NAME_SIZE;
	od->option[i].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
	od->option[i].value.s = EngFncs->engine_alloc(EVMS_NAME_SIZE + 1);
	if (!od->option[i].value.s) {
		rc = ENOMEM;
		goto out;
	}

	/* Initialize the "Size" option. The constraint range and initial size
	 * cannot be filled in until a freespace region is chosen.
	 */
	i = LVM2_OPTION_CREATE_REGION_SIZE_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_CREATE_REGION_SIZE_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Size for the new LVM2 region."));
	od->option[i].tip = EngFncs->engine_strdup(_("Size must be a multiple of the container's "
						     "extent-size and cannot exceed the amount of "
						     "freespace in the container. If not, it will "
						     "be rounded down as appropriate."));
	od->option[i].type = EVMS_Type_Unsigned_Int64;
	od->option[i].unit = EVMS_Unit_Sectors;
	od->option[i].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
			      EVMS_OPTION_FLAGS_AUTOMATIC;

	/* Initialize the "Stripes" option. The constraint range cannot be
	 * filled in until a freespace region is chosen. The initial number
	 * of stripes is always one.
	 */
	i = LVM2_OPTION_CREATE_REGION_STRIPES_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_CREATE_REGION_STRIPES_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Number of stripes for the new LVM2 region."));
	od->option[i].tip = EngFncs->engine_strdup(_("One stripe implies a linear region. Number "
						     "of stripes cannot exceed the number of "
						     "objects consumed by the container."));
	od->option[i].type = EVMS_Type_Unsigned_Int64;
	od->option[i].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
			      EVMS_OPTION_FLAGS_AUTOMATIC;
	od->option[i].value.ui64 = LVM2_DEFAULT_STRIPES;

	/* Initialize the "Stripe-size" option. This option is inactive until
	 * the user selects more than one stripe. The constraint list cannot be
	 * filled in until a freespace region is chosen.
	 */
	i = LVM2_OPTION_CREATE_REGION_STRIPE_SIZE_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_CREATE_REGION_STRIPE_SIZE_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Stripe-size for the new LVM2 region."));
	od->option[i].tip = EngFncs->engine_strdup(_("Size of each stripe \"chunk\". Only available "
						     "when \"stripes\" option is greater than 1."));
	od->option[i].type = EVMS_Type_Unsigned_Int64;
	od->option[i].unit = EVMS_Unit_Sectors;
	od->option[i].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
			      EVMS_OPTION_FLAGS_AUTOMATIC |
			      EVMS_OPTION_FLAGS_INACTIVE;

	/* Initialize the "PVs" option. The constraint list cannot be
	 * filled in until a freespace region is chosen.
	 */
	i = LVM2_OPTION_CREATE_REGION_PVS_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_CREATE_REGION_PVS_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Objects (PVs) to place the new LVM2 region on."));
	od->option[i].tip = EngFncs->engine_strdup(_("Region will be allocated on only these objects. "
						     "Leave blank for automatic allocation."));
	od->option[i].min_len = 1;
	od->option[i].max_len = EVMS_NAME_SIZE;
	od->option[i].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
			      EVMS_OPTION_FLAGS_AUTOMATIC;

	od->count = LVM2_OPTION_CREATE_REGION_COUNT;
	context->min_selected_objects = 1;
	context->max_selected_objects = 1;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * create_region_set_objects
 *
 * Validate that the freespace region in the selected-objects
 * list is valid for creating a new region. Make adjustments
 * to the option-descriptor as appropriate.
 **/
int create_region_set_objects(task_context_t *context, task_effect_t *effect)
{
	option_desc_array_t *od = context->option_descriptors;
	storage_object_t *freespace, *object;
	storage_container_t *container;
	container_data_t *c_data;
	list_element_t iter;
	u_int32_t count;
	int i, j = 0, rc = 0;

	LOG_ENTRY();

	/* There should only be one selected freespace region,
	 * so just get the first thing in the list.
	 */
	freespace = get_freespace_region(context->selected_objects);
	if (!freespace) {
		LOG_ERROR("No freespace region selected.\n");
		rc = EINVAL;
		goto out;
	}
	container = freespace->producing_container;
	c_data = container->private_data;

	/* Make sure there's space available in this container. */
	if (freespace->size == 0) {
		LOG_ERROR("No freespace avilable in container %s.\n",
			  container->name);
		rc = ENOSPC;
		goto out;
	}

	/* Initialize the "Size" option to the total available freespace. The
	 * allowable range of sizes goes from one extent to the total available
	 * freespace, in increments of extent-size.
	 */
	i = LVM2_OPTION_CREATE_REGION_SIZE_IDX;
	od->option[i].value.ui64 = freespace->size;
	od->option[i].constraint_type = EVMS_Collection_Range;
	SET_RANGE64(od->option[i].constraint.range, c_data->pe_size,
		    freespace->size, c_data->pe_size);

	/* Initialize the "Stripes" option's constraint-range to the total
	 * number of objects that have at least one unused extent.
	 */
	i = LVM2_OPTION_CREATE_REGION_STRIPES_IDX;
	count = count_available_pvs(container->objects_consumed);
	od->option[i].constraint_type = EVMS_Collection_Range;
	SET_RANGE64(od->option[i].constraint.range, 1, count, 1);

	/* Initialize the "Stripe-size" option value and constraint-list. The
	 * stripe-size cannot be larger than the container's extent-size.
	 */
	i = LVM2_OPTION_CREATE_REGION_STRIPE_SIZE_IDX;
	od->option[i].constraint_type = EVMS_Collection_List;
	SET_POWER2_LIST64(od->option[i].constraint.list, LVM2_MIN_STRIPE_SIZE,
			  min(LVM2_MAX_STRIPE_SIZE, c_data->pe_size));
	od->option[i].value.ui64 = min(LVM2_DEFAULT_STRIPE_SIZE, c_data->pe_size);

	/* Initialize the "PVs" option's constraint list. Add all objects that
	 * have any free extents.
	 */
	i = LVM2_OPTION_CREATE_REGION_PVS_IDX;
	od->option[i].type = EVMS_Type_String;
	od->option[i].constraint_type = EVMS_Collection_List;
	od->option[i].flags |= EVMS_OPTION_FLAGS_VALUE_IS_LIST;
	od->option[i].constraint.list = EngFncs->engine_alloc(sizeof(value_list_t) +
							      sizeof(value_t) * count);
	od->option[i].value.list = EngFncs->engine_alloc(sizeof(value_list_t) +
							 sizeof(value_t) * count);
	if (!od->option[i].constraint.list ||
	    !od->option[i].value.list) {
		rc = ENOMEM;
		goto out;
	}
	LIST_FOR_EACH(container->objects_consumed, iter, object) {
		rc = count_available_extents_in_pv(object);
		if (rc) {
			od->option[i].constraint.list->value[j++].s = EngFncs->engine_strdup(object->name);
		}
	}
	od->option[i].constraint.list->count = j;
	od->option[i].value.list->count = 0;

	*effect |= EVMS_Effect_Reload_Options;
	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * validate_lv_name
 *
 * Make sure the proposed region name is not already in use.
 **/
static int validate_lv_name(char *lv_name, char *container_name)
{
	char region_name[EVMS_NAME_SIZE+1];
	int rc;

	LOG_ENTRY();

	if (!lv_name) {
		LOG_ERROR("No region name specified.\n");
		rc = EINVAL;
		goto out;
	}

	if (strchr(lv_name, ' ')) {
		LOG_ERROR("Region name (%s) cannot contain spaces.\n", lv_name);
		rc = EINVAL;
		goto out;
	}

	lv_name_to_region_name(lv_name, region_name, container_name);
	rc = EngFncs->register_name(region_name);
	if (rc) {
		LOG_ERROR("Region name \"%s\" is already in use or "
			  "too long.\n", region_name);
		goto out;
	}

	EngFncs->unregister_name(region_name);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * create_region_set_option
 *
 * Validate the specified option-value for a create-region task. If the
 * option-value is valid, update the appropriate entry in the option-descriptor.
 **/
int create_region_set_option(task_context_t *context,
			     u_int32_t index,
			     value_t *value,
			     task_effect_t *effect)
{
	option_desc_array_t *od = context->option_descriptors;
	storage_object_t *freespace;
	storage_container_t *container;
	container_data_t *c_data;
	u_int64_t available_size, size_increment;
	value_list_t *current_pvs;
	list_anchor_t objects = NULL;
	u_int32_t i, j, k;
	int rc = 0;

	LOG_ENTRY();

	/* There should only be one selected freespace region,
	 * so just get the first thing in the list.
	 */
	freespace = get_freespace_region(context->selected_objects);
	if (!freespace) {
		LOG_ERROR("No freespace region selected.\n");
		rc = EINVAL;
		goto out;
	}
	container = freespace->producing_container;
	c_data = container->private_data;

	switch (index) {

	/* Name */
	case LVM2_OPTION_CREATE_REGION_NAME_IDX:
		/* Make sure this name is not in use and that isn't too long.
		 * The selected name has no effect on any other options.
		 */
		rc = validate_lv_name(value->s, container->name);
		if (!rc) {
			LOG_DEBUG("Setting name option: %s\n", value->s);
			strncpy(od->option[index].value.s, value->s, EVMS_NAME_SIZE);
			od->option[index].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		}
		break;

	/* Size */
	case LVM2_OPTION_CREATE_REGION_SIZE_IDX:
		/* Selected size must be within the constraint-range and must
		 * be a multiple of the constraint-range increment. The selected
		 * size has no effect on any other options.
		 */
		if (value->ui64 < od->option[index].constraint.range->min.ui64) {
			value->ui64 = od->option[index].constraint.range->min.ui64;
		} else if (value->ui64 > od->option[index].constraint.range->max.ui64) {
			value->ui64 = od->option[index].constraint.range->max.ui64;
		} else {
			value->ui64 -= value->ui64 %
				       od->option[index].constraint.range->increment.ui64;
		}

		LOG_DEBUG("Setting size option: %"PRIu64" sectors\n", value->ui64);
		od->option[index].value.ui64 = value->ui64;
		break;

	/* Stripes */
	case LVM2_OPTION_CREATE_REGION_STRIPES_IDX:
		/* Selected stripes must be within the constraint-range. */
		if (value->ui64 < od->option[index].constraint.range->min.ui64) {
			value->ui64 = od->option[index].constraint.range->min.ui64;
		} else if (value->ui64 > od->option[index].constraint.range->max.ui64) {
			value->ui64 = od->option[index].constraint.range->max.ui64;
		}

		LOG_DEBUG("Setting stripes option: %"PRIu64"\n", value->ui64);
		od->option[index].value.ui64 = value->ui64;

		/* If the number of stripes is greater than one,
		 * activate the stripe-size option.
		 */
		if (value->ui64 > 1) {
			od->option[LVM2_OPTION_CREATE_REGION_STRIPE_SIZE_IDX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
		} else {
			od->option[LVM2_OPTION_CREATE_REGION_STRIPE_SIZE_IDX].flags |= EVMS_OPTION_FLAGS_INACTIVE;
		}

		/* Adjust the "size" option's constraint range, which must be
		 * a multiple of the extent-size and the number of stripes. Then
		 * set the "size" option to re-validate the "size" value.
		 */

		current_pvs = od->option[LVM2_OPTION_CREATE_REGION_PVS_IDX].value.list;
		objects = pv_names_to_list(current_pvs, container);
		size_increment = c_data->pe_size * value->ui64;
		available_size = count_available_extents_in_pvs(objects) * c_data->pe_size;
		available_size -= available_size % size_increment;
		EngFncs->destroy_list(objects);

		/* FIXME: This doesn't take into account the actual allocation
		 *        of extents, which might further limit the available
		 *        size of the new region.
		 */
		i = LVM2_OPTION_CREATE_REGION_SIZE_IDX;
		EngFncs->engine_free(od->option[i].constraint.range);
		SET_RANGE64(od->option[i].constraint.range,
			    size_increment, available_size, size_increment);

		rc = create_region_set_option(context, i,
					      &(od->option[i].value), effect);

		break;

	/* Stripe-Size */
	case LVM2_OPTION_CREATE_REGION_STRIPE_SIZE_IDX:
		/* The stripe-size must be within the constraint-list. The
		 * stripe-size has no effect on any other options.
		 */
		for (i = 0; i < od->option[index].constraint.list->count; i++) {
			if (value->ui64 == od->option[index].constraint.list->value[i].ui64) {
				break;
			}
		}
		if (i == od->option[index].constraint.list->count) {
			value->ui64 = min(LVM2_DEFAULT_STRIPE_SIZE, c_data->pe_size);
		}

		LOG_DEBUG("Setting stripe-size option: %"PRIu64"\n", value->ui64);
		od->option[index].value.ui64 = value->ui64;
		break;

	/* PVs */
	case LVM2_OPTION_CREATE_REGION_PVS_IDX:
		/* Each selected PV must be in the constraint-list. */
		for (i = 0, k = 0; i < value->list->count; i++) {
			for (j = 0; j < od->option[index].constraint.list->count; j++) {
				rc = strcmp(value->list->value[i].s,
					    od->option[index].constraint.list->value[j].s);
				if (!rc) {
					LOG_DEBUG("Setting PVs option entry: %s\n", value->list->value[i].s);
					od->option[index].value.list->value[k++].s = EngFncs->engine_strdup(value->list->value[i].s);
					break;
				}
			}
		}
		od->option[index].value.list->count = k;

		/* The "stripes" option's constraint-range must be adjusted
		 * based on the number of selected PVs. Then set the "stripes"
		 * option to re-validate the "stripes" value, which will in
		 * turn adjust and re-validate the "size" option.
		 */
		i = LVM2_OPTION_CREATE_REGION_STRIPES_IDX;
		if (!k) {
			k = od->option[index].constraint.list->count;
		}
		EngFncs->engine_free(od->option[i].constraint.range);
		SET_RANGE64(od->option[i].constraint.range, 1, k, 1);

		rc = create_region_set_option(context, i,
					      &(od->option[i].value), effect);

		break;

	default:
		rc = EINVAL;
		break;
	}

	*effect |= EVMS_Effect_Reload_Options;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * create_region_parse_options
 *
 * Parse the option-values from the option-array.
 **/
void create_region_parse_options(storage_container_t *container,
				 option_array_t *options,
				 char **lv_name,
				 u_int64_t *size,
				 u_int64_t *stripes,
				 u_int64_t *stripe_size,
				 list_anchor_t *objects)
{
	container_data_t *c_data = container->private_data;
	storage_object_t *freespace;
	u_int i;

	LOG_ENTRY();

	/* Default values. */
	freespace = get_freespace_region(container->objects_produced);
	*lv_name = NULL;
	*size = freespace->size;
	*stripes = 1;
	*stripe_size = 0;
	*objects = NULL;

	for (i = 0; i < options->count; i++) {
		/* If only the name-based index is specified, get the number. */
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name,
				    LVM2_OPTION_CREATE_REGION_NAME_STR)) {
				options->option[i].number = LVM2_OPTION_CREATE_REGION_NAME_IDX;
			} else if (!strcmp(options->option[i].name,
					   LVM2_OPTION_CREATE_REGION_SIZE_STR)) {
				options->option[i].number = LVM2_OPTION_CREATE_REGION_SIZE_IDX;
			} else if (!strcmp(options->option[i].name,
					   LVM2_OPTION_CREATE_REGION_STRIPES_STR)) {
				options->option[i].number = LVM2_OPTION_CREATE_REGION_STRIPES_IDX;
			} else if (!strcmp(options->option[i].name,
					   LVM2_OPTION_CREATE_REGION_STRIPE_SIZE_STR)) {
				options->option[i].number = LVM2_OPTION_CREATE_REGION_STRIPE_SIZE_IDX;
			} else if (!strcmp(options->option[i].name,
					   LVM2_OPTION_CREATE_REGION_PVS_STR)) {
				options->option[i].number = LVM2_OPTION_CREATE_REGION_PVS_IDX;
			} else {
				continue;
			}
		}

		/* Use the number-based index to record the option. */
		switch (options->option[i].number) {

		case LVM2_OPTION_CREATE_REGION_NAME_IDX:
			*lv_name = options->option[i].value.s;
			LOG_DEBUG("Name option: %s\n", *lv_name);
			break;

		case LVM2_OPTION_CREATE_REGION_SIZE_IDX:
			*size = options->option[i].value.ui64;
			LOG_DEBUG("Size option: %"PRIu64"\n", *size);
			break;

		case LVM2_OPTION_CREATE_REGION_STRIPES_IDX:
			*stripes = options->option[i].value.ui64;
			LOG_DEBUG("Stripes option: %"PRIu64"\n", *stripes);
			break;

		case LVM2_OPTION_CREATE_REGION_STRIPE_SIZE_IDX:
			*stripe_size = options->option[i].value.ui64;
			LOG_DEBUG("Stripe-size option: %"PRIu64"\n", *stripe_size);
			break;

		case LVM2_OPTION_CREATE_REGION_PVS_IDX:
			*objects = pv_names_to_list(options->option[i].value.list,
						    container);
			LOG_DEBUG("PVs option.\n");
			break;

		default:
			break;
		}
	}

	/* More defaults. */
	if (!*objects) {
		*objects = pv_names_to_list(NULL, container);
	}

	if (*stripes > 1 && !*stripe_size) {
		*stripe_size = min(LVM2_DEFAULT_STRIPE_SIZE, c_data->pe_size);
	}

	LOG_EXIT_VOID();
}

/**
 * create_region_validate_options
 *
 * Verify that these options are valid for creating a new region in this
 * container. This does not validate that an approprite extent-allocation
 * exists. It simply verifies that each option adheres to the proper
 * limitations.
 **/
int create_region_validate_options(storage_container_t *container,
				   char *lv_name,
				   u_int64_t *size,
				   u_int64_t *stripes,
				   u_int64_t *stripe_size,
				   list_anchor_t objects)
{
	container_data_t *c_data = container->private_data;
	storage_object_t *object;
	list_element_t iter1, iter2;
	u_int64_t increment, count, free_extents = 0;
	int rc;

	LOG_ENTRY();

	/* Validate the region's name. */
	rc = validate_lv_name(lv_name, container->name);
	if (rc) {
		goto out;
	}

	/* All selected objects must have at least one free extent. Simply
	 * remove all objects that have no freespace. Count the total number
	 * of free-extents while we're here.
	 */
	LIST_FOR_EACH_SAFE(objects, iter1, iter2, object) {
		count = count_available_extents_in_pv(object);
		if (!count) {
			EngFncs->remove_element(iter1);
			LOG_DEBUG("Removing %s from objects list - no free "
				  "extents.\n", object->name);
		}
		free_extents += count;
	}
	if (!free_extents) {
		LOG_ERROR("No freespace in list of selected objects.\n");
		rc = ENOSPC;
		goto out;
	}

	/* Number of stripes cannot exceed the number of selected objects. */
	count = EngFncs->list_count(objects);
	if (*stripes < 1) {
		*stripes = 1;
		LOG_DEBUG("Rounding number of stripes up to %"PRIu64".\n",
			  *stripes);
	} else if (*stripes > count) {
		*stripes = count;
		LOG_DEBUG("Rounding number of stripes down to %"PRIu64".\n",
			  *stripes);
	}

	/* Size must be a multiple of the extent-size
	 * and the number of stripes.
	 */
	increment = *stripes * c_data->pe_size;
	if (*size < increment) {
		*size = increment;
		LOG_DEBUG("Rounding size up to %"PRIu64".\n", *size);
	} else if (*size > free_extents * c_data->pe_size) {
		*size = free_extents * c_data->pe_size;
		LOG_DEBUG("Rounding size down to %"PRIu64".\n", *size);
	} else if (*size % increment) {
		*size -= *size % increment;
		LOG_DEBUG("Rounding size down to %"PRIu64".\n", *size);
	}

	/* If the region is striped, stripe-size must
	 * be in the proper range and a power-of-2.
	 */
	if (*stripes > 1) {
		if (*stripe_size < LVM2_MIN_STRIPE_SIZE) {
			*stripe_size = LVM2_MIN_STRIPE_SIZE;
			LOG_DEBUG("Rounding stripe-size up to %"PRIu64".\n",
				  *stripe_size);

		} else if (*stripe_size > min(LVM2_MAX_STRIPE_SIZE, c_data->pe_size)) {
			*stripe_size = min(LVM2_MAX_STRIPE_SIZE, c_data->pe_size);
			LOG_DEBUG("Rounding stripe-size down to %"PRIu64".\n",
				  *stripe_size);

		} else if (*stripe_size & (*stripe_size - 1)) {
			u_int64_t mask = 1;
			while (*stripe_size & (*stripe_size - 1)) {
				*stripe_size &= ~mask;
				mask <<= 1;
			}
			LOG_DEBUG("Rounding stripe-size down to %"PRIu64".\n",
				  *stripe_size);
		}
	} else {
		*stripe_size = 0;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}


/**
 * Expand-region routines.
 **/


/**
 * expand_region_init_task
 *
 * Initialize the acceptable-objects list and option-descriptor array for an
 * expand-region task.
 **/
int expand_region_init_task(task_context_t *context)
{
	option_desc_array_t *od = context->option_descriptors;
	storage_object_t *object, *region = context->object;
	storage_container_t *container = region->producing_container;
	container_data_t *c_data = container->private_data;
	region_data_t *r_data = region->private_data;
	region_mapping_t *r_map;
	list_element_t iter;
	u_int64_t count, available_size, increment;
	u_int64_t available_objects, stripes;
	u_int32_t i, j = 0;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Initializing expand task for region %s.\n", region->name);

	rc = can_expand_region(region);
	if (rc) {
		goto out;
	}

	available_objects = count_available_pvs(container->objects_consumed);
	available_size = count_available_extents_in_pvs(container->objects_consumed) * c_data->pe_size;
	r_map = EngFncs->last_thing(r_data->mappings, NULL);

	if (!available_objects) {
		/* No PVs with free extents. Expand is not possible. */
		LOG_WARNING("No freespace in container %s.\n", container->name);
		rc = ENOSPC;
		goto out;
	}

	/* Ask the engine if it's ok to expand by the max amount. */
	rc = EngFncs->can_expand_by(region, &available_size);
	if (rc) {
		if (rc != EAGAIN) {
			LOG_ERROR("Expand of region %s rejected by the "
				  "engine.\n", region->name);
			goto out;
		}
		rc = 0;

		/* Make sure the size is still a multiple of the PE-size. */
		LOG_DEBUG("Engine will only allow max expand size of %"PRIu64
			  " sectors.\n", available_size);
		available_size -= available_size % c_data->pe_size;
		LOG_DEBUG("Reduced max expand size to %"PRIu64" sectors.\n",
			  available_size);
	}

	/* Initialize the "PVs" option. */
	i = LVM2_OPTION_EXPAND_REGION_PVS_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_EXPAND_REGION_PVS_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Objects (PVs) to use for the new portion of the region."));
	od->option[i].tip = EngFncs->engine_strdup(_("Region will be expanded only onto these objects. "
						     "Leave blank for automatic allocation."));
	od->option[i].type = EVMS_Type_String;
	od->option[i].min_len = 1;
	od->option[i].max_len = EVMS_NAME_SIZE;
	od->option[i].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
			      EVMS_OPTION_FLAGS_AUTOMATIC |
			      EVMS_OPTION_FLAGS_VALUE_IS_LIST;
	od->option[i].constraint_type = EVMS_Collection_List;
	od->option[i].constraint.list = EngFncs->engine_alloc(sizeof(value_list_t) +
							      sizeof(value_t) * available_objects);
	od->option[i].value.list = EngFncs->engine_alloc(sizeof(value_list_t) +
							 sizeof(value_t) * available_objects);
	if (!od->option[i].constraint.list || !od->option[i].value.list) {
		rc = ENOMEM;
		goto out;
	}
	LIST_FOR_EACH(container->objects_consumed, iter, object) {
		count = count_available_extents_in_pv(object);
		if (count) {
			od->option[i].constraint.list->value[j++].s = EngFncs->engine_strdup(object->name);
		}
	}
	od->option[i].constraint.list->count = j;
	od->option[i].value.list->count = 0;


	/* Initialize the "Stripes" option. The initial number of stripes
	 * should be the same as in the last region-mapping (or the number
	 * of objects that have free extents, if that's less).
	 */
	i = LVM2_OPTION_EXPAND_REGION_STRIPES_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_EXPAND_REGION_STRIPES_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Number of stripes for the new portion of the region."));
	od->option[i].tip = EngFncs->engine_strdup(_("One stripe implies a linear region. Number "
						     "of stripes cannot exceed the number of "
						     "objects consumed by the container."));
	od->option[i].type = EVMS_Type_Unsigned_Int64;
	od->option[i].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
			      EVMS_OPTION_FLAGS_AUTOMATIC;
	od->option[i].constraint_type = EVMS_Collection_Range;
	SET_RANGE64(od->option[i].constraint.range, 1, available_objects, 1);
	od->option[i].value.ui64 = stripes
				 = min(r_map->stripe_count, available_objects);


	/* Initialize the "Size" option. */
	increment = stripes * c_data->pe_size;
	available_size -= available_size % increment;
	i = LVM2_OPTION_EXPAND_REGION_SIZE_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_EXPAND_REGION_SIZE_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Extra size for the LVM2 region."));
	od->option[i].tip = EngFncs->engine_strdup(_("Extra size must be a multiple of the container's "
						     "extent-size and cannot exceed the amount of "
						     "freespace in the container. If not, it will "
						     "be rounded down as appropriate."));
	od->option[i].type = EVMS_Type_Unsigned_Int64;
	od->option[i].unit = EVMS_Unit_Sectors;
	od->option[i].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
			      EVMS_OPTION_FLAGS_AUTOMATIC;
	od->option[i].constraint_type = EVMS_Collection_Range;
	SET_RANGE64(od->option[i].constraint.range, increment, available_size, increment);
	od->option[i].value.ui64 = available_size;


	/* Initialize the "Stripe-size" option. This option is
	 * inactive if there's only one stripe.
	 */
	i = LVM2_OPTION_EXPAND_REGION_STRIPE_SIZE_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_EXPAND_REGION_STRIPE_SIZE_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Stripe-size for the new portion of the region."));
	od->option[i].tip = EngFncs->engine_strdup(_("Size of each stripe \"chunk\". Only available "
						     "when \"stripes\" option is greater than 1."));
	od->option[i].type = EVMS_Type_Unsigned_Int64;
	od->option[i].unit = EVMS_Unit_Sectors;
	od->option[i].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
			      EVMS_OPTION_FLAGS_AUTOMATIC |
			      (stripes == 1) ? EVMS_OPTION_FLAGS_INACTIVE : 0;
	od->option[i].constraint_type = EVMS_Collection_List;
	SET_POWER2_LIST64(od->option[i].constraint.list, LVM2_MIN_STRIPE_SIZE,
			  min(LVM2_MAX_STRIPE_SIZE, c_data->pe_size));
	od->option[i].value.ui64 = r_map->stripe_size;


	od->count = LVM2_OPTION_EXPAND_REGION_COUNT;
	context->min_selected_objects = 0;
	context->max_selected_objects = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * expand_region_set_option
 *
 * Validate the specified option-value for an expand-region task. If the
 * option-value is valid, update the appropriate entry in the option-descriptor.
 **/
int expand_region_set_option(task_context_t *context,
			     u_int32_t index,
			     value_t *value,
			     task_effect_t *effect)
{
	option_desc_array_t *od = context->option_descriptors;
	storage_container_t *container = context->object->producing_container;
	container_data_t *c_data = container->private_data;
	value_list_t *current_pvs;
	list_anchor_t objects;
	u_int64_t size_increment, available_size;
	u_int32_t i, j, k;
	int rc = 0;

	LOG_ENTRY();

	switch (index) {

	/* Size */
	case LVM2_OPTION_EXPAND_REGION_SIZE_IDX:
		/* Selected size must be within the constraint-range and must
		 * be a multiple of the constraint-range increment. The selected
		 * size has no effect on any other options.
		 */
		if (value->ui64 < od->option[index].constraint.range->min.ui64) {
			value->ui64 = od->option[index].constraint.range->min.ui64;
		} else if (value->ui64 > od->option[index].constraint.range->max.ui64) {
			value->ui64 = od->option[index].constraint.range->max.ui64;
		} else {
			value->ui64 -= value->ui64 %
				       od->option[index].constraint.range->increment.ui64;
		}

		LOG_DEBUG("Setting size option: %"PRIu64" sectors\n", value->ui64);
		od->option[index].value.ui64 = value->ui64;
		break;

	/* Stripes */
	case LVM2_OPTION_EXPAND_REGION_STRIPES_IDX:
		/* Selected stripes must be within the constraint-range. */
		if (value->ui64 < od->option[index].constraint.range->min.ui64) {
			value->ui64 = od->option[index].constraint.range->min.ui64;
		} else if (value->ui64 > od->option[index].constraint.range->max.ui64) {
			value->ui64 = od->option[index].constraint.range->max.ui64;
		}

		LOG_DEBUG("Setting stripes option: %"PRIu64"\n", value->ui64);
		od->option[index].value.ui64 = value->ui64;

		/* If the number of stripes is greater than one,
		 * activate the stripe-size option.
		 */
		if (value->ui64 > 1) {
			od->option[LVM2_OPTION_EXPAND_REGION_STRIPE_SIZE_IDX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
		} else {
			od->option[LVM2_OPTION_EXPAND_REGION_STRIPE_SIZE_IDX].flags |= EVMS_OPTION_FLAGS_INACTIVE;
		}

		/* Adjust the "size" option's constraint range, which must be
		 * a multiple of the extent-size and the number of stripes. Then
		 * set the "size" option to re-validate the "size" value.
		 */

		current_pvs = od->option[LVM2_OPTION_EXPAND_REGION_PVS_IDX].value.list;
		objects = pv_names_to_list(current_pvs, container);
		size_increment = c_data->pe_size * value->ui64;
		available_size = count_available_extents_in_pvs(objects) * c_data->pe_size;
		available_size -= available_size % size_increment;
		EngFncs->destroy_list(objects);

		/* FIXME: This doesn't take into account the actual allocation
		 *        of extents, which might further limit the available
		 *        size of the new region.
		 */
		i = LVM2_OPTION_EXPAND_REGION_SIZE_IDX;
		EngFncs->engine_free(od->option[i].constraint.range);
		SET_RANGE64(od->option[i].constraint.range,
			    size_increment, available_size, size_increment);

		rc = expand_region_set_option(context, i,
					      &(od->option[i].value), effect);

		break;

	/* Stripe-Size */
	case LVM2_OPTION_EXPAND_REGION_STRIPE_SIZE_IDX:
		/* The stripe-size must be within the constraint-list. The
		 * stripe-size has no effect on any other options.
		 */
		for (i = 0; i < od->option[index].constraint.list->count; i++) {
			if (value->ui64 == od->option[index].constraint.list->value[i].ui64) {
				break;
			}
		}
		if (i == od->option[index].constraint.list->count) {
			value->ui64 = min(LVM2_DEFAULT_STRIPE_SIZE, c_data->pe_size);
		}

		LOG_DEBUG("Setting stripe-size option: %"PRIu64"\n", value->ui64);
		od->option[index].value.ui64 = value->ui64;
		break;

	/* PVs */
	case LVM2_OPTION_EXPAND_REGION_PVS_IDX:
		/* Each selected PV must be in the constraint-list. */
		for (i = 0, k = 0; i < value->list->count; i++) {
			for (j = 0; j < od->option[index].constraint.list->count; j++) {
				rc = strcmp(value->list->value[i].s,
					    od->option[index].constraint.list->value[j].s);
				if (!rc) {
					LOG_DEBUG("Setting PVs option entry: %s\n", value->list->value[i].s);
					od->option[index].value.list->value[k++].s = EngFncs->engine_strdup(value->list->value[i].s);
					break;
				}
			}
		}
		od->option[index].value.list->count = k;

		/* The "stripes" option's constraint-range must be adjusted
		 * based on the number of selected PVs. Then set the "stripes"
		 * option to re-validate the "stripes" value, which will in
		 * turn adjust and re-validate the "size" option.
		 */
		i = LVM2_OPTION_EXPAND_REGION_STRIPES_IDX;
		if (!k) {
			k = od->option[index].constraint.list->count;
		}
		EngFncs->engine_free(od->option[i].constraint.range);
		SET_RANGE64(od->option[i].constraint.range, 1, k, 1);

		rc = expand_region_set_option(context, i,
					      &(od->option[i].value), effect);

		break;

	default:
		rc = EINVAL;
		break;
	}

	*effect |= EVMS_Effect_Reload_Options;

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * expand_region_parse_options
 *
 * Parse the option-values from the option-array.
 **/
void expand_region_parse_options(storage_object_t *region,
				 option_array_t *options,
				 u_int64_t *size,
				 u_int64_t *stripes,
				 u_int64_t *stripe_size,
				 list_anchor_t *objects)
{
	storage_container_t *container = region->producing_container;
	container_data_t *c_data = container->private_data;
	region_data_t *r_data = region->private_data;
	region_mapping_t *r_map;
	storage_object_t *freespace;
	u_int i;

	LOG_ENTRY();

	/* Default values. */
	freespace = get_freespace_region(container->objects_produced);
	r_map = EngFncs->last_thing(r_data->mappings, NULL);
	*size = freespace->size;
	*stripes = r_map->stripe_count;
	*stripe_size = r_map->stripe_size;
	*objects = NULL;

	for (i = 0; i < options->count; i++) {
		/* If only the name-based index is specified, get the number. */
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name,
				    LVM2_OPTION_EXPAND_REGION_SIZE_STR)) {
				options->option[i].number = LVM2_OPTION_EXPAND_REGION_SIZE_IDX;
			} else if (!strcmp(options->option[i].name,
					   LVM2_OPTION_EXPAND_REGION_STRIPES_STR)) {
				options->option[i].number = LVM2_OPTION_EXPAND_REGION_STRIPES_IDX;
			} else if (!strcmp(options->option[i].name,
					   LVM2_OPTION_EXPAND_REGION_STRIPE_SIZE_STR)) {
				options->option[i].number = LVM2_OPTION_EXPAND_REGION_STRIPE_SIZE_IDX;
			} else if (!strcmp(options->option[i].name,
					   LVM2_OPTION_EXPAND_REGION_PVS_STR)) {
				options->option[i].number = LVM2_OPTION_EXPAND_REGION_PVS_IDX;
			} else {
				continue;
			}
		}

		/* Use the number-based index to record the option. */
		switch (options->option[i].number) {

		case LVM2_OPTION_EXPAND_REGION_SIZE_IDX:
			*size = options->option[i].value.ui64;
			LOG_DEBUG("Size option: %"PRIu64"\n", *size);
			break;

		case LVM2_OPTION_EXPAND_REGION_STRIPES_IDX:
			*stripes = options->option[i].value.ui64;
			LOG_DEBUG("Stripes option: %"PRIu64"\n", *stripes);
			break;

		case LVM2_OPTION_EXPAND_REGION_STRIPE_SIZE_IDX:
			*stripe_size = options->option[i].value.ui64;
			LOG_DEBUG("Stripe-size option: %"PRIu64"\n", *stripe_size);
			break;

		case LVM2_OPTION_EXPAND_REGION_PVS_IDX:
			*objects = pv_names_to_list(options->option[i].value.list,
						    container);
			LOG_DEBUG("PVs option.\n");
			break;

		default:
			break;
		}
	}

	/* More defaults. */
	if (!*objects) {
		*objects = pv_names_to_list(NULL, container);
	}

	if (*stripes > 1 && !*stripe_size) {
		*stripe_size = min(LVM2_DEFAULT_STRIPE_SIZE, c_data->pe_size);
	}

	LOG_EXIT_VOID();
}

/**
 * expand_region_validate_options
 *
 * Verify that these options are valid for expanding this region. This does not
 * validate that an approprite extent-allocation exists. It simply verifies
 * that each option adheres to the proper limitations.
 **/
int expand_region_validate_options(storage_object_t *region,
				   u_int64_t *size,
				   u_int64_t *stripes,
				   u_int64_t *stripe_size,
				   list_anchor_t objects)
{
	storage_container_t *container = region->producing_container;
	container_data_t *c_data = container->private_data;
	storage_object_t *object;
	list_element_t iter1, iter2;
	u_int64_t increment, count, free_extents = 0;
	int rc = 0;

	LOG_ENTRY();

	/* All selected objects must have at least one free extent. Simply
	 * remove all objects that have no freespace. Count the total number
	 * of free-extents while we're here.
	 */
	LIST_FOR_EACH_SAFE(objects, iter1, iter2, object) {
		count = count_available_extents_in_pv(object);
		if (!count) {
			EngFncs->remove_element(iter1);
			LOG_DEBUG("Removing %s from objects list - no free "
				  "extents.\n", object->name);
		}
		free_extents += count;
	}
	if (!free_extents) {
		LOG_ERROR("No freespace in list of selected objects.\n");
		rc = ENOSPC;
		goto out;
	}

	/* Number of stripes cannot exceed the number of selected objects. */
	count = EngFncs->list_count(objects);
	if (*stripes < 1) {
		*stripes = 1;
		LOG_DEBUG("Rounding number of stripes up to %"PRIu64".\n",
			  *stripes);
	} else if (*stripes > count) {
		*stripes = count;
		LOG_DEBUG("Rounding number of stripes down to %"PRIu64".\n",
			  *stripes);
	}

	/* Size must be a multiple of the extent-size
	 * and the number of stripes.
	 */
	increment = *stripes * c_data->pe_size;
	if (*size < increment) {
		*size = increment;
		LOG_DEBUG("Rounding size up to %"PRIu64".\n", *size);
	} else if (*size > free_extents * c_data->pe_size) {
		*size = free_extents * c_data->pe_size;
		LOG_DEBUG("Rounding size down to %"PRIu64".\n", *size);
	} else if (*size % increment) {
		*size -= *size % increment;
		LOG_DEBUG("Rounding size down to %"PRIu64".\n", *size);
	}

	/* If the region is striped, stripe-size must
	 * be in the proper range and a power-of-2.
	 */
	if (*stripes > 1) {
		if (*stripe_size < LVM2_MIN_STRIPE_SIZE) {
			*stripe_size = LVM2_MIN_STRIPE_SIZE;
			LOG_DEBUG("Rounding stripe-size up to %"PRIu64".\n",
				  *stripe_size);

		} else if (*stripe_size > min(LVM2_MAX_STRIPE_SIZE, c_data->pe_size)) {
			*stripe_size = min(LVM2_MAX_STRIPE_SIZE, c_data->pe_size);
			LOG_DEBUG("Rounding stripe-size down to %"PRIu64".\n",
				  *stripe_size);

		} else if (*stripe_size & (*stripe_size - 1)) {
			u_int64_t mask = 1;
			while (*stripe_size & (*stripe_size - 1)) {
				*stripe_size &= ~mask;
				mask <<= 1;
			}
			LOG_DEBUG("Rounding stripe-size down to %"PRIu64".\n",
				  *stripe_size);
		}
	} else {
		*stripe_size = 0;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}


/**
 * Shrink-region routines.
 **/


/**
 * shrink_region_init_task
 *
 * Initialize the acceptable-objects list and option-descriptor array for a
 * shrink-region task.
 **/
int shrink_region_init_task(task_context_t *context)
{
	option_desc_array_t *od = context->option_descriptors;
	storage_object_t *region = context->object;
	storage_container_t *container = region->producing_container;
	container_data_t *c_data = container->private_data;
	region_data_t *r_data = region->private_data;
	region_mapping_t *r_map;
	u_int64_t max_delta, min_delta;
	u_int32_t i;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Initializing shrink task for region %s.\n", region->name);

	rc = can_shrink_region(region);
	if (rc) {
		goto out;
	}

	r_map = EngFncs->last_thing(r_data->mappings, NULL);
	min_delta = c_data->pe_size * r_map->stripe_count;

	r_map = EngFncs->first_thing(r_data->mappings, NULL);
	max_delta = region->size - c_data->pe_size * r_map->stripe_count;

	LOG_DEBUG("Allowable range for shrink size: %"PRIu64" to %"PRIu64
		  " sectors in %"PRIu64" sector increments.\n",
		  min_delta, max_delta, c_data->pe_size);

	/* Ask the engine if it's ok to shrink by the max amount. */
	rc = EngFncs->can_shrink_by(region, &max_delta);
	if (rc) {
		if (rc != EAGAIN) {
			LOG_ERROR("Shrink of region %s rejected by the "
				  "engine.\n", region->name);
			goto out;
		}
		rc = 0;

		/* Make sure max_delta is still a multiple of the PE-size. */
		LOG_DEBUG("Engine will only allow max shrink size of %"PRIu64
			  " sectors.\n", max_delta);
		max_delta -= max_delta %
			     (c_data->pe_size * r_map->stripe_count);
		LOG_DEBUG("Reduced max shrink size to %"PRIu64" sectors.\n",
			  max_delta);
	}

	/* Initialize the "Size" option. */
	i = LVM2_OPTION_SHRINK_REGION_SIZE_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_SHRINK_REGION_SIZE_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Size to remove from the LVM2 region."));
	od->option[i].tip = EngFncs->engine_strdup(_("Removed size must be a multiple of the "
						     "container's extent-size. If not, it will "
						     "be rounded down as appropriate."));
	od->option[i].type = EVMS_Type_Unsigned_Int64;
	od->option[i].unit = EVMS_Unit_Sectors;
	od->option[i].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
			      EVMS_OPTION_FLAGS_AUTOMATIC;
	od->option[i].constraint_type = EVMS_Collection_Range;
	SET_RANGE64(od->option[i].constraint.range,
		    min_delta, max_delta, c_data->pe_size);
	od->option[i].value.ui64 = min_delta;

	od->count = LVM2_OPTION_SHRINK_REGION_COUNT;
	context->min_selected_objects = 0;
	context->max_selected_objects = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * shrink_region_set_option
 *
 * Validate the specified option-value for a shrink-region task. If the
 * option-value is valid, update the appropriate entry in the option-descriptor.
 **/
int shrink_region_set_option(task_context_t *context,
			     u_int32_t index,
			     value_t *value,
			     task_effect_t *effect)
{
	option_desc_array_t *od = context->option_descriptors;
	int rc = 0;

	LOG_ENTRY();

	switch (index) {

	case LVM2_OPTION_SHRINK_REGION_SIZE_IDX:
		/* Selected size must be within the constraint-range and
		 * must be a multiple of the constraint-range increment.
		 */
		if (value->ui64 < od->option[index].constraint.range->min.ui64) {
			value->ui64 = od->option[index].constraint.range->min.ui64;
		} else if (value->ui64 > od->option[index].constraint.range->max.ui64) {
			value->ui64 = od->option[index].constraint.range->max.ui64;
		} else {
			value->ui64 -= value->ui64 %
				       od->option[index].constraint.range->increment.ui64;
		}

		LOG_DEBUG("Setting size option: %"PRIu64" sectors\n", value->ui64);
		od->option[index].value.ui64 = value->ui64;
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * shrink_region_parse_options
 *
 * Parse the option-values from the option-array.
 **/
void shrink_region_parse_options(storage_object_t *region,
				 option_array_t *options,
				 u_int64_t *size)
{
	container_data_t *c_data = region->producing_container->private_data;
	region_data_t *r_data = region->private_data;
	region_mapping_t *r_map = EngFncs->last_thing(r_data->mappings, NULL);
	u_int i;

	LOG_ENTRY();

	/* Default values. */
	*size = c_data->pe_size * r_map->stripe_count;

	for (i = 0; i < options->count; i++) {
		/* If only the name-based index is specified, get the number. */
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name,
				    LVM2_OPTION_SHRINK_REGION_SIZE_STR)) {
				options->option[i].number = LVM2_OPTION_SHRINK_REGION_SIZE_IDX;
			} else {
				continue;
			}
		}

		/* Use the number-based index to record the option. */
		switch (options->option[i].number) {

		case LVM2_OPTION_SHRINK_REGION_SIZE_IDX:
			*size = options->option[i].value.ui64;
			LOG_DEBUG("Size option: %"PRIu64"\n", *size);
			break;

		default:
			break;
		}
	}

	LOG_EXIT_VOID();
}

/**
 * shrink_region_validate_options
 *
 * Verify that these options are valid for shrinking this region.
 **/
int shrink_region_validate_options(storage_object_t *region,
				   u_int64_t *size)
{
	container_data_t *c_data = region->producing_container->private_data;
	region_data_t *r_data = region->private_data;
	region_mapping_t *r_map = EngFncs->last_thing(r_data->mappings, NULL);
	int rc = 0;

	LOG_ENTRY();

	/* Size must be a multiple of the extent-size and less than the
	 * size of the entire region.
	 */
	if (*size < c_data->pe_size * r_map->stripe_count) {
		*size = c_data->pe_size * r_map->stripe_count;
		LOG_DEBUG("Rounding size up to %"PRIu64".\n", *size);

	} else if (*size > region->size - c_data->pe_size) {
		*size = region->size - c_data->pe_size;
		LOG_DEBUG("Rounding size down to %"PRIu64".\n", *size);

	} else if (*size % c_data->pe_size) {
		*size -= *size % c_data->pe_size;
		LOG_DEBUG("Rounding size down to %"PRIu64".\n", *size);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/**
 * Set-region-info routines.
 **/


/**
 * set_region_info_init_task
 *
 * Initialize the option-descriptor array for a set-region-info task. There
 * is no acceptable-objects list for set-info.
 **/
int set_region_info_init_task(task_context_t *context)
{
	option_desc_array_t *od = context->option_descriptors;
	storage_object_t *region = context->object;
	u_int32_t i;

	LOG_ENTRY();
	LOG_DEBUG("Initializing set-info task for region %s.\n", region->name);

	/* No info to set for freespace regions. */
	if (region->data_type != DATA_TYPE) {
		LOG_ERROR("No information to set for freespace region %s.\n",
			  region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* Initialize the "Name" option. */
	i = LVM2_OPTION_SET_REGION_INFO_NAME_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_SET_REGION_INFO_NAME_STR);
	od->option[i].title = EngFncs->engine_strdup(_("New name for this LVM2 region."));
	od->option[i].type = EVMS_Type_String;
	od->option[i].min_len = 1;
	od->option[i].max_len = EVMS_NAME_SIZE;
	od->option[i].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
	od->option[i].value.s = EngFncs->engine_alloc(EVMS_NAME_SIZE+1);

	od->count = LVM2_OPTION_SET_REGION_INFO_COUNT;
	context->min_selected_objects = 0;
	context->max_selected_objects = 0;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * set_region_info_set_option
 *
 * Validate the specified option-value for a set-region-info task. If the
 * option-value is valid, update the appropriate entry in the option-descriptor.
 **/
int set_region_info_set_option(task_context_t *context,
			       u_int32_t index,
			       value_t *value,
			       task_effect_t *effect)
{
	storage_container_t *container = context->object->producing_container;
	option_desc_array_t *od = context->option_descriptors;
	int rc;

	LOG_ENTRY();

	switch (index) {

	case LVM2_OPTION_SET_REGION_INFO_NAME_IDX:
		/* Make sure this name is not in use and that isn't too long. */
		rc = validate_lv_name(value->s, container->name);
		if (!rc) {
			LOG_DEBUG("Setting name option: %s\n", value->s);
			strncpy(od->option[index].value.s, value->s, EVMS_NAME_SIZE);
			od->option[index].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		}
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * set_region_info_parse_options
 *
 * Parse the option-values from the option-array.
 **/
void set_region_info_parse_options(storage_object_t *region,
				   option_array_t *options,
				   char  **lv_name)
{
	u_int i;

	LOG_ENTRY();

	/* Default values. */
	*lv_name = NULL;

	for (i = 0; i < options->count; i++) {
		/* If only the name-based index is specified, get the number. */
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name,
				    LVM2_OPTION_SET_REGION_INFO_NAME_STR)) {
				options->option[i].number = LVM2_OPTION_SET_REGION_INFO_NAME_IDX;
			} else {
				continue;
			}
		}

		/* Use the number-based index to record the option. */
		switch (options->option[i].number) {

		case LVM2_OPTION_SET_REGION_INFO_NAME_IDX:
			*lv_name = options->option[i].value.s;
			LOG_DEBUG("Name option: %s\n", *lv_name);
			break;

		default:
			break;
		}
	}

	LOG_EXIT_VOID();
}

/**
 * set_region_info_validate_options
 *
 * Verify that these options are valid for setting info for this region.
 **/
int set_region_info_validate_options(storage_object_t *region,
				     char *lv_name)
{
	int rc;

	LOG_ENTRY();

	/* Validate the new region name. */
	rc = validate_lv_name(lv_name, region->producing_container->name);

	LOG_EXIT_INT(rc);
	return rc;
}


/**
 * Set-container-info routines.
 **/


/**
 * set_container_info_init_task
 *
 * Initialize the option-descriptor array for a set-container-info task. There
 * is no acceptable-objects list for set-info.
 **/
int set_container_info_init_task(task_context_t *context)
{
	option_desc_array_t *od = context->option_descriptors;
	storage_container_t *container = context->container;
	u_int32_t i;

	LOG_ENTRY();
	LOG_DEBUG("Initializing set-info task for container %s.\n",
		  container->name);

	/* Initialize the "Name" option. */
	i = LVM2_OPTION_SET_CONTAINER_INFO_NAME_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_SET_CONTAINER_INFO_NAME_STR);
	od->option[i].title = EngFncs->engine_strdup(_("New name for this LVM2 container."));
	od->option[i].type = EVMS_Type_String;
	od->option[i].min_len = 1;
	od->option[i].max_len = EVMS_NAME_SIZE;
	od->option[i].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
	od->option[i].value.s = EngFncs->engine_alloc(EVMS_NAME_SIZE+1);

	od->count = LVM2_OPTION_SET_CONTAINER_INFO_COUNT;
	context->min_selected_objects = 0;
	context->max_selected_objects = 0;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * set_container_info_set_option
 *
 * Validate the specified option-value for a set-container-info task. If the
 * option-value is valid, update the appropriate entry in the option-descriptor.
 **/
int set_container_info_set_option(task_context_t *context,
				  u_int32_t index,
				  value_t *value,
				  task_effect_t *effect)
{
	option_desc_array_t *od = context->option_descriptors;
	int rc;

	LOG_ENTRY();

	switch (index) {

	case LVM2_OPTION_SET_CONTAINER_INFO_NAME_IDX:
		/* Make sure this name is not in use and that isn't too long. */
		rc = validate_vg_name(value->s, context->container->disk_group);
		if (!rc) {
			LOG_DEBUG("Setting name option: %s\n", value->s);
			strncpy(od->option[index].value.s, value->s, EVMS_NAME_SIZE);
			od->option[index].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		}
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * set_container_info_parse_options
 *
 * Parse the option-values from the option-array.
 **/
void set_container_info_parse_options(storage_container_t *container,
				      option_array_t *options,
				      char  **vg_name)
{
	u_int i;

	LOG_ENTRY();

	/* Default values. */
	*vg_name = NULL;

	for (i = 0; i < options->count; i++) {
		/* If only the name-based index is specified, get the number. */
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name,
				    LVM2_OPTION_SET_CONTAINER_INFO_NAME_STR)) {
				options->option[i].number = LVM2_OPTION_SET_CONTAINER_INFO_NAME_IDX;
			} else {
				continue;
			}
		}

		/* Use the number-based index to record the option. */
		switch (options->option[i].number) {

		case LVM2_OPTION_SET_CONTAINER_INFO_NAME_IDX:
			*vg_name = options->option[i].value.s;
			LOG_DEBUG("Name option: %s\n", *vg_name);
			break;

		default:
			break;
		}
	}

	LOG_EXIT_VOID();
}

/**
 * set_container_info_validate_options
 *
 * Verify that these options are valid for setting info for this container.
 **/
int set_container_info_validate_options(storage_container_t *container,
					char *vg_name)
{
	int rc;

	LOG_ENTRY();

	/* Validate the new container name. */
	rc = validate_vg_name(vg_name, container->disk_group);

	LOG_EXIT_INT(rc);
	return rc;
}


/**
 * Split-region-mapping routines.
 **/


/**
 * split_region_mapping_init_task
 *
 * Initialize the option-descriptor array for a split-region-mapping task.
 **/
int split_region_mapping_init_task(task_context_t *context)
{
	option_desc_array_t *od = context->option_descriptors;
	storage_object_t *region = context->object;
	region_data_t *r_data = region->private_data;
	u_int32_t mapping_count = EngFncs->list_count(r_data->mappings);
	region_mapping_t *r_map;
	list_element_t iter;
	u_int32_t j = 0;
	int rc, i;

	LOG_ENTRY();
	LOG_DEBUG("Initializing split-mapping task for region %s.\n",
		  region->name);

	/* Check that there are mappings that can be split. */
	rc = can_split_a_region_mapping(region);
	if (rc) {
		LOG_DEBUG("No mappings can be split for region %s.\n",
			  region->name);
		goto out;
	}

	/* Find the first mapping that can be split. */
	LIST_FOR_EACH(r_data->mappings, iter, r_map) {
		rc = can_split_region_mapping(r_map);
		if (!rc) {
			break;
		}
		j++;
	}

	/* "Mapping" option. Set up a constraint-range based on the number
	 * of mappings in this region. Every mapping may not be able to be
	 * split, so this may include some that aren't valid, but this
	 * presents a simpler list to the user. Plus, in practice it will
	 * be unlikely to have a mapping that cannot be split.
	 */
	i = LVM2_OPTION_SPLIT_MAPPING_MAP_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_SPLIT_MAPPING_MAP_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Index of the logical-mapping to split."));
	od->option[i].tip = EngFncs->engine_strdup(_("Display extended details for this region to see information about the mappings and determine which mapping you wish to split."));
	od->option[i].type = EVMS_Type_Unsigned_Int32;
	od->option[i].value.ui32 = j;
	od->option[i].constraint_type = EVMS_Collection_Range;
	SET_RANGE32(od->option[i].constraint.range, j, mapping_count - 1, 1);

	/* "Extent" option. Assume mapping "j" in this region is initially
	 * selected and set up a constraint-range based on the number of
	 * extents in that mapping. Each side of the split mapping must have
	 * at least one extent, so zero is not a valid choice for this option.
	 */
	i = LVM2_OPTION_SPLIT_MAPPING_EXTENT_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_SPLIT_MAPPING_EXTENT_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Extent within the selected mapping."));
	od->option[i].tip = EngFncs->engine_strdup(_("Display extended details for this region to determine where within this mapping to make the split."));
	od->option[i].type = EVMS_Type_Unsigned_Int64;
	od->option[i].value.ui64 = r_map->stripe_count;
	od->option[i].constraint_type = EVMS_Collection_Range;
	SET_RANGE64(od->option[i].constraint.range, r_map->stripe_count,
		    r_map->le_count - r_map->stripe_count, r_map->stripe_count);

	od->count = LVM2_OPTION_SPLIT_MAPPING_COUNT;

out:
	LOG_EXIT_INT(rc);
	return rc;
}


/**
 * split_region_mapping_set_option
 *
 * Validate the specified option-value for a split-mapping task. If the
 * option-value is valid, update the appropriate entry in the option-descriptor.
 **/
int split_region_mapping_set_option(task_context_t *context, u_int32_t index,
				    value_t *value, task_effect_t *effect)
{
	option_desc_array_t *od = context->option_descriptors;
	storage_object_t *region = context->object;
	region_mapping_t *r_map;
	int i, rc = EINVAL;

	LOG_ENTRY();

	switch (index) {

	case LVM2_OPTION_SPLIT_MAPPING_MAP_IDX:
		/* Make sure the selected value is
		 * within the acceptable range.
		 */
		if (value->ui32 < od->option[index].constraint.range->min.ui32 ||
		    value->ui32 > od->option[index].constraint.range->max.ui32) {
			rc = EINVAL;
			break;
		}

		/* Find the desired mapping and verify that it can be split. */
		r_map = find_mapping_by_index(region, value->ui32);

		rc = can_split_region_mapping(r_map);
		if (rc) {
			LOG_ERROR("Mapping %u in region %s cannot be split.\n",
				  value->ui32, region->name);
			break;
		}

		LOG_DEBUG("Setting \"map\" option to %u.\n", value->ui32);
		od->option[index].value.ui32 = value->ui32;

		/* Reset the constraint range and initial
		 * value for the extent option.
		 */
		i = LVM2_OPTION_SPLIT_MAPPING_EXTENT_IDX;
		EngFncs->engine_free(od->option[i].constraint.range);
		SET_RANGE64(od->option[i].constraint.range, r_map->stripe_count,
			    r_map->le_count - r_map->stripe_count,
			    r_map->stripe_count);
		od->option[i].value.ui64 = r_map->stripe_count;
		*effect |= EVMS_Effect_Reload_Options;

		break;

	case LVM2_OPTION_SPLIT_MAPPING_EXTENT_IDX:
		/* Round down to nearest stripe_count boundary. */
		if (value->ui64 % od->option[index].constraint.range->increment.ui64) {
			value->ui64 -= value->ui64 % od->option[index].constraint.range->increment.ui64;
			*effect |= EVMS_Effect_Inexact;
		}

		/* Make sure the selected value is
		 * within the acceptable range.
		 */
		if (value->ui64 < od->option[index].constraint.range->min.ui64 ||
		    value->ui64 > od->option[index].constraint.range->max.ui64 ) {
			rc = EINVAL;
			break;
		}

		LOG_DEBUG("Setting \"extent\" option to %"PRIu64".\n",
			  value->ui64);
		od->option[index].value.ui64 = value->ui64;
		rc = 0;
		break;

	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * split_mapping_parse_options
 *
 * Parse the option-values from the option-array.
 **/
void split_mapping_parse_options(option_array_t *options,
				 u_int32_t *r_map_index,
				 u_int64_t *extent_index)
{
	u_int i;

	LOG_ENTRY();

	/* Default values. */
	*r_map_index = 0;
	*extent_index = 1;

	for (i = 0; i < options->count; i++) {
		/* If only the name-based index is specified, get the number. */
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name,
				    LVM2_OPTION_SPLIT_MAPPING_MAP_STR)) {
				options->option[i].number = LVM2_OPTION_SPLIT_MAPPING_MAP_IDX;
			} else if (!strcmp(options->option[i].name,
					   LVM2_OPTION_SPLIT_MAPPING_EXTENT_STR)) {
				options->option[i].number = LVM2_OPTION_SPLIT_MAPPING_EXTENT_IDX;
			} else {
				continue;
			}
		}

		/* Use the number-based index to record the option. */
		switch (options->option[i].number) {

		case LVM2_OPTION_SPLIT_MAPPING_MAP_IDX:
			*r_map_index = options->option[i].value.ui32;
			LOG_DEBUG("Map option: %u\n", *r_map_index);
			break;

		case LVM2_OPTION_SPLIT_MAPPING_EXTENT_IDX:
			*extent_index = options->option[i].value.ui64;
			LOG_DEBUG("Extent option: %"PRIu64"\n", *extent_index);
			break;

		default:
			break;
		}
	}

	LOG_EXIT_VOID();
}

/**
 * split_mapping_validate_options
 *
 * Verify that these options are valid for splitting a mapping in the
 * specified region. Return a pointer to the mapping to be split if it
 * is found correctly.
 **/
int split_mapping_validate_options(storage_object_t *region,
				   u_int32_t r_map_index,
				   u_int64_t *extent_index,
				   region_mapping_t **map_to_split)
{
	region_data_t *r_data = region->private_data;
	region_mapping_t *r_map;
	int rc;

	LOG_ENTRY();

	/* Find the desired region-map. */
	r_map = find_mapping_by_index(region, r_map_index);
	if (!r_map) {
		LOG_ERROR("Mapping %u does not exist. Region %s only contains "
			  "%u mappings.\n", r_map_index, region->name, 
			  EngFncs->list_count(r_data->mappings));
		rc = EINVAL;
		goto out;
	}

	/* Make sure this mapping can be split. */
	rc = can_split_region_mapping(r_map);
	if (rc) {
		LOG_ERROR("Mapping %u in region %s cannot be split - not "
			  "large enough.\n", r_map_index, region->name);
		goto out;
	}

	/* Make sure the specified extent is valid for splitting this mapping.
	 * It must be less than the number of extents in the mapping (but not
	 * zero) and a multiple of the number of stripes.
	 */
	if (*extent_index % r_map->stripe_count) {
		/* Round down to the nearest stripe_count multiple. */
		*extent_index -= *extent_index % r_map->stripe_count;
	}

	if (*extent_index == 0 ||
	    *extent_index >= r_map->le_count) {
		LOG_ERROR("Extent %"PRIu64" is not a valid location to split "
			  "mapping %u on region %s.\n",
			  *extent_index, r_map_index, region->name);
		rc = EINVAL;
		goto out;
	}

	*map_to_split = r_map;

out:
	LOG_EXIT_INT(rc);
	return rc;
}


/**
 * Merge-region-mappings routines.
 **/


/**
 * merge_region_mappings_init_task
 *
 * Check if there are any mappings in this region that can be merged. There
 * are no options or selected-objects, so there's not much to do here.
 **/
int merge_region_mappings_init_task(task_context_t *context)
{
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Initializing merge-mappings task for region %s.\n",
		  context->object->name);

	rc = can_merge_region_mappings(context->object);

	LOG_EXIT_INT(rc);
	return rc;
}


/**
 * Move-region-mapping routines.
 **/


/**
 * move_mapping_init_map_option_list
 *
 * Create a constraint list of all mappings in this region that could be
 * moved. Return a pointer to the first available mapping that is found.
 **/
static int move_mapping_init_map_option_list(storage_object_t *region,
					     value_list_t **list,
					     region_mapping_t **selected_r_map)
{
	storage_container_t *container = region->producing_container;
	region_data_t *r_data = region->private_data;
	region_mapping_t *r_map;
	list_element_t iter;
	u_int64_t max_con_extents;
	u_int32_t i;
	int rc, j;

	LOG_ENTRY();

	*selected_r_map = NULL;
	max_con_extents = max_consecutive_extents_in_container(container);

	/* Allocate the constraint list. */
	*list = EngFncs->engine_alloc(sizeof(value_list_t) + sizeof(value_t) *
				      EngFncs->list_count(r_data->mappings));
	if (!*list) {
		rc = ENOMEM;
		goto out;
	}

	/* Check each mapping in this region. */
	i = j = 0;
	LIST_FOR_EACH(r_data->mappings, iter, r_map) {
		rc = can_move_region_mapping(r_map, max_con_extents);
		if (!rc) {
			(*list)->value[j++].ui32 = i;
			if (!*selected_r_map) {
				/* Return the first valid mapping.*/
				*selected_r_map = r_map;
			}
		}
		i++;
	}
	(*list)->count = j;

	/* Need at least one mapping which can be moved. */
	rc = (*selected_r_map) ? 0 : ENOSPC;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * move_mapping_init_stripe_option_list
 *
 * Create a constraint list of all stripes in this mapping that aren't
 * already scheduled to be moved.
 **/
static int move_mapping_init_stripe_option_list(region_mapping_t *r_map,
						value_list_t **list)
{
	u_int64_t i;
	int rc, j;

	LOG_ENTRY();

	/* Allocate the constraint list. */
	*list = EngFncs->engine_alloc(sizeof(value_list_t) +
				      sizeof(value_t) * r_map->stripe_count);
	if (!*list) {
		rc = ENOMEM;
		goto out;
	}

	/* Each stripe can be moved as long as it
	 * isn't already scheduled to be moved.
	 */
	for (i = 0, j = 0; i < r_map->stripe_count; i++) {
		rc = can_move_stripe(r_map->le_maps + i);
		if (!rc) {
			(*list)->value[j++].ui64 = i;
		}
	}
	(*list)->count = j;

	/* Need at least one stripe which can be moved. */
	rc = j ? 0 : EBUSY;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * move_mapping_init_object_option_list
 *
 * Create a constraint list of all objects in this mapping's container that
 * have enough consecutive freespace to move this mapping to. Return a pointer
 * to the first valid object that is found.
 **/
static int move_mapping_init_object_option_list(region_mapping_t *r_map,
						value_list_t **list,
						storage_object_t **selected_object)
{
	storage_container_t *container = r_map->r_data->region->producing_container;
	u_int64_t extents_per_stripe = r_map->le_count / r_map->stripe_count;
	u_int64_t max_con_extents;
	storage_object_t *object;
	list_element_t iter;
	int rc, i;

	LOG_ENTRY();

	*selected_object = NULL;

	/* Allocate the constraint list. */
	*list = EngFncs->engine_alloc(sizeof(value_list_t) + sizeof(value_t) *
				      EngFncs->list_count(container->objects_consumed));
	if (!*list) {
		rc = ENOMEM;
		goto out;
	}

	/* Check every PV object in this container. */
	i = 0;
	LIST_FOR_EACH(container->objects_consumed, iter, object) {
		max_con_extents = max_consecutive_extents_on_object(object);
		if (extents_per_stripe <= max_con_extents) {
			(*list)->value[i++].s = EngFncs->engine_strdup(object->name);
			if (!*selected_object) {
				/* Return the first valid object. */
				*selected_object = object;
			}
		}
	}
	(*list)->count = i;

	/* Need at least one stripe which can be moved. */
	rc = (*selected_object) ? 0 : ENOSPC;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * move_mapping_init_extent_option_list
 *
 * Create a constraint list of all PEs in the selected object that begin a
 * consecutive area of freespace large enough to move the specified mapping.
 **/
static int move_mapping_init_extent_option_list(region_mapping_t *r_map,
						storage_object_t *object,
						value_list_t **list)
{
	pv_data_t *pv_data = object->consuming_private_data;
	u_int64_t extents_per_stripe = r_map->le_count / r_map->stripe_count;
	u_int64_t i, j, max_con_extents;
	int rc, k;

	LOG_ENTRY();

	/* Allocate the constraint list. */
	*list = EngFncs->engine_alloc(sizeof(value_list_t) +
				      sizeof(value_t) * pv_data->pe_count);
	if (!*list) {
		rc = ENOMEM;
		goto out;
	}

	for (i = 0, k = 0; i < pv_data->pe_count; i++) {
		max_con_extents = consecutive_extents_at_pe(pv_data, i);
		if (max_con_extents >= extents_per_stripe) {
			for (j = 0; j < max_con_extents - extents_per_stripe + 1; j++) {
				(*list)->value[k++].ui64 = i + j;
			}
		}
		i += max_con_extents;
	}
	(*list)->count = k;

	/* Need at least one destination extent. */
	rc = k ? 0 : ENOSPC;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * move_region_mapping_init_task
 *
 * Initialize the option-descriptor array for a move-region-mapping task.
 **/
int move_region_mapping_init_task(task_context_t *context)
{
	option_desc_array_t *od = context->option_descriptors;
	storage_object_t *object, *region = context->object;
	region_mapping_t *r_map;
	int i, rc;

	LOG_ENTRY();
	LOG_DEBUG("Initializing move-mapping task for region %s.\n",
		  region->name);

	/* "Mapping" option.
	 * Check each mapping in this region to determine which can be moved.
	 */
	i = LVM2_OPTION_MOVE_MAPPING_MAP_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_MOVE_MAPPING_MAP_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Index of the logical-mapping to move."));
	od->option[i].tip = EngFncs->engine_strdup(_("Display extended details for this region to see information "
						     "about the mappings and determine which mapping you wish to move."));
	od->option[i].type = EVMS_Type_Unsigned_Int32;
	od->option[i].constraint_type = EVMS_Collection_List;

	/* Set up the constraint list. */
	rc = move_mapping_init_map_option_list(region,
					       &(od->option[i].constraint.list),
					       &r_map);
	if (rc) {
		goto out;
	}

	/* Set the default value to the first item in the constraint list. */
	od->option[i].value.ui32 = od->option[i].constraint.list->value[0].ui32;


	/* "Stripe" option.
	 * See which stripes in the selected mapping can be moved. If the
	 * selected mapping is linear, we don't need to display this option.
	 */
	i = LVM2_OPTION_MOVE_MAPPING_STRIPE_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_MOVE_MAPPING_STRIPE_STR);
	od->option[i].title = EngFncs->engine_strdup(_("The stripe within this mapping to move."));
	od->option[i].tip = EngFncs->engine_strdup(_("Only one stripe per mapping can be moved at a time. Display extended "
						     "details for this region to see information about the mappings and "
						     "determine which stripe within this mapping you wish to move."));
	od->option[i].type = EVMS_Type_Unsigned_Int64;
	if (r_map->stripe_count == 1) {
		od->option[i].flags |= EVMS_OPTION_FLAGS_INACTIVE;
		od->option[i].value.ui64 = 0;
	} else {
		/* Set up the constraint list. */
		od->option[i].constraint_type = EVMS_Collection_List;
		rc = move_mapping_init_stripe_option_list(r_map,
							  &(od->option[i].constraint.list));
		if (rc) {
			goto out;
		}

		/* Set the default value to the first item in the constraint list. */
		od->option[i].value.ui64 = od->option[i].constraint.list->value[0].ui64;
	}


	/* "Object" option.
	 * Find all the PVs in this container that could be
	 * used as a destination for moving this mapping.
	 */
	i = LVM2_OPTION_MOVE_MAPPING_OBJECT_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_MOVE_MAPPING_OBJECT_STR);
	od->option[i].title = EngFncs->engine_strdup(_("The PV object to move this mapping to."));
	od->option[i].tip = EngFncs->engine_strdup(_("Display extended details for this region's container and PVs to "
						     "determine which PV has space available to move this mapping."));
	od->option[i].type = EVMS_Type_String;
	od->option[i].min_len = 1;
	od->option[i].max_len = EVMS_NAME_SIZE;
	od->option[i].value.s = EngFncs->engine_alloc(EVMS_NAME_SIZE+1);
	od->option[i].constraint_type = EVMS_Collection_List;

	/* Set up the constraint list. */
	rc = move_mapping_init_object_option_list(r_map,
						  &(od->option[i].constraint.list),
						  &object);
	if (rc) {
		goto out;
	}

	/* Set the default value to the name of the selected PV object. */
	strncpy(od->option[i].value.s, object->name, EVMS_NAME_SIZE);


	/* "Extent" option.
	 * Look for all PEs on the selected object which could
	 * start the destination area for moving this mapping.
	 */
	i = LVM2_OPTION_MOVE_MAPPING_EXTENT_IDX;
	od->option[i].name = EngFncs->engine_strdup(LVM2_OPTION_MOVE_MAPPING_EXTENT_STR);
	od->option[i].title = EngFncs->engine_strdup(_("The starting PE of the destination area for the move."));
	od->option[i].tip = EngFncs->engine_strdup(_("Display extended details for this region's container "
						     "and the selected PV object to determine where within "
						     "the PE-map has space available to move this mapping."));
	od->option[i].type = EVMS_Type_Unsigned_Int64;
	od->option[i].constraint_type = EVMS_Collection_List;

	/* Set up the constraint list. */
	rc = move_mapping_init_extent_option_list(r_map, object,
						  &(od->option[i].constraint.list));
	if (rc) {
		goto out;
	}

	/* Set the default value to the first item in the constraint list. */
	od->option[i].value.ui64 = od->option[i].constraint.list->value[0].ui64;


	od->count = LVM2_OPTION_MOVE_MAPPING_COUNT;
	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * move_region_mapping_set_option
 *
 * Validate the specified option-value for a move-mapping task. If the
 * option-value is valid, update the appropriate entry in the option-descriptor.
 **/
int move_region_mapping_set_option(task_context_t *context, u_int32_t index,
				   value_t *value, task_effect_t *effect)
{
	option_desc_array_t *od = context->option_descriptors;
	storage_object_t *object, *region = context->object;
	region_mapping_t *r_map;
	int i, rc;

	LOG_ENTRY();

	switch (index) {

	/* "Map" option. */
	case LVM2_OPTION_MOVE_MAPPING_MAP_IDX:

		/* Make sure the selected mapping is in the constraint-list. */
		for (i = 0; i < od->option[index].constraint.list->count; i++) {
			if (value->ui32 ==
			    od->option[index].constraint.list->value[i].ui32) {
				break;
			}
		}
		if (i == od->option[index].constraint.list->count) {
			LOG_ERROR("Invalid mapping index specified: %u.\n",
				  value->ui32);
			rc = EINVAL;
			break;
		}

		LOG_DEBUG("Setting \"map\" option to %u.\n", value->ui32);
		od->option[index].value.ui32 = value->ui32;

		/* Find the selected mapping in the region. */
		r_map = find_mapping_by_index(region, value->ui32);

		/* Update "stripe" option. */
		index = LVM2_OPTION_MOVE_MAPPING_STRIPE_IDX;

		EngFncs->engine_free(od->option[index].constraint.list);
		od->option[index].constraint.list = NULL;
		if (r_map->stripe_count == 1) {
			od->option[index].flags |= EVMS_OPTION_FLAGS_INACTIVE;
			od->option[index].constraint_type = 0;
			od->option[index].value.ui64 = 0;
		} else {
			/* Set up a new constraint list. */
			od->option[index].constraint_type = EVMS_Collection_List;
			rc = move_mapping_init_stripe_option_list(r_map,
								  &(od->option[index].constraint.list));
			if (rc) {
				break;
			}

			/* Set the default value to the first item in the constraint list. */
			od->option[index].value.ui64 = od->option[index].constraint.list->value[0].ui64;
		}

		/* Update "object" option. */
		index = LVM2_OPTION_MOVE_MAPPING_OBJECT_IDX;

		/* Set up a new constraint list. */
		for (i = 0; i < od->option[index].constraint.list->count; i++) {
			EngFncs->engine_free(od->option[index].constraint.list->value[i].s);
		}
		EngFncs->engine_free(od->option[index].constraint.list);

		rc = move_mapping_init_object_option_list(r_map,
							  &(od->option[index].constraint.list),
							  &object);
		if (rc) {
			break;
		}

		/* Set the default value to the name of the selected PV object. */
		strncpy(od->option[index].value.s, object->name, EVMS_NAME_SIZE);

		/* Update "extent" option. */
		index = LVM2_OPTION_MOVE_MAPPING_EXTENT_IDX;

		/* Set up a new constraint list. */
		EngFncs->engine_free(od->option[index].constraint.list);

		/* Set up the constraint list. */
		rc = move_mapping_init_extent_option_list(r_map, object,
							  &(od->option[index].constraint.list));
		if (rc) {
			break;
		}

		/* Set the default value to the first item in the constraint list. */
		od->option[index].value.ui64 = od->option[index].constraint.list->value[0].ui64;

		*effect |= EVMS_Effect_Reload_Options;
		break;

	/* "Stripe" option. */
	case LVM2_OPTION_MOVE_MAPPING_STRIPE_IDX:

		/* Make sure the selected stripe is in the constraint-list. */
		for (i = 0; i < od->option[index].constraint.list->count; i++) {
			if (value->ui64 ==
			    od->option[index].constraint.list->value[i].ui64) {
				break;
			}
		}
		if (i == od->option[index].constraint.list->count) {
			LOG_ERROR("Invalid stripe index specified: %"
				  PRIu64".\n", value->ui64);
			rc = EINVAL;
			break;
		}

		LOG_DEBUG("Setting \"stripe\" option to %"PRIu64".\n",
			  value->ui64);
		od->option[index].value.ui64 = value->ui64;

		/* Since the selected mapping has not changed (only the stripe
		 * within that mapping) none of the other options are affected.
		 */

		rc = 0;
		break;

	/* "Object" option. */
	case LVM2_OPTION_MOVE_MAPPING_OBJECT_IDX:

		/* Make sure the selected object is in the constraint-list. */
		for (i = 0; i < od->option[index].constraint.list->count; i++) {
			rc = strncmp(value->s,
				     od->option[index].constraint.list->value[i].s,
				     EVMS_NAME_SIZE);
			if (!rc) {
				break;
			}
		}
		if (i == od->option[index].constraint.list->count) {
			LOG_ERROR("Invalid object name specified: %s\n",
				  value->s);
			rc = EINVAL;
			break;
		}

		LOG_DEBUG("Setting \"object\" option to %s.\n", value->s);
		strncpy(od->option[index].value.s, value->s, EVMS_NAME_SIZE);

		/* Get the selected mapping and PV object. */
		index = LVM2_OPTION_MOVE_MAPPING_MAP_IDX;
		r_map = find_mapping_by_index(region, od->option[index].value.ui32);
		object = find_pv_by_name(region->producing_container, value->s);

		/* Update "extent" option. */
		index = LVM2_OPTION_MOVE_MAPPING_EXTENT_IDX;

		/* Set up a new constraint list. */
		EngFncs->engine_free(od->option[index].constraint.list);

		/* Set up the constraint list. */
		rc = move_mapping_init_extent_option_list(r_map, object,
							  &(od->option[index].constraint.list));
		if (rc) {
			break;
		}

		/* Set the default value to the first item in the constraint list. */
		od->option[index].value.ui64 = od->option[index].constraint.list->value[0].ui64;

		*effect |= EVMS_Effect_Reload_Options;
		break;

	/* "Extent" option. */
	case LVM2_OPTION_MOVE_MAPPING_EXTENT_IDX:

		/* Make sure the selected extent is in the constraint-list. */
		for (i = 0; i < od->option[index].constraint.list->count; i++) {
			if (value->ui64 ==
			    od->option[index].constraint.list->value[i].ui64) {
				break;
			}
		}
		if (i == od->option[index].constraint.list->count) {
			LOG_ERROR("Invalid physical extent specified: %"
				  PRIu64".\n", value->ui64);
			rc = EINVAL;
			break;
		}

		LOG_DEBUG("Setting \"extent\" option to %"PRIu64".\n",
			  value->ui64);
		od->option[index].value.ui64 = value->ui64;

		rc = 0;
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * move_mapping_parse_options
 *
 * Parse the option-values from the option-array.
 **/
void move_mapping_parse_options(option_array_t *options,
				u_int32_t *r_map_index,
				u_int64_t *stripe_index,
				char **object_name,
				u_int64_t *extent_index)
{
	u_int i;

	/* Default values. */
	*r_map_index = 0;
	*stripe_index = 0;
	*object_name = NULL;	
	*extent_index = 0;

	LOG_ENTRY();

	for (i = 0; i < options->count; i++) {
		/* If only the name-based index is specified, get the number. */
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name,
				    LVM2_OPTION_MOVE_MAPPING_MAP_STR)) {
				options->option[i].number = LVM2_OPTION_MOVE_MAPPING_MAP_IDX;
			} else if (!strcmp(options->option[i].name,
					   LVM2_OPTION_MOVE_MAPPING_STRIPE_STR)) {
				options->option[i].number = LVM2_OPTION_MOVE_MAPPING_STRIPE_IDX;
			} else if (!strcmp(options->option[i].name,
					   LVM2_OPTION_MOVE_MAPPING_OBJECT_STR)) {
				options->option[i].number = LVM2_OPTION_MOVE_MAPPING_OBJECT_IDX;
			} else if (!strcmp(options->option[i].name,
					   LVM2_OPTION_MOVE_MAPPING_EXTENT_STR)) {
				options->option[i].number = LVM2_OPTION_MOVE_MAPPING_EXTENT_IDX;
			} else {
				continue;
			}
		}

		/* Use the number-based index to record the option. */
		switch (options->option[i].number) {

		case LVM2_OPTION_MOVE_MAPPING_MAP_IDX:
			*r_map_index = options->option[i].value.ui32;
			LOG_DEBUG("Map option: %u\n", *r_map_index);
			break;

		case LVM2_OPTION_MOVE_MAPPING_STRIPE_IDX:
			*stripe_index = options->option[i].value.ui64;
			LOG_DEBUG("Stripe option: %"PRIu64"\n", *stripe_index);
			break;

		case LVM2_OPTION_MOVE_MAPPING_OBJECT_IDX:
			*object_name = options->option[i].value.s;
			LOG_DEBUG("Object option: %s\n", *object_name);
			break;

		case LVM2_OPTION_MOVE_MAPPING_EXTENT_IDX:
			*extent_index = options->option[i].value.ui64;
			LOG_DEBUG("Extent option: %"PRIu64"\n", *extent_index);
			break;

		default:
			break;
		}
	}

	LOG_EXIT_VOID();
}

/**
 * move_mapping_validate_options
 *
 * Verify that the move-mapping options are valid. If they are, return
 * pointers to the selected mapping and destination PV.
 **/
int move_mapping_validate_options(storage_object_t *region,
				  u_int32_t r_map_index,
				  u_int64_t stripe_index,
				  char *object_name,
				  u_int64_t extent_index,
				  region_mapping_t **r_map,
				  storage_object_t **object)
{
	storage_container_t *container = region->producing_container;
	u_int64_t max_con_extents;
	pv_data_t *pv_data;
	int rc;

	LOG_ENTRY();

	/* Find the selected mapping. */
	*r_map = find_mapping_by_index(region, r_map_index);
	if (!*r_map) {
		LOG_ERROR("Could not find mapping %u in region %s.\n",
			  r_map_index, region->name);
		rc = EINVAL;
		goto out;
	}

	/* Find the selected object. */
	*object = find_pv_by_name(container, object_name);
	if (!*object) {
		LOG_ERROR("Could not find object %s in container %s.\n",
			  object_name, container->name);
		rc = EINVAL;
		goto out;
	}
	pv_data = (*object)->consuming_private_data;

	/* Make sure the selected mapping can be moved. */
	max_con_extents = consecutive_extents_at_pe(pv_data, extent_index);
	rc = can_move_region_mapping(*r_map, max_con_extents);
	if (rc) {
		LOG_ERROR("Cannot move mapping %u in region %s.\n",
			  r_map_index, region->name);
		goto out;
	}

	/* Make sure the selected stripe is valid and can be moved. */
	if (stripe_index >= (*r_map)->stripe_count) {
		LOG_ERROR("Selected stripe %"PRIu64". Mapping %u in region %s "
			  "only has %"PRIu64" stripes.\n", stripe_index,
			  r_map_index, region->name, (*r_map)->stripe_count);
		rc = EINVAL;
		goto out;
	}

	rc = can_move_stripe((*r_map)->le_maps + stripe_index);
	if (rc) {
		LOG_ERROR("Stripe %"PRIu64" in mapping %u in region "
			  "%s cannot be moved at this time.\n",
			  stripe_index, r_map_index, region->name);
		goto out;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

