/*
 *
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <glib.h>
#include <ncurses.h>
#include <panel.h>
#include <frontend.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "dialog.h"
#include "clist.h"
#include "selwin.h"
#include "views.h"
#include "save.h"
#include "callbacks.h"
#include "settings.h"
#include "logging.h"

extern int open_engine(char *node);

/**
 *	int get_number_of_nodes (void)
 *
 *	This routine gets the number of nodes if running in a clustered
 *	environment.
 */
int get_number_of_nodes(void)
{
	int rc = 0;
	node_list_t *nodes;
	int count;

	rc = evms_get_node_list(ACTIVE_NODES_ONLY, &nodes);

	if (rc == 0) {
		count = nodes->count;
		evms_free(nodes);
		return count;
	}

	return 0;
}

/**
 *	open_remote_engine - open the engine on a given node
 *	@node: the new node to administer
 *
 *	This routine handles the opening on an engine on another node in
 *	the cluster.
 */
int open_remote_engine(char *node)
{
	int rc;
	
	enable_message_queuing();
	rc = evms_set_current_node(node);
	disable_message_queuing();
	display_queued_messages();
	if (rc == 0) {
		char *message;

		message = g_strdup_printf(_("Now administering node %s"), node);
		print_statusbar_text(message);
		g_free(message);

		refresh_views();
	} else {
		update_status(_("Engine open"), rc);
		show_message_dialog(_("Error"),
			_("Failed to open remote node engine. Please exit and restart evmsn."));
	}

	return rc;
}

/**
 *	administer_node_button_activated - switch to selected node by closing and reopening engine
 *	@item: the menu item that was activated
 *
 *	This routine handles switching to the new remote node. We first check to see if there
 *	are pending changed before closing the engine. If so, we prompt the user to see if they
 *	want to changes saved. After getting the response, we save changes if necessary then
 *	close the engine and reopen the engine with the new node. Once the engine has reopened,
 *	we schedule to have the views refreshed.
 */
int administer_node_button_activated(struct menu_item *item)
{
	int rc;
	char *new_node;
	boolean changes_pending;
	struct selwin *selwin = item->user_data;

	new_node = get_selected_data(selwin->clist);

	rc = evms_changes_pending(&changes_pending, NULL);
	if (rc == 0) {
		if (changes_pending) {
			char *title;

			title = g_strdup_printf(_("Save changes before switching to node %s?"), new_node);
			show_save_confirmation_dialog(title, _("_Continue without Saving"),
						_("Do you wish to save changes before switching to the new node?"));
			g_free(title);
		}
		
		rc = open_remote_engine(new_node);
		if (rc == 0)
			((struct dialog_window *)selwin)->status = DLG_STATUS_CLOSING;

		rc = evms_set_current_node(new_node);
		refresh_views();

	} else {
		show_message_dialog(_("Error"), _("Unable to determine if changes are pending!"));
		log_error("%s: evms_changes_pending() returned error code %d.\n", __FUNCTION__, rc);
	}

	return 0;
}

/**
 *	on_delete_node_clist_item - invoked when a node clist item is deleted
 *	@item: the clist item being deleted
 *
 *	This routine is a clist_item_delete_cb function invoked when a clist item for
 *	a nodename is deleted. This allows us to free the node name string associated
 *	with the item.
 */
void on_delete_node_clist_item(struct clist_item *item)
{
	g_free(item->user_data);
}

/**
 *	populate_clist_with_active_nodes - populate the remote nodes to administer clist
 *	@clist: the active nodes selection list
 *
 *	This routine retieves the list of active remote nodes that we can switch to and
 *	appends each to the given clist.
 */
void populate_clist_with_active_nodes(struct clist *clist)
{
	int rc;
	struct node_list_s *nodes;

	rc = evms_get_node_list(ACTIVE_NODES_ONLY, &nodes);
	if (rc == 0) {
		int i;

		for (i = 0; i < nodes->count; i++) {
			char *node_name;
			GPtrArray *text;

			text = g_ptr_array_new();
			node_name = g_strdup(nodes->node_info[i].node_name);
			g_ptr_array_add(text, " ");
			g_ptr_array_add(text, node_name);
    			append_item(clist, text, node_name, (clist_item_delete_cb)on_delete_node_clist_item);
			g_ptr_array_free(text, FALSE);
		}
		evms_free(nodes);
	}
}

/**
 *	show_nodes_dialog_menuitem_activated - display nodes that we can administer
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to display a selection list of nodes to remote administer.
 *	When the user attempts to switch to another node we will check for pending changes
 *	and prompt to have them saved before closing the engine and opening the engine on
 *	the remote node. We'll also refresh the views to coincide with the new node being
 *	administered.
 */
int show_nodes_dialog_menuitem_activated(struct menu_item *item)
{
	struct selwin *selwin;
	struct dialog_window *dialog;

	selwin = create_selection_window(_("Administer Remote Node"),
					NULL, NULL,
					_("_Administer"),
					(menuitem_activate_cb)administer_node_button_activated,
					NULL, NULL, NULL);

	dialog = (struct dialog_window *)selwin;

	set_clist_column_count(selwin->clist, 2);
	set_clist_column_info(selwin->clist, 0, calc_clist_column_width(selwin->clist, 0.05),
				0,
				CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(selwin->clist, 1, calc_clist_column_width(selwin->clist, 0.95),
				get_clist_column_end(selwin->clist, 0),
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(selwin->clist, 0, "");
	print_clist_column_title(selwin->clist, 1, _("Nodename"));

	set_menu_item_visibility(dialog->prev_button, FALSE);

	populate_clist_with_active_nodes(selwin->clist);

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	process_modal_dialog(dialog);
	return 0;
}

/**
 *	set_log_level_button_activated - set the engine log level to match the current selection level
 *	@item: the menu item button that was activated
 *
 *	This routine sets the engine log to match the level selected by the user.
 */
int set_log_level_button_activated(struct menu_item *item)
{
	int rc;
	struct selwin *selwin = item->user_data;
	struct dialog_window *dialog = item->user_data;

	rc = evms_set_debug_level(GPOINTER_TO_UINT(get_selected_data(selwin->clist)));
	update_status(_("Set Log Level"), rc);
	dialog->status = DLG_STATUS_CLOSING;

	return 0;
}

/**
 *	append_log_level_to_clist - append an item corresponding to the given log level
 *	@clist: the log level selection list
 *	@description: the description of the log level
 *	@level: the log level code
 *	@current_level: the current debug level set in the engine
 *
 *	This routine appends one log level item to the log level selection list.
 */
void append_log_level_to_clist(struct clist *clist, char *description, debug_level_t level,
				debug_level_t current_level)
{
	GPtrArray *text;
	struct clist_item *item;

	text = g_ptr_array_new();

	g_ptr_array_add(text, " ");
	g_ptr_array_add(text, description);
 	item = append_item(clist, text, GUINT_TO_POINTER(level), NULL);

	if (level == current_level)
		select_item(clist, item);

	g_ptr_array_free(text, FALSE);
}

/**
 *	populate_clist_with_log_levels - append log level descriptions to the selection list
 *	@clist: the log level selection list
 *
 *	This routine appends items for the log level selection list. It selects the log level
 *	that matches the current log level.
 */
void populate_clist_with_log_levels(struct clist *clist)
{
	debug_level_t current_level = DEFAULT;

	evms_get_debug_level(&current_level);

	append_log_level_to_clist(clist, _("Critical failures only"),
				CRITICAL, current_level);
	append_log_level_to_clist(clist, _("Previous item plus serious errors"),
				SERIOUS, current_level);
	append_log_level_to_clist(clist, _("Previous item plus user errors"),
				ERROR, current_level);
	append_log_level_to_clist(clist, _("Previous item plus warnings"),
				WARNING, current_level);
	append_log_level_to_clist(clist, _("Previous item plus recoverable errors"),
				DEFAULT, current_level);
	append_log_level_to_clist(clist, _("Previous item plus detailed system information"),
				DETAILS, current_level);
	append_log_level_to_clist(clist, _("Previous item plus information to debug a problem"),
				DEBUG, current_level);
	append_log_level_to_clist(clist, _("Previous item plus more debug information"),
				EXTRA, current_level);
	append_log_level_to_clist(clist, _("Previous item plus trace function entry and exit"),
				ENTRY_EXIT, current_level);
	append_log_level_to_clist(clist, _("Verbose output (includes all log events)"),
				EVERYTHING, current_level);
}

/**
 *	show_log_level_dialog_menuitem_activated - display dialog to allow setting engine log level
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to display a selection list of log levels for the engine
 *	logging. We pre-select the current log level.
 */
int show_log_level_dialog_menuitem_activated(struct menu_item *item)
{
	struct selwin *selwin;
	struct dialog_window *dialog;

	selwin = create_selection_window(_("Set Log Level"),
					NULL, NULL,
					_("_Set"),
					(menuitem_activate_cb)set_log_level_button_activated,
					NULL, NULL, NULL);

	dialog = (struct dialog_window *)selwin;

	set_clist_column_count(selwin->clist, 2);
	set_clist_column_info(selwin->clist, 0, calc_clist_column_width(selwin->clist, 0.05),
				0,
				CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(selwin->clist, 1, calc_clist_column_width(selwin->clist, 0.95),
				get_clist_column_end(selwin->clist, 0),
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(selwin->clist, 0, "");
	print_clist_column_title(selwin->clist, 1, _("Log Level"));

	set_menu_item_visibility(dialog->prev_button, FALSE);

	populate_clist_with_log_levels(selwin->clist);

	process_modal_dialog(dialog);
	return 0;
}
