/***************************************************************************
                          librarypanel.cpp  -  description
                             -------------------
    begin                : Wed Jan 26 2000
    copyright            : (C) 2000 by John Donoghue
    email                : donoghue@chariot.net.au
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "knap_librarypanel.h"
#include "support_funcs.h"

#include "knap_player.h"
#include "knap_playlist.h"

#include <knap_listview.h>
#include <qlayout.h>
#include <qpopupmenu.h>
#include <qlabel.h>
#include <kapp.h>
#include <qpushbutton.h>
#include <qregexp.h>
#include <klocale.h>
#include <qcursor.h>

#include <stdlib.h>

#include "songinfo.h"

#include <iostream>
using namespace std;

#include <qdir.h>
#include <qfile.h>

#include "mp3.xpm"

#include <math.h>

class LibraryItem : public QListViewItem
{
public:
  LibraryItem(QListView *parent,const char *s1,const char *s2=0,const char *s3=0,
		   const char *s4=0,const char *s5=0,const char *s6=0,const char *s7=0)
    : QListViewItem(parent,s1,s2,s3,s4,s5,s6,s7)
  { }
  virtual QString text(int col) const
  {
    static QString ret; // Copied on return from function so ok, even if ugly.
    if(col==3) {
      ret=convertSize(atol(QListViewItem::text(col)));
      return ret;
    }
    else
      return QListViewItem::text(col);
  }

  virtual QString key(int col,bool asc) const
  {
    static QString ret;
    if(col==3) { // sort the size
        ret.sprintf( "%012ld", atol(QListViewItem::text(col)) );
        return ret;
    }
    else if(col==5)     // time
    {
       int min,sec;
       sscanf(QListViewItem::text(col),"%d:%d",&min,&sec);

       ret.sprintf( "%09ld", (min * 60) + sec);
       return ret;
    }

    else if(col==1) { // bitrate
        ret.sprintf( "%06ld", atol(QListViewItem::text(col)) );
        return ret;
    }
    return QListViewItem::text(col);
  }
};


KNAP_LibraryPanel::KNAP_LibraryPanel(QWidget *parent, const char *name )
//: QWidget(parent,name)
  : QFrame(parent,name)
{
  showsharedpath=false;
  sharedcount=0;
  totalgig = 0.0;

  setFrameStyle( QFrame::Panel | QFrame::Sunken );

  QGridLayout *grid=new QGridLayout(this,3,3,5,5);

  stats=new QLabel(i18n("No files"),this,"stats");
  stats->setFrameStyle(QFrame::Panel|QFrame::Sunken);

  grid->addMultiCellWidget(stats,0,0,0,1);

  QPushButton *rescanbutton=new QPushButton(i18n("Rescan"),this,"rescanbutton");
  rescanbutton->setMaximumSize( rescanbutton->sizeHint() );
  connect(rescanbutton,SIGNAL(clicked()),this,SLOT(rescan()));
  grid->addWidget(rescanbutton,0,2);

  sharedlistview=new KNAP_ListView(this,"libraryview");
  sharedlistview->setFrameStyle(QFrame::WinPanel|QFrame::Sunken);
  grid->addMultiCellWidget(sharedlistview,1,1,0,2);

  sharedlistview->addColumn(i18n("file"),200);
  sharedlistview->setColumnWidthMode(0,QListView::Manual);

  sharedlistview->addColumn(i18n("bitrate"),80);
  sharedlistview->setColumnWidthMode(1,QListView::Manual);
  sharedlistview->addColumn(i18n("frequency"),80);
  sharedlistview->setColumnWidthMode(2,QListView::Manual);
  sharedlistview->addColumn(i18n("size"),80);
  sharedlistview->setColumnWidthMode(3,QListView::Manual);
  sharedlistview->addColumn(i18n("time"),80);
  sharedlistview->setColumnWidthMode(4,QListView::Manual);

  sharedlistview->addColumn(i18n("shared"),80);
  sharedlistview->setColumnWidthMode(5,QListView::Manual);

  sharedlistview->loadWidths();


  // the player
  player = new KNAP_Player(this,"player");
  grid->addWidget(player,2,0);
  grid->setColStretch(0,0);
//  player->setMaximumHeight(50);
  player->setMaximumWidth(200);
//  player->setMaximumSize(player->sizeHint());
  connect(this,SIGNAL( play(const char *) ),player,SLOT( play(const char *) ) );

  // coonect player sigs to here for sending to
  connect(player,SIGNAL( nextSong() ),this,SLOT( nextSong() ) );
  connect(player,SIGNAL( prevSong() ),this,SLOT( prevSong() ) );
  connect(player,SIGNAL( playSong() ),this,SLOT( currSong() ) );

  // playlist
  playlist = new KNAP_PlayList(this,"playlist");
  grid->addMultiCellWidget(playlist,2,2,1,2);
  
  connect(this,SIGNAL( queuesong(const char *) ),playlist,SLOT( addSong(const char *) ) );
  connect(playlist,SIGNAL( play(const char *) ),player,SLOT( play(const char *) ) );
//  playlist->setMaximumHeight(player->sizeHint().height());
//  connect(this,SIGNAL( queuesong(const char *) ),player,SLOT( play(const char *) ) );
  
  grid->setColStretch(1,2);
  grid->setRowStretch(1,2);
  //grid->setRowStretch(2,0);
  
  // create the popup menu for the downloadlist view
  sharedpopupmenu=new QPopupMenu();
  if(sharedpopupmenu) {
    sharedpopupmenu->insertItem(i18n("Queue Song"),this,SLOT(popupQueue())) ;
    sharedpopupmenu->insertItem(i18n("Play Song"),this,SLOT(popupPlay())) ;
  }
  connect(sharedlistview,SIGNAL(rightButtonPressed( QListViewItem *, const QPoint &, int )),
       this,SLOT(popupMenu( QListViewItem *, const QPoint &, int )));

  connect(sharedlistview,SIGNAL( dblClicked(const char *) ),player,SLOT( play(const char *) ) );
}

KNAP_LibraryPanel::~KNAP_LibraryPanel()
{

}

void KNAP_LibraryPanel::popupPlay()
{
  QListViewItem *selecteditem=sharedlistview->currentItem();

  if(selecteditem) 
  {
     QString filename = selecteditem->text(0);
     filename.replace(QRegExp("~"),QDir::homeDirPath());
     emit play(filename.ascii());   //--- FULL PATHNAME was .data()
  }
}
void KNAP_LibraryPanel::popupQueue()
{
  QListViewItem *selecteditem=sharedlistview->currentItem();

  if(selecteditem) 
  {
     QString filename = selecteditem->text(0);
     filename.replace(QRegExp("~"),QDir::homeDirPath());

     emit queuesong(filename.ascii());   //--- FULL PATHNAME

     // TODO remove this
     cerr << "queued song: " << filename << endl;
  }
}
void KNAP_LibraryPanel::clear()
{
  sharedlistview->clear();
  sharedcount=0;
  totalgig = 0.0;
}

void KNAP_LibraryPanel::popupMenu( QListViewItem *i, const QPoint &pos, int col)
{
  /* select the item we are currently over */

  QListViewItem *selecteditem=i; //sharedlistview->itemAt(pos);

  if(sharedpopupmenu && selecteditem) {
     sharedlistview->setSelected(selecteditem,true);
     sharedpopupmenu->exec(QCursor::pos());
  }
}

void KNAP_LibraryPanel::addItem(const QString &name,int bitrate,int freq,long size,unsigned long seconds,bool shared)
{
  QString s_filename;
  char buffer[30];

  if(!showsharedpath) s_filename=getFileName(name);
  else  s_filename=name;

// jade - turned off homedir translation - as server doesn't like being sent ~/
//  s_filename.replace(QRegExp(QDir::homeDirPath()),"~");

  LibraryItem *i;

  // put the item in the listview
  // field 6 is invisible & holds the FULL path name of the song

  QString s_time = QString().setNum(floor((double)seconds / 60.0));
  s_time += ":";
  int sec = seconds % 60;
  if (sec < 10) s_time += "0";
  s_time += QString().setNum(sec);

  i=new LibraryItem(sharedlistview,
        s_filename,
        QString().setNum(bitrate),
        QString().setNum(freq),
        QString().setNum(size),
        s_time,
        shared == true ? i18n("Yes") : i18n("No"),
        name);
  // set the pixmap just to add a bit of 'color'
  if(i) i->setPixmap(0,QPixmap((const char **)mp3_xpm));

  if(shared) sharedcount++;

  totalgig += ((double)size/(1024.0*1024.0*1024.0));

  sprintf(buffer,i18n("%.2f Gig"),totalgig);

  QString statstr;
  statstr.sprintf(i18n(" total: %d songs, %d shared, %s"),
	sharedlistview->childCount(),sharedcount,buffer);

  stats->setText(statstr);

  return;
}


// find the song if it exists in the list and return some info
//
SongInfo *KNAP_LibraryPanel::findSongInfo(const char *fname, bool getall)
{
  static SongInfo asong;

  QListViewItem *it=sharedlistview->firstChild();

  while(it) {
     if(strcmp(fname,it->text(0))==0) {
       // found it
       if(!getall) {
         asong.filename=fname;
         asong.bitrate = QString(it->text(1)).toInt();
         asong.freq = QString(it->text(2)).toInt();
         asong.size = getFileSize(fname);

//         QString s_time = it->text(4);
         int min,sec;
         sscanf(it->text(4),"%d:%d",&min,&sec);

         asong.seconds = (min * 60) + sec;

#ifdef DEBUG_1
         cerr<<"size = "<<asong.size<<" qs: "<<it->text(3)<<endl<<flush;
#endif 
         if(strcmp(it->text(5),i18n("Yes"))==0) asong.shared=true;
         else asong.shared=false;

         return &asong;
       }
       else { // load it in
          SongInfo *nsong = SongInfo::loadSongInfo(fname,getall);
          if(nsong) { 
             if(strcmp(it->text(5),i18n("Yes"))==0) nsong->shared=true;
             else nsong->shared=false;

#ifdef DEBUG_1
             cerr<<"size = "<<asong.size<<" qs: "<<it->text(3)<<endl<<flush;
#endif
          }
          return nsong;
       }
     }
     it=it->nextSibling();
  }
  return NULL;
}
int KNAP_LibraryPanel::addDir(const char *name,bool shared,int depth)
{
  if(!name) return 0;
  SongInfo *si;
  QDir dir;
  int cnt=0;

  dir.setPath(name);
  if(!dir.exists()) return 0;

#ifdef DEBUG_2
        cout<<"enter dir: "<<name<<" depth was "<<depth<<endl;
#endif

  dir.setFilter( QDir::Dirs | QDir::Files | QDir::Readable );  //  QDir::NoSymLinks |
  dir.setNameFilter("*.mp?");
  dir.setMatchAllDirs(true);

  const QFileInfoList  *list = dir.entryInfoList();
  QFileInfoListIterator it( *list );      // create list iterator
  QFileInfo *fi;                          // pointer for traversing

  while ( (fi=it.current()) ) {           // for each file...

    // this may take a while so allow events to continue
    kapp->processEvents(500);

    if(fi->isDir()) {
      // if its not a .. and we want to enter this dir - do it
      if(depth!=1 && fi->fileName()!="." && fi->fileName()!="..") {
        cnt += addDir(fi->absFilePath(),shared,(depth == 0 ? 0 : depth-1));
      }
    }
    else
    //// if(!fi->isSymLink ()) TODO - get real name not symlink
    {
      QString nm = fi->absFilePath();

      si=findSongInfo(nm);
      if(si==NULL) {
// LOADING MD5 as well!!
//cerr<<"loading songinfo: "<<nm<<endl;
        si=SongInfo::loadSongInfo(nm,false);
        if(si) {
          addItem(nm,si->bitrate,si->freq,si->size,si->seconds,shared);
          cnt++;
        }
      }
    }
    ++it;                               // goto next list element
  }

  return cnt;
}

void KNAP_LibraryPanel::rescan()
{
//  emit rescan();
   KNAP_Preferences p;

   clear();

   PathInfoList::Iterator it;

   for( it = p.sharepaths.begin(); it != p.sharepaths.end(); ++it )
   {
      addDir((*it).path,(*it).shared,(*it).depth);
   } 
}

SongInfo *KNAP_LibraryPanel::getSong(int v)
{
   SongInfo *asong;
   int curr=0;
   // invalid index
   if(v<0 || v>=sharedlistview->childCount()) return NULL;

   QListViewItem *it=sharedlistview->firstChild();

   while(it) {
      if(curr==v) {

	 asong = SongInfo::loadSongInfo(it->text(0),true);
	 if(asong) { 
	    if(strcmp(it->text(5),i18n("Yes"))==0) asong->shared=true;
	    else asong->shared=false;
	 }
	 return asong;
      }
      curr++;
      it=it->nextSibling();
   }
   return NULL;
}

SongList KNAP_LibraryPanel::getAllSongs()
{
   SongList slist;
   SongInfo *song;

   QListViewItem *it=sharedlistview->firstChild();

   while(it) 
   {
      song = SongInfo::loadSongInfo(it->text(0),true); 
      if(song) 
      {
	 if(it->text(5).compare("Yes")==0) song->shared = true;
	 else song->shared = false;

	 slist.append(*song);
      }
      it=it->nextSibling();
   }
   return slist;
}

QStringList KNAP_LibraryPanel::getAllSongNames()
{
   QStringList slist;
   QListViewItem *it=sharedlistview->firstChild();

   while(it) {
      slist.append(it->text(0));
      it=it->nextSibling();
   }
   return slist;
}

bool KNAP_LibraryPanel::isSharedSong(const char *name)
{
   if(!name) return false;

   QListViewItem *it=sharedlistview->firstChild();

   while(it) {
      if(strcmp(it->text(0),name)==0)
      {
	 // found it
	 return true;
      }
      it=it->nextSibling();
   }
   return false;
}
/**
 * slot called on pressing the next button on the 
 * player
 */
void KNAP_LibraryPanel::nextSong()
{
   const char * s = playlist->getNextSong();
   if(s) player->play(s);
   else player->stop();
}
/**
 * slot called on pressing the prev button on the 
 * player
 */
void KNAP_LibraryPanel::prevSong()
{
   const char * s = playlist->getPrevSong();
   if(s) player->play(s);
   else player->stop();
}
/**
 * slot called on pressing the play button on the 
 * player when there is no song loaded
 */
void KNAP_LibraryPanel::currSong()
{
   const char * s = playlist->getCurrSong();
   if(s==0) playlist->resetSongPos();
   s = playlist->getCurrSong();

   if(s) player->play(s);
}
