// fftdata.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include "controller.h"
#include "header.h"
#include "fftdata.h"
#include "request.h"

void
FFTData::print(FILE *out) {
	int npoints = nPoints();
	double nyquist = sRate() / 2.0;
	fprintf(out, "FFT Data Dump\n");
	for(int frame=0; frame < length(); frame++) {
		fprintf(out, "Frame %d:\n", frame);
		for(int i = 0; i < npoints; ++i) {
			fprintf(out, "\tFreq:\t%0.2f\tAmp:\t%0.4f\n",
				double(i)/npoints * nyquist, get(frame, i));
		}
	}
	fflush(out);
}

void
FFTData::information(Controller *controller) {
	AlertRequest request("FFT Data Information:");
	request.setBell(false);
	char str[128];
	request.appendLabel("------------");
	request.appendLabel("Filename: ", controller->fileName());
	request.appendLabel("Length (frames): ", toString(str, length()));
	request.appendLabel("Number of Freq. Points: ", toString(str, nPoints()));
	request.appendLabel("Frame Rate: ", toString(str, frameRate()));
	request.appendLabel("Sound Sample Rate: ", toString(str, sRate()));
	request.appendLabel("File Size (bytes): ", toString(str, sizeInBytes()));
	controller->handleRequest(request);
}

Data *
FFTData::newData(int length) {
	Data *d = new FFTData(this, length);
	if (d->length() != length) {
		d->unref();
		d = nil;
	}
	return d;
}

Data *
FFTData::clone(const Range &r) { return new FFTData(this, r); }

Data *
FFTData::clone(const Range &r, const Range &c) {
	return new FFTData(this, r, c);
}

Header *
FFTData::createHeader(DataFile *, boolean reading) {
	Header *hdr = new FFTHeader(
		nPoints(),
		frameRate(),
		sRate(),
		duration()
	);
	if(!reading) configureHeader(hdr);
	return hdr;
}

const char *
FFTData::channelName(int chan) const {
	static char chan_name[32];
	double freq = double(chan)/nPoints() * sRate()/2;
	double pfreq = (freq < 1000.0) ? freq : freq/1000.0;
	sprintf(chan_name, freq < 1000.0 ? "%0.1f hz" : "%0.2fk hz", pfreq);
	return chan_name;
}

Range
FFTData::limits(int chan, boolean real) const {
	double peak = maxValue();
	Range r;
	return r.set(0.0, peak);  // same for all chans
}
