/*
 * renderer-dali.h --
 *
 *      A Dali-object "renderer": it gets a frame, transforms it into 
 *      a set of Dali-like structures, and calls the effect implementation
 *
 * Copyright (c) 1993-2001 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* @(#) $Header: /usr/mash/src/repository/mash/mash-1/fx/renderer-dali.h,v 1.1 2002/02/07 04:17:01 chema Exp $ */


#include <stdlib.h>
#include "renderer.h"
#include "vw.h"
#include "rv.h"

#include "tclcl.h"
#include "module.h"
#include "fx/vidreps.h"
#include "fx/effects/effect-module.h"


class DaliRenderer : public Renderer {
public:
	DaliRenderer(int decimation);
	virtual int command(int argc, const char*const* argv);
	virtual void set_frame_buffer(VidRep *fb) {
		frame_ = (VidRep *) fb;
	};
	virtual void init_frame(VideoFrame* vf) = 0;
	virtual int sameframe(VideoFrame* vf) = 0;

protected:
	virtual void recv(Buffer*);
	virtual void recv(Buffer*, int, int);
	virtual void resize(int w, int h);
	virtual void render(const u_char* frm, int off, int x,
			int w, int h) {};
	virtual void push(const u_char* frm, int miny, int maxy,
			int minx, int maxx) {};
	VidRep *frame_;

	// input_id_ stores to which effect input our frames will be delivered
	//	in multiple-inputs effects
	int input_id_;

	// redefinition of target_ to permit dispatching recv(frame, input_id)
	EffectModule *target_;

	char *effect_;
	char *frame_handler_;

	// this is the color subsampling scheme of the input. The name is 
	//	clearly confusing: decimation is also used to specify a frame 
	//	size of 1 (4CIF), 2 (CIF), or 4 (QCIF). I use it here for 
	//	coherence with the rest of the rendering code.
	int decimation_;  // 422, 411, or 420
};


class UncompressedDaliRenderer : public DaliRenderer {
public:
	UncompressedDaliRenderer(int decimation);
	virtual int command(int argc, const char*const* argv);
	virtual void set_frame_buffer(VidRep *fb) {
		frame_ = (Uncompressed *) fb;
	};
	virtual void init_frame(VideoFrame* vf);
	inline virtual int sameframe(VideoFrame* vf) {
		if ((vf->width_ == frame_->w_) && (vf->height_ == frame_->h_) &&
				(vf->bp_ == frame_->lum_->firstByte)) {
			return 1;
		}
		return 0;
	};

protected:
	virtual void recv(Buffer*);
	virtual void render(const u_char* frm, int off, int x,
			int w, int h) {};
	virtual void push(const u_char* frm, int miny, int maxy,
			int minx, int maxx) {};
	Uncompressed *frame_;
};

