/*
 * rep-decoder-jpeg2semicompressed.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "inet.h"
#include "rtp.h"
#include "rep-decoder.h"
#include "bsd-endian.h"
#include "tclcl.h"
#include "jpeg/jpeg.h"
#include "renderer.h"
#include "pktbuf.h"
#include "vidreps.h"

class JpegReassembler {
public:
	JpegReassembler();
	~JpegReassembler();
	u_char* reassemble(const rtphdr* rh, const u_char* bp, int& len);
	inline int badoff() const { return (badoff_); }
	inline int hugefrm() const { return (hugefrm_); }
protected:
	int decimate_;
	int ndec_;

	/*
	 * Reassembly buffer.  This should be a dma buffer in the
	 * jvdriver case but it's not clear if jvdriver allows buffers
	 * to be shared across sockets. FIXME ask Lance
	 * If it does, then we can pass dma buffers betwee
	 * the decoder and renderers.
	 */
#define JPEG_SLOTS 64
#define JPEG_SLOTMASK (JPEG_SLOTS - 1)
	struct slot {
		int seqno;
		int eof;	/* nbytes in last pkt of frame, o.w. 0 */
		u_int32_t off;
		u_int32_t ts;
	} slots_[JPEG_SLOTS];

	/*
	 * Reassembly buffers.  We do double-buffering, which allows
	 * packets to arrive out of order across frame boundaries,
	 * but not across an entire frame (i.e., we don't want to see
	 * any packets from frame k+2 until we're done with frame k).
	 * We use RTP timestamps to keep track of which frame is
	 * allocated to which buffer.  (RTP guarantees that the
	 * timestamps are constant across a frame, and increase
	 * between succesive frames.  FIXME is latter really true?)
	 */
	struct rbuf {
		int drop;
		u_int32_t ts;
		u_char* bp;
	};
	rbuf rb0_;
	rbuf rb1_;
	int rbsize_;
	int hugefrm_;
	int badoff_;
};

/*
 * Initial size of each reassembly buffer.
 */
#define JPEG_BUFSIZE (16*1024)

class JpegToSemicompressedDecoder : public RepDecoder {

public:
  JpegToSemicompressedDecoder();
  virtual ~JpegToSemicompressedDecoder();
  virtual int command(int argc, const char*const* argv);

protected:
  virtual void post_delay_recv(pktbuf*);
  virtual void set_frame_buffer(VidRep *fb) {
    output_ = (Semicompressed *) fb;
  };
  virtual void resize(int inw, int inh);

  void lum_copy_into_sc_block(short *coeffs, ScBlock *scb);
  void chrom_copy_into_sc_block(short *coeffs, ScBlock *scb);

  int inq_;		/* input quantization */
  int type_;		/* JPEG/RTP parameters type code */
  int decimation_;
  u_char *rvts_;

  void configure();
  JpegDCTDecoder* codec_;
  JpegDecoder::config config_;
  JpegReassembler reasm_;
  Semicompressed *output_;
};

static class JpegToSemicompressedDecoderClass : public TclClass {
public:
  JpegToSemicompressedDecoderClass() : TclClass("Module/VideoDecoder/JPEGToSemicompressed") {}
  TclObject* create(int argc, const char*const* argv) {
    return (new JpegToSemicompressedDecoder());
  }
} swd_jpeg_to_semicompressed_;

JpegToSemicompressedDecoder::JpegToSemicompressedDecoder()
  : RepDecoder(sizeof(jpeghdr)), rvts_(0), codec_(0)
{
  JpegDecoder::defaults(config_);

  inq_ = 0;
  /* guess type 0 */
  type_ = 0;
  decimation_ = 422;
  output_ = 0;
}

JpegToSemicompressedDecoder::~JpegToSemicompressedDecoder()
{
  delete codec_;
}

int
JpegToSemicompressedDecoder::command(int argc, const char*const* argv)
{
  return (RepDecoder::command(argc,argv));
}

void JpegToSemicompressedDecoder::configure()
{
  config_.comp[0].hsf = 2;
  int old_decimation = decimation_;
  if (type_ == 1) {
    decimation_ = 411;
    config_.comp[0].vsf = 2;
  } else {
    decimation_ = 422;
    config_.comp[0].vsf = 1;
  }
  config_.comp[1].hsf = 1;
  config_.comp[1].vsf = 1;
  config_.comp[2].hsf = 1;
  config_.comp[2].vsf = 1;
  config_.width = inw_;
  config_.height = inh_;
  JpegDecoder::quantizer(config_, inq_);

  delete codec_;
  /*
   * Can't use normal JpegDCTDecoder::create because the 422 decoder doesn't
   * do what we want. So we added true_create to definition of JpegDCTDecoder
   * that does what we want and use it instead.
   */

  codec_ = JpegDCTDecoder::true_create(config_, inw_, inh_);

  /*
   * If we have been given name of Semicompressed object
   * to put result in, set up ScImages correctly.
   */
  if (output_ != 0) {
    if (decimation_ == 411) {
      output_->set(inw_, inh_, 2, 2, inw_, inh_, 0, 0);
    } else {
      output_->set(inw_, inh_, 2, 1, inw_, inh_, 0, 0);
    }
  }

  Tcl& tcl = Tcl::instance();
  int q;
  q = JpegDecoder::q_to_thresh(inq_);
  codec_->thresh(q);

#ifdef notyet
  int ct = (toi(tcl.attr("softJPEGcthresh")));
#else
  int ct = 6;
#endif

  codec_->cthresh(ct);

  if (old_decimation != decimation_)
    tcl.evalf("%s parameters_changed", name());
}

void JpegToSemicompressedDecoder::resize(int inw, int inh) {
  rvts_ = new u_char[(inw*inh)/64];
  RepDecoder::resize(inw, inh);
}

void JpegToSemicompressedDecoder::post_delay_recv(pktbuf* pb)
{
  int mb_width, mb_height, mby, mbx, frm_idx, sc_idx, sc_w;

  rtphdr* rh = (rtphdr*)pb->dp;
  const jpeghdr* p = (const jpeghdr*)(rh + 1);
  int needConfig = 0;
  if (p->q != inq_ || p->type != type_) {
    type_ = p->type;
    inq_ = p->q;
    needConfig = 1;
  }
  int inw = p->width << 3;
  int inh = p->height << 3;
  if (inw_ !=  inw || inh_ != inh) {
    resize(inw, inh);
    needConfig = 1;
  }
  if (needConfig)
    configure();

  u_int8_t* bp = (u_int8_t*)(p + 1);
  int cc = pb->len - (sizeof(*rh) + sizeof(*p));
  bp = reasm_.reassemble(rh, bp, cc);
  if (bp != 0) {
    if (output_ != 0) {
      output_->ts_ = ntohl(rh->rh_ts);
      output_->ssrc_ = ntohl(rh->rh_ssrc);
    }

    codec_->decode(bp, cc, rvts_, 0);
    codec_->resetndblk();

    if (output_ != 0) {
      short *frm;

      frm = codec_->frame();

      mb_width = inw_ / 16;
      if (decimation_ == 411) {
	mb_height = inh_ / 16;
      } else {
	mb_height = inh_ / 8;
      }

      for (mby=0; mby<mb_height; mby++) {
	for (mbx=0; mbx<mb_width; mbx++) {
	  if (decimation_ == 411) {
	    frm_idx = ((mby*mb_width)+mbx)*6*64;
	    sc_idx = mby*mb_width*4+mbx*2;
	    sc_w = output_->lum_->width;
	    lum_copy_into_sc_block(frm+frm_idx,
				   &(output_->lum_->firstBlock[sc_idx]));
	    lum_copy_into_sc_block(frm+frm_idx+64,
				   &(output_->lum_->firstBlock[sc_idx+1]));
	    lum_copy_into_sc_block(frm+frm_idx+128,
				   &(output_->lum_->firstBlock[sc_idx+sc_w]));
	    lum_copy_into_sc_block(frm+frm_idx+192,
				   &(output_->lum_->firstBlock[sc_idx+sc_w+1]));

	    sc_idx = mby*mb_width+mbx;
	    chrom_copy_into_sc_block(frm+frm_idx+256,
				     &(output_->cr_->firstBlock[sc_idx]));
	    chrom_copy_into_sc_block(frm+frm_idx+320,
				     &(output_->cb_->firstBlock[sc_idx]));
	  } else {
	    frm_idx = ((mby*mb_width)+mbx)*4*64;
	    sc_idx = mby*mb_width*2+mbx*2;
	    sc_w = output_->lum_->width;
	    lum_copy_into_sc_block(frm+frm_idx,
				   &(output_->lum_->firstBlock[sc_idx]));
	    lum_copy_into_sc_block(frm+frm_idx+64,
				   &(output_->lum_->firstBlock[sc_idx+1]));

	    sc_idx = mby*mb_width+mbx;
	    chrom_copy_into_sc_block(frm+frm_idx+128,
				     &(output_->cr_->firstBlock[sc_idx]));
	    chrom_copy_into_sc_block(frm+frm_idx+192,
				     &(output_->cb_->firstBlock[sc_idx]));
	  }
	}
      }

      if (callback_ != 0) {
	Tcl& tcl = Tcl::instance();
	tcl.eval(callback_);
      }
    }
  }
  pb->release();
}

void
JpegToSemicompressedDecoder::lum_copy_into_sc_block(short *coeffs,
						    ScBlock *scb)
{
  int col, row;
  int num_ac = 0;
  u_int32_t *coeff_int;

  /* Note: the coeffs are stored in column-row order */

  scb->dc = *coeffs++ >> 3;

  if (*coeffs != 0) {
    scb->index[0] = 8;
    scb->value[0] = *coeffs;
    num_ac = 1;
  }
  coeffs++;
  coeff_int = (u_int32_t *) coeffs;
  row = 2;

  for (col = 0; col < 8; col++) {
    while(row < 8) {
      u_int32_t coeff_int_val;

      coeff_int_val = *coeff_int++;

      if (coeff_int_val != 0) {
	int bidx = row*8+col;
	if (coeff_int_val & 0xffff0000) {
	  scb->index[num_ac] = bidx;
	  scb->value[num_ac++] = (coeff_int_val >> 16);
	}
	if (coeff_int_val & 0x0000ffff) {
	  scb->index[num_ac] = bidx+8;
	  scb->value[num_ac++] = (coeff_int_val & 0x0000ffff);
	}
      }
      row+=2;
    }
    row = 0;
  }

  for(int i=0; i<num_ac; i++) {
    for(int j=i+1; j<num_ac; j++) {
      if (scb->index[i] > scb->index[j]) {
	char tmp_idx;
	short tmp_val;
	tmp_idx = scb->index[i];
	tmp_val = scb->value[i];
	scb->index[i] = scb->index[j];
	scb->value[i] = scb->value[j];
	scb->index[j] = tmp_idx;
	scb->value[j] = tmp_val;
      }
    }
  }

  scb->numOfAC = num_ac;
  scb->intracoded = 1;
  scb->skipMB = 0;
  scb->skipBlock = 0;
}

void
JpegToSemicompressedDecoder::chrom_copy_into_sc_block(short *coeffs,
						      ScBlock *scb)
{
  int col, row;
  int num_ac = 0;
  u_int32_t *coeff_int;

  /* Note: the coeffs are stored in column-row order */

  /* Rebias chrom dc component by subtracting 128 */

  scb->dc = (*coeffs++ >> 3) - 128;

  if (*coeffs != 0) {
    scb->index[0] = 8;
    scb->value[0] = *coeffs;
    num_ac = 1;
  }
  coeffs++;
  coeff_int = (u_int32_t *) coeffs;
  row = 2;

  for (col = 0; col < 8; col++) {
    while(row < 8) {
      u_int32_t coeff_int_val;

      coeff_int_val = *coeff_int++;

      if (coeff_int_val != 0) {
	int bidx = row*8+col;
	if (coeff_int_val & 0xffff0000) {
	  scb->index[num_ac] = bidx;
	  scb->value[num_ac++] = (coeff_int_val >> 16);
	}
	if (coeff_int_val & 0x0000ffff) {
	  scb->index[num_ac] = bidx+8;
	  scb->value[num_ac++] = (coeff_int_val & 0x0000ffff);
	}
      }
      row+=2;
    }
    row = 0;
  }

  for(int i=0; i<num_ac; i++) {
    for(int j=i+1; j<num_ac; j++) {
      if (scb->index[i] > scb->index[j]) {
	char tmp_idx;
	short tmp_val;
	tmp_idx = scb->index[i];
	tmp_val = scb->value[i];
	scb->index[i] = scb->index[j];
	scb->value[i] = scb->value[j];
	scb->index[j] = tmp_idx;
	scb->value[j] = tmp_val;
      }
    }
  }

  scb->numOfAC = num_ac;
  scb->intracoded = 1;
  scb->skipMB = 0;
  scb->skipBlock = 0;
}

