# recv-cues.tcl --
#
#       An object for receiving cues in a collaborative meeting.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#
# An object for receiving cues in a collaborative meeting.
#
# Similar to CuesSender, this class makes a gui for the
# cues, which is just a list of images of the cues. It
# registers and listens for cues messages on the global
# coordination bus. The client of this object needs to
# pass in the bus.
#
# See also: CuesSender
#
# Examples of usage can be found in vat.
#

Class CuesReceiver

#
# Initialize the global coordination bus. All instances of
# CuesReceiver (for a single application) share the same
# bus. The object to use should be CoordBus_ and the associated
# CBChannel/Site object. There are examples of them in
# application-{vic,vat}.tcl.
#
CuesReceiver proc set_cb { cb } {
        # initialize global coordination bus
	CuesReceiver set glob_chan_ $cb

	# register for global coordination bus events
        $cb register AWARE_ear "$self recv"
        $cb register AWARE_hand "$self recv"
	$cb register AWARE_yes "$self recv"
	$cb register AWARE_no "$self recv"
        $cb register UNAWARE_ear "$self recv"
        $cb register UNAWARE_hand "$self recv"
	$cb register UNAWARE_yes "$self recv"
	$cb register UNAWARE_no "$self recv"
}

#
# Called when received an event from the bus.
# It dispatches the event to the right function. This
# is either telling the gui to start or stop blinking
# the specified cue.
#
CuesReceiver proc recv { list msg } {
	array set info $list

        set event [split $info(event) "_"]
        set cue [lindex $event 1]
	switch -- [lindex $event 0] {
		AWARE { set fn start_blink }
		UNAWARE { set fn stop_blink }
	}

        # dispatch the event to all instances of
        # receiver
        CuesReceiver instvar all_
        if [info exists all_] {
		foreach cr $all_ {
			$cr $fn $cue $msg
		}
	}
}

#
# Create a CuesReceiver object and the gui
# path: the gui is construct using this path, the
#       client should pass in the name of a frame
# size: either lg (large, for something like switcher windows) or
#              sm (small, for something like vic thumbnails)
#
CuesReceiver instproc init { path size } {
	# initialize some variables
	# cname_: the current source displayed and associated to
	#          the cues
	$self instvar cname_ size_ cues_
	set cname_ 0
	set size_ $size
	# FIXME change so client can specify
	set cues_ "hand ear yes no"

        # add this new instance to a list
	CuesReceiver instvar all_
	lappend all_ $self

        # create the cues images
	$self instvar top_ colors_
        set top_ $path
	foreach c $cues_ {
		$self set ${c}_l [label ${top_}.${c} \
				  -bitmap ${size_}_${c}]
		pack ${top_}.${c} -side left -padx 10
	}

	# blink colors
	# FIXME change so client can specify
        set colors_(1) red
        set colors_(0) gray
}

CuesReceiver instproc destroy { } {
	# remove from group
	CuesReceiver instvar all_
        set i [lsearch $all_ $self]
        set all_ [lreplace $all_ $i $i]

	# cancel all scheduled events
	$self instvar cues_ cnames_
	foreach cue $cues_ {
		foreach cname $cnames_ {
			$self stop_blink $cue $cname
		}
	}

	$self next
}

#
# This method is used only in one special circumstance:
# for a window that share multiple videos for different
# participants. We want the cues to blink only if they
# are from the person currently showing in the window.
# The client call enable whenever the video changes.
# cname: the cname of the source currently displayed on
# the window
#
CuesReceiver instproc enable { cname } {
	$self set cname_ $cname
	$self refresh
}

CuesReceiver instproc refresh { } {
	$self instvar top_ cues_ colors_
	foreach c $cues_ {
		${top_}.${c} conf -background $colors_(0)
	}
}

#
# Start blinking the cue image for the person with
# cname
#
CuesReceiver instproc start_blink { cue cname } {
	$self instvar ${cname}_info_ accum_ cnames_

        # no need to blink if this cname's cue is already blinking
	if [info exists ${cname}_info_($cue)] {
 	        return
        }

        set ${cname}_info_($cue) 1
	set ${cname}_info_(${cue}_accum) 0
	set ${cname}_info_(${cue}_after) 0
	set ${cname}_info_(${cue}_color) 0

	if { ![info exists cnames_] || [lsearch cnames_ $cname] == -1 } {
		lappend cnames_ $cname
        }

        $self blinklite 100 $cue $cname
}

#
# Stop blinking the cue image for the person with
# cname
#
CuesReceiver instproc stop_blink { cue cname } {
        $self instvar top_ colors_ ${cname}_info_

        if { [info exists ${cname}_info_($cue)] } {
	        unset ${cname}_info_($cue)
		set w ${top_}.${cue}
		if [winfo exists $w] {
			$w conf -background $colors_(0)
		}
	}
	if { [info exists ${cname}_info_(${cue}_after)] } {
		after cancel [set ${cname}_info_(${cue}_after)]
	}
}

#
# Recursive function to continously blink the cue
# image until N seconds (FIXME now set to 30, probably
# should let the client decide)
#
CuesReceiver instproc blinklite { interval cue cname } {
        $self instvar ${cname}_info_ colors_ cname_ top_

        # if stop_blink has been called, don't continue
        if ![info exists ${cname}_info_($cue)] {
		return
	}

	# if it has been blinking for more than 30 seconds, stop
	if { [set ${cname}_info_(${cue}_accum)] >= 30000 } {
	        $self stop_blink $cue $cname
	        return
	}

        set ${cname}_info_(${cue}_color) \
		[expr {([set ${cname}_info_(${cue}_color)] + 1) % 2}]

	set ${cname}_info_(${cue}_accum) \
		[expr [set ${cname}_info_(${cue}_accum)] + $interval]

	# only blink if it is enabled
        if { "$cname_" == "$cname" } {
		${top_}.${cue} conf \
			-background $colors_([set ${cname}_info_(${cue}_color)])
	}

        set ${cname}_info_(${cue}_after) \
		[after $interval \
		 "$self blinklite [expr int($interval * 1.05)] $cue $cname"]
}

