# ui-Real.tcl --
#
#       This is the code that creates the windows used to add/configure a new
#       Real Transcoder.
#
# This file has the main ui component of the tgw application. Also, it contains
# many of the slightly modified superclasses to make things work... This is the
# main file with all the code in it.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import ServerProfileWindow ServerProfileSave StreamProfileSave StreamProfileWindow AudioAgentTGW TopLevelWindow OutgoingSession

# this class is not currently used, but I didn't want to delete it...
Class NewTGWtranscoderScreen -superclass TopLevelWindow -configuration { }

NewTGWtranscoderScreen instproc init { mainUI } {
    $self next .newOutput
    $self instvar ui_ 
    set ui_ $mainUI
}

NewTGWtranscoderScreen instproc build w {
    $self instvar ui_
    $self create-window $w "New Output"
    wm withdraw $w

    set x [winfo rootx .]
    set y [winfo rooty .]
    incr y [winfo height .]
    incr y -[winfo reqheight $w]
    incr y -20
    # adjust for virtual desktops
    incr x [winfo vrootx .]
    incr y [winfo vrooty .]
    if { $y < 0 } { set y 0 }
    if { $x < 0 } {
	set x 0
    } else {
	set right [expr [winfo screenwidth .] - \
		[winfo reqwidth $w]]
	if { $x > $right } {
	    set x $right
	}
    }
    wm geometry $w +$x+$y
    
    button $w.newReal -text "New Real Transcoder" -relief raised \
	    -font [$self get_option smallfont] -highlightthickness 1 \
	    -command "$ui_ showNewRealScreen"
    button $w.newH261 -text "New Bandwidth Limiter" -relief raised \
	    -font [$self get_option smallfont] -highlightthickness 1 \
	    -command "$ui_ showNewH261screen"
    button $w.newRedirector -text "New Redirector" -relief raised \
	    -font [$self get_option smallfont] -highlightthickness 1 \
	    -command "$ui_ showNewRedirector"
    button $w.cancel -text "Cancel" -relief raised \
	    -font [$self get_option smallfont] -highlightthickness 1 \
	    -command "wm withdraw $w"
    pack $w.newReal $w.newH261 $w.newRedirector $w.cancel -padx 1 -pady 1
}

# asks for the server,port, username and password of the Real
# server you are going to use.
Class RealServerUI -superclass TopLevelWindow -configuration { }

RealServerUI instproc init { mainUI } {
    $self next .realServer
    $self instvar ui_ loadScreen_ saveServer_
    set ui_ $mainUI
    set loadScreen_ [new ServerProfileWindow $self]
    set saveServer_ [new ServerProfileSave]
}

RealServerUI instproc setServerInfo { sName sPort uName uPass } {
    global inputtedSName inputtedSPort inputtedUName inputtedPass
    set inputtedSName $sName
    set inputtedSPort $sPort
    set inputtedUName $uName
    set inputtedPass $uPass
}

RealServerUI instproc build w {
    $self instvar ui_ loadScreen_ saveServer_
    $self create-window $w "RealServer Info"
    wm withdraw $w
    global inputtedSName inputtedSPort inputtedUName inputtedPass

    frame $w.top
    label $w.top.sname -anchor w -text "Server Name:"
    entry $w.top.snameEntry -textvariable inputtedSName -relief sunken
    grid $w.top.sname $w.top.snameEntry -stick news
    label $w.top.sport -anchor w -text "Server Port:"
    entry $w.top.sportEntry -textvariable inputtedSPort -relief sunken
    grid $w.top.sport $w.top.sportEntry -stick news
    label $w.top.uname -anchor w -text "User Name:"
    entry $w.top.unameEntry -textvariable inputtedUName -relief sunken
    grid $w.top.uname $w.top.unameEntry -stick news
    label $w.top.pass -anchor w -text "Password:"
    entry $w.top.passEntry -textvariable inputtedPass -relief sunken -show *
    grid $w.top.pass $w.top.passEntry -stick news
#    label $w.warning -anchor w -text "Leave blank if not broadcasting live"
    frame $w.bottom
    button $w.bottom.load -text "Load" -relief raised \
	    -font [$self get_option smallfont] -highlightthickness 1 \
	    -command "$loadScreen_ toggle"
    button $w.bottom.save -text "Save" -relief raised \
	    -font [$self get_option smallfont] -highlightthickness 1 \
	    -command "$saveServer_ toggle"
    button $w.bottom.ok -text "OK" -relief raised \
	    -font [$self get_option smallfont] -highlightthickness 1 \
	    -command "$self callNewRealSession"
    button $w.bottom.cancel -text "Cancel" -relief raised \
	    -font [$self get_option smallfont] -highlightthickness 1 \
	    -command "$self toggle"
    grid $w.bottom.load $w.bottom.save $w.bottom.ok $w.bottom.cancel -stick news
#    pack $w.top $w.warning $w.bottom
    pack $w.top $w.bottom
}

RealServerUI instproc toggle { {comb {}} } {
    global inputtedSName inputtedSPort inputtedUName inputtedPass
    $self instvar combine_

    set inputtedSName ""
    set inputtedSPort ""
    set inputtedUName ""
    set inputtedPass ""
    set combine_ $comb

    $self next
}

RealServerUI instproc callNewRealSession { } {
    $self instvar ui_ combine_
    global inputtedSName inputtedSPort inputtedUName inputtedPass
    if { $combine_ == "combine"} {
	$ui_ newCombiner $inputtedSName $inputtedSPort $inputtedUName $inputtedPass
    } else {
	$ui_ newRealSession $inputtedSName $inputtedSPort $inputtedUName $inputtedPass
    }
    $self toggle
}

# this is the window used to configure different specifications
# when transcoding a stream to RealMedia...
Class RealConfigUI -superclass TopLevelWindow -configuration { }

RealConfigUI instproc init { mainUI } {
    $self next .realConfig
    $self instvar ui_ saveStream_ loadStream_
    set ui_ $mainUI
    set saveStream_ [new StreamProfileSave]
    set loadStream_ [new StreamProfileWindow]
}

RealConfigUI instproc build w {
    global use56K use128K use256K use512K other otherInputted useSS outputFPS title author copyright sendtoRealServer streamName saveAsFile fileName sendAudio audioAddr
    $self instvar saveStream_ loadStream_
    
    $self create-window $w "Config RealStream"
    wm withdraw $w
    frame $w.datarate
    label $w.datarate.label -anchor w -text "DataRate:"
    checkbutton $w.datarate.56K -text "56K" -variable use56K
    checkbutton $w.datarate.512K -text "512K" -variable use512K
    grid $w.datarate.label $w.datarate.56K $w.datarate.512K -sticky news
    checkbutton $w.datarate.128K -text "128K" -variable use128K
    checkbutton $w.datarate.other -text "Other:" -variable other -state disabled
    entry $w.datarate.otherEntry -textvariable otherInputted -relief sunken -width 4
    grid $w.datarate.128K -col 1 -row 1
    grid $w.datarate.other -col 2 -row 1
    grid $w.datarate.otherEntry -col 3 -row 1
    checkbutton $w.datarate.256K -text "256K" -variable use256K
    grid $w.datarate.256K -sticky news -col 1
    frame $w.streamsettings
    checkbutton $w.streamsettings.useSS -text "Use Sure Stream" -variable useSS
    grid $w.streamsettings.useSS -sticky w
    label $w.streamsettings.fps -anchor w -text "Frames Per Sec:"
    entry $w.streamsettings.fpsEntry -textvariable outputFPS -relief sunken -width 2
    grid $w.streamsettings.fps $w.streamsettings.fpsEntry -sticky w
    frame $w.authorsettings
    label $w.authorsettings.titLabel -anchor w -text "Title:"
    entry $w.authorsettings.title -textvariable title -relief sunken
    grid $w.authorsettings.titLabel $w.authorsettings.title -sticky w
    label $w.authorsettings.autLabel -anchor w -text "Author:"
    entry $w.authorsettings.author -textvariable author -relief sunken
    grid $w.authorsettings.autLabel $w.authorsettings.author -sticky w
    label $w.authorsettings.copyLabel -anchor w -text "Copyright:"
    entry $w.authorsettings.copyright -textvariable copyright -relief sunken
    grid $w.authorsettings.copyLabel $w.authorsettings.copyright -sticky w
    frame $w.sendto
    checkbutton $w.sendto.server -text "Send to RealServer" -variable sendtoRealServer
    grid $w.sendto.server -sticky w
    label $w.sendto.servlabel -anchor c -text "Streamname:"
    entry $w.sendto.streamName -textvariable streamName -relief sunken
    grid $w.sendto.servlabel $w.sendto.streamName
    checkbutton $w.sendto.file -text "Save as file" -variable saveAsFile
    grid $w.sendto.file -sticky w
    label $w.sendto.fileLabel -anchor c -text "Filename:"
    entry $w.sendto.fileName -textvariable fileName -relief sunken
    button $w.sendto.browseButt -text "Browse" -relief raised \
	    -font [$self get_option smallfont] -highlightthickness 1 \
	    -command { set fileName [tk_getSaveFile -defaultextension .rm -filetypes {{{RealMedia File} {.rm}}} -title "Save As..." ] }
    grid $w.sendto.fileLabel $w.sendto.fileName $w.sendto.browseButt
    checkbutton $w.sendto.audio -text "Send Audio" -variable sendAudio
    grid $w.sendto.audio -sticky w
    label $w.sendto.audioLabel -anchor c -text "Audio Source:"
    entry $w.sendto.audioAddr -textvariable audioAddr -relief sunken
    grid $w.sendto.audioLabel $w.sendto.audioAddr
    frame $w.buttons
    button $w.buttons.load -text "Load" -relief raised \
	    -font [$self get_option smallfont] -highlightthickness 1 \
	    -command "$loadStream_ toggle"
    button $w.buttons.save -text "Save" -relief raised \
	    -font [$self get_option smallfont] -highlightthickness 1 \
	    -command "$saveStream_ toggle"
    button $w.buttons.ok -text "OK" -relief raised \
	    -font [$self get_option smallfont] -highlightthickness 1 \
	    -command "$self okPressed"
    button $w.buttons.cancel -text "Cancel" -relief raised \
	    -font [$self get_option smallfont] -highlightthickness 1 \
	    -command "$self toggle"
    pack $w.buttons.load $w.buttons.save $w.buttons.ok $w.buttons.cancel -side left
    pack $w.datarate $w.streamsettings $w.authorsettings $w.sendto $w.buttons -anchor w
}

RealConfigUI instproc okPressed { } {
    $self instvar currentSrc_
    global use56K use128K use256K use512K other otherInputted useSS outputFPS title author copyright sendtoRealServer streamName saveAsFile fileName sendAudio audioAddr

    $currentSrc_ setOptions $use56K $use128K $use256K $use512K $other $otherInputted $useSS $outputFPS $title $author $copyright $sendtoRealServer $streamName $saveAsFile $fileName $sendAudio $audioAddr
    $self toggle
}

RealConfigUI instproc toggle { { src 0 } } {
    $self instvar currentSrc_
    if { $src != 0 } { 
	set currentSrc_ $src
	$currentSrc_ getOptions
    }
    $self next
}

# the default outgoing session (parent class for all specific
# outgoing sessions for each kind of transcoder).
Class OutgoingSession

OutgoingSession instproc setFormat { type } {
    $self instvar format_
    set format_ $type
}

OutgoingSession instproc queryFormat { } {
    $self instvar format_
    return $format_
}

# the outgoingSession when using a Real transcoder.
Class OutgoingReal -superclass OutgoingSession

# when the cell is created, you are passed this info from the main tgw-ui class.
OutgoingReal instproc init { sname sport uname pass } {
    $self instvar server_ port_ user_ pass_
    $self setFormat "Real"
    set server_ $sname
    set port_ $sport
    set user_ $uname
    set pass_ $pass
}

# called by updateUI
OutgoingReal instproc makeUI { w ui src rownum} {
    $self instvar server_ port_ user_ pass_
    new RealMatrixUI $server_ $port_ $user_ $pass_ $w $ui $src
}

# the actually cell in the matrix.
Class RealMatrixUI

RealMatrixUI instproc init { sname sport uname pass window ui src } {
    $self instvar server_ port_ user_ pass_ w ui_ src_
    set server_ $sname
    set port_ $sport
    set user_ $uname
    set pass_ $pass
    set w $window
    set ui_ $ui
    set src_ $src
    
    frame $w.buttons
    button $w.buttons.start -text Start -relief raised -font [$self get_option smallfont] -highlightthickness 1 -command "$self startRealTranscoding" -state disabled -width 10
    button $w.buttons.config -text Config -relief raised -font [$self get_option smallfont] -highlightthickness 1 -command "[$ui_ set configReal_] toggle $self" -width 10
    pack $w.buttons.start $w.buttons.config -side left
    grid $w.buttons -row 4
#    pack $w -side left
}

# gets the specifications of what to display in the cell.
RealMatrixUI instproc getOptions { } {
    global use56K use128K use256K use512K other otherInputted useSS outputFPS title author copyright sendtoRealServer streamName saveAsFile fileName sendAudio audioAddr
    $self instvar datarate_ fps_ title_ author_ copyright_ sendToServer_ streamName_ saveAsFile_ fileName_ sendAudio_ audioAddr_ w

    if { [info exists fps_] } {
	set outputFPS $fps_
    } else {
	set outputFPS ""
    }
    if { [info exists title_] } {
	set title $title_	
    } else {
	set title ""
    }
    if { [info exists author_] } {
	set author $author_
    } else {
	set author ""
    }
    if { [info exists copyright_] } {
	set copyright $copyright_
    } else {
	set copyright ""
    }
    if { [info exists sendToServer_] } {
	set sendtoRealServer $sendToServer_
    } else {
	set sendtoRealServer 0
    }
    if { [info exists streamName_] } {
	set streamName $streamName_
    } else {
	set streamName ""
    }
    if { [info exists saveAsFile_] } {
	set saveAsFile $saveAsFile_
    } else {
	set saveAsFile 0
    }
    if { [info exists fileName_] } {
	set fileName $fileName_
    } else {
	set fileName ""
    }
    if { [info exists sendAudio_] } {
	set sendAudio $sendAudio_
    } else {
	set sendAudio 0
    }
    if { [info exists audioAddr_] } {
	set audioAddr $audioAddr_
    } else {
	set audioAddr ""
    }
    if { ![info exists datarate_] } {
	set datarate_ ""
    }

    set useSS 0
    set use56K 0
    set use128K 0
    set use256K 0
    set use512K 0
    set tempDataRate $datarate_
    while { [string length $tempDataRate] > 0 } {
	if { [string compare "SS" [string range $tempDataRate 0 1]] == 0 } {
	    set useSS 1
	    set tempDataRate [string range $tempDataRate 2 [string length $tempDataRate]]
	} elseif { [string compare "56," [string range $tempDataRate 0 2]] == 0 } {
	    set use56K 1
	    set tempDataRate [string range $tempDataRate 3 [string length $tempDataRate]]
	} elseif { [string compare "128," [string range $tempDataRate 0 3]] == 0 } {
	    set use128K 1
	    set tempDataRate [string range $tempDataRate 4 [string length $tempDataRate]]
	} elseif { [string compare "256," [string range $tempDataRate 0 3]] == 0 } {
	    set use256K 1
	    set tempDataRate [string range $tempDataRate 4 [string length $tempDataRate]]
	} elseif { [string compare "512," [string range $tempDataRate 0 3]] == 0 } {
	    set use512K 1
	    set tempDataRate [string range $tempDataRate 4 [string length $tempDataRate]]
	} else {
	    set other 1
	    set tempDataRate [string trimright $tempDataRate ","]
	    set otherInputted $tempDataRate
	    set tempDataRate ""
	}
    }
}

# sets the specifications for what to display in the cell
# called when the RealConfigUI window is closed.
RealMatrixUI instproc setOptions { use56K use128K use256K use512K other otherInputted useSS outputFPS title author copyright sendtoRealServer streamName saveAsFile fileName sendAudio audioAddr } {
    $self instvar datarate_ fps_ title_ author_ copyright_ sendToServer_ streamName_ saveAsFile_ fileName_ sendAudio_ audioAddr_ w
    set fps_ $outputFPS
    set title_ $title
    set author_ $author
    set copyright_ $copyright
    set sendToServer_ $sendtoRealServer
    set streamName_ $streamName
    set saveAsFile_ $saveAsFile
    set fileName_ $fileName
    set sendAudio_ $sendAudio
    set audioAddr_ $audioAddr
    if { $useSS == 1 } {
	set datarate_ "SS"
    } else {
	set datarate_ ""
    }
    if { $use56K == 1 } {
	append datarate_ "56,"
    }
    if { $use128K == 1 } {
	append datarate_ "128,"
    }
    if { $use256K == 1 } {
	append datarate_ "256,"
    }
    if { $use512K == 1 } {
	append datarate_ "512,"
    }
    if { $other == 1 } {
	append datarate_ "$otherInputted"
	append datarate_ ","
    }

    if { [llength [info commands $w.sname]] != 0 } {
	destroy $w.sname
    }
    if { $sendToServer_ == 1 } {
	set temp "Stream: $streamName_"
	label $w.sname -anchor w -text $temp -font [$self get_option smallfont] -pady 0 -borderwidth 0
	grid $w.sname -row 0
    }
    if { [llength [info commands $w.fname]] != 0 } {
	destroy $w.fname
    }
    if { $saveAsFile_ == 1 } {
	set temp2 "File: $fileName_"
	label $w.fname -anchor w -text $temp2 -font [$self get_option smallfont] -pady 0 -borderwidth 0
	grid $w.fname -row 1
    }
    if { [llength [info commands $w.specs]] != 0 } {
	destroy $w.specs
    }
    set temp3 "Spec: $fps_"
    set printedDatarate [string trimright $datarate_ ","]
    append temp3 "fps, $printedDatarate"
    label $w.specs -anchor w -text $temp3 -font [$self get_option smallfont] -pady 0 -borderwidth 0
    grid $w.specs -row 2
    if { [llength [info commands $w.audio]] != 0 } {
	destroy $w.audio
    }
    if { $sendAudio_ == 1 } {
	set temp4 "Audio: $audioAddr_"
	label $w.audio -anchor w -text $temp4 -font [$self get_option smallfont] -pady 0 -borderwidth 0
	grid $w.audio -row 3
    }    
    $w.buttons.start configure -state normal
}

# called when start is clicked.
RealMatrixUI instproc startRealTranscoding { } {
    $self instvar datarate_ fps_ title_ author_ copyright_ sendToServer_ streamName_ saveAsFile_ fileName_ sendAudio_ audioAddr_ ui_ src_ server_ port_ user_ pass_ w rend_ audio_

    set outputTypes 0
    if { $sendToServer_ == 1 } {
	if { $saveAsFile_ == 1 } {
	    set outputTypes 1
	}    
    } else {
	set outputTypes 2
    }

    if { $sendAudio_ == 1 } {
	set avOptions "A/V"
    } else {
	set avOptions "V"
    }

    set rend_ [new RealWindow $server_ $streamName_ $user_ $pass_ $port_ $fps_ $datarate_ $outputTypes $fileName_ $title_ $author_ $copyright_ $avOptions]
    if { $sendAudio_ == 1 } {
	set audio_ [new AudioAgentTGW TGW $audioAddr_ $rend_]
    }
    set dec [$src_ data-handler]
    $rend_ setColorModel [$dec decimation]
    $dec attach $rend_
    $w.buttons.start configure -text Stop -command "$self stopRealTranscoding"
    $w.buttons.config configure -state disabled
}

# called when stop is clicked.
RealMatrixUI instproc stopRealTranscoding { } {
    $self instvar rend_ ui_ src_ sendAudio_ audio_ w
    [$src_ data-handler] detach $rend_
    if { $sendAudio_ == 1 } {
	$audio_ destroy
    }
    $w.buttons.start configure -state disabled
    delete $rend_    
}

# =======================================

Class OutgoingCombiner -superclass OutgoingSession

OutgoingCombiner instproc init { comb sname sport uname pass } {
    $self instvar realEnc_ server_ port_ user_ pass_
    $self setFormat "Real"
    set realEnc_ $comb
    set server_ $sname
    set port_ $sport
    set user_ $uname
    set pass_ $pass
}

OutgoingCombiner instproc makeUI { w ui src rownum } {
    $self instvar realEnc_ server_ port_ user_ pass_ w_
    set w_ $w
    set dec [$src data-handler]
    set combiner_link [new CombinerSource $realEnc_ $src [$dec decimation]]
    
    #if this is the first row then display a little gui to set up the real transcoder
    if { $rownum == 1 } {
	new RealMatrixUIComb $realEnc_ $server_ $port_ $user_ $pass_ $w $ui $src
    }

    frame $w.sideselector
    button $w.sideselector.left -text L -relief raised -font [$self get_option smallfont] -highlightthickness 1 -command "$self sideSelectButtonClick 1 $src $w" -width 3
    button $w.sideselector.right -text R -relief raised -font [$self get_option smallfont] -highlightthickness 1 -command "$self sideSelectButtonClick 2 $src $w" -width 3
    pack $w.sideselector.left $w.sideselector.right -side left
    grid $w.sideselector -row 5

    [$src data-handler] attach $combiner_link
}

OutgoingCombiner instproc sideSelectButtonClick { pos src w } {
    $self instvar realEnc_
    $realEnc_ setSide $pos $src
    if { $pos == 1 } {
	$w.sideselector.right configure -state disabled
    } else {
	$w.sideselector.left configure -state disabled
    }
}

Class RealMatrixUIComb -superclass RealMatrixUI

RealMatrixUIComb instproc init { comb sname sport uname pass window ui src } {
    $self instvar combiner_
    set combiner_ $comb
    $self next $sname $sport $uname $pass $window $ui $src
}

# called when start is clicked.
RealMatrixUIComb instproc startRealTranscoding { } {
    $self instvar datarate_ fps_ title_ author_ copyright_ sendToServer_ streamName_ saveAsFile_ fileName_ sendAudio_ audioAddr_ ui_ src_ server_ port_ user_ pass_ w rend_ audio_ combiner_

    set outputTypes 0
    if { $sendToServer_ == 1 } {
	if { $saveAsFile_ == 1 } {
	    set outputTypes 1
	}    
    } else {
	set outputTypes 2
    }

    if { $sendAudio_ == 1 } {
	set avOptions "A/V"
    } else {
	set avOptions "V"
    }

    if { [$combiner_ ready] == "yes" } {
	set rend_ [new RealWindow $server_ $streamName_ $user_ $pass_ $port_ $fps_ $datarate_ $outputTypes $fileName_ $title_ $author_ $copyright_ $avOptions]
	if { $sendAudio_ == 1 } {
	    set audio_ [new AudioAgentTGW TGW $audioAddr_ $rend_]
	}
	$rend_ setColorModel 411
    
	$combiner_ linkEncoder $rend_
	$w.buttons.start configure -text Stop -command "$self stopRealTranscoding"
	$w.buttons.config configure -state disabled
    } else {
	puts "You need to select the location of the two sources first"
    }
}

# called when stop is clicked.
RealMatrixUIComb instproc stopRealTranscoding { } {
    $self instvar rend_ ui_ src_ sendAudio_ audio_ w combiner_
    $combiner_ unlinkEncoder
    if { $sendAudio_ == 1 } {
	$audio_ destroy
    }
    $w.buttons.start configure -state disabled
    delete $rend_    
}
