# client-405.tcl --
#
#       Provides abstract access to some amxd commands; user doesn't need to
#       know where the amxd is.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

Import enable

import DpClient

Class 405Client


#-----------------------------------------------------------------------------
#
# 405Client constructor
#
# Input:
#   amxHost, amxPort - the host/port of amxd server.  Must NOT be "".
#
# Description:
#   Creates DP clients to talk to the servers.
#   FIXME: hostnames and ports for vcc3 server should be parameters and
#   not hardcoded.
#
#-----------------------------------------------------------------------------

405Client instproc init { amxHost amxPort } {
    $self instvar amx_ cams_

    # AMXd
    if {[catch {new DpClient $amxHost $amxPort} amx_]} {
        puts stderr "Error:  could not connext to AMX server"
	set amx_ ""
    }

    # VCC3d
    set hostname htsr2.bmrc.berkeley.edu
    set port 6905
    if {[catch {new DpClient $hostname $port} cams_(speaker)]} {
        puts stderr "Error:  could not connext to Speaker Camera server, will use AMX control"
	unset cams_(speaker)
    }

    # VCC3d
    set hostname htsr2.bmrc.berkeley.edu
    set port 6906
    if {[catch {new DpClient $hostname $port} cams_(audience)]} {
        puts stderr "Error:  could not connext to Audience Camera server, will use AMX control"
	unset cams_(audience)
    }
}


#-----------------------------------------------------------------------------
#
# 405Client isConnectedToAMX
#
# Description:
#   If connection to amx_ is successful, return 1, else return 0.
#
#-----------------------------------------------------------------------------

405Client public isConnectedToAMX {} {
    $self instvar amx_ 
    if {$amx_ != ""} {
	return 1
    } else {
	return 0
    }
}

# this should only be called when the 405Client object will no longer be used
405Client public closeConnection {} {
    $self instvar amx_ cams_

    catch {$amx_ closeRPC}
    catch {$cams_(speaker) closeRPC}
    catch {$cams_(audience) closeRPC}
}

# FIXME - there should be a better way to do this, possibly pass in a
#    do/doNoWait option to the functions which defaults to "do"
# these are provided to allow the user to call doNoWait explicitly
405Client public getAmxClient {} {
    $self instvar amx_

    return $amx_
}

405Client public getVcc3Client {id} {
    $self instvar cams_

    if {[info exists cams_($id)]} {
	return $cams_($id)
    }
    return -code error "405Client::getVcc3Client: id $id not supported"
}

# FIXME - all these catch statements should print a warning when they fail

405Client public matrix_switchVideoStream {input output {projectorScanConverted 0}} {
    $self instvar amx_

    catch {$amx_ do matrix_switchVideoStream $input $output $projectorScanConverted} result
    return $result
}

405Client public matrix_getInputSource {output} {
    $self instvar amx_

    catch {$amx_ do matrix_getInputSource $output} result
    return $result
}

405Client public camera_pulseMove {id dir {numTimes 1}} {
    $self instvar amx_ cams_

    if {[info exists cams_($id)]} {
	switch -exact -- $dir {
	    in {
		set temp [$cams_($id) do vcc3_getZoomPosition]
		set temp [expr $temp + 20]
		return [$cams_($id) do vcc3_setZoomPosition $temp]
	    }
	    out {
		set temp [$cams_($id) do vcc3_getZoomPosition]
		set temp [expr $temp - 20]
		return [$cams_($id) do vcc3_setZoomPosition $temp]
	    }
	    default {
		return [$cams_($id) do vcc3_moveRelative $dir 100]
	    }
	}
    }
    # if we got here, the vcc3d is not working, or the id is invalid
    catch {$amx_ do camera_pulseMove $id $dir $numTimes} result
    return $result
}

405Client public camera_center {id} {
    $self instvar amx_ cams_

    if {[info exists cams_($id)]} {
	catch {$cams_($id) do vcc3_home} result
	return $result
    }
    # if we got here, the id is invalid, or the vcc3d wasn't up when
    #   we initialized, try to use AMX
    catch {$amx_ do camera_center $id} result
    return $result
}

405Client public camera_startMove {id dir} {
    $self instvar amx_ cams_

    if {[info exists cams_($id)]} {
	catch {$cams_($id) do vcc3_startMove $dir} result
	return $result
    }
    # if we got here, the id is invalid, or the vcc3d wasn't up when
    #   we initialized, try to use AMX
    catch {$amx_ do camera_startMove $id $dir} result
    return $result
}

# keep for backward compatibility
405Client public camera_stopMove {id} {
    $self camera_stop $id
}

405Client public camera_stop {id} {
    $self instvar amx_ cams_

    if {[info exists cams_($id)]} {
	catch {$cams_($id) do vcc3_stop} result
	return $result
    }
    # if we got here, the id is invalid, or the vcc3d wasn't up when
    #   we initialized, try to use AMX
    catch {$amx_ do camera_stopMove $id} result
    return $result
}

405Client public camera_setPreset {id num} {
    $self instvar amx_ cams_

    if {[info exists cams_($id)]} {
	catch {$cams_($id) do vcc3_setPreset $num} result
	return $result
    }
    # if we got here, the id is invalid, or the vcc3d wasn't up when
    #   we initialized, try to use AMX
    catch {$amx_ do camera_setPreset $id $num} result
    return $result
}

405Client public camera_goPreset {id num} {
    $self instvar amx_ cams_

    if {[info exists cams_($id)]} {
	catch {$cams_($id) do vcc3_goPreset $num} result
	return $result
    }
    # if we got here, the id is invalid, or the vcc3d wasn't up when
    #   we initialized, try to use AMX
    catch {$amx_ do camera_goPreset $id $num} result
    return $result
}

405Client public camera_fadeIn {id} {
    $self instvar amx_ cams_

    if {[info exists cams_($id)]} {
	catch {$cams_($id) doNoWait vcc3_fadeIn} result
	return $result
    }
    # if we got here, the id is invalid, or the vcc3d wasn't up when
    #   we initialized, do nothing
    return -1
}

405Client public camera_fadeOut {id} {
    $self instvar amx_ cams_

    if {[info exists cams_($id)]} {
	catch {$cams_($id) doNoWait vcc3_fadeOut} result
	return $result
    }
    # if we got here, the id is invalid, or the vcc3d wasn't up when
    #   we initialized, do nothing
    return -1
}

405Client public lights_set {setting} {
    $self instvar amx_

    catch {$amx_ do lights_set $setting} result
    return $result
}

405Client public callback_register {callback {filter ""}} {
    $self instvar amx_

    catch {$amx_ do callback_register $callback $filter} result
    return $result
}

405Client public callback_unregister {callback} {
    $self instvar amx_

    catch {$amx_ do callback_unregister $callback} result
    return $result
}

405Client public callback_unregisterClient {sock} {
    $self instvar amx_

    catch {$amx_ do callback_unregisterClient $sock} result
    return $result
}

405Client public callback_disable {callback} {
    $self instvar amx_

    catch {$amx_ do callback_disable $callback} result
    return $result
}

405Client public callback_enable {callback} {
    $self instvar amx_

#   catch {$amx_ do callback_enable $callback} result
# WEITSANG
    catch {$amx_ doNoWait callback_enable $callback} result
    return $result
}

405Client public amxd_getVersion {} {
    $self instvar amx_

    catch {$amx_ do amxd_getVersion} result
    return $result
}

405Client public vcr_doFunction { id func} {
    $self instvar amx_

    catch {$amx_ do vcr_doFunction $id $func} result
    return $result
}

405Client public scanConv_adjustSize {dir} {
    $self instvar amx_

    catch {$amx_ do scanConv_adjustSize $dir} result
    return $result
}


405Client public scanConv_adjustPosition {dir} {
    $self instvar amx_

    catch {$amx_ do scanConv_adjustPosition $dir} result
    return $result
}

405Client public scanConv_switchMode {mode} {
    $self instvar amx_

    catch {$amx_ do scanConv_switchMode $mode} result
    return $result
}

405Client public onAir_turnOn {} {
    $self instvar amx_

    catch {$amx_ do onAir_turnOn} result
    return $result
}

405Client public onAir_turnOff {} {
    $self instvar amx_

    catch {$amx_ do onAir_turnOff} result
    return $result
}

#405Client public {} {
#    $self instvar amx_
#}

